proc()
  local genjacobi;
begin
  
genjacobi:=
proc(notation, g0, g1)
  option escape;
  local p, fenv, fn;
begin
  fn := hold(jacobi).notation;
             
  p:=proc(u,m)
  begin
    if args(0)<>2 then
      error("expecting 2 arguments");
    end_if;
    
    if iszero(u) then
      return(g0(0));
    elif iszero(m) then
      return(g0(u));
    elif iszero(m-1) then
      return(g1(u));
    end_if;
    
    if type(u)=DOM_FLOAT or (type(u)=DOM_COMPLEX and (type(Re(u))=DOM_FLOAT or type(Im(u))=DOM_FLOAT)) or
      type(m)=DOM_FLOAT or (type(m)=DOM_COMPLEX and (type(Re(m))=DOM_FLOAT or type(Im(m))=DOM_FLOAT))  then
      return((eval(procname))::float(u,m));
    end_if;

    if not testtype(u,Type::Arithmetical) then
      if testtype(u, Type::Set) then
        if testtype(m, Type::Set) and not testtype(m, Type::Arithmetical) then
          return(Dom::ImageSet(eval(procname)(#u, #m), [#u, #m], [u, m]));
        else
          return(Dom::ImageSet(eval(procname)(#u, m), [#u], [u]));
        end_if;
      end_if;

      error("first argument must be of 'Type::Arithmetical'");
    end_if;

    if not testtype(m,Type::Arithmetical) then
      if testtype(m, Type::Set) then
        return(Dom::ImageSet(eval(procname)(u, #m), [#m], [m]));
      end_if;

      error("second argument must be of 'Type::Arithmetical'")
    end_if;

    return(procname(u,m));
  end_proc;

  fenv:=funcenv(subsop(p, 6=fn));

  fenv::type:=expr2text(fn);

  fenv::Content := stdlib::genOutFunc("Cjacobi".notation, 2):
  evalassign(jacobi.notation, fenv);
  evalassign(jacobi.notation, prog::remember(eval(jacobi.notation), () -> [property::depends(args()), DIGITS, slotAssignCounter("jacobi".notation)]));

  return(null());
end_proc:


genjacobi("SN", sin,  tanh):
genjacobi("CN", cos,  sech):
genjacobi("DN", u->1, sech):
genjacobi("CD", cos,  u->1):
genjacobi("SD", sin,  sinh):
genjacobi("ND", u->1, cosh):
genjacobi("DC", sec,  u->1):
genjacobi("NC", sec,  cosh):
genjacobi("SC", tan,  sinh):
genjacobi("NS", csc,  coth):
genjacobi("DS", csc,  csch):
genjacobi("CS", cot,  csch):
end_proc():

/* differentiation */

// Note: sn(u,m)=sin(am(u,m)), cs(u,m)=cos(am(u,m)) => diff(sn(u,m), m)=diff(am(u,m), m)*cs(u,m)

jacobiSN::diff:=proc(f)
begin
  return(
    diff(jacobiAM(op(f)), args(2..args(0))) * jacobiCN(op(f)));
end_proc:


jacobiCN::diff:=proc(f)
begin
  return(
    - diff(jacobiAM(op(f)), args(2..args(0))) * jacobiSN(op(f)));
end_proc:


// Note: dn(u,m)=f(am(u,m), m) with f(phi,m)=sqrt(1-m*sin(phi)^2) (for appropriate phi and m),
// therefore diff(dn(u,m), m)=diff(am(u,m), m)*diff(f(phi,m), phi) + diff(f(phi,m), m) @ phi=am(u,m)
jacobiDN::diff:=proc(f)
begin
  return(
    diff(jacobiAM(op(f)), args(2..args(0))) * (-op(f,2)) * jacobiCN(op(f)) * jacobiSD(op(f)) +
    diff(op(f,2), args(2..args(0))) * (-1/2) * jacobiSN(op(f)) * jacobiSD(op(f)));
end_proc:


jacobiCD::diff:=proc(f)
begin
  return(
    diff(jacobiAM(op(f)), args(2..args(0))) * jacobiSD(op(f)) * (op(f,2)*jacobiCD(op(f))^2 - 1) +
    diff(op(f,2), args(2..args(0))) * (1/2) * jacobiSD(op(f))^2 * jacobiCD(op(f)));
end_proc:


jacobiSD::diff:=proc(f)
begin
  return(
    diff(jacobiAM(op(f)), args(2..args(0))) * jacobiCD(op(f)) * (1 + op(f,2)*jacobiSD(op(f))^2) +
    diff(op(f,2), args(2..args(0))) * (1/2) * jacobiSD(op(f))^3);
end_proc:


jacobiND::diff:=proc(f)
begin
  return(
    diff(jacobiAM(op(f)), args(2..args(0))) * op(f,2) * jacobiSD(op(f)) * jacobiCD(op(f)) * jacobiND(op(f)) +
    diff(op(f,2), args(2..args(0))) * (1/2) * jacobiSD(op(f))^2 * jacobiND(op(f)));
end_proc:


jacobiDC::diff:=proc(f)
begin
  return(
    diff(jacobiAM(op(f)), args(2..args(0))) * (jacobiSC(op(f))*jacobiDC(op(f)) - op(f,2)*jacobiSD(op(f))) +
    diff(op(f,2), args(2..args(0))) * (-1/2) * jacobiSC(op(f)) * jacobiSD(op(f)));
end_proc:


jacobiNC::diff:=proc(f)
begin
  return(
    diff(jacobiAM(op(f)), args(2..args(0))) * jacobiSC(op(f)) * jacobiNC(op(f)));
end_proc:


jacobiSC::diff:=proc(f)
begin
  return(
    diff(jacobiAM(op(f)), args(2..args(0))) * jacobiNC(op(f))^2);
end_proc:


jacobiNS::diff:=proc(f)
begin
  return(
    diff(jacobiAM(op(f)), args(2..args(0))) * (-1) * jacobiCS(op(f)) * jacobiNS(op(f)));
end_proc:


jacobiDS::diff:=proc(f)
begin
  return(
    diff(jacobiAM(op(f)), args(2..args(0))) * (-jacobiCS(op(f))*jacobiDS(op(f)) - op(f,2)*jacobiCD(op(f))) +
    diff(op(f,2), args(2..args(0))) * (-1/2) * jacobiSD(op(f)));
end_proc:


jacobiCS::diff:=proc(f)
begin
  return(
    diff(jacobiAM(op(f)), args(2..args(0))) * (-1) * jacobiNS(op(f))^2);
end_proc:



/* For float-evaluation apply Gauss' transformation (Abramowitz&Stegun 16.12):

    k[0]=sqrt(m)
    k[n]=(1-sqrt(1-k[n-1]^2)) / (1+sqrt(1-k[n-1]^2))

    u[0]=u
    u[n]=u[n-1]/(1+k[n])

    sn(u[n],k[n]^2)=(1+k[n+1])*sn(u[n+1],k[n+1]^2) / (1+k[n+1]*sn(u[n+1],k[n+1]^2)^2)
    cn(u[n],k[n]^2)=cn(u[n+1],k[n+1]^2)*dn(u[n+1],k[n+1]^2) / (1+k[n+1]*sn(u[n+1],k[n+1]^2)^2)
    dn(u[n],k[n]^2)=(dn(u[n+1],k[n+1]^2)^2-1+k[n+1]) / (1+k[n+1]-dn(u[n+1],k[n+1]^2)^2)

   k[n] converges to 0 quickly (if m<>1).
*/

// FIXME: Fehlerabschtzung
    
jacobiSN::float:=proc(u,m)
local k1,sn,cs,err;
begin
  u:=float(u);
  m:=float(m);

  if (type(u)<>DOM_FLOAT and type(u)<>DOM_COMPLEX) or (type(m)<>DOM_FLOAT and type(m)<>DOM_COMPLEX) then
    return(hold(jacobiSN)(u,m));
  end_if;

  if iszero(m-1) then
    return(tanh(u));
  end_if;

  sn:=sin(u);
  cs:=cos(u);
  err:=specfunc::abs(m) * (1 + specfunc::abs(u) + specfunc::abs(sn*cs)) * (1 + specfunc::abs(cs));
  if err<float(10^(-DIGITS)) then
    return(sn);
  end_if;

  k1:=m/(1+sqrt(1-m))^2;
  sn:=jacobiSN::float(u/(1+k1), k1^2);

  return((1+k1)*sn/(1+k1*sn^2));
end_proc:


jacobiCN::sncndn:=proc(u,m)
local k1, sncndn, d, err, sn, cs;
begin
  sn:=sin(u);
  cs:=cos(u);
  err:=specfunc::abs(m) * (1 + specfunc::abs(u) + specfunc::abs(sn*cs)) * (1 + specfunc::abs(sn) + specfunc::abs(cs));
  if err<float(10^(-DIGITS)) then
    return(sn, cs, 1.0);
  end_if;

  k1:=m/(1+sqrt(1-m))^2;
  sncndn:=jacobiCN::sncndn(u/(1+k1), k1^2);
  d:=1+k1*sncndn[1]^2;

  return((1+k1)*sncndn[1]/d, sncndn[2]*sncndn[3]/d, (1-k1*sncndn[1]^2)/d);
end_proc:

jacobiCN::float:=proc(u,m)
begin
  u:=float(u);
  m:=float(m);

  if (type(u)<>DOM_FLOAT and type(u)<>DOM_COMPLEX) or (type(m)<>DOM_FLOAT and type(m)<>DOM_COMPLEX) then
    return(hold(jacobiCN)(u,m));
  end_if;

  if iszero(m-1) then
    return(sech(u));
  end_if;

  return(jacobiCN::sncndn(u,m)[2]);
end_proc:


jacobiDN::float:=proc(u,m)
local k1,sn;
begin
  u:=float(u);
  m:=float(m);

  if (type(u)<>DOM_FLOAT and type(u)<>DOM_COMPLEX) or (type(m)<>DOM_FLOAT and type(m)<>DOM_COMPLEX) then
    return(hold(jacobiDN)(u,m));
  end_if;

  if iszero(m-1) then
    return(sech(u));
  end_if;

  k1:=m/(1+sqrt(1-m))^2;
  sn:=jacobiSN::float(u/(1+k1), k1^2);

  return((1-k1*sn^2) / (1+k1*sn^2));
end_proc:


jacobiCD::float:=proc(u,m)
begin
  u:=float(u);
  m:=float(m);

  if (type(u)<>DOM_FLOAT and type(u)<>DOM_COMPLEX) or (type(m)<>DOM_FLOAT and type(m)<>DOM_COMPLEX) then
    return(hold(jacobiCD)(u,m));
  end_if;

  if iszero(m-1) then
    return(1);
  end_if;

  return(((sn,cn,dn)->cn/dn)(jacobiCN::sncndn(u,m)));
end_proc:


jacobiSD::float:=proc(u,m)
begin
  u:=float(u);
  m:=float(m);

  if (type(u)<>DOM_FLOAT and type(u)<>DOM_COMPLEX) or (type(m)<>DOM_FLOAT and type(m)<>DOM_COMPLEX) then
    return(hold(jacobiSD)(u,m));
  end_if;

  if iszero(m-1) then
    return(sinh(u));
  end_if;

  return(((sn,cn,dn)->sn/dn)(jacobiCN::sncndn(u,m)));
end_proc:


jacobiND::float:=proc(u,m)
begin
  u:=float(u);
  m:=float(m);

  if (type(u)<>DOM_FLOAT and type(u)<>DOM_COMPLEX) or (type(m)<>DOM_FLOAT and type(m)<>DOM_COMPLEX) then
    return(hold(jacobiND)(u,m));
  end_if;

  return(1.0/jacobiDN::float(u,m));
end_proc:


jacobiDC::float:=proc(u,m)
begin
  u:=float(u);
  m:=float(m);

  if (type(u)<>DOM_FLOAT and type(u)<>DOM_COMPLEX) or (type(m)<>DOM_FLOAT and type(m)<>DOM_COMPLEX) then
    return(hold(jacobiDC)(u,m));
  end_if;

  if iszero(m-1) then
    return(1);
  end_if;

  return(((sn,cn,dn)->dn/cn)(jacobiCN::sncndn(u,m)));
end_proc:


jacobiNC::float:=proc(u,m)
begin
  u:=float(u);
  m:=float(m);

  if (type(u)<>DOM_FLOAT and type(u)<>DOM_COMPLEX) or (type(m)<>DOM_FLOAT and type(m)<>DOM_COMPLEX) then
    return(hold(jacobiNC)(u,m));
  end_if;

  if iszero(m-1) then
    return(cosh(u));
  end_if;

  return(1.0/jacobiCN::sncndn(u,m)[2]);
end_proc:


jacobiSC::float:=proc(u,m)
begin
  u:=float(u);
  m:=float(m);

  if (type(u)<>DOM_FLOAT and type(u)<>DOM_COMPLEX) or (type(m)<>DOM_FLOAT and type(m)<>DOM_COMPLEX) then
    return(hold(jacobiSC)(u,m));
  end_if;

  if iszero(m-1) then
    return(sinh(u));
  end_if;

  return(((sn,cn,dn)->sn/cn)(jacobiCN::sncndn(u,m)));
end_proc:


jacobiNS::float:=proc(u,m)
begin
  u:=float(u);
  m:=float(m);

  if (type(u)<>DOM_FLOAT and type(u)<>DOM_COMPLEX) or (type(m)<>DOM_FLOAT and type(m)<>DOM_COMPLEX) then
    return(hold(jacobiNS)(u,m));
  end_if;

  return(1.0/jacobiSN::float(u,m));
end_proc:


jacobiDS::float:=proc(u,m)
begin
  u:=float(u);
  m:=float(m);

  if (type(u)<>DOM_FLOAT and type(u)<>DOM_COMPLEX) or (type(m)<>DOM_FLOAT and type(m)<>DOM_COMPLEX) then
    return(hold(jacobiDS)(u,m));
  end_if;

  if iszero(m-1) then
    return(csch(u));
  end_if;

  return(((sn,cn,dn)->dn/sn)(jacobiCN::sncndn(u,m)));
end_proc:


jacobiCS::float:=proc(u,m)
begin
  u:=float(u);
  m:=float(m);

  if (type(u)<>DOM_FLOAT and type(u)<>DOM_COMPLEX) or (type(m)<>DOM_FLOAT and type(m)<>DOM_COMPLEX) then
    return(hold(jacobiCS)(u,m));
  end_if;

  if iszero(m-1) then
    return(csch(u));
  end_if;

  return(((sn,cn,dn)->cn/sn)(jacobiCN::sncndn(u,m)));
end_proc:



