/* -----------------------------------------

meijerG -- Meijer's G function

Calls: 
  meijerG([[a1,..,a.n], [a.(n+1), .. , a.p]], [[b1,..,b.m], [b.(m+1), .. , b.q]], x) 
  meijerG([a1,..,a.n], [a.(n+1), .. , a.p], [b1,..,b.m], [b.(m+1), .. , b.q], x) 
  meijerG(m, n, [a1,..,a.n, a.(n+1), .. , a.p], [b1,..,b.m, b.(m+1), .. , b.q], x) 

Author: W. Oevel
First shipped with: 2009a
--------------------------------------------*/

alias(isInt = (x -> _lazy_or(domtype(x) = DOM_INT, 
                             _lazy_and(domtype(float(x)) = DOM_FLOAT,
                                       iszero(frac(x)))))):


meijerG:= proc(a, b, x)
local cancelIndices, p, q, i, j, r;
begin
  if args(0) = 5 then
    [a, b, x]:= meijerG::checkParameters(args());
    if a = FAIL then
       return(hold(meijerG)(op(b)));
    end_if:
  elif args(0) <> 3 then
    error("expecting 3 or 5 arguments");
  end_if;

  //===============================================
  // We now should have the internal standard form:
  //   a = [[a[1], .. , a[n]], [a[n+1], .. , a[p]], 
  //   b = [[b[1], .. , b[m]], [b[m+1], .. , b[p]]. 
  //===============================================

  //=============
  // Overloading:
  //=============
  if x::dom::meijerG <> FAIL then
     return(x::dom::meijerG(a, b, x));
  end_if:

  if nops(a[1]) = 0 and nops(a[2]) = 0 and 
     nops(b[1]) = 0 and nops(b[2]) = 0 then
     if domtype(x) = DOM_FLOAT or
       (domtype(x) = DOM_COMPLEX and domtype(op(x, 1)) = DOM_FLOAT) then
       return(float(0));
     else
       return(0);
     end_if;
  end_if;

  //===========================================
  // Here is the strategy for 'normalizing' the
  // parameter lists. Let A, B, C, D denote a 
  // sequence of parameters, let a, b, c, d denote
  // single parameters. consider calls of the
  // form G([[A], [B]], [[C], [D]], z):
  //
  // 0) sort the parameter lists A, B, C, D
  //
  // 1)  G([[a, A], [B]], [[a, C], [D]], z)
  //   = G([[A], [B]], [[C], [D]], z)
  //
  // 2)  G([[A], [b, B]], [[c, C], [D]], x) 
  //   = (-1)^(b-c)*G([[b, A], [B]], [[C], [c, D]], (-1)^(b-c)*x) 
  //   if b - c is an integer < 0.
  //
  // 3)  G([[A], [B]], [[c, C], [d, D]], x) 
  //   =(-1)^(d-c)* G([[A], [B]], [[d, C], [c, D]], (-1)^(c-d)*x) 
  //   if c - d is an integer < 0.
  //===========================================

  //===========================================
  // utility for eliminating all entries in the 
  // lists l1 and l2 that occur in both lists:
  //===========================================
  cancelIndices:= proc(l1, l2)
  local i, j, r1;
  begin
    r1:= []:
    for i in l1 do
      if (j:= contains(l2, i)) <> 0 then
        l2:= subsop(l2, j = null());
      else
        r1:= r1.[i];
      end_if;
    end_for;
    return([r1, l2]);
  end_proc:

  //=======================================================
  // simplify the parameter lists by sorting and cancelling
  //=======================================================
  [a[1], b[2]]:= cancelIndices(sort(a[1]), sort(b[2]));
  [a[2], b[1]]:= cancelIndices(sort(a[2]), sort(b[1]));

  //=======================================================
  // Apply various simplifications. This aims at the
  // 'normal form' p < q or (p = q, |x| < 1) 
  // and minimal m
  //=======================================================
  p:= nops(a[1]) + nops(a[2]):
  q:= nops(b[1]) + nops(b[2]):

  //=======================================================
  // apply the 'analytic continuation'
  //  G(m, n, p, q, a, b, x) = G(n, m, q, b, 1-b, 1-a, 1/x)
  // to map the case p > q to the case p < q
  // (and, for p = q,  map |x| > 1 to |x| < 1).
  //=======================================================
  if p > q or
     (p = q and 
      contains({DOM_FLOAT, DOM_COMPLEX}, domtype(float(x))) and 
      specfunc::abs(float(x)) > 1
      // the flip formula is valid throughout the complex 
      // plain apart from the negative real semi axis!
      and not (domtype(float(x)) = DOM_FLOAT and float(x) < 0))
     then
     if not iszero(x) then
       [a, b]:= map([b, a], map, map, x -> 1 - x): 
       return(meijerG(a, b, 1/x));
     end_if:
  end_if:

/*
  //=======================================================
  // Let A = a1,a2,..., B = b1,b2,... etc. Apply 
  // G([[A], [b, B]], [[c, C]], [D], x) = 
  // (-1)^(b-c) * G([[b, A], [B]], [[C], [c, D]], x)
  // if b - c is an integer. Stategy: minimize the
  // number m = length(list C). (If m = 0, the result is 0.)
  //=======================================================
  for j from 1 to nops(b[1]) do
    for i from 1 to nops(a[2]) do 
      r:= a[2][i] - b[1][j];
      if isInt(r) then
         a[1]:= a[1] . [a[2][i]]:
         a[2]:= subsop(a[2], i = null()):
         b[2]:= [b[1][j]] . b[2]:
         b[1]:= subsop(b[1], j = null()):
         return((-1)^r*meijerG(a, b, x));
      end_if;
    end_for;
  end_for;
*/

  //=======================================================
  // Let A = a1,a2,..., B = b1,b2,... etc. Apply 
  // G([[a, A], [b, B]], [[C]], [D], x) = 
  // (-1)^(a-b) * G([[b, A], [a, B]], [[C], [D]], x)
  // if a - b is a positive integer. Stategy: make a - b <= 0.
  // Idea: move the high indices from A to B, so that
  // cancelIndices has a chance of cancelling elements 
  // in B and C
  //=======================================================
  for i from 1 to nops(a[1]) do 
    for j from 1 to nops(a[2]) do
      r:= a[1][i] - a[2][j];
      if isInt(r) and r > 0 then
         [a[1][i], a[2][j]]:= [a[2][j], a[1][i]];
         return((-1)^r*meijerG(a, b, x));
      end_if;
    end_for;
  end_for;

  //=======================================================
  // Let A = a1,a2,..., B = b1,b2,... etc. Apply 
  // G([[A], [B]], [[c, C]], [d, D], x) = 
  // (-1)^(c-d) * G([[A], [B]], [[d, C], [c, D]], x)
  // if d - c is a positive integer. Stategy: make d - c <= 0.
  // Idea: move the small indices from C to D, so that
  // cancelIndices has a chance of cancelling elements 
  // in A and D
  //=======================================================
  for i from 1 to nops(b[1]) do 
    for j from 1 to nops(b[2]) do
      r:= b[1][i] - b[2][j];
      if isInt(r) and r < 0 then
         [b[1][i], b[2][j]]:= [b[2][j], b[1][i]];
         return((-1)^r*meijerG(a, b, x));
      end_if;
    end_for;
  end_for;

/*
  //=======================================================
  // If m = 0, there are no residues inside the contour
  // and the result is 0:
  //=======================================================
  if (p <= q and m = 0) or
     (p >= q and n = 0) then
     return(0);
  end_if:
*/

  //=======================================================
  // We require that the poles of gamma(c[i] - z) and 
  // gamma(1 - a[j] + z) are distinct (the integration
  // contour includes one type and excludes the other),
  // i.e. a[i] - c[j] must not be a positive integer.
  //=======================================================
  for i from 1 to nops(a[1]) do 
   for j from 1 to nops(b[1]) do 
      r:= a[1][i] - b[1][j];
      if isInt(r) and r > 0 then
         error("the meijerG function is not defined for the given parameters");
      end_if;
    end_for:
  end_for:

  //=======================================================
  // search for a float entry in the arguments (complex
  // numbers x + I*y are split into x, y):
  //=======================================================
  if has([map((op(map(a[1],op)),
               op(map(a[2],op)),
               op(map(b[1],op)),
               op(map(b[2],op)), op(x)), type)],DOM_FLOAT) then
     return(meijerG::float([a[1], a[2]], [b[1], b[2]], x, "go") );
  end;

  //=======================================================
  // some automatic simplifications
  //=======================================================
/*
  if a = [[], []] and b = [[0], []] then
    return(exp(-x));
  elif a = [[0, 0], []] and b = [[0], [-1]] then
    return(ln(1 + x)/x);
  end_if:
*/

  //=======================================================
  // no further automatic simplifications: symbolic return
  //=======================================================
  return(procname(a, b, x));
end_proc:

meijerG := funcenv(meijerG):
meijerG := prog::remember(meijerG, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("meijerG")]):

meijerG::type:= "meijerG":
meijerG::print:= "meijerG":
meijerG::info:=
"meijerG([a,b],[c,d],x) -- Meijer G function with parameter lists a, b, c, d and argument x":

//=====================================================================
// meijerG::checkParameters = utility producing an internal normal form 
// a = [[a1,..,a.n], [a.(n+1),..,a.p]]
// b = [[b1,..,b.m], [b.(m+1),..,b.q]]
// from the input formats
//   meijerG([[a1,..,a.n], [a.(n+1),..,a.p]],[[b1,..,b.m], [b.(m+1),..,b.q]], x)
//   meijerG( [a1,..,a.n], [a.(n+1),..,a.p] , [b1,..,b.m], [b.(m+1),..,b.q] , x)
//   meijerG(m, n, [a.1,..,a.p] , [b1,..,b.q] , x)
//=====================================================================
meijerG::checkParameters:= proc(a, b, x)
local m, n;
begin
    if domtype(a) <> DOM_LIST then
      // the call is meijerG(m, n, [a[1], .. , a[p]], [b[1], .. , b[q]], x)
      [m, n, a, b, x]:= [args(1..5)]:
      if domtype(a) <> DOM_LIST then
        error("3rd argument: expecting a list");
      end_if;
      if domtype(b) <> DOM_LIST then
        error("4th argument: expecting a list");
      end_if;
      if (not isInt(m)) or (not isInt(n)) then
        // m or n is symbolic. Can't do anything.
        return([FAIL, [m, n, a, b, x], FAIL]);
      end_if;
      if m < 0 then
        error("1st argument: expecting a list or a positive integer");
      elif m > nops(b) then
        error("1st argument: expecting an integer not exceeding ".
              "the length of the 2nd parameter list"); 
      else
        b:= [b[1..m], b[m+1 .. nops(b)]];
      end_if:
      if n < 0 then
        error("2nd argument: expecting a list or a positive integer");
      elif n > nops(a) then
        error("2nd argument: expecting an integer not exceeding ".
              "the length of the 1st parameter list"); 
      else
        a:= [a[1..n], a[n+1 .. nops(a)]];
      end_if:
    else
      // the call is meijerG([a[1],..,a[n], [a[n+1,..,a[p]], [b[1],..,b[m]], b[m+1],..,b[q]], x)
      [a, b, x]:= [[args(1), args(2)], [args(3), args(4)], args(5)]:
    end_if:
    //===============================================
    // We now should have the internal standard form:
    //   a = [[a[1], .. , a[n]], [a[n+1], .. , a[p]], 
    //   b = [[b[1], .. , b[m]], [b[m+1], .. , b[p]]. 
    //===============================================
    if type(a) <> DOM_LIST or 
     nops(a) <> 2 or
     type(a[1]) <> DOM_LIST or
     type(a[2]) <> DOM_LIST then
     error("illegal arguments");
    end_if:
    if type(b) <> DOM_LIST or 
     nops(b) <> 2 or
     type(b[1]) <> DOM_LIST or
     type(b[2]) <> DOM_LIST then
     error("illegal arguments");
    end_if:
    return([a, b, x]):
end_proc:


meijerG::Content:=
proc(Out, data)
  local m, n, p, q, z, l1, l2;
begin
  if nops(data) = 3 and testtype(op(data ,1), Type::ListOf(DOM_LIST)) and
     testtype(op(data ,2), Type::ListOf(DOM_LIST)) then
    n := nops(op(data, 1)[1]);
    p := n + nops(op(data, 1)[2]);
    m := nops(op(data, 2)[1]);
    q := m + nops(op(data, 2)[2]);
    z := op(data, 3);
    
    l1 := _concat(op(op(data, 1)));
    l2 := _concat(op(op(data, 2)));
  elif nops(data) = 5 and
       testtype(op(data, 1), DOM_LIST) and 
       testtype(op(data, 2), DOM_LIST)  and
       testtype(op(data, 3), DOM_LIST)  and
       testtype(op(data, 4), DOM_LIST) then
    n := nops(op(data, 1));
    l1 := op(data, 1).op(data, 2);
    p := nops(l1);
    m := nops(op(data, 3));
    l2 := op(data, 3).op(data, 4);
    q := nops(l2);
    z := op(data, 5);
  elif nops(data) = 5 and
       testtype(op(data, 1), DOM_INT) and 
       testtype(op(data, 2), DOM_INT)  and
       testtype(op(data, 3), DOM_LIST)  and
       testtype(op(data, 4), DOM_LIST) then
    m := op(data, 1);
    n := op(data, 2);
    l1 := op(data, 3);
    p := nops(l1);
    l2:= op(data, 4);
    q := nops(l2);
    z := op(data, 5);
  elif nops(data) = 7 and
       not testtype(op(data, 1), DOM_LIST) and 
       not testtype(op(data, 2), DOM_LIST) and 
       not testtype(op(data, 3), DOM_LIST) and 
       not testtype(op(data, 4), DOM_LIST) then
    m := op(data, 1);
    n := op(data, 2);
    p := op(data, 3);
    q := op(data, 4);
    l1:= op(data, 5);
    l2:= op(data, 6);
    z := op(data, 7);
  else
    return(Out::stdFunc(data));
  end_if;
    
  Out::Capply(Out::CmeijerG,
              Out(if nops(l1) = 0 then hold(``) else op(l1) end),
              Out(if nops(l2) = 0 then hold(``) else op(l2) end),
              Out(z), Out(m), Out(n), Out(p), Out(q)):
end:

alias(MeijerG=specfunc::MeijerG):
MeijerG:=newDomain("specfunc::MeijerG"):
MeijerG::create_dom:=hold(specfunc::MeijerG):

autoload(specfunc::MeijerG::convertTo_patterns):
autoload(specfunc::MeijerG::reduceAdamchik1996):
autoload(specfunc::MeijerG::reduceDGL):
autoload( specfunc::MeijerG::convolutionInt ): /* Integration */
stdlib::deferredAlias(meijerG::convolutionInt, specfunc::MeijerG::convolutionInt):

meijerG::float:=
    loadproc(meijerG::float, pathname("STDLIB", "FLOAT"), "meijerG"):

meijerG::diff:=
    loadproc(meijerG::diff, pathname("STDLIB", "DIFF"), "meijerG"):

// IaS = Integrals and Series, Prudnikov etc.
// meijerG([[a[1..n], a[n+1..p]], [b[1..m],b[m+1..q]], x)
// hold([m, n, p, q] = [
meijerG::convertFrom_patterns:= table(
  hold(
  [0,1,1,0] = [
    [ [#a[1]=1], exp(-1/#x) ] /* IaS v3 1990, 8.4.3.(2) */
  ],
  [0,1,1,1] = [
    [ [#a[1]-#b[1]=1], #x^#b[1]*heaviside(#x-1) ] /* IaS v3 1990, 8.4.2.(2) */
  ],
/* Do check !!!  */
  [0,1,2,1] = [
    [ [#a[2]=#b[1], #a[1]=1, not #a[2] in Z_ ], exp(1/#x)/PI*sin((#a[2])*PI) ] /* IaS v3 1990, 8.4.3.(6) */
  ],
  [1,0,0,1] = [
    [ [#b[1]=0], exp(-#x) ] /* IaS v3 1990, 8.4.3.(1) */
  ],
  [1,0,0,2] = [
    [ [#b[1]= 0 , #b[2]=1/2], cos(2*sqrt(#x))/sqrt(PI) ], /* IaS v3 1990, 8.4.5.(1) */
    [ [#b[1]=1/2, #b[2]= 0 ], sin(2*sqrt(#x))/sqrt(PI) ]  /* IaS v3 1990, 8.4.5.(2) */
  ],
  [1,0,1,1] = [
    [ [#a[1]-#b[1]=1], #x^#b[1]*heaviside(1-#x) ], /* IaS v3 1990, 8.4.2.(1) */
    [ [#a[1]>0 or not #a[1] in Z_, #b[1]=0], heaviside(1-#x)*(1-#x)^(#a[1]-1)/gamma(#b[1]) ] /* IaS v3 1990, 8.4.2.(3) */
  ],

/* Do check !!!  */
  [1,0,1,2] = [
    [ [#a[1]=#b[2], #b[1]=0, not (1-#a[1]) in Z_ ], exp(#x)/PI*sin((1-#a[1])*PI) ] /* IaS v3 1990, 8.4.3.(5) */
  ],

  [1,1,1,1] = [
    [ [#a[1]<=1 or not #a[1] in Z_, #b[1]=0 ], (1+#x)^(#a[1]-1)*gamma(1-#a[1]) ]
//  [ [#a[1]-#b[1]<>1 ], (#x)^(#b[1])*(1+#x)^(#a[1]-#b[1]-1)*gamma(1-#a[1]+#b[1]) ] /* IaS v3 1990, 8.4.2.(5) */
  ],

/* Do check !!! */
  [1,1,1,2] = [
    [ [#b[2]=0,#a[1]=#b[1],#a[1] in N_], (-1)^#a[1]*
      // (exp(-#x)-sum( (-#x)^k/k!, k=0..(#a[1]-1) ))
          exp(-#x)*(1 - igamma(#a[1],-#x)/gamma(#a[1]))
      ] /* IaS v3 1990, 8.4.3.(3) */
  ],

  [1,1,1,3] = [
    [ [#a[1]=1, #b[1]=1/2, #b[2]=0, #b[3]=0 ], Si(2*sqrt(#x))*2/sqrt(PI) ]
//  [ [], gamma(1-#a[1]+#b[1])*#x^#b[1]/gamma(1+#b[1]-#b[2])/gamma(1+#b[1]-#b[3])*
//       hypergeom([1+#b[1]-#a[1]],[1+#b[1]-#b[2],1+#b[1]-#b[3]],-#x)]
  ],

/* Do check !!! */
  [1,1,2,1] = [
    [ [#a[2]=1,#a[1]=#b[1],#a[1]<=0,#a[1] in Z_], (-1)^(#a[1]+1)*
    // (exp(-1/#x)-sum( (-#x)^(-k)/k!, k=0..(-#a[1]) ))
        exp(-1/#x)*(1 - igamma(-#a[1]+1,-1/#x)/gamma(-#a[1]+1))
    ], /* IaS v3 1990, 8.4.3.(4) */
    [ [#a[2]=#b[1]+1, #b[1]-#a[1] in N_ ], 
       -(1/#x)^(-#b[1])* (igamma(#b[1] - #a[1] + 1, 1/#x) - gamma(#b[1] - #a[1] + 1))
    ]
  ],
  [1,1,2,2] = [
    [ [#a[1]=0, #a[2]=1/2, #b[1]=0, #b[2]=1/2, #x<>1], 1/(1-#x)/PI ], /* IaS v3 1990, 8.4.2.(6) */
    [ [#a[1]<>1, #a[1]/2=#a[2], #a[2]=#b[2], #b[1]=0,not #a[1]/2 in Z_ ], 
         abs(1-#x)^(#a[1]-1)/PI*gamma(1-#a[1])*cos((1-#a[1])*PI/2) ] /* IaS v3 1990, 8.4.2.(7) */
  ],

/* Do check !!!  */
  [1,1,2,3] = [
    [ [#a[1]=#b[1], #a[1] in Z_, #a[1]>=1, #a[2]=#b[2], #b[3]=0, not (1-#a[2]) in Z_], 
     //  (exp(#x)-sum(#x^k/k!,k=0..(#a[1]-1)) )
          exp(#x)*(1 - igamma(#a[1],#x)/gamma(#a[1]))
	 *sin((1-#a[2])*PI) / PI *(-1)^(#a[1]) ] /* IaS v3 1990, 8.4.3.(7) */
  ],

  [1,2,2,2] = [
    [ [#a[1]=1/2, #a[2]=1, #b[1]=1/2, #b[2]=0 ], 2*arccos( 1/sqrt(1+#x) ) ]
//  [ [not -(1-#a[1]+#b[1]) in Z_ intersect Dom::Interval( [0],infinity ), 
//     not -(1-#a[2]+#b[1]) in Z_ intersect Dom::Interval( [0],infinity ), 
//     not -(#b[1]-#b[2]+1) in Z_ intersect Dom::Interval( [0],infinity )], 
//      #x^#b[1]*gamma(1-#a[1]+#b[1])*gamma(1-#a[2]+#b[1])*
//       hypergeom( [1-#a[1]+#b[1],1-#a[2]-#b[1]], [#b[1]-#b[2]+1], -#x )/gamma(#b[1]-#b[2]+1) ] /* Wolfram */
  ],
  [2,0,1,3] = [
    [ [#b[2]=#a[1]+1, #b[1] = 1/2, #b[3]=0], 
       #a[1]/sqrt(PI)*( sin( 2*sqrt(#x) ) ) - sqrt(#x)/sqrt(PI)*cos( 2*sqrt(#x) ) ]
  ]
  )
):

meijerG::convertTo:= proc( xpr, var )
  local F;
begin
  if type(xpr)="meijerG" then return(xpr); end_if;
  F:= matchlib::einwohner( xpr, var, op(specfunc::MeijerG::convertTo_patterns) );
  if F=FAIL and type(xpr)="_mult" then return( map(xpr,meijerG::convertTo,var) ); end_if;
  if F<>[] and args(0)=3 and args(3)=All then return( F ); end_if;
  F:= matchlib::addpatterns::handle_results( F );
  if F=FAIL then
    F:= matchlib::einwohner( subs(xpr,var=1/var), var, op(specfunc::MeijerG::convertTo_patterns) );
    if args(0)=3 and args(3)=All then return( F ); end_if;
    F:= matchlib::addpatterns::handle_results( F );
    F:= subs( F, var=1/var );
    userinfo( 10, "Cannot find meijerG-representation for ".expr2text(xpr) );
  else
    userinfo( 10, expr2text(xpr)." rewritten to ".expr2text(F) );
  end_if;
  F;
end_proc:

meijerG::checkPatterns:= proc( patterns, as, bs, z )
  local sigma;
begin
  sigma:= genident();
  as:= map( as, X->X-sigma );
  bs:= map( bs, X->X-sigma );
//   patterns:= subs( patterns, a=as, b=bs, x=z );
  patterns:= map( patterns, X->subsop(X,1=subs( op(X,1), [#a=as, #b=bs, #x=z] )) );
  patterns:= map( patterns, X->X.[simplify::simplifyCondition( _and(op(eval(X[1]))) )] );
  patterns:= select( patterns, X->X[3]=TRUE or 
                             type(X[3])="_equal" and 
			     op(X,[3,1])=sigma or 
			     is(subs(X[3],sigma=0),Goal=TRUE ) );
  map( patterns, proc(X)
    local pre, r;
    begin
      X[2]:= subs( X[2], #a=as, #b=bs, #x=z );
      if type(X[3])="_equal" and op(X,[3,1])=sigma then
        X[2]:= subs( X[2], op(X,[3]) );
        pre:= z^(op(X,[3,2]));
      else
        X[2]:= subs( X[2], sigma=0 );
        pre:= 1;
      end;
      if traperror( ( r:= eval(X[2]*pre) ) )=0 then
        r;
      else
        null();
      end_if;
    end_proc );
end_proc:

// set to FAIL to avoid problem with logarithmic discontinuity at zero
meijerG::series:= X -> FAIL:

meijerG::simplify:= X -> meijerG::Simplify(X):

meijerG::Simplify:= proc(G)
  local m, n, p, q, a, b, x, z, patterns, F;
begin
  if nops(G) = 5 then 
    [a, b, x]:= meijerG::checkParameters(op(G));
    if a = FAIL then
       return(hold(meijerG)(Simplify(a) $ a in [op(b)]));
    end_if:
    G:= meijerG(a, b, x):
  elif nops(G) <> 3 then
    error("expecting 3 or 5 arguments in meijerG");
  end_if;
  G:= specfunc::MeijerG::reduceAdamchik1996(G);
  if type(G)<>"meijerG" then return(G); end_if;
  m:= nops(op(G,[2,1]));
  n:= nops(op(G,[1,1]));
  p:= nops(op(G,[1,1])) + nops(op(G,[1,2]));
  q:= nops(op(G,[2,1])) + nops(op(G,[2,2]));

  a:= op(G,[1,1]).op(G,[1,2]);
  b:= op(G,[2,1]).op(G,[2,2]);
  z:= op(G,3);

  if contains( meijerG::convertFrom_patterns, [m,n,p,q] ) then
    patterns:= meijerG::convertFrom_patterns[ [m,n,p,q] ];
    F:= meijerG::checkPatterns( patterns, a, b, z );
    if F<>FAIL and nops(F)>0 then
      return( op(F,1) );
    end_if;
  end_if;

  /* Transform G-Function */
  if contains( meijerG::convertFrom_patterns, [n,m,q,p] ) then
    patterns:= meijerG::convertFrom_patterns[ [n,m,q,p] ];
    F:= meijerG::checkPatterns( patterns, map(b,X->1-X), map(a,X->1-X), 1/z );
    if F<>FAIL and nops(F)>0 then
      return( op(F,1) );
    end_if;
  end_if;

  G:= specfunc::MeijerG::reduceDGL(G);

  if type(G)="meijerG" then
    G:= meijerG::toHypergeom( G );
  end_if;
  G;
end_proc:

meijerG::toHypergeom:= proc( G )
  local m, n, p, q, a, b, x, z, res, h, j, tmp;
begin
  if nops(G) = 5 then 
    [a, b, x]:= meijerG::checkParameters(op(G));
    if a = FAIL then
       return(hold(meijerG)(op(b)));
    end_if:
    G:= meijerG(a, b, x):
  elif nops(G) <> 3 then
    error("expecting 3 or 5 arguments in meijerG");
  end_if;
  G:= specfunc::MeijerG::reduceAdamchik1996(G);
  if type(G)<>"meijerG" then return(G); end_if;
  m:= nops(op(G,[2,1]));
  n:= nops(op(G,[1,1]));
  p:= nops(op(G,[1,1])) + nops(op(G,[1,2]));
  q:= nops(op(G,[2,1])) + nops(op(G,[2,2]));

  a:= op(G,[1,1]).op(G,[1,2]);
  b:= op(G,[2,1]).op(G,[2,2]);
  z:= op(G,3);

  if p<q then
    [n, m, q, p]:= [m, n, p, q];
    [a, b]:= [map( b, X->1-X), map( a, X->1-X )];
    z:= 1/z;
  end_if;
  if p=q then
    if simplify::simplifyCondition( abs(z) < 1 ) = TRUE then 
      [n, m, q, p]:= [m, n, p, q];
      [a, b]:= [map( b, X->1-X), map( a, X->1-X )];
      z:= 1/z;
    end_if;
  end_if;
  res:= 0;
  for h from 1 to n do
    tmp:= (1/z)^(1-a[h]);
    for j from 1 to m do
       tmp:= tmp*gamma(b[j]-a[h]+1):
    end_for:
    for j from m+1 to q do
       if is(a[h]-b[j]<=0 and a[h]-b[j] in Z_, Goal=TRUE ) then 
         tmp:= 0:
         break;
       else
         tmp:= tmp/gamma(a[h] - b[j]):
       end_if:
    end_for:
    if iszero(tmp) then
       next; // proceed to the next value of h
    end_if;
    for j from 1 to n do
      if j<>h then
        if is(a[h]-a[j]<=0 and a[h]-a[j] in Z_, Goal=TRUE ) then return( G ); end_if;
        tmp:= tmp * gamma( a[h]-a[j] );
      end_if;
    end_for;
    for j from n+1 to p do
       if is(1+a[j]-a[h]<=0 and a[h]-a[j] in Z_, Goal=TRUE ) then 
         tmp:= 0:
         break;
       else
         tmp:= tmp/gamma(1+a[j]-a[h]):
       end_if;
    end_for:
    if iszero(tmp) then
       next; // proceed to the next value of h
    end_if;
    if traperror((tmp:= tmp * 
                        hypergeom(map(b,X->X+1-a[h]), 
                                  map(subsop(a,h=null()), X->1+X-a[h]), 
                                  (-1)^(q-m-n)/z)))<>0 then
      return(G);
    end_if;
    res:= res + tmp;
  end_for;
  res;
end_proc:

unalias(isInt):

// setuserinfo(meijerG,20):
