/*=====================================================================*/
/*                    function polylog                                 */
/*=====================================================================*/
/*
 *  References:
 *
 *     [1] Leonard Lewin: Polylogarithms and related functions.
 *         North Holland (1981).
 *
 *     Other smarter references (provided by Gerhard Niklasch
 *     <nikl@mathematik.tu-muenchen.de> (april 1998)):
 *
 *     [2] H. Cohen, L. Lewin, D. Zagier: article pp. 25-34 of 
 *         Experimental Mathematics vol.1 (1992).
 *
 *     [3] Leonard Lewin (ed.): Structural properties of
 *         Polylogarithms. AMS. (Mathematical Surveys and Monographs
 *         vol.37) (1991).
 *
 *     See also the Reference Manual [4] 
 *     (and the source code in C programming language) of PARI-GP:
 *         http://pari.home.ml.org
 *     or
 *         ftp://megrez.math.u-bordeaux.fr/pub/pari/    
 */


polylog :=
proc(n,x)
begin
   if args(0)<>2 then
     error("Wrong number of arguments")
   elif x::dom::polylog<>FAIL then
     return(x::dom::polylog(args()))
   end_if;

   case type(x)
    of DOM_SET do
    of "_union" do
      return(map(x, u-> polylog(n,u)))
  end_case;

  if not testtype(x, Type::Arithmetical) then
    if testtype(x, Type::Set) then
      if testtype(n, Type::Set) and not testtype(n, Type::Arithmetical) then
        return(Dom::ImageSet(eval(procname)(#n, #x), [#n, #x], [n, x]));
      else
        return(Dom::ImageSet(eval(procname)(n, #x), [#x], [x]));
      end_if;
    end_if;

    error("first argument must be of 'Type::Arithmetical'");
  end_if;

  if not testtype(n, Type::Arithmetical) then
    if testtype(n, Type::Set) then
      return(Dom::ImageSet(eval(procname)(#n, x), [#n], [n]));
    end_if;

    error("second argument must be of 'Type::Arithmetical'")
  end_if;

   if (type(n)= DOM_INT)then

      if ( n<=1 and  iszero(x-1) ) then error("singularity"): end_if:

      case domtype(x)
         of DOM_FLOAT do
            return(polylog::float(n,x)):
         of DOM_COMPLEX do
            if type(op(x,1)) = DOM_FLOAT or type(op(x,2)) = DOM_FLOAT then
                return(polylog::float(n,x)):
            end_if:
      end_case;

      if (n=1) then return(-ln(1-x)): end_if:
      if (n=0) then return(x/(1-x)): end_if:
      if (n<0) then return(specfunc::polylog_neg(n,x)): end_if:
      if (iszero(x)) then return(0): end_if:
      if (iszero(x-1)) then return(zeta(n)): end_if:
      // the exception polylog(1, -1)=-ln(2) was returned above
      if (iszero(x+1)) then return(-(1-2^(1-n))*zeta(n)): end_if;
      if ((iszero(x-1/2))and(iszero(n-3))) then   
        return(7*zeta(3)/8-PI^2*ln(2)/12+ln(2)^3/6):
      end_if:
      if n=2 then
         // op(dilog,[1,5]) is the remember table of dilog.
         // It contains all special values implemented for
         // dilog(1-x) = polylog(2,x):
         if contains(op(dilog,[1,5]), 1-x) then
            return(dilog(1-x));
         end_if;
      end_if:
   else
      // properties that hold for arbitrary n:
      if (iszero(x)) then return(0): end_if:
      if (iszero(x-1)) then return(zeta(n)): end_if:
      if (iszero(x+1)) and is(n<>1)=TRUE then
         return(-(1-2^(1-n))*zeta(n)): end_if:

      if testtype(n,Type::Numeric) then 
        error("first argument must be an integer")
      end_if;
   end_if:

   // properties that hold for arbitrary n:
   if (iszero(x)) then return(0): end_if:
   if (iszero(x-1)) then return(zeta(n)): end_if:
   if (iszero(x+1)) and is(n<>1)=TRUE then
       return(-(1-2^(1-n))*zeta(n)): end_if:

   procname(n,x)
end_proc:

polylog := prog::remember(polylog, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("polylog")]):

polylog:= funcenv(polylog):
polylog::print:= "polylog":
polylog::info:= "polylog -- the polylogarithm function.":
polylog::type:= "polylog":

polylog::diff:=
  proc()
    local op11, op12, otherarg;
  begin
    otherarg := args(2..args(0));
    op11 := op(args(1),1);
    op12 := op(args(1),2);
    if has(op11, otherarg) then
      hold(diff)(polylog(op11,op12), otherarg)
    else
      (polylog(op11 -1,op12)/op12)*diff(op12,otherarg)
    end_if;
  end_proc:

polylog::series:= loadproc(polylog::series, pathname("SERIES"), "polylog"):

polylog::rectform:= loadproc(polylog::rectform, pathname("STDLIB", "RECTFORM"),
                             "polylog"):

/*
TO DO ??
rewrite combine
*/

/************************************************************************/
specfunc::numer_polylog_neg:=proc(n,x)
/************************************************************************/
/*
 * Computes the numerator of polylog(n,x) when n<0. 
 * Not user-callable.
 * Must be domain element of specfunc to make it protected 
 */
local i,a0,a1,_var;

begin
  _var:=genident();
  a0:=_var;
  for i from 1 to -n do
    a1:=_var*(1-_var)*diff(a0,_var)+i*_var*a0;
    a0:=normal(a1);
  end_for;
  return((subs(a0,_var=x)));
end_proc:


/************************************************************************/
specfunc::polylog_neg:=proc(n,x)
/************************************************************************/
/*
 * Computes polylog(n,x) when n<0. (polylog(n,x) is a rational fraction).
 * Not user-callable. 
 */

begin
  specfunc::numer_polylog_neg(n,x)/(1-x)^(-n+1);
end_proc:

/************************************************************************/
specfunc::fpolylog_sz:=proc(p,z) 
/************************************************************************/
/* 
 * The name of the function is specfunc::fpolylog_sz for float(polylog)(small |z|). 
 * Not user-callable. 
 */

local a,v,lnv,ln2,ln10,n,fi,fz,old_DIGITS,s,i,j,cst1,np;

begin
   /* 
    * We compute an approximation of polylog(p,z) by
    * truncating the power series to n terms.
    * Then the absolute value of the remainder of the series
    * is  <= |z|^(n+1)/(n+1)^p/(1-|z|)
    * The condition is choosen to be:
    * |z|^(n+1)/(n+1)^p/(1-|z|) <= 10^(-DIGITS) /2
    * or:
    * (n+1) * ln(|z|)-p*ln(n+1) <= -DIGITS * ln(10)+ ln(1-|z|) -ln(2)
    * or: 
    * (n+1) * (-ln(|z|))+p*ln(n+1) >= DIGITS * ln(10)+ ln(1-|z|) +ln(2)
    */
   if(p=1) then
      return(float(-ln(1-z)));
   end_if;
   old_DIGITS:=DIGITS;
   DIGITS:=10; // 10 should be enough for just computing n 
   v:=float(abs(z));
   lnv:=ln(v);
   n:=10; // at least 10 terms in the series 
   ln2:=ln(2.0);
   ln10:=ln(10.0);
   cst1:=old_DIGITS*ln10-ln(1-v)+ln2;
   /*
    * Rather surprinsingly, searching n is quite time-consumming,
    * even with DIGITS:=10.
    * The idea is here is to accelerate the search by doing
    * big jumps. Several thousands of terms should be very rare.
    */
   repeat
      n:=n+1000
   until
      (n+1.0)*(-lnv)+p*ln(n+1.0) > cst1
   end_repeat;
   n:=n-1000;
   repeat
      n:=n+100
   until
      (n+1.0)*(-lnv)+p*ln(n+1.0) > cst1
   end_repeat;
   n:=n-100;
   repeat
      n:=n+10
   until
      (n+1.0)*(-lnv)+p*ln(n+1.0) > cst1
   end_repeat;
   n:=n-10;
   repeat
      n:=n+1
   until
      (n+1.0)*(-lnv)+p*ln(n+1.0) > cst1
   end_repeat;
   /*
    *   We compute with more precision (i.e. new_DIGITS)!
    *   It is expected that on every loop
    *   a upperbound of the error is 10^(-new_DIGITS+1).
    *   The condition is then:
    *   n * 10^(- new_DIGITS+1) < 10^(-DIGITS) / 2
    *   or
    *   ln(n)+(-new_DIGITS+1)*ln(10.0)<(-DIGITS)*ln(10.0)-ln(2)
    *   (new_DIGITS-1)*ln(10.0)>(DIGITS)*ln(10.0)+ln(n)+ln(2)
    *   new_DIGITS-1> DIGITS + (ln(n)+ln(2))/ln(10.0) 
    */
   a:=round((ln(n)+ln2)/ln10);
   DIGITS:=old_DIGITS+a+3; // 3 is safest :^) 
   /*
    *  We Compute now the (finite) sum of terms of the series
    *  for polylog(p,z):
    */
   fz:=float(z);
   fi:=float(n);
   np:=_mult(fi $ j=1..p);
   s:=1.0/np;
   for i from n-1 downto 1 do
      fi:=float(i);
      np:=_mult(fi $ j=1..p);
      s:=s*fz+1.0/np;
   end_for;
   s:=s*fz;
   DIGITS:=old_DIGITS;
   s;
end_proc:


/************************************************************************/
specfunc::fpolylog_bz:=proc(p,z) 
/************************************************************************/
/* 
 * The name of the function is specfunc::fpolylog_bz for float(polylog)(big |z|). 
 * Not user-callable.
 */

local k,old_DIGITS,s,lnz,lnz2,lnzp,two_power_k;

begin
   /*
    * If |z| is big, we use the formula:
    *  polylog(p,z)+(-1)^{p} polylog(p,1/z)=
    * -1/p! ln(-z)^p 
    * +2 \sum_{j=1}^{[p/2]} {ln(-z)^{p-2j}\over (p-2j)!} polylog(2j,-1)
    * [1] formula 7.20 p.192)
    * or, since 
    * polylog(2j,-1)=-(1-2^{1-2j})polylog(2j,1)= -(1-2^{1-2j}) \zeta(2j),
    *
    * polylog(p,z)=(-1)^{p-1} polylog(p,1/z) -1/p! ln(-z)^p 
    * + 2 \sum_{j=1}^{[p/2]} {ln(-z)^{p-2j}\over (p-2j)!} (2^{1-2j}-1) \zeta(2j)
    *
    * j from 1 to [p/2]=p/2  if p is even 
    * 2j from 2 to p, 2k= p-2j from p-2 to 0, j=p/2 -k
    * thus k from (p-1)/2 to 0
    *
    * (2k+1=p-2j)  j from 1 to [p/2]=(p-1)/2 if p is odd
    * 2j from 2 to p-1, 2k+1=p-2j from p-2 to 1, j= (p-1)/2 -k
    * thus k from (p-3)/2 to 0
    *
    * if p is even: j=p/2 -k
    * polylog(p,z)=(-1)^{p-1} polylog(p,1/z) -1/p! ln(-z)^p 
    * + 2 \sum_{k=(p)/2}^0 {ln(-z)^{2k}\over (2k)!} 
    *                                    (2^{1-p+2k}-1) \zeta(p-2k)
    *
    * if p is odd: j= (p-1)/2 -k
    * polylog(p,z)=(-1)^{p-1} polylog(p,1/z) -1/p! ln(-z)^p 
    * + 2 \sum_{k=(p-1)/2}^0{ln(-z)^{2k+1}\over (2k+1)!}
    *                                      (2^{1-p +2k+1}-1)\zeta(p-1-2k)
    *
    * (\zeta is a builtin function in MuPAD, coded in C programming
    * language.)
    */
   old_DIGITS:=DIGITS;
   DIGITS:=DIGITS+p+5;
/* TO DO: a better choice for DIGITS
*/ 
   lnz:=float(ln(-z));
   lnz2:=lnz*lnz;
   lnzp:=_mult(lnz $k=1..p);
   if (p mod 2 = 0) then
      /* 
       * if p is even:
       * polylog(p,z)=(-1)^{p-1} polylog(p,1/z) -1/p! ln(-z)^p 
       * + 2 \sum_{k=(p-2)/2}^0 {ln(-z)^{2k}\over (2k)!} 
       *                                    (2^{1-p+2k}-1) \zeta(p-2k)
       */
      two_power_k:=1/2;
      // since the first value of the exponent is: 1- p +(p-2) = -1 
      s:=(two_power_k - 1.0)*float(zeta(2))/float(fact(p-2));
      if  (p>2) then
         for k from ((p-2)/2)-1 downto 0 do
            two_power_k:=two_power_k / 4;       
            s:=s*lnz2+(two_power_k - 1.0)*float(zeta(p-2*k))/float(fact(2*k));  
         end_for;    
      end_if;
      s:=2.0*s -(lnzp/float(fact(p))) - specfunc::fpolylog_sz(p,1/z);
   else
      /*
       * if p is odd:
       * polylog(p,z)=(-1)^{p-1} polylog(p,1/z) -1/p! ln(-z)^p
       * + 2 \sum_{k=(p-1)/2}^0{ln(-z)^{2k+1}\over (2k+1)!}
       *                                      (2^{1-p+2k+1}-1)\zeta(p-1-2k)
       */
      two_power_k:=1/2;
      // since the first value of the exponent is: 1-p +p-3+1 = -1 
      s:=(two_power_k - 1.0)*float(zeta(2))/float(fact(p-2));
      if( p>3) then
         for k from ((p-3)/2)-1 downto 0 do
            two_power_k:=two_power_k / 4;       
            s:=s*lnz2+(two_power_k - 1.0)*float(zeta(p-1-2*k))/float(fact(2*k+1));  
         end_for;  
      end_if;  
      s:=2.0*s*lnz;
      s:=s-lnzp/float(fact(p)) + specfunc::fpolylog_sz(p,1/z);
   end_if;
   DIGITS:=old_DIGITS;
   s;
end_proc:

/************************************************************************/
specfunc::fpolylog_slnz:=proc(p,z) 
/************************************************************************/
/* 
 * The name of the function is specfunc::fpolylog_slnz for 
 * float(polylog)(small |ln(z)|). 
 * Not user-callable.
 */

local j,old_DIGITS,a,lnz,lnz2,w,n,cst1,s1,s2,s3,s;

begin
   /*
    * If |z| is near 1, we use the following wonderful formula from
    * [3] (formula at the top of the page 387):
    *   polylog(p,z)= 
    *     \sum_{j=0}^{\infty} 1/j! (ln(z))^j zeta1 (p-j) ]
    * where zeta1  (k) is the value on the integer k of the
    * Riemann's \zeta function if k different of 1 and
    * where zeta1 (1) = 1+1/2+1/3+... 1/(p-1) -ln(-ln(z))
    * (H. Cohen and D. Zagier)
    * This formula is rewritten 
    * polylog(p,z)= s1 +s2 +s3
    *   s1=  \sum_{j=0}^{p-2} 1/j! (ln(z))^j \zeta (p-j) ]
    *   s2= (1+1/2+1/3+... 1/(p-1) -ln(-ln(z))* ln(z)^(p-1)/(p-1)!
    *   s3=  \sum_{j=p}^{\infty}} 1/j! (ln(z))^j \zeta (p-j) ]
    *     =  \sum_{k=0}^{\infty}} 1/(k+p)! (ln(z))^(k+p) \zeta (-k) ]
    *     =  1/p! (ln(z))^(p) (-1/2) +
    *        \sum_{j=1}^{\infty}} 1/(2j-1+p)! (ln(z))^(2j-1+p) \zeta(-2j+1) ]
    *     =  1/p! (ln(z))^(p) (-1/2) +
    *        \sum_{j=1}^{\infty}} 1/(2j-1+p)! (ln(z))^(2j-1+p) (-1)^j B_j /2j ]
    *  where B_j is the j-th Bernoulli number.
    *  (Whittaker and Watson. Modern Analysis, 
    *  Cambridge University Press 4th ed. 1927 (13.14 p. 268))
    * Thus s3 is also:
    *  s3=1/p! (ln(z))^(p) (-1/2) +
    *    \sum_{j=1}^{\infty}} 1/(2j-1+p)! (ln(z))^(2j-1+p) (-1)^j 
    * {2 (2j)! \over 2j (2\pi )^{2j}}  \zeta (2j) ]
    * Thus the radius of convergence in |ln(z)| is  2\pi
    */

   /*
    * Computing s3.
    */ 

   /*
    *  For an estimate of the remainder we use the formula:
    *     \sum_{j=1}^{\infty}} 1/(2j-1+p)! (ln(z))^(2j-1+p) (-1)^j 
    * {2 (2j)! \over 2j (2\pi )^{2j}}  \zeta (2j) ]
    * thus the absolute value of the remainder 
    * \sum_{j=n+1}^{\infty}} (idem) is smaller than
    * {1.1 |ln(z)|^{p-1} \over n+1} {w^{n+1}\over 1-w}
    * where w= {|ln(z)|^2 \over (2 \pi)^2}
    * We want the remainder smaller than 10^{-DIGITS}/2.
    */ 

   /*
    * Rather surprinsingly, searching n is quite time-consumming,
    * even with DIGITS:=10.
    * The idea is here is to accelerate the search by doing
    * big jumps. Several thousands of terms should be very rare.
    */
   old_DIGITS:=DIGITS;
   DIGITS:=10;
   lnz:=float(ln(z));
   lnz2:=lnz*lnz; 
   w:=float(abs(lnz2)/(2*PI)^2);
   cst1:=float(0.5*10^(-old_DIGITS)/1.1)*abs(lnz)^(1-p)*(1.0-w);
   n:=10;   /* at least 10 terms */
   repeat
      n:=n+1000;
   until
      w^(n+1)/(n+1.0) < cst1
   end_repeat;
   n:=n-1000;
   repeat
      n:=n+100;
   until
      w^(n+1)/(n+1.0) < cst1
   end_repeat;
   n:=n-100;
   repeat
      n:=n+10;
   until
      w^(n+1)/(n+1.0) < cst1
   end_repeat;
   n:=n-10;
   repeat
      n:=n+1;
   until
      w^(n+1)/(n+1.0) < cst1
   end_repeat;
   /*
    *   We compute with more precision (i.e. new_DIGITS)!
    *   It is expected that on every loop
    *   a bound of the error is 10^(-new_DIGITS+1).
    *   The condition is then:
    *   n*10^(- new_DIGITS+1 ) < 10^(-DIGITS)/2
    *   or
    *   ln(n)+(-new_DIGITS+1)*ln(10.0)<(-DIGITS)*ln(10.0)-ln(2)
    *   (new_DIGITS-1)*ln(10.0)>(DIGITS)*ln(10.0)+ln(n)+ln(2)
    *   new_DIGITS-1> DIGITS + (ln(n)+ln(2))/ln(10.0) 
    */
   a:=round((ln(n)+ln(2.0))/ln(10.0));
   DIGITS:=old_DIGITS+a+3; // 3 is safest :^) 
   /*
    *  Computing the (finite) sum of terms of the series in ln(z)
    *  for s3:
    */
   lnz:=float(ln(z));
   lnz2:=lnz*lnz;
   s3:=float(zeta(-2*n+1)/(2*n+p-1)!);
   for j from n-1 downto 1 do
      s3:=s3*lnz2+float(zeta(-2*j+1)/(2*j+p-1)!);
   end_for;
   s3:=s3*(lnz)^(p+1);
   s3:=s3-0.5*lnz^p/float(p!);
   /*
    * Computing s1.
    */ 
   s1:=float(zeta(2)/(p-2)!);
   if (p>2) then
      for j from 3 to p  do
         s1:=s1 * lnz + float(zeta(j)/(p-j)!);
      end_for;
   end_if;
   /*
    * Computing s2.
    */ 
   s2:=-float(ln(-lnz));
   for j from 1 to p-1 do
      s2:=s2+float(1/j);
   end_for;
   s2:=s2*lnz^(p-1)/float((p-1)!);
   s:=s1+s2+s3;
   DIGITS:=old_DIGITS;
   s;
end_proc:



/************************************************************************/
polylog::float:= proc(p,z) 
/************************************************************************/
  local r1,fz,s,absfz,old_DIGITS,_variable,z2;
begin
  /*
   * Fist we compute some particular cases.
   */

   if (type(p)<> DOM_INT)then
       if testtype(float(p),Type::Numeric)
       then error("First argument must be an integer."):
       else return(hold(polylog)(p,float(z)))
       end_if
   end_if:

  if ( p<=1 and  iszero(z-1) ) then error("singularity"): end_if:

  fz:= float(z);                 
  if has(fz, {RD_INF, RD_NINF}) then
     if iszero(p) then
        return(float(-1))
     elif p < 0 then
        return(float(0))
     else 
        return(RD_NAN)
     end_if;
  end_if;

  if z = RD_NAN then
     return(RD_NAN)
  end_if;

  if (p <= 0) then
    _variable:=genident();
    fz:=polylog(p,_variable):
    old_DIGITS:=DIGITS:
    DIGITS:=DIGITS - p + 2;
    fz:=float(subs(fz,_variable=z)):
    DIGITS:=old_DIGITS:
    return(fz);
  end_if:

  if iszero(p-1) then 
     fz:= float(z):
     if domtype(fz) = DOM_FLOAT or 
        domtype(fz) = DOM_COMPLEX then
        // Numerical stabilization around the root z = 0:
        // The relative error is fz^5/5/fz = fz^4/4 < 0.25*10^(-DIGITS)
        if specfunc::abs(fz) < 10.0^(-DIGITS/4) then
           return(fz + fz^2/2 + fz^3/3 + fz^4/4)
        end_if;
     end_if;
     return(float(-ln(1 - z))); 
  end_if;

  if (iszero(z)) then return(float(0));
  elif  (iszero(z-1)) then return(float(zeta(p)));
  elif  (iszero(z+1)) then return((2^(1-p)-1)*float(zeta(p)));
  end_if;

  /*
   *  We now choose the function to use:
   */

  if not contains({DOM_FLOAT,DOM_COMPLEX},type(fz))
     then return( hold(polylog)(p,fz) );
  end_if;
  absfz:=abs(fz);
  r1:=0.7;
/* TO DO: a better choice for r1?
*/
  if  (absfz<r1) then
  /*
   *     If |z| is small 
   *     we use the power series in z
   */
     return(specfunc::fpolylog_sz(p,z));
  elif  (absfz>1/r1) then
   /* 
    *  If |z| is big 
    *  we use the functional relation:
    */
     return(specfunc::fpolylog_bz(p,z)); 
  elif (abs(float(z+1.0))<0.5)then
   /*
    * If z is near -1 (i.e. |ln(z)| is big) we use
    * z*z and -z which are near 1.
    * (formula 7.42 p.197 of [1])
    */
     old_DIGITS:=DIGITS;
     DIGITS:=DIGITS+1;

     /* dirty: PARI uses Karatsuba for multiplying complex numbers,
        not  (a+I*b)*(A+I*B)= (a*A-b*B) + I*(a*B+b*A). For this
        reason (1.0+I*1.0/10^100)^2 yields 1.0, not 1.0 + I*2.0/10^100,
        because the imaginary part of the result from Karatsuba suffers
        from severe cancellation.
        Consequently, for z = 1.0 + I*tiny the square z^2 may become
        a real number and thus may end up on the wrong side of
        the branch cut of polylog. We need z^2 = 1.0 + 2*I*tiny to
        get the correct sign of the imaginary part. For this reason,
        do the square z2=fz^2 properly:
     */
     if domtype(fz) = DOM_COMPLEX 
        then z2:= (op(fz,1)^2 -op(fz,2)^2) +2*I*op(fz,1)*op(fz,2);
        else z2:= z^2;
     end_if;
     s:= polylog::float(p,z2)/float(2^(p-1)) - polylog::float(p,-z);
     DIGITS:=old_DIGITS;
     return(s);
  else
  /*
   *     If |ln(z)| is small we use specfunc::fpolylog_slnz(p,z) 
   */
     return(specfunc::fpolylog_slnz(p,z));
  end_if;
end_proc:

polylog::Content:= stdlib::genOutFunc("Cpolylog", 2):

polylog::TeX := (p, data, prio) -> "\\mathop{\\mathrm{Li}}\\nolimits_{".
				  generate::tex(op(data, 1), output::Priority::Noop).
				  "}\\!\\left(".
				  generate::tex(op(data, 2), output::Priority::Noop).
				  "\\right)":

/************************************************************************/
