// PARI's round is ok, but we still need an interface function
// to handle sets ;-(

// Walter, 6.11.06: new functionality introduced for MuPaD 5.0
// 
//  round(x, n)   -- round a float to n decimal digits
//                   after the decimal point and insert
//                   all further decimal digits by 0
//  x -- an arithmetical expression
//  n -- an integer (can be negative)

round := proc(x /*, n */)
  option noDebug;
  save DIGITS;
  local n, tmp;
begin
  if args(0) < 1 then
    error("one argument expected")
  elif x::dom::round <> FAIL then
    return(x::dom::round(args()))
  end_if;

  case args(0) 
  of 1 do 
     // the classical code until MuPAD 4
     case type(x)
       of DOM_INT do return(x);
       of DOM_RAT do
       of DOM_FLOAT do
       of DOM_COMPLEX do
          return(specfunc::round(x));
       of DOM_SET do
       of "_union" do
          return(map(x, round))
     end_case;
     if not testtype(x,Type::Arithmetical) then
        /* generic handling of sets */
        if testtype(x, Type::Set) then
          if type(x)=Dom::ImageSet then
            return(map(x, round));
          else
            return(Dom::ImageSet(round(#x), #x, x));
          end_if;
        end_if;
        error("argument must be of 'Type::Arithmetical'")
     end_if;
     return(specfunc::round(x)):
     break;


  of 2 do //additional code for MuPAD beyond 4.01
     n := args(2);
     if domtype(n) <> DOM_INT then
        error("2nd argument: expecting an integer")
     end_if:
     case type(x)
       of DOM_INT do 
       of DOM_RAT do 
          return(float(10^(-n)*specfunc::round(10^n*x)));
       of DOM_FLOAT do
          // Beware: if 10^n*x > 10^DIGITS, then
          // roundoff trash from padding x with
          // binary zeroes will become visible by
          // specfunc::round. Example:
          // specfunc::round(10^20 * 12345.67)
          //     -->   1234566999999999999868928
          // Avoid this by the following construct:
          tmp:= 10^n*x:
          if specfunc::abs(tmp) <= 10.0^DIGITS then
             // we need to call float here, because
             // specfunc::round might return 0 producing
             // 0 instead of 0.0
             return(float(10^(-n)*specfunc::round(tmp)));
          else
             // this emulates padding of x with decimal digits:
             return(float(10^(-DIGITS)*specfunc::round(10^DIGITS*x)));
          end_if:
       of DOM_COMPLEX do
          return( round(Re(x), n) + round(Im(x), n)*I );
       of DOM_SET do
       of "_union" do
          return(map(x, round, args(2..args(0))))
     end_case;

     if not testtype(x,Type::Arithmetical) then
        /* generic handling of sets */
        if testtype(x, Type::Set) then
          if type(x)=Dom::ImageSet then
            return(map(x, round, n));
          else
            return(Dom::ImageSet(round(#x, n), #x, x));
          end_if;
        end_if;
        error("the 1st argument must be of 'Type::Arithmetical'")
     end_if;

     // the following code is for exact expressions
     // such as PI*(sin(1) + sqrt(3)):
     tmp:= specfunc::round(10^n*x):
     if contains({DOM_INT, DOM_COMPLEX}, domtype(tmp)) then
        // We do not expect roundoff trash in x, because it is exact.
        // So we do not need to distinguish between large and small
        // arguments
        return(float(10^(-n)*tmp));
     else
        return(hold(round)(args()));
     end_if:

  otherwise // more than 2 arguments
     error("expecting 1 or 2 arguments");
  end_case:
end_proc:

round:= funcenv(round):
round::type:= "round":
round::print:= "round":
round::float:= specfunc::round:
round::hull := DOM_INTERVAL::round@hull:

round::simplify := 
proc(x)
  local iv, res;
begin
  iv := hull(op(x)) intersect hull(getprop(op(x)));
  if op(iv, 0) = hold(hull) then // real interval
    res := {round(op(iv, 1)), round(op(iv, 2))};
    if nops(res) = 1 then
      return(op(res));
    end_if;
  elif op(iv, 0) = FAIL then // simple complex interval
    res := {round(op(iv, [1,1]) + op(iv, [2,1])*I),
      round(op(iv, [1,2]) + op(iv, [2,2])*I)};
    if nops(res) = 1 then
      return(op(res));
    end_if;
  end_if;
  x;
end_proc:

round::undefined:= {}:

round::realDiscont:=
loadproc(round::realDiscont, pathname("STDLIB", "DISCONT"), "round"):

round::rectform:=
loadproc(round::rectform, pathname("STDLIB","RECTFORM"), "round"):

round::series:= loadproc(round::series, pathname("SERIES"), "round"):
