/*

surd -  n-th root

surd(x, n)

returns that solution y of the equation y^n = x the argument of which
is closest to x

n is understood to be an integer

*/


surd:=
proc(x: Type::Union(Type::Arithmetical, Type::Set), n: Type::Union(Type::Arithmetical, Type::Set))
  local k;
begin
  if args(0) = 0 then
    error("surd called without arguments")
  end_if;

  if x::dom::surd <> FAIL then
    return(x::dom::surd(args()))
  end_if;

  if args(0) > 2 then
    error("Too many arguments")
  end_if;
    
  if testtype(n, Type::Numeric) and type(n) <> DOM_INT then
    error("Second argument must be integer")
  end_if;

  // special cases
  case n
    of 0 do
      error("Second argument must be non-zero")
    of 1 do
      return(x)
    of 2 do
      return(sqrt(x))
  end_case;
  
  if contains({DOM_INT, DOM_RAT, DOM_FLOAT}, domtype(x)) and x >= 0 then
    return(x^(1/n))
  end_if;
    
  if domtype(n) = DOM_INT then
    case type(x)
      of DOM_INT do
      of DOM_RAT do
        // we know that x < 0
        return((-1)^(1 - modp(n + 1, 2)/n)*(-x)^(1/n))
      of DOM_FLOAT do  
        // we know that x < 0
        return(float((-1)^(1 - modp(n + 1, 2)/n)*(-x)^(1/n)))
      of DOM_COMPLEX do
        if domtype(op(x, 2)) = DOM_FLOAT then
           return(surd::float(x, n));
        end_if;
        if iszero(op(x,1)) and n mod 4 = 1 then
          return(I*sign(op(x, 2))*specfunc::abs(x)^(1/n))
        end_if;
        k := round(1/2*arg(x)*(n - 1)/PI);
        if type(k) = DOM_INT then
          return(exp(2*I*k*PI/n)*x^(1/n))
        end_if;
    end_case;

    // exact numerical values such as x = sqrt(3) etc:
    if contains({DOM_FLOAT, DOM_COMPLEX}, domtype(float(x))) then
       // x is floatable, so the following 'is' 
       // should not be too expensive
       if is(x >= 0) = TRUE then
          return(x^(1/n));
       elif is(x <= 0) = TRUE then
          return((-1)^(1 - modp(n + 1, 2)/n)*(-x)^(1/n));
       end_if;
    end_if;

    //----------------
    // symbolic values
    //----------------
    if is(x >= 0) = TRUE then
       return(x^(1/n));
    elif is(x <= 0) = TRUE then
       return((-1)^(1 - modp(n + 1, 2)/n)*(-x)^(1/n));
    end_if;

  end_if;

  if not testtype(x, Type::Arithmetical) then
    if testtype(x, Type::Set) then
      if testtype(n, Type::Set) and not testtype(n, Type::Arithmetical) then
        return(Dom::ImageSet(eval(procname)(#x, #n), [#x, #n], [x, n intersect Z_ minus {0}]));
      else
        return(Dom::ImageSet(eval(procname)(#x, n), [#x], [x]));
      end_if;
    end_if;

    error("first argument must be of 'Type::Arithmetical'");
  end_if;

  if not testtype(n, Type::Arithmetical) then
    if testtype(n, Type::Set) then
      return(Dom::ImageSet(eval(procname)(x, #n), [#n], [n intersect Z_ minus {0}]));
    end_if;

    error("second argument must be of 'Type::Arithmetical'")
  end_if;

  // symbolic x or symbolic n

  procname(args())
end_proc:

surd := prog::remember(surd, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("surd")]):


surd:= funcenv(surd):

surd::print:= "surd":
surd::type:= "surd":
surd::info:= "surd -- nth root":

//-----------------------------------------------
surd::float:= proc(x, n)
local k, fx;
begin
  fx:= float(x):
  if domtype(fx) = DOM_FLOAT then
     if fx >= 0 then
        return(fx^(1/n));
     else // fx < 0 
        if n mod 2 = 0 then
             return(exp(I*(1 - 1/n)*float(PI))*(-fx)^(1/n));
        else return(-(-fx)^(1/n));
        end_if;
     end_if;
  end_if;
  if domtype(fx) = DOM_COMPLEX then
     // Copy the symbolic code above
     // special case: Re(x) = 0
     if iszero(op(fx,1)) and n mod 4 = 1 then
        return(I*sign(op(fx, 2))*specfunc::abs(fx)^(1/n))
     end_if;
     k := round(1/2*arg(fx)*(n - 1)/PI);
     if type(k) = DOM_INT then
        return(exp(float(2*I*k*PI/n))*fx^(1/n))
     end_if;
  end_if;
  hold(surd)(fx, n);
end_proc:
//-----------------------------------------------


surd::diff:= (f, x) -> f/op(f,1)/op(f,2) * diff(op(f,1), x):

surd::TeX := (f, arg, prio) -> "\\sqrt[".generate::TeX(op(arg,2)).
			      "]{".generate::TeX(op(arg,1))."}":

surd::Content :=
proc(Out, data)
begin
  if nops(data) <> 2 then
    return(Out::stdFunc(data));
  end_if;
  Out::Capply(Out::Croot,
	      Out::Cdegree(Out(op(data, 2))),
	      Out(op(data, 1)));
end_proc:



surd::expand:= 
proc(a: "surd")
  local x, newx, n, pos, others, dummy, ignore;
begin
  ignore:= contains({args()}, IgnoreAnalyticConstraints);
  x:= op(a, 1);
  n:= op(a, 2);
  
  case type(x) 
  of "_mult" do
    if ignore then
      return(expand(map(x, surd, n), args(2..args(0))))
    else
      [pos, others, dummy]:= split(x, u -> is(u>0, Goal = TRUE));
      return(expand(map(pos, surd, n), args(2..args(0))) * surd(others, n))
    end_if
  of "_power" do
    if stdlib::hasmsign(op(x, 2)) and ignore then
      return(1/ expand(surd(op(x, 1)^(-op(x, 2)), n), args(2..args(0))))
    end_if
  end_case;	
  // expand recursively only now
  
  newx:= expand(x, args(2..args(0)));
  
  if newx <> x then
    surd(newx, n)
  else
    a
  end_if  
end_proc:



surd::simplify:=
proc(s: "surd", options = simplify::defaultOptions)
begin
  if options[IgnoreAnalyticConstraints] then
    if type(op(s, 1)) = "_power" then
      op(s, [1, 1])^(op(s, [1, 2])/op(s, 2))
    elif type(op(s, 1)) = "surd" then
      // surd(surd(x, n), m) = (x^(1/n))^(1/m) = x^(1/(n*m))
      op(s, [1, 1])^(1/(op(s, [1, 2]) * op(s, 2)))
    else   
      op(s, 1)^(1/op(s, 2))
    end_if  
  else
    s
  end_if;
end_proc:



/* some attributes still missing :

hull
conjugate
rectform
series
Re
Im
laplace


*/



// end of file
