/*
 theta(t)

 t -  complex number with Re(t) > 0

 theta(t) is defined as
       sum_{k=-\infty}^{\infty} exp(n^2*PI*t)

*/


theta:=
proc(t)
begin
  if args(0) <> 1 then
    error("wrong number of arguments")
  end_if;

  if contains({0, -1}, specfunc::sign(Re(t))) then
     error("argument must have positive real part")
  end_if;
  
  case type(t)
    of DOM_FLOAT do
      return(theta::float(t))
    of DOM_COMPLEX do
      if type(op(t, 1)) = DOM_FLOAT then
        return(theta::float(t))
      else
        // theta is 2*I-periodic; normalize to 0<= Im(t) < 2
        t:= t - I*2*floor(op(t, 2)/2);
        if t = 1 then
           return(PI^(1/4)/gamma(3/4)):
        end_if;
        break
      end_if
    of DOM_SET do
    of "_union" do
      return(map(t, theta))
  end_case;

  if not testtype(t, Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(t, Type::Set) then
      if type(t)=Dom::ImageSet then
        return(map(t, theta));
      else
        return(Dom::ImageSet(theta(#t), #t, t));
      end_if;
    end_if;

    error("argument must be of 'Type::Arithmetical'")
  end_if;
  
  procname(t)
end_proc:

theta := prog::remember(theta, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("theta")]):

theta(1):= PI^(1/4) / gamma(3/4):

theta:= funcenv(theta):

theta::print:= "theta":
theta::type := "theta":


theta::float:=
proc(t)
  local a, b, c,  summe, absc, fPI,Ret, N, i, digits;
  save DIGITS;
begin
  t:= float(t);
  if not contains({DOM_FLOAT, DOM_COMPLEX}, domtype(t)) then
    return(hold(theta)(t))
  end_if;

  if testargs() then
    if Re(t) <= 0 then
      error("real part of argument must be positive")
    end_if
  end_if;

  if domtype(t) = DOM_COMPLEX then
// theta is 2*PI*I-periodic; normalize to 0<= Im(t) < 2*PI
    t:= t - I*2*floor(op(t, 2)/2)
  end_if;
  
  if specfunc::abs(t) < 0.8 then
    return(theta::float(1/t)*sqrt(1/t))
  end_if;
  
  if type(t) = DOM_COMPLEX and
    op(t, 1) < 0.2 and specfunc::abs(frac(op(t, 2)/2) - 0.5) < 0.1 then
    // t is close to I; use another method
    return(
           dedekindEta::float(1/2*(I*t + 1))^2/dedekindEta::float(I*t + 1)
           )
  end_if;
  
  digits:= DIGITS;
  
  fPI:= float(PI);
  c:= exp(-fPI*t);
  summe:= 0.5 +c;
  a:= c^2;
  b:= a*c;
  // b = c^3

  if iszero(c) then // rounded off to zero -> every summand c^(i^2)
                    // except i=0 will vanish
    return(1.0)
  end_if;



  // compute bound with small value of DIGITS
  DIGITS:= 8;
  // |sum(c^(n^2), n=N..infinity)| <= |c|^(N^2)*(1/(1-|c|))
  // it is sufficient if this is < 10^(-DIGITS)
  // i.e. N^2 >= ln((1-|c|)/10^DIGITS)/ln(|c|)
  absc:= specfunc::abs(c);
  if specfunc::abs((Ret:= Re(t))) > 10^(-DIGITS/2) then
    N:= ceil(sqrt(ln((1-absc)/10.0^(DIGITS+5))/ln(absc)))
  else
     // |c| is close to 1
      // of course, we must not compute |c| = 1.0 here
     // |c| = exp(-PI*Re(t)) <= 1 - PI*Re(t) + PI^2*Re(t)^2/2
     // ln(1-|c|)/ln(|c|) <= ln(1-|c|)* (1/(|c|-1)) 
     // <= ln(PI*Re(t)) / (PI*Re(t) - PI^2*Re(t)^2 / 2)
     N:= ceil(sqrt((ln(fPI*Ret)+10.0*ln::float(digits)/
         (fPI*Ret - fPI^2*Ret^2/2))))
  end_if;

  
  DIGITS:= digits;
  for i from 1 to N do
    // let C:= exp(-PI*t)
    // loop invariant: b = C^(2*i+1), c = C^(i^2)
    c:= c*b;
    b:= b*a;
    summe:= summe + c
  end_for;
  
  2* summe 
end_proc:

// end of file
