/* --------------------------------------------------
Call:        stats::geometricQuantile(p)

Parameters:  p  - an arithmetical expression representing
                  a real number 0 <= p <= 1

Returns:  a function

Details:
  - The geometric distribution describes the number of
    Bernoulli trials with success probability p
    up to and including the first success
  - f:= stats::geometricQuantile(p) can be called in the
    form f(x) with an arithmetical expression x representing
    a real number 0 <= x <= 1

     x symbolic             --> symbolic call
     x float and p symbolic --> symbolic expression
     x float and p numeric  --> float
------------------------------------------------*/

stats::geometricQuantile:= proc(p)
local fp, ln10;
option escape;
begin
  if args(0) <> 1 then
     error("expecting one parameter");
  end_if;

  // ------------- check p -------------
  fp:= float(p):
  if domtype(fp) = DOM_FLOAT then
     if fp < 0 or fp > 1 then
        error("the 'probability parameter' p must satisfy 0 <= p <= 1"):
     end_if;
  end_if;
  if domtype(fp) = DOM_COMPLEX then
     error("the 'probability parameter' must be real");
  end_if;

  // The following approximation of ln(10) is used
  // by the returned procedure
  ln10:= 2.302585093;

  //------------------------------------------
  // return the following procedure
  //------------------------------------------

  proc(x)
  local pp, fp, q, fq, fx, fw, k, oldDIGITS, tmp;
  save DIGITS;
  begin
    if args(0)<>1 then
       error("expecting one argument")
    end_if:

    // double check the parameter p. It might have
    // changed since this procedure was created:

    // ------------- check p -------------
    pp:= context(p):
    fp:= float(pp):
    q:= 1 - pp;
    if domtype(fp) = DOM_FLOAT then
       if fp < 0 or fp > 1 then
          error("the 'probability parameter' p must satisfy 0 <= p <= 1"):
       end_if;
    end_if;
    if domtype(fp) = DOM_COMPLEX then
       error("the 'probability parameter' must be real");
    end_if;

    // -------------- check x ---------------
    fx:= float(x):

    if domtype(fx) = DOM_COMPLEX then
       error("expecting a real argument"):
    end_if;

    if domtype(fx) = DOM_FLOAT then
         if fx < 0 or fx > 1 then
            error("expecting an argument 0 <= x <= 1");
         end_if;
    end_if;

    //---------------------------------------------
    // handle the exceptional cases p = 0 and p = 1
    //---------------------------------------------
    if iszero(pp) then
       return(infinity);
    end_if;

    if iszero(q) then
       return(1);
    end_if;

    //---------------------------------------------
    // handle the exceptional cases x = 0 and x = 1
    //---------------------------------------------
    if iszero(x) then
       return(1);
    end_if;

    if iszero(x-1) then 
       return(infinity)
    end_if:

    //---------------------------------------------
    // Find the smallest integer k such that
    // geometricCDF(p)(k) = 1 - q^k >= x
    // -->  q^k <= 1 - x --> k = ceil( ln(1-x)/ln(q) );
    //---------------------------------------------

    if domtype(fp) <> DOM_FLOAT then
       return(hold(stats::geometricQuantile)(pp)(x));
    end_if;

    //---------------------------------------------
    // now we are sure that
    // --  p is a numerical value 0 < p < 1
    //---------------------------------------------

    if domtype(fx) <> DOM_FLOAT then
       // x is symbolic, nothing can be done
       // The special cases p = 0 and p = 1 were
       // handled above.
       return(hold(stats::geometricQuantile)(pp)(x));
    end_if;

    //---------------------------------------------
    // now we are sure that
    // --  p is numerical and 0 < p < 1
    // --  x is numerical and 0 < x < 1
    //---------------------------------------------
    // We know that x <> 1, p <> 1, q <> 1.
    // We need to avoid fx = 1.0, fq = 1.0,
    // so resolve 1 - x, 1 - p by boosting DIGITS:
    //---------------------------------------------

    oldDIGITS:= DIGITS;
    fw:= float(1 - x):
    fq:= float(q):
    while iszero(fw) or iszero(fq) or iszero(1 - fq) do
      DIGITS:= 2*DIGITS;
      fw:= float(1 - x):
      fq:= float(q):
    end_while;
    DIGITS:= oldDIGITS;

    k := ln::float(fw)/ln::float(fq);
    // if ceil(k) < 1 then the result is
    // infinity. However, we are not sure
    // that the current precision suffices.
    // Accept k < 1/2 as realistic:
    if k < 1/2 then 
       return(1);
    end_if;

    //---------------------------------------------
    // we need to boost DIGITS to make sure that
    // the floating point estimate k is accurate
    // to log(10, k) digits. Otherwise, floor would
    // produce a wrong result:
    //---------------------------------------------
    tmp:= 3 + ceil(ln::float(k))/ln10;
    if tmp > DIGITS then
       DIGITS := tmp;
       // recompute k with higher DIGITS
       k := ln::float(fw)/ln::float(fq);
    end_if; 
    if specfunc::abs(frac(k)) <= specfunc::abs(k)/10^oldDIGITS then
       k:= round(k);
    else
       k:= ceil(k);
    end_if:
    if k < 1 then
         return(1);
    else return(k);
    end_if;
  end_proc:
end_proc:
