/*--
stats::tTest -- the classical tTest for testing an estimate
                of an expected mean

Call: 

stats::tTest(x1, x2, ..., mu <, Normal)
stats::tTest([x1, x2, ...], mu <, Normal)
stats::tTest(s <, c>, mu <, Normal>)

Parameter:

x1, x2, ... -- arithmetical expressions (symbolic
               objects are allowed!)
mu          -- the hypothesized mean: an arithmetical
               expression (symbolic object is allowed)
s           -- a stats::sample
c	    -- a column index: positive integer.
               Column c of s provides the data x1, x2, ..

Option:  Normal -- use stats::normalCDF(0, 1) as an approximation
                   of stats::tdist(samplesize - 1) for the
                   t-test

Synopsis:

stats::tTest(..data.., mu <, Normal>) determine how good of a guess
'mu' is for the expected mean of values of the data.
If the option 'Normal' is not specified, the t-distribution is used.

Examples:
  Determine how good of a guess is 5 for the mean of x, where 10
  experiments gave 1, -2, 3, -4, 5, -6, 7, -8, 9, 10.

>> data := [1, -2, 3, -4, 5, -6, 7, -8, 9, 10]:

>> float(stats::tTest(data, 5));

		5.756660091e-2

>> float(stats::Test(data, 5, Normal));

		4.058346175e-2

See also:

stats::mean, stats::stdev, stats::normal, stats::tCDF,
stats::csGOFT, stats::ksGOFT, stats::swGOFT

--*/

stats::tTest := proc()
   local usenormal, data, mu, n, sigma, m, x;
begin
   if args(0) < 2 then
      error("expecting at least 2 arguments");
   end_if; 

   if args(args(0)) = Normal then
        usenormal:= TRUE;
   else usenormal:= FALSE;
   end_if;

   if usenormal then
        if args(0) < 3 then
           error("expecting at least 3 arguments ".
                 " if the option 'Normal' is specified");
        end_if; 
        data:= stats::getdata(testargs(), "all_data", 1, args(1..args(0) - 2)):
        if domtype(data) = DOM_STRING then
           error(data)
        end_if:
        mu:= args(args(0) - 1):
   else
        data:= stats::getdata(testargs(), "all_data", 1, args(1..args(0) - 1)):
        mu:= args(args(0));
   end_if;

   // Now, the data are a list [x1, x2, ..]

   n:= nops(data): // the sample size
   if n = 0 then
      error("no sample data");
   end_if;
   if n = 1 then
      // n = 1 --> stddev(data) = 0
      return(FAIL);
   end_if;

   if domtype(mu) = DOM_FLOAT then
      data:= float(data);
   end_if:
   sigma:= stats::stdev(data, Sample);
   if sigma = FAIL or iszero(sigma) then 
      return(FAIL)
   end_if:
   m := stats::mean(data):
   if domtype(m) = DOM_FLOAT then
        x := float( (m - mu)/(sigma/sqrt(n)));
   else x := (m - mu)/(sigma/sqrt(n));
   end_if;
   if usenormal then
        return([PValue = stats::normalCDF(0, 1)(x), StatValue = x])
   else if n < 2 then
           error("expecting at least 2 data");
        end_if;
        return([PValue = stats::tCDF(n - 1)(x), StatValue = x]);
   end_if;
end_proc:
