/* --------------------------------------------------
Call:        stats::geometricPF(p)

Parameters:  p  - an arithmetical expression representing
                  real numbers 0 <= p <= 1

Returns:  a function

Details:
  - The geometric distribution describes the number of
    Bernoulli trials with success probability p
    up to and including the first success
  - f:= stats::geometricPF(p) can be called in the
    form f(x) with an arithmetical expression x representing
    a real number.

     x symbolic             --> symbolic call
     x float and p symbolic --> symbolic expression
     x float and p numeric  --> float
-------------------------------------------------- */

stats::geometricPF:=proc(p)
local fp;
option escape;
begin
  if args(0) <> 1 then
     error("expecting one argument")
  end_if:

  // check the parameters at the time when the
  // function is generated

  // ------------- check p -------------
  fp:= float(p):
  if domtype(fp) = DOM_FLOAT then
     if fp < 0 or fp > 1 then
        error("the 'probability parameter' p must satisfy 0 <= p <= 1"):
     end_if;
  end_if;
  if domtype(fp) = DOM_COMPLEX then
     error("the 'probability parameter' must be real");
  end_if;

  //-------------------------------
  // return the following procedure
  //-------------------------------

  proc(x)
  local pp, fp, q, fx;
  begin

    // -------------------------------------------------
    // Return
    // P(X = x) = p*q^(x -1), x = 1, 2, 3, ...
    // where q = 1 - p and x >= 1
    // -------------------------------------------------

    if args(0) <> 1 then
       error("expecting one argument")
    end_if:

    // ---------------------------------------------
    // DO NOT MESS UP THE ORDERING OF THE FOLLWING
    // CHECKS! YOU WILL CHANGE THE FUNCTIONALITY!
    // ---------------------------------------------

    // double check the parameter p. It might have
    // changed since this procedure was created:

    // ------------- check p -------------
    pp:= context(p):
    fp:= float(pp):
    q:= 1 - pp;
    if domtype(fp) = DOM_FLOAT then
       if fp < 0 or fp > 1 then
          error("the 'probability parameter' p must satisfy 0 <= p <= 1"):
       end_if;
    end_if;
    if domtype(fp) = DOM_COMPLEX then
       error("the 'probability parameter' must be real");
    end_if;

    // -------------- check x ---------------
    if x = -infinity then return(0); end_if;
    if x =  infinity then return(0); end_if;
    fx:= float(x):
    if domtype(fx) = DOM_COMPLEX then
       error("expecting a real argument"):
    end_if;

    // ---------------------------------------------
    // special cases: p = 0 and p = 1
    // Note that 0^x will not simplify to 0
    // if x is symbolic ;-(
    // ---------------------------------------------
    if iszero(pp) then
       if domtype(x) = DOM_FLOAT then
            return(float(0));
       else return(0);
       end_if;
    end_if;

    if iszero(q) then // p = 1
       if (domtype(fx) = DOM_FLOAT and not iszero(x - 1)) then
          if domtype(x) = DOM_FLOAT then
               return(float(0));
          else return(0);
          end_if;
       else
          // Note  is(1.0 <> 1) = TRUE !
          if domtype(x) <> DOM_FLOAT and (is(x <> 1) = TRUE) then
            if domtype(x) = DOM_FLOAT then
                 return(float(0));
            else return(0);
            end_if;
          end_if;
       end_if;
       if iszero(x - 1) then
          if domtype(x) = DOM_FLOAT then
              return(float(1));
          else return(1);
          end_if;
       end_if;
       return(hold(stats::geometricPF)(pp)(x));
    end_if;

    //-------------------------------------
    // handle symbolic x:
    //-------------------------------------
    if domtype(fx) <> DOM_FLOAT then
       // x is symbolic, nothing can be done
       return(hold(stats::geometricPF)(pp)(x));
    end_if;

    //-------------------------------------
    // Now we are sure that x is numerical 
    //-------------------------------------
    if fx < 1 or not iszero(frac(x)) then
       if domtype(x) = DOM_FLOAT then
          return(float(0));
       else return(0);
       end_if;
    end_if;

    //-------------------------------------
    // now we are sure that x is a positive
    // integer of float(integer)
    //-------------------------------------

    if domtype(x) = DOM_FLOAT then
         return(fp*float(q)^(x - 1));
    else return(pp*q^(x - 1));
    end_if;
  end_proc:
end_proc:
