/* -------------------------------------------------------------------------
Stephan Huckemann Sept.5.01
hypergeometric probability for exactly x successes in a sample of size n where 
population size is N with X successes in entire population
= binomial(X, x)* binomial(N-X, n-x) / binomial(N, n)
-------------------------------------------------------------------------*/

stats::hypergeometricPF:=proc(N, X, n)

local fN, fX, fn;

option escape;
begin

/*prog::check(stats::hypergeometricPF,6);*/

  if args(0)<>3 then
     error("expecting three arguments")
  end_if:

  // ------------- check N -------------
  fN := float(N);
  if domtype(N) = DOM_INT and N < 1 then
     error("the 'population size' must be symbolic or a positive integer"):
  end_if;
  if domtype(fN) = DOM_FLOAT and domtype(N) <> DOM_INT then
     error("the 'population size' must be symbolic or a positive integer"):
  end_if;
  if domtype(fN) = DOM_COMPLEX then
     error("the 'population size' must be real"):
  end_if;
  
 // ------------- check X -------------
  fX := float(X);
  if domtype(X) = DOM_INT then
	if X < 0 or (domtype(N)= DOM_INT and X > N) then
	     error("the 'success population size' must be symbolic or an integer".
		" between 0 and 'population size'"):
	end_if;
  end_if;
  if domtype(fX) = DOM_FLOAT and domtype(X) <> DOM_INT then
	     error("the 'success population size' must be symbolic or an integer".
		" between 0 and 'population size'"):  
  end_if;
  if domtype(fX) = DOM_COMPLEX then
     error("the 'success population size' must be real"):
  end_if;
 
 // ------------- check n -------------
  fn := float(n);
  if domtype(n) = DOM_INT then
	if n < 0 or (domtype(N) = DOM_INT and n > N) then
	     error("the 'sample size' must be symbolic or an integer".
		" between 0 and 'population size'"):
	end_if;
  end_if;
  if domtype(fn) = DOM_FLOAT and domtype(n) <> DOM_INT then
     error("the 'sample size' must be symbolic or an integer".
	" between 0 and 'population size'"):
  end_if;
  if domtype(fn) = DOM_COMPLEX then
     error("the 'sample size' must be real"):
  end_if;
 

  //-------------------------------
  // return the following procedure
  //-------------------------------
  proc(x)
  local NN, XX, nn, fNN, fXX, fnn, fx, tx, tmp;
  begin
    if args(0)<>1 then
       error("expecting one argument")
    end_if:

    // double check the parameters N, X and n. They
    // might have changed since this procedure was
    // created:


    // ------------- check N -------------
    NN  := context(N):
    fNN := float(NN):
    if domtype(NN) = DOM_INT and NN < 1 then
     error("the 'population size' must be symbolic or a positive integer"):
    end_if;
    if domtype(fNN) = DOM_FLOAT and domtype(NN) <> DOM_INT then
     error("the 'population size' must be symbolic or a positive integer"):
    end_if;
    if domtype(fNN) = DOM_COMPLEX then
     error("the 'population size' must be real"):
    end_if;

    // ------------- check X -------------
    XX := context(X):
    fXX := float(XX):
    if domtype(XX) = DOM_INT then
	if XX < 0 or (domtype(NN)= DOM_INT and XX > NN) then
	     error("the 'success population size' must be symbolic or an integer".
		" between 0 and 'population size'"):
	end_if;
    end_if;
    if domtype(fXX) = DOM_FLOAT and domtype(XX) <> DOM_INT then
     error("the 'success population size' must be symbolic or an integer".
	" between 0 and 'population size'"):
    end_if;
    if domtype(fXX) = DOM_COMPLEX then
     error("the 'success population size' must be real"):
    end_if;

    // ------------- check n -------------
    nn:= context(n):
    fnn := float(nn):
    if domtype(nn) = DOM_INT then
	if nn < 0 or (domtype(NN) = DOM_INT and nn > NN) then
	     error("the 'sample size' must be symbolic or an integer".
		" between 0 and 'population size'"):
	end_if
    end_if;
    if domtype(fnn) = DOM_FLOAT and domtype(nn) <> DOM_INT then
     error("the 'sample size' must be symbolic or an integer".
	" between 0 and 'population size'"):
    end_if;
    if domtype(fnn) = DOM_COMPLEX then
     error("the 'sample size' must be real"):
    end_if;

    // now we can assume that N, X and n are of type integer or symbolic

    if x = -infinity then return(0); end_if;
    if x =  infinity then return(0); end_if;

    // P(x) = binomial(X, x)* binomial(N-X, n-x) / binomial(N, n)
    // if x is an integer with 0 <= x <= n,
    // P(x) = 0 else

    // trivial cases:
    // if x is numeric or symbolic > N, X, n => P(x) = 0
    // if x is numeric < 0. then P(x) = 0
    // if n is numeric = 0 => P(x) = 1 if x = 0, P(x) = 0 else
    // another  still pretty trivial case
    // if n + X > N and x < n + X - N we have P(x) = 0

    // It appears there are no nice special cases only with regards to x:
    // If x = 0, then P(x) = binomial(N-X, n) / binomial(N, n)
    // If x = n, then P(x) = binomial(X, n) / binomial(N, n)
    // If x = X, then P(x) = binomial(N-X, n-X) / binomial(N, n)

    // considering X leads to some nice cases:
    // if X = N, then P(x) = 1 if x = n, = 0 else        
    // if X = N-1, then P(x) = (N-n)/N if x = n, = n/N if x = n-1, 0 else    
    // if X = 0, then P(x) = 1 if x = 0, = 0 else
    // if X = 1, then P(x) = (N-n)/N if x = 0, = n/N if x = 1, 0 else    
    // in those latter four symbolic cases we will return a symbolic result
    // else we will return only numeric results if everything is numerical

    fx := float(x);
    // -------- If x is neither symbolic nor integer -----------------------
    if domtype(fx) = DOM_COMPLEX then 
	error("the argument must be real"); 
    end_if;

    if (domtype(fx) = DOM_FLOAT and iszero(frac(x))) or (domtype(fx) <> DOM_FLOAT)
		  then 
		// nothing we're just fine
    else
	if domtype(x) = DOM_FLOAT then return(float(0)) end_if;
	return(0);
    end_if;

    // --------- any variable is now symbolic or integer -------------

    // ------------------------trivial cases: ----------------------
    // if x numeric or symbolic < 0 or > N, X, n => P(x) = 0
    // check 1st whether domtypes allow comparison

    if domtype(fx) = DOM_FLOAT then

	if fx < 0 then 
		if domtype(x) = DOM_FLOAT then return(float(0)) end_if;
		return(0); 
	end_if;
	if (domtype(NN) = DOM_INT and (fx - NN) > 0) then 
		if domtype(x) = DOM_FLOAT then return(float(0)) end_if;
		return(0); 
	end_if;
    	if (domtype(XX) = DOM_INT and (fx - XX) > 0) then 
		if domtype(x) = DOM_FLOAT then return(float(0)) end_if;
		return(0); 
	end_if;
    	if (domtype(nn) = DOM_INT and (fx - nn) > 0) then 
		if domtype(x) = DOM_FLOAT then return(float(0)) end_if;
		return(0); 
	end_if;

    // if n numeric = 0 => P(x) = 1 if x = 0, P(x) = 0 else

	if (domtype(nn) = DOM_INT and nn = 0) then
		if iszero(x) then 
			if domtype(x) = DOM_FLOAT then return(float(1)) end_if;
			return(1);
		else 
			if domtype(x) = DOM_FLOAT then return(float(0)) end_if;
			return(0);
		end_if;
	end_if;		

    else  // x is symbolic and cannot be convertet to float
	if (domtype(NN) <> DOM_INT and is((x - NN) > 0) = TRUE) then 
		return(0); 
	end_if;
    	if (domtype(XX) <> DOM_INT and  is((x - XX) > 0) = TRUE) then 
		return(0); 
	end_if;
    	if (domtype(nn) <> DOM_INT and  is((x - nn) > 0) = TRUE) then 
		return(0); 
	end_if;
    end_if;

    // ---------------- now we know that x is symbolic or >= 0 ---------

    // --------------- pretty trivial case -----------------------
    // if n + X > N and x < n + X - N then P(x) = 0
    // check if n + X - N is numeric
     if	domtype(nn + XX - NN) = DOM_INT and domtype(fx) = DOM_FLOAT
	and (nn + XX - NN) > 0 and (nn + XX - NN - fx) > 0 then
		if domtype(x) = DOM_FLOAT then return(float(0)) end_if;
		return(0);
	// note that actually (nn + XX - NN - x) > 0 implies that  (nn + XX - NN) > 0
    end_if;

    // For reasons of efficiency the symbolic case is continued at the end

    // -------------- check X = N --------------- 
    if iszero(NN - XX) then
	if iszero(x - nn) then
		if domtype(x) = DOM_FLOAT then
			return(float(1));
		else
			if domtype(x) = DOM_FLOAT then return(float(1)) end_if;
			return(1);
		end_if;
	else
		// if x is symbolic and n is not or vice versa
		// 		or
		// or both are symbolic but cannot be compared
		// 		nothing can be said
		if (domtype(fx) <> DOM_FLOAT and domtype(nn) = DOM_INT) or
		   (domtype(fx) = DOM_FLOAT and domtype(nn) <> DOM_INT) or
	    	   ((domtype(fx) <> DOM_FLOAT and domtype(nn) <> DOM_INT)
		     and is((nn - x) > 0) <> TRUE ) then
		       return(hold(stats::hypergeometricPF)(NN, XX, nn)(x));
		else
			if domtype(x) = DOM_FLOAT then return(float(0)) end_if;
			return(0);
		end_if;
    	end_if; 
    end_if;   
    // -------------- check X = N - 1 ---------------
    if iszero(NN - 1 - XX) then
	if iszero(x - nn) then
		if domtype(x) = DOM_FLOAT then
			return(float((NN-nn)/NN));
		else
			return((NN-nn)/NN);
		end_if;
	else
		if iszero(nn - 1 - x) then
			if domtype(x) = DOM_FLOAT then
				return(float(nn/NN));
			else
				return(nn/NN);
			end_if;
		else
		// if x is symbolic and n is not or vice versa
		// 		or
		// or both are symbolic but cannot be compared
		// 		nothing can be said
		if (domtype(fx) <> DOM_FLOAT and domtype(nn) = DOM_INT) or
		   (domtype(fx) = DOM_FLOAT and domtype(nn) <> DOM_INT) or
	    	   ((domtype(fx) <> DOM_FLOAT and domtype(nn) <> DOM_INT)
		     and is((nn - x) > 0) <> TRUE ) then
			return(hold(stats::hypergeometricPF)(NN, XX, nn)(x));
		    else
			if domtype(x) = DOM_FLOAT then return(float(0)) end_if;
			return(0);
		    end_if;
		end_if;
    	end_if; 
    end_if;   
    // -------------- check X = 0 ---------------
    if iszero(XX) then
	if iszero(x) then
		if domtype(x) = DOM_FLOAT then return(float(1)) end_if;
		return(1);
	else
		// if x is symbolic
		if domtype(fx) <> DOM_FLOAT then
		       return(hold(stats::hypergeometricPF)(NN, XX, nn)(x));
		end_if;
		if domtype(x) = DOM_FLOAT then return(float(0)) end_if;
		return(0);
    	end_if; 
    end_if;
    // -------------- check X = 1 --------------
    if iszero(1 - XX) then
	if iszero(x) then
		if domtype(x) = DOM_FLOAT then
			return(float((NN-nn)/NN));
		else
			return((NN-nn)/NN);
		end_if;
	else
		if iszero(1 - x) then
			if domtype(x) = DOM_FLOAT then
				return(float(nn/NN));
			else
				return(nn/NN);
			end_if;
		else
		// if x is symbolic
		    if domtype(fx) <> DOM_FLOAT then
		       return(hold(stats::hypergeometricPF)(NN, XX, nn)(x));
		    else
			if domtype(x) = DOM_FLOAT then return(float(0)) end_if;
			return(0);
		    end_if;
		end_if;
    	end_if; 
    end_if;

	// now the same for n ...

   // -------------- check n = N --------------- 
    if iszero(NN - nn) then
	if iszero(x - XX) then
		if domtype(x) = DOM_FLOAT then
			return(float(1));
		else
			if domtype(x) = DOM_FLOAT then return(float(1)) end_if;
			return(1);
		end_if;
	else
		// if x is symbolic and X is not or vice versa
		// 		or
		// or both are symbolic but cannot be compared
		// 		nothing can be said
		if (domtype(fx) <> DOM_FLOAT and domtype(XX) = DOM_INT) or
		   (domtype(fx) = DOM_FLOAT and domtype(XX) <> DOM_INT) or
	    	   ((domtype(fx) <> DOM_FLOAT and domtype(XX) <> DOM_INT)
		     and is((XX - x) > 0) <> TRUE ) then
		       return(hold(stats::hypergeometricPF)(NN, XX, nn)(x));
		else
			if domtype(x) = DOM_FLOAT then return(float(0)) end_if;
			return(0);
		end_if;
    	end_if; 
    end_if;   
    // -------------- check n = N - 1 ---------------
    if iszero(NN - 1 - nn) then
	if iszero(x - XX) then
		if domtype(x) = DOM_FLOAT then
			return(float((NN-XX)/NN));
		else
			return((NN-XX)/NN);
		end_if;
	else
		if iszero(XX - 1 - x) then
			if domtype(x) = DOM_FLOAT then
				return(float(XX/NN));
			else
				return(XX/NN);
			end_if;
		else
		// if x is symbolic and X is not or vice versa
		// 		or
		// or both are symbolic but cannot be compared
		// 		nothing can be said
		if (domtype(fx) <> DOM_FLOAT and domtype(XX) = DOM_INT) or
		   (domtype(fx) = DOM_FLOAT and domtype(XX) <> DOM_INT) or
	    	   ((domtype(fx) <> DOM_FLOAT and domtype(XX) <> DOM_INT)
		     and is((XX - x) > 0) <> TRUE ) then
			return(hold(stats::hypergeometricPF)(NN, XX, nn)(x));
		    else
			if domtype(x) = DOM_FLOAT then return(float(0)) end_if;
			return(0);
		    end_if;
		end_if;
    	end_if; 
    end_if;   
    // -------------- check n = 0 ---------------
    if iszero(nn) then
	if iszero(x) then
		if domtype(x) = DOM_FLOAT then return(float(1)) end_if;
		return(1);
	else
		// if x is symbolic
		if domtype(fx) <> DOM_FLOAT then
		       return(hold(stats::hypergeometricPF)(NN, XX, nn)(x));
		end_if;
		if domtype(x) = DOM_FLOAT then return(float(0)) end_if;
		return(0);
    	end_if; 
    end_if;
    // -------------- check n = 1 --------------
    if iszero(1 - nn) then
	if iszero(x) then
		if domtype(x) = DOM_FLOAT then
			return(float((NN-XX)/NN));
		else
			return((NN-XX)/NN);
		end_if;
	else
		if iszero(1 - x) then
			if domtype(x) = DOM_FLOAT then
				return(float(XX/NN));
			else
				return(XX/NN);
			end_if;
		else
		// if x is symbolic
		    if domtype(fx) <> DOM_FLOAT then
		       return(hold(stats::hypergeometricPF)(NN, XX, nn)(x));
		    else
			if domtype(x) = DOM_FLOAT then return(float(0)) end_if;
			return(0);
		    end_if;
		end_if;
    	end_if; 
    end_if;


    // ----------------------------------------------------------------------
    // now we shall only compute a result if all NN, XX, nn, x are numerical
    // we do already know that NN, XX, nn and x are either integer or symbolic
    // but first let's check some more symbolic cases ...
    // ----------------------------------------------------------------------

    // in the corresponding hypergeometricCDF (cf. mu-file) we display a 
    // symbolic result if X, n and x are numerical and N > X
    // for consistency we shall do the same here.
    // we could also do compute one binomial for any 3 variables numerical 
    // and the fourth symbolic. We shall not though.
 
    if domtype(NN) <> DOM_INT and domtype(XX) = DOM_INT and domtype(nn) = DOM_INT 
	and domtype(fx) = DOM_FLOAT  then
	if is((NN-XX) > 0 ) = TRUE then
	    tx := trunc(x);
	    tmp :=  binomial(NN-XX, nn-tx); // beware of binomials as e.g.
		// float(binomial(6,3)) - binomial (6,3.0) = 1.7 e-18 ....

	    if domtype(x) = DOM_FLOAT then
	    	return(float(binomial(XX,tx) *  tmp / binomial(NN, nn))); 
            else
	    	return(binomial(XX,x) * tmp / binomial(NN, nn));
	    end_if;
	end_if;
    end_if;


    // in case any is symbolic

    if domtype(NN) <> DOM_INT or domtype(XX) <> DOM_INT or domtype(nn) <> DOM_INT 
	or domtype(fx) <> DOM_FLOAT  then
    // --------------- pretty trivial case -----------------------
    // if n + X > N and x < n + X - N then P(x) = 0
    // what was left was that case in which at least one of the N,X,n was symbolic
     if
	is((nn + XX - NN) > 0) = TRUE and 
	is((nn + XX - NN - x) > 0) = TRUE then
		if domtype(x) = DOM_FLOAT then return(float(0)) end_if;
		return(0);
     end_if;
    // nothing can be said
       return(hold(stats::hypergeometricPF)(NN, XX, nn)(x));
    end_if;

    // now everything is completly numeric

    // speeding up if floating point arithmetic is desired:
    if domtype(x) = DOM_FLOAT then
	    return(binomial(fXX, x) * binomial(fNN - fXX, fnn - x) / binomial(fNN, fnn));
    end_if;
    // else rational output is desired
    return(binomial(XX, x) * binomial(NN - XX, nn - x) / binomial(NN, nn));
   
  end_proc:
end_proc:
