
/* ------------------------------------------
Walter Oevel, 7.7.01
  - now 
     betaQuantile(a, b)(0) = 0.0
     betaQuantile(a, b)(1) = 1.0
    for any value of a and b
------------------------------------------ */

stats::betaQuantile:=proc(a, b)
local fa, fb;
option escape;
begin
  if args(0)<>2 then
      error("expecting two arguments")
  end_if:

  // ------------- check a -------------
  fa:= float(a):
  if domtype(fa) = DOM_FLOAT and fa <= 0 then  
     error("the first shape parameter must be positive"):
  end_if;
  
  // ------------- check b -------------
  fb:= float(b):
  if domtype(fb) = DOM_FLOAT and fb <= 0 then
     error("the second shape parameter must be positive"):
  end_if;

  //-------------------------------
  // return the following procedure
  //-------------------------------
  proc(x)
  local aa, bb, fa, fb, fx, z, numericEvaluation,
        z1, z2, oldDIGITS, macheps;
  save DIGITS;
  begin 
    if args(0)<>1 then error("one argument expected")
    end_if:

    macheps:= 10.0^(-DIGITS):
    DIGITS:= DIGITS + 2:

    // ------------- check a -------------
    // enforce full evaluation of the parameters
    aa:= context(a):
    fa:= float(aa): 
    if domtype(fa) = DOM_FLOAT and fa <= 0 then
       error("the first shape parameter must be positive"):
    end_if;
       
    // ------------- check b -------------
    // enforce full evaluation of the parameters
    bb:= context(b):
    fb:= float(bb):
    if domtype(fb) = DOM_FLOAT and fb <= 0 then
       error("the second shape parameter must be positive"):
    end_if; 
       
    // ----------- check x ----------
    fx:= float(x):

    if domtype(fx) <> DOM_FLOAT then
       // x is symbolic, nothing can be done
       return(hold(stats::betaQuantile)(aa, bb)(x));
    end_if;

    // now we are sure that x is numeric

    if fx > 1 then
       error("expecting an argument 0 <= x <= 1"):
    end_if;
    if iszero(1 - x) then
       return(float(1)) 
    end_if;
    if iszero(x) then
       return(float(0)) 
    end_if;
    if fx < 0 then
       error("expecting an argument 0 <= x <= 1"):
    end_if;

    //---------------------------------------------
    // now we are sure that x is numerical and 0 < x < 1
    //---------------------------------------------

    // --------- float evaluation ? ------------
    // Presently, we do not have a specialfunction ibeta
    // that may be used to return a symbolic result or an
    // exact numerical result. We *must* try floating point
    // evaluation. In the following, we can change
    //   if domtype(fx)=DOM_FLOAT  
    // to
    //   if domtype(x)=DOM_FLOAT,
    // when we have a symbolic ibeta: 

    if domtype(fx)=DOM_FLOAT and 
       domtype(fa)=DOM_FLOAT and 
       domtype(fb)=DOM_FLOAT
    then numericEvaluation:=TRUE:
    else numericEvaluation:=FALSE:
    end_if:

    // ---------------------
    // the work starts
    // ---------------------

    if numericEvaluation then

       //---- special case b = 1 --------
       // x = betaCDF(a, 1)(z) = z^a 
       //     --> z = x^(1/a)
       //------------------------------------

       if iszero(fb - 1) then
          return(fx^(1/fa));
       end_if;

       //---- special case a = 1 --------
       // betaCDF(1, b)(z) = 1 - (1 - z)^b = x
       // --> z = 1 - (1 - x)^(1/b)
       // = x/b  + (b-1)/2*x^2/b^2 + (2*b-1)/6*(b-1)*x^3/b^3 + ...
       // = x/b * (1 + (b-1)/2*x/b + (2*b-1)/6*(b-1)*x^2/b^2 + ... )
       //  |-z-|      |----z1----|   |-------- z2 ---------|
       //------------------------------------

       if iszero(fa - 1) then
          // first, we need to stabilize the potential
          // cancellation if x is very small: use the
          // Taylor expansion above:

          z:= fx/fb;
          z1:= (fb - 1)/2*z;
          if specfunc::abs(z1) < macheps then
             return(z);
          end_if;
          z2:= (2*fb - 1)/3*z*z1;
          if specfunc::abs(z2) < macheps then
             return(z*(1 + z1));
          end_if;
          // We know that iszero(1 - x) = FALSE.
          // However, float(1 - x) can still yield 0.0.
          // Boost DIGITS to resolve float(1 - x):
          fx:= float(1 - x);
          oldDIGITS:= DIGITS;
          while iszero(fx) do
             DIGITS:= 2*DIGITS;
             fx:= float(1 - x);
          end_while;
          // boost DIGITS to get 1 - x correctly
          // to oldDIGITS decimal places
          DIGITS:= oldDIGITS + ceil(-ln(fx)/2.30285093);
                     // 2.302.. = ln(10):
          fx:= float(1 - x);
          DIGITS:= oldDIGITS;
          return(1 - fx^(1/fb));
       end_if;
        
       //------------------------------------
       // general case a <> 1, b <> 1
       //------------------------------------

       //------------------------------------
       // First, try to get away without calling
       // numeric::realroot for very small solutions.
       // Try a solution via a Taylor expansion:

       // If the solution z is very small, we
       // should not call numeric::realroots,
       // because small results are returned only with
       // an *absolute* precision of macheps, but we
       // wish a *relative* precision here.
       // Use the continued fraction expansion
       //  a * beta(a, b)* ibeta(a, b, z)
       //  = z^a*(1 - z)^(b -1) *
       //     (1 + (b-1)/(a+1)*z/(1-z) *
       //     (1 + (b-2)/(a+2)*z/(1-z) *
       //     (1 + (b-3)/(a+3)*z/(1-z) * (1 + O(z)))))
       //  = a * beta(a, b) * x.
       //  With z << 1, we end up with the following fixed point
       //  equation
       //  z = (a*beta(a,b)*x)^(1/a) *
       //        (1 + (b-1)/(a+1)*z *(1 + (a+b)/a+2)*z))
       //  z = (a*beta(a,b)*x)^(1/a) *
       //      (1 + (b-1)/(a+1)*z + (b-1)/(a+1)*(a+b)/a+2*z^2 + ..)
       //  
       // Beware of underflow! Do not(!) use
       // z:= (fa*beta(fa, fb)*fx)^(1/fa), because
       // fa*beta(fa, fb) may overflow easily for a >> 1
       // Use instead:

       z:= (fa*fx)^(1/fa)*beta(fa, fb)^(1/fa):
       z1:= (fb - 1)/(fa + 1) * z; 
       if specfunc::abs(z1) < macheps then
          return(z);
       end_if;
       // for moderate values of a, b the 
       // following z2 is of the order of
       // magnitude of z^2, i.e., we get a
       // fast result for z < sqrt(macheps) as well:
       z2:= (fa + fb)/(fa + 1)/2 * z1 * z; 
          
       if specfunc::abs(z2) <= macheps then
          return(z * (1 + z1));
       end_if;

       //--------------------------------------
       // Next, do a conclusive check whether
       // the solution z satisfies z < macheps
       //--------------------------------------
       z:=genident():
       if specfunc::ibeta(aa, bb, macheps) >= fx then
          // The solution z is definitely larger than macheps.
          // We have |b-1|/(a+1)*z > macheps and macheps >= z.
          // This can only happen for |b - 1| >= a + 1,
          // --> b - 1 >= a + 1 --> b > a + 2.
          // Well, in this case, accept that
          // numeric::realroots switches
          // to *absolute* precision:
          return(numeric::realroot(
                 hold(specfunc::ibeta)(aa, bb, z) = x,
                 z = 0 .. macheps));
       else
          // the solution z is definitely larger than macheps
          if float(1 - x) > macheps then
             return(numeric::realroot(
                    hold(specfunc::ibeta)(aa, bb, z) = x, 
                    z = macheps .. 1)
                   );

          else // 1 - x <= macheps
               // Use
               //       ibeta(a, b, z) = 1 - ibeta(b, a, 1 - z)
               // to rewrite the equation
               //    ibeta(a, b, z) = x 
               // as
               //    ibeta(b, a, zz) = 1 - x 
               // and return z = 1 - zz.
               // There is no problem, if zz << 1 is computed
               // only to absolute precision, because z = 1 - zz
               // is still accuracte to relative precision.
               // However, 1 - x suffers from cancellation:

               return(1 - numeric::realroot(
                      hold(specfunc::ibeta)(bb, aa, z) = 1 - x, 
                      z = 0 .. 1)
                     );
          end_if;
       end_if; // if specfunc::ibeta(aa, bb, macheps) < fx 
    end_if; // if numericalEvaluation

    // ------------------------------------------
    // if no numerical evaluation is possible, we
    // cannot do anything, because we do not have
    // a symbolic ibeta to represent the result:

    return(hold(stats::betaQuantile)(aa,bb)(x));

  end_proc:
end_proc:
// -------- end of file ---------
