/* ------------------------------------------------------------------------

stats::normalRandom -- produce a generator of normally distributed random numbers

Call(s): 
    stats::normalRandom(m, v, <Seed = n>)

Parameters:
      m -- the mean: a real number
      v -- the variance: a positive real number
      n -- the seed: an integer

Options: 
     With Seed = n, the generator returned by stats::normalRandom
     is initialized with the seed n. With this option, the 
     parameters m, v must both be convertible to real floating
     point numbers.

     Two generators created with the same seed produce the
     same sequence of numbers.

Returns: a procdure of type DOM_PROC
          
Details:
 - stats::normalRandom produces a generator of normal deviates 
   with mean m and variance v;
 - After f:= stats::normalRandom(m,v), the call f() returns a random 
   floating point number.
 - If m or v cannot be converted to real floats (v > 0),
   then normalRandom(m, v) returns a symbolic call.
 - If m or v cannot be converted to real floats (v > 0),
   then normalRandom(m, v, Seed = n) raises an error.
 - The generators produced by normalRandom do not react
   to the global variable SEED!
 - For efficiency, it is recommended to create random sequences
   via
   >> f:= stats::XXXRandom(m, v)
   >> f() $ i = 1..10
   rather than via
   >>  stats::XXXRandom(m, v)() $ i = 1..10
   Also note that
   >>  stats::XXXRandom(m, v, Seed = n)() $ i = 1..10
   does not produce a random sequence, because
   in each call a freshly initialized generator
   would be created, all of them producing the same number

Examples:
 >> f:= stats::normalRandom(0, 1)
 >> f(), f()
                -0.5905080773, -0.4790082948

 >> f:= stats::normalRandom(0, 1, Seed = 1):
 >> f(), f()
                 -1.506518279, 1.274387583
 >> f(), f()
                 -1.312078264, 0.8517156157

 >> g:= stats::normalRandom(0, 1, Seed = 1)
 >> g(), g()
                  -1.506518279, 1.274387583

 >> f:= stats::normalRandom(m, v)

                   stats::normalRandom(m, v)

 >> m:= 0: v:= 1: f(), f(), f()

          -0.3966534696, 1.420342475, -0.7737378823
-----------------------------------------------------------------*/

stats::normalRandom:= proc(m, v)
local fm, fv, r1, r2;
option escape;
begin
  if args(0) < 2 then
     error("expecting at least two arguments")
  end_if:
  if args(0) > 3 then 
     error("expecting no more than three arguments")
  end_if:

  // ------------- check m -------------
  fm:= float(m):
  if domtype(fm) = DOM_COMPLEX then
     error("the mean must be real");
  end_if;

  // ------------- check v -------------
  fv:= float(v):
  if domtype(fv) = DOM_FLOAT and fv <= 0 then
     error("the variance must be positive"):
  end_if;
  if domtype(fv) = DOM_COMPLEX then
     error("the variance must be real");
  end_if;

  // -----------  check option Seed = s ---------------
  if args(0)=3 then
    if type(args(3))<>"_equal" then
       error("the 3rd argument must be of the form 'Seed = integer' or 'Seed = CurrentTime'"):
    end_if:
    if op(args(3),1)<>Seed then
       error("the 3rd argument must be of the form 'Seed = integer' or 'Seed = CurrentTime'"):
    end_if:
    if domtype(op(args(3),2))<>DOM_INT and op(args(3),2)<>CurrentTime then
       error("the 3rd argument must be of the form 'Seed = integer' or 'Seed = CurrentTime'"):
    end_if:
  end_if:

  // ------ unevaluated return ? --------
  if domtype(fm) <> DOM_FLOAT or
     domtype(fv) <> DOM_FLOAT
  then
    if args(0) = 2 then
         return(procname(args()));
    else // do not accept symbolic m and/or v in conjunction
         // with Seed = n, because otherwise the following would
         // happen:
         // delete m, v: f:= stats::normalRandom(m, v, Seed = 1):
         // m:= 0: v:= 1: f(), f(), f()
         //     -1.506518279, -1.506518279, -1.506518279
         error("mean and variance must be numerical ".
               "if 'Seed = ...' is specified"):
    end_if;
  end_if:

  fv:= fv^(1/2): 
  //produce an uc(0,1) random generator: 
  if args(0)=3 then
    if op(args(3),2)=CurrentTime then
      // get an integer number instead:
      r2 := stdlib::frandom(CurrentTime)[1];
      r1:= frandom(r2):
      r2:= frandom(r2 + 10^10):
    else
      r1:= frandom(op(args(3),2)):
      r2:= frandom(op(args(3),2) + 10^10):
    end_if;
  else r1:= frandom:
       r2:= frandom:
  end_if:

  //----------------------------------------------------------
  // the following algorithm is an implementation of the
  // Box-Mueller method using polar coordinates.
  // See: Knuth, Seminumerical Algorithms, Vol. 2, p. 122,123
  //----------------------------------------------------------
  proc()
  local uc1, uc2, s, xmax, x;
  begin
    xmax:= max(100, DIGITS*4):
    for x from 1 to xmax do
      // note that even with Seed = n, two consecutive
      // calls of r = frandom should produce independent
      // random values. Nevertheless, use 2 independent
      // uc(0, 1) generators r1, r2:
      uc1:= 2*r1() - 1: // uc(-1,1) deviate
      uc2:= 2*r2() - 1: // uc(-1,1) deviate
      s:= uc1^2 + uc2^2:
      // The probability of s = 0 is very small. 
      // If s = 0, try again.
      if iszero(s) then next end_if:
      if s < 1 then
        return(fm + fv*uc1*(-2*ln::float(s)/s)^(1/2)):
      end_if:
    end_for:
    //If we arrive here, the mean fm is returned:
    return(fm);
  end_proc:
end_proc:
