/*---------------------------------------------------------------
uniformRandom(a, b) -- generator uniform random generators

Call:        stats::uniformRandom(a, b, Seed = s)
Parameters:  a -- a arithmetical expression
             b -- a arithmetical expression  > a

Returns:  a procedure producing UC(a, b) random floats

Details:   uniformRandom(a, b) = a + (b - a)*frandom
---------------------------------------------------------------*/
stats::uniformRandom:=proc(a, b)
local fa, fbma, r;
option escape;
begin
  if args(0) < 2 then
     error("expecting at least two arguments")
  end_if:
  if args(0) > 3 then 
     error("expecting no more than three arguments")
  end_if:

  // ------------- check a -------------
  fa:= float(a):
  if domtype(fa) = DOM_COMPLEX then
     error("the left border must be real");
  end_if;

  // ------------- check b -------------
  if domtype(float(b)) = DOM_COMPLEX then
     error("the right border must be real");
  end_if;

  fbma:= float(b - a):

  if domtype(fbma) = DOM_FLOAT and fbma < 0 then
     error("the left border must not exceed the right border"):
  end_if;

  // -----------  check option Seed = s ---------------
  if args(0)=3 then
    if type(args(3))<>"_equal" then
       error("the 3rd argument must be of the form 'Seed = integer' or 'Seed = CurrentTime'"):
    end_if:
    if op(args(3),1)<>Seed then
       error("the 3rd argument must be of the form 'Seed = integer' or 'Seed = CurrentTime'"):
    end_if:
    if domtype(op(args(3),2))<>DOM_INT and op(args(3),2)<>CurrentTime then
       error("the 3rd argument must be of the form 'Seed = integer' or 'Seed = CurrentTime'"):
    end_if:
    r:=frandom(op(args(3),2)):
  else
    r:=frandom:
  end_if:

  // ------ unevaluated return ? --------
  if domtype(fa) <> DOM_FLOAT or
     domtype(fbma) <> DOM_FLOAT
  then
    if args(0) = 2 then
         return(procname(args()));
    else // do not accept symbolic a and/or b in conjunction
         // with Seed = s, because otherwise the following would
         // happen:
         // delete a, b: f:= stats::XXXRandom(a, b, Seed = 1):
         // a:= 1: b:= 2: f(), f(), f()
         //     -1.506518279, -1.506518279, -1.506518279
         error("all parameters must be numerical ".
               "if 'Seed = ...' is specified"):
    end_if;
  end_if:

  //-------------------------------
  // return the following procedure
  //-------------------------------

  if iszero(a) and iszero(b - 1) then
       // important special case: a = 0, b = 1.
       // Return frand directly to
       // avoid unnecessary arithmetic
       return(r);
  else return(proc() 
              begin
                 fa + fbma*r(); 
              end_proc
             );
  end_if;
end_proc:
