
/*--

stats::Tdist -- Compute the T distribution

Call:

stats::Tdist(x, v)

x -- value
v -- degrees of freedom (positive integer)

f(u,v) := (1+u^2/v)^(-1/2*v-1/2) * gamma((v+1)/2)/(sqrt(PI*v)*gamma(v/2))

stats::Tdist(x, v) = int(f(u), u = -infinity..x)
                   = int((1+u^2/v)^(-1/2*v-1/2), u = -infinity..x) *
                     gamma((v+1)/2)/(sqrt(PI*v)*gamma(v/2))
--*/

stats::Tdist := proc(x, v)
   local F, r, y, fv;
begin
   if args(0) <> 2 then
      error("Two arguments expected")
   end_if;

   case type(x)
     of DOM_SET do
     of "_union" do
       return(map(x, stats::Tdist, v))
   end_case;

   if testtype(x, DOM_FLOAT) and 
      testtype( (fv:= float(v)), DOM_FLOAT) then
      if fv <=0 then 
         error("Second argument must be positive");
      end_if;
      if x >= 0 then
        return(1 - 1/2*specfunc::ibeta(v/2, 1/2, v/(v+x^2)));
      else
        return(1/2*specfunc::ibeta(v/2, 1/2, v/(v+x^2)));
      end_if;
   end_if;

   if not testtype(v, Type::PosInt) then
      return(procname(args()));
   end_if;

   y:=genident();
   F:=stats::Int(v, y) * gamma((v+1)/2)/(sqrt(PI*v)*gamma(v/2));
   r:=subs(F, y=x, EvalChanges);
   if testtype(x, DOM_FLOAT) then r := float(r) end_if;
   return(r);
end_proc:

stats::Tdist:= funcenv(stats::Tdist):
stats::Tdist:= slot(stats::Tdist, "float", 
                    proc(x, v) begin stats::Tdist(float(x), v) end_proc):
stats::Tdist:= slot(stats::Tdist, "diff", 
  proc(f, y) local x, v; begin   // f = stats::Tdist(x, v)
    x:= op(f, 1);
    v:= op(f, 2);
    1/sqrt(PI)/sqrt(v)*gamma((v+1)/2)/gamma(v/2)*
    1/(1+x^2/v)^( (v+1)/2 ) * diff(x, y);
  end_proc):

/*--
  returns Int(v,x)=int((1+u^2/v)^(-1/2*v-1/2),u=-infinity..x)
  where v is a positive integer.
 
  Int(v,x)=int(1/(1+u^2/v)^(v/2+1/2),u=-infinity..x)
          =sqrt(v)*int(1/(1+u^2)^(v/2+1/2),u=-infinity..x/sqrt(v))
 
  Reference: Calcul formel : mode d'emploi. Exemples en Maple.
        Gomez, Salvy, Zimmermann, 1995, Masson.
--*/
 
stats::Int := proc(v,x) 
   local g,y;
begin
   y:=genident();
   if v mod 2 = 1 then 
      g:=stats::In((v+1)/2,y) else g:=stats::Jn(v/2,y) 
   end_if;
   expand(sqrt(v)*(subs(g,y=x/sqrt(v),EvalChanges)-limit(g,y=-infinity)))
end_proc:
 
/*--
  returns int(1/(1+x^2)^n,x) where n>=1 
--*/

stats::In := proc(n,x)
   option remember; // to accelerate further calls 
begin
   if n=1 then 
      arctan(x)
   else    
      // using integration by parts on 1/(1+x^2)^n, we get the recurrence 
      // I(n)=x/(1+x^2)^n+2*n*I(n)-2*n*I(n+1), see [GoSaZi95] p. 252      
      (x/(1+x^2)^(n-1)+(2*n-3)*stats::In(n-1,x))/(2*n-2)
   end_if
end_proc:

/*--
  returns int(1/(1+x^2)^(n+1/2),x) where n>=1
  Using integration by parts, we get the recurrence
  J(n)=x/(1+x^2)^(n+1/2)+(2*n+1)*J(n)-(2*n+1)*J(n+1)
--*/

stats::Jn := proc(n,x)
  option remember; // to accelerate further calls 
begin
   if n=1 then 
      x/(1+x^2)^(1/2)
   else 
      (x/(1+x^2)^(n-1/2)+(2*n-2)*stats::Jn(n-1,x))/(2*n-1)
   end_if
end_proc:
