/*
combine::gamma:
slot for combining expressions involving the gamma function

We use the rules gamma(-z) * gamma(z) = -PI / (z * sin(PI*z))
and
gamma(z) * gamma(1-z) = PI / sin(PI*z)
(see Abramowitz / Stegun 6.1.17)


On combining gamma(n-z) * gamma(z) for higher integer values, we
would have to exclude the possibility that z is an integer with 0 < z < n

We also use the rule
gamma(z + n)^k * gamma(z)^l = gamma(z)^(k+l) * pochhammer(z, n)^k 

and

z * gamma(z) = gamma(z+1)


This slot does not react to option IgnoreAnalyticConstraints

*/

combine::gamma:=
proc(e, options)
  local gammas, gammapowers, gammaargs, others, factors, bases, dummy,
  i, j, n, z, powdiff, dx, dx2, x, nums, dens, doGauss, done;
begin  
  if type(e) = "_mult" and hastype(e, "gamma") then
    [gammas, others, dummy]:= split([op(e)], x-> type(x) = "gamma" );
    [gammapowers, others, dummy]:= split(others,
                                         x-> type(x) = "_power"
                                         and type(op(x, 1)) = "gamma"
                                         );
    // others:= expr(factor(_mult(op(others))));
    others:= maprat(_mult(op(others)), expr@factor);
    // represent each a^b as [a, b] and non-powers a as [a, 1]
    gammas:= map(gammas, x -> [x, 1]).map(gammapowers, x->[op(x)]);
    // for each gamma(z)^k, extract z
    gammaargs:= map(gammas, x -> op(x[1], 1));
    
    
    i:= 0;
    while (i:= i+1) < nops(gammas) do
      // try to combine with gammas[j] where j > i
      j:= i + 1;
      while j <= nops(gammas) do

        // eliminate all gamma(z)^k * gamma(z+n)^l
        n:= gammaargs[i] - gammaargs[j];
        if domtype(n) = DOM_INT then
          // express both gammas in terms of that one with smaller argument
          if n>=0 then
            gammas[j][2]:= gammas[j][2] + gammas[i][2];
            others:= others*
                     expand(pochhammer(gammaargs[j], n))^gammas[i][2];
            delete gammas[i];
            delete gammaargs[i];
            j:= i+1;
            next // next i
          else          
            gammas[i][2]:= gammas[j][2] + gammas[i][2];
            others:= others*
                     expand(pochhammer(gammaargs[i], -n))^gammas[j][2];
            delete gammas[j];
            delete gammaargs[j];
            next // next j
          end_if
        end_if;

      
        n:= (z:= gammaargs[i]) + gammaargs[j];
        powdiff:= gammas[i][2] - gammas[j][2];
        if n=0 then
          // combine gamma(z)^k * gamma(-z)^l
          if stdlib::hasmsign(powdiff) then
            // swap
            powdiff:= -powdiff;
            z:= -z;
            [gammas[i], gammas[j]]:= [gammas[j], gammas[i]];
            [gammaargs[i], gammaargs[j]]:= [gammaargs[j], gammaargs[i]]
          end_if;
          others:= others *
                   ((-PI) / (z * sin(PI*z)))^gammas[j][2];
          delete gammas[j], gammaargs[j];
          if powdiff = 0 then // both gammas disappear
            delete gammas[i], gammaargs[i];
            j:= i+1;
            next // next i
          else
            gammas[i][2]:= powdiff;
            next // next j
          end_if
        elif n = 1 then
          // combine gamma(z)^k * gamma(1-z)^l
          // = (gamma(z)*gamma(1-z))^k * gamma(1-z)^(l-k)
          // = (PI / sin(PI*z))^k * gamma(1-z)^(l-k)
          others:= others * (PI / sin(PI*z))^gammas[i][2];
          gammas[j][2]:= gammas[j][2] - gammas[i][2];
          if gammas[j][2] = 0 then // delete gammas[j]
            delete gammas[j], gammaargs[j];
          end_if;
          delete gammas[i], gammaargs[i];
          j:= i+1
        else 
          j:= j + 1
        end_if
      end_while;
    end_while;

    //--------------------------------------------------
    // Use Gauss' Mult Formula
    //   gamma(n*x) = (2*PI)^(1/2-n/2)*n^(n*x-1/2)*
    //                product(gamma(x+k/n),k=0..n)
    // to combine products of several gamma calls.
    // The functionality of doGauss is the sideeffect of
    // changing the lists 'gammas', 'gammaargs' and the 
    // expression 'others'. 
    //--------------------------------------------------
    doGauss:= proc(x, n)
    local replace_indices, N, i, j, y, dx, p, g, shifts;
    option remember;
    begin
      if iszero(n) or iszero(n-1) then
         return();
      end_if:
      N:= nops(gammas);
      replace_indices:= [NIL $ N]:
      for i from 1 to N do
        y:= op(gammas[i][1]); // we are dealing with gamma(y)
	dx:= expand(y - x);
	if contains({DOM_INT, DOM_RAT}, domtype(dx)) then
	  if dx < 0 or 1 <= dx then
	    // In gamma(x + dx), reduce dx to the interval [0, 1):
	    // split dx = trunc(dx) + frac(dx).
	    // Use gamma(x + m + dx) = pochhammer(x, m)*gamma(x+dx)
	    // and move the pochhammer term to 'others'.
	    others:= others* pochhammer(x + frac(dx), trunc(dx))^gammas[i][2];
	    dx:= frac(dx);
	    gammas[i][1]:= gamma(x + dx);
	    gammaargs[i]:= x + dx;
	  end_if:

	  replace_indices[i] := [i, dx]:


	  dx:= expand(y - n*x);
          if contains({DOM_INT}, domtype(dx)) then
	    others:= others*pochhammer(n*x, dx)^gammas[i][2];
	    gammas[i][1]:= gamma(n*x);
	    gammaargs[i]:= n*x;
	  end_if;
	end_if:
      end_for;

      replace_indices:= subs(replace_indices, NIL = null(),EvalChanges);

      // Combine 
      //  product(gamma(x + k/n), k = 0..n-1)^p = 
      //      some_factor * gamma(n*x)^p
      // First, determine the smallest power of the
      // gamma term, i.e., finally generate gamma(n*x)^p
      p:= min(gammas[i][2] $ i in map(replace_indices, op, 1));

      //===========================================================
      // Replace the product gamma(x)*gamma(x+1/n)*.. by gamma(n*x)
      //===========================================================
      // Step 1: Find all gamma terms gamma(x + i/n)^expo
      // in the product and reduce the exponents by p.
      // If there is a factor gamma(n*x)^expo, increase the
      // exponent by p:

      g:= [gamma(n*x), p];
      shifts:= {i/n $ i = 0..n-1};
      for i from 1 to nops(gammas) do
         if (j:= contains(map(replace_indices, op, 1), i)) > 0 then
            gammas[i][2]:= gammas[i][2] - p;
	    shifts:= shifts minus {replace_indices[j][2]};
	 end_if;
         y:= op(gammas[i][1]); 
	 if g <> 1 and iszero(y - n*x) then 
            gammas[i][2]:= gammas[i][2] + p;
	    g:= 1;
	 end_if;
      end_for;

      // Step 2: Add factors gamma(x + i/n) for those values
      // i that are not contained in the original expression
      // (they are stored in shifts)
      for dx in shifts do
        gammas:= append(gammas, [gamma(x + dx), -p]);
        gammaargs:= append(gammaargs, x + dx);
      end_for:

      // Add the factor gamma(n*x)^p  (stored in g)
      if g<>1 then // if g = 1, this term was in the original
                   // product and was dealt with before
         gammas:= append(gammas, g);
         gammaargs:= append(gammaargs, n*x);
      end_if;

      others:= others * (2*PI)^(p*(n-1)/2)*n^(p*(1/2 - n*x));

      // clean up:
      for i from 1 to nops(gammas) do
        if gammas[i][2] = 0 then
	   gammas[i]:= NIL;
	   gammaargs[i]:= NIL;
	end_if;
      end_for;
      gammas:= subs(gammas, NIL = null(),EvalChanges):
      gammaargs:= subs(gammaargs, NIL = null(),EvalChanges):
      null():
    end_proc:
    //--------------------------------------------------
    // end of doGauss()
    //--------------------------------------------------

    done:= {};
    j:= 0:
    while j <= nops(gammaargs) do
     j:= j + 1;
     for i from 1 to nops(gammaargs) do
      dx:= map(gammaargs, expand@_subtract, gammaargs[i]);
      dx:= select(dx, x -> contains({DOM_INT, DOM_RAT}, domtype(x)));
      if nops(dx) = 0 then
         next;
      end_if:
      dx:= sort(dx); // after sorting, dx[1] = min(dx) !
      x:= gammaargs[i] +  dx[1];
      dx:= map(dx, _subtract, dx[1]);
      // dens = list of denominators of the rational shifts
      dens:= map(select(dx, x -> (domtype(x) = DOM_RAT)), op, 2); 
      n:= lcm(op(dens));
      if contains(done, [x, n]) then 
         // We did the mult formula gamma(n*x) before
         next;
      else
         // We are currently dealing with the mult formula for gamma(n+x).
	 // Mark this as 'done' for the next tries.
         done:= done union {[x, n]};
      end_if:
      nums:= map(dx, _mult, n);

      assert(map({op(nums)}, domtype) = {DOM_INT});

      // If nums = [0, 1, 2, .., n-1, ...], then we
      // have a perfect match for Gauss' Multiplication Formula.
      // More generally, we can replace nops(dx) gamma calls by
      // a total of (1 + number of missing numerators) gamma calls
      // when using Gauss.
      // Further, if there are factors of the form gamma(n*x + integer) 
      // in the product, they can be written as 
      // pochhammer(nx, integer)*gamma(n*x) and (the power of) gamma(nx) 
      // will absorb/cancel the term produced by the Gauss formula.
      // So, assume they will cancel and subtract the number nops(dx2)
      // of all such terms:
      
      dx2:= map(gammaargs, expand@_subtract, n*x);
      dx2:= select(dx2, x -> contains({DOM_INT}, domtype(x)));
      if 1 + nops({i $ i = 0 .. n-1} minus map({op(nums)}, modp, n)) - nops(dx2) < nops(dx) then
         // We reduce the number of gamma calls by using Gauss.
	 doGauss(x, n);
	 // We need to restart the loop 'for i from 1 to nops(gammaargs)',
	 // since gammaargs may have been changed by doGauss(x, n).
	 break;
      end_if;
     end_for:
    end_while;

    // for each gammaarg, try whether others contains the same factor
    if type(others) <> "_mult" then
      factors:= [others]
    else
      factors:= [op(others)]
    end_if;
    factors:= map(factors, x -> if type(x) = "_power" then
                              [op(x)]
                            else
                              [x, 1]
                            end_if
                );
    bases:= map(factors, op, 1);

    i:= 1;
    while i <= nops(factors) do
      if (j:= contains(gammaargs, bases[i])) > 0 then
        // our input contains some gamma(z)^k and some z^l.
        // we combine only if l>=k
        powdiff:= factors[i][2] - gammas[j][2];
        if testtype(powdiff, Type::NonNegative) = TRUE and powdiff >= 0 then
          // there are enough copies of factors[i]
          z:= gammaargs[j] + 1;
          if (n:= contains(gammaargs, z)) = 0 then
            gammaargs[j]:= z;
            gammas[j][1]:= gamma(z)
          else
            // combine by hand
            gammas[n][2]:= gammas[n][2] + gammas[j][2];
            delete gammas[j], gammaargs[j]
          end_if;
          if powdiff = 0 then
            delete bases[i], factors[i];
            next
          else
            factors[i][2]:= powdiff
          end_if;
        end_if
      end_if;
      i:= i+1
    end_while;
    return(_mult(gammas[i][1]^gammas[i][2] $i=1..nops(gammas)) *
           _mult(factors[i][1]^factors[i][2] $i=1..nops(factors)))
  end_if;
  e
end_proc:

// end of file
