/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    Factored -- domain of objects kept in factored form

    A Dom::Factored object f consists of three operands:
    i)   a list [e,f1,e1,f2,e2,...], where f = e*f1^e1*f2^e2*...
         The entry e and the factors fi are of type R, e1,e2, ... are 
         integers.
    ii)  a string, indicating what is known about the factorization. 
         Currently, the following types are known: 
         "unknown"      --  nothing is known about the factorization
         "irreducible"  --  e is an unit in R, fi are irreducible over R
         "squarefree"   --  e is an unit in R, fi are square-free over R
    iii) the ring R (must be an integral domain).

    Please not that the factors fi are not ordered.

    The following methods are defined:

    ring             --   R
    setRing          --   reset the ring
    factors          --   the fi
    exponents        --   the ei
    expand           --   returns the object [1,f,1]
    getType          --   returns the 'type' of the factorization
    setType          --   sets a type of the factorization
    irreducible      --   returns TRUE if the object is irreducible, or FALSE.
    factor           --   returns a factorization of type "irreducible", if possible
    sqrfree          --   returns a factorization of type "squarefree", if possible
    op(o,...)        --   works on the first operand of o as op for lists
    subsop(o,...)    --   works on the first operand of o as subsop for lists
    _index(o,...)    --   works on the first operand of o as _index for lists  
    set_index(o,...) --   works on the first operand of o as set_index for lists
    subs(o,...)      --   works on the first operand of o as set_index for lists
    expr             --   returns an object of type DOM_EXPR
    convert(o)       --   tries to convert o to an object of dom (with type "unknown")
    convert_to(o,T)  --   tries to convert o to an object of type T
    _concat          --
    select           --
    zip              --
    unapply          --   for overloading the function fp::unapply
    map              --
    iszero           --   zero test
+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

Factored:= newDomain("Factored"):
Factored::create_dom:=hold(Factored):
Factored::interface:= {}:
alias(DOM = Factored):

// -----------------------------------------
//                entries
// -----------------------------------------    
DOM::info:= "domain of objects kept in factored form":

// -----------------------------------------
//                methods 
// -----------------------------------------    
DOM::new:= proc(a,flag="unknown",R=Dom::ExpressionField())
    local i;
begin
    case args(0)
    of 0 do
        error("no arguments given")
    of 3 do
        if domtype(R) <> DOM_DOMAIN or R::hasProp(Cat::IntegralDomain) = FALSE then
            error("expecting a domain (more precisely, an integral domain)")
        end_if
    of 2 do
        if not contains({"squarefree","unknown","irreducible"},flag) then
            error("type must either be \"irreducible\", \"squarefree\" or \"unknown\"")
        end_if;
    of 1 do
        if a = FAIL then 
            return( FAIL ) 
        elif domtype(a) <> DOM_LIST then 
            if domtype(a) <> R then 
                a:= R::coerce(a);
                if a = FAIL then
                    error("not an object of correct type") 
                end_if
            end_if;
            return( new(DOM,[R::one,a,1],flag,R) )
        elif (a:= DOM::convert(a,R)) = FAIL then
            error("not an object of correct type") 
        else
            if nops(a) <> 0 and R::hasProp(Ax::systemRep) and a[1] = -1 then
                // for system domains, avoid factorizations such as -(b-a)*(b+a):
                for i from 1 to nops(a) div 2 do
                    if stdlib::hasmsign(a[2*i]) then
                        // use expr here, because a[j] can be of domain type DOM_POLY!
                        a[1]:= 1;
                        a[2*i]:= -a[2*i];
                        break
                    end_if
                end_for
            end_if;

            return( extsubsop(a,2=flag) )
        end_if
    otherwise
        error("expecting one or two arguments")
    end_case
end_proc:

DOM::create:= proc(x, flag="unknown":DOM_STRING, R=Dom::ExpressionField())
    local i;
begin
    if x = FAIL then 
        return(FAIL) 
    elif domtype(x) = DOM_LIST then
        if nops(x) <> 0 and R::hasProp(Ax::systemRep) and x[1] = -1 then
            // for system domains, avoid factorizations such as -(b-a)*(b+a):
            for i from 1 to nops(x) div 2 do
                if stdlib::hasmsign(x[2*i]) and x[2*i+1] mod 2 = 1 then
                    x[1]:= 1;
                    x[2*i]:= -x[2*i];
                    break
                end_if
            end_for
        end_if;
        return( new(DOM,x,flag,R) ) 
    else
        return( new(DOM,[R::one,x,1],flag,R) )
    end_if
end_proc:

DOM::getRing:= x -> extop(x,3):

DOM::getType:= proc(x:DOM)
begin   
    return( extop(x,2) )
end_proc:

DOM::setType:= proc(x:DOM,t:DOM_STRING)
begin
    return( extsubsop(x,2=t) )
end_proc:

DOM::setRing:= proc(x:DOM,R)
begin
    return( extsubsop(x,3=R) )
end_proc:

DOM::factors:= proc(x)
    local i;
begin
    x:= extop(x,1);
    return( [x[2*i] $ i=1..nops(x) div 2] )
end_proc:

DOM::exponents:= proc(x)
    local i;
begin
    x:= extop(x,1);
    return( [x[2*i+1] $ i=1..nops(x) div 2] )
end_proc:

DOM::factor:=
proc(x:DOM)
  local R;
begin
  if extop(x,2) = "irreducible" then
    return( x )
  else
    R:= extop(x,3);
    x:= R::factor(DOM::convert_to(x,R));
    if x = FAIL then
      return( FAIL )
    elif domtype(x) = DOM then
      // R returns factored-format
      return(x)
    else
      assert(type(x) = DOM_LIST);
      return( new(DOM,x,"irreducible",R) )
    end_if
  end_if
end_proc:

DOM::sqrfree:=
proc(x)
  local R;
begin
  if extop(x,2) = "squarefree" then
    return( x )
  else
    R:= extop(x,3);
    x:= R::sqrfree(DOM::convert_to(x,R));
    if x = FAIL then
      return( FAIL )
    elif domtype(x) = DOM then
      return(x)
    else
      assert(type(x) = DOM_LIST);
      return( new(DOM,x,"squarefree",R) )
    end_if
  end_if
end_proc:

DOM::irreducible:= proc(x)
    local R;
begin
    if extop(x,2) <> "irreducible" then 
        // R is an integral domain; hence, f is irreducible if f is prim!
        R:= extop(x,3);
        return( R::irreducible(DOM::convert_to(x,R)) )
    else
        x:= extop(x,1);
        if nops(x) = 3 and x[3] = 1 then 
            return( TRUE )
        else 
            return( FALSE )
        end_if
    end_if
end_proc:

DOM::expr:= proc(x)
    local i, R;
begin
    R:= extop(x,3);
    x:= extop(x,1); 
    if nops(x) = 1 then 
        return( R::expr(x[1]) )
    else
        return( _mult(R::expr(x[1]),_power(R::expr(x[2*i]),x[2*i+1]) $ i=1..nops(x) div 2) )
    end_if
end_proc:

DOM::print:=
proc(l)
  local i, R, pow;
begin
  R:= extop(l,3);
  l:= extop(l,1); 
  if nops(l) = 1 then 
    return( l[1] )
  else
    pow:= (a,b) -> if b=1 then generate::sortSums(a)
		   else hold(_power)(generate::sortSums(a),b) end_if;
    if expr(l[1]) = 1 then
      if nops(l) = 3 then 
        return(pow(l[2], l[3]))
      else    
        return(hold(_mult)(pow(l[2*i], l[2*i+1]) $ i=1..nops(l) div 2))
      end_if
    elif expr(l[1]) = -1 then
      if nops(l) = 3 then 
        return(hold(_negate)(pow(l[2],l [3])))
      else 
        return(hold(_negate)(hold(_mult)(pow(l[2*i], l[2*i+1])
				$ i=1..nops(l) div 2)))
      end_if
    else
      if contains([DOM_INT, DOM_RAT], type(l[1])) <> 0 then
        return(hold(_mult)(pow(l[2*i], l[2*i+1]) $ i=1..nops(l) div 2, l[1]))
      else
        return(hold(_mult)(l[1], pow(l[2*i], l[2*i+1]) $ i=1..nops(l) div 2))
      end_if
    end_if
  end_if
end_proc:
DOM::expr2text:= l -> "Factored(".expr2text(extop(l)).")":

DOM::isNeg := x -> generate::isNeg(op(x, 1)):
DOM::_negate := x -> extsubsop(x, 1 =subsop(extop(x, 1), 1=_negate(op(x,1)))):
DOM::isInverted := FAIL:
DOM::sortSums := x->x:


DOM::testtype:=
proc(x, T = DOM)
  local i: DOM_INT;
begin
  if T = DOM then
    if DOM::convert(x) <> FAIL then
      TRUE
    else
      FAIL
    end;
  else
    assert(domtype(x) = DOM);
    if T = Type::Arithmetical then
      _lazy_and(testtype(op(x, i), Type::Arithmetical) $i=1..nops(x))
    else 
      FAIL
    end_if
  end_if;
end_proc:

DOM::op:= x -> op( extop(x,1),args(2..args(0)) ):

DOM::nops:= x -> nops( extop(x,1) ):

DOM::subsop:= x -> extsubsop( x,1=subsop( extop(x,1),args(2..args(0)) ),2="unknown" ):

DOM::subs:= x -> extsubsop( x,1=subs( extop(x,1),args(2..args(0)) ),2="unknown" ):

//------------------------------------------------------
// old version
// DOM::_index:= (x,i) -> (_index( extop(x,1),i ): 
//   //  warning("in Factored::_index, called by ".expr2text(context(hold(procname))))
// ):

DOM::_index:= proc(x, i)
                local ext, op1, base, expo;
              begin
                ext := extop(x, 1);
                op1 := op(ext, 1);
                if op1 <> 1 then
                  if i - 1 > nops(ext) div 2 then
                    FAIL
                  elif i = 1 then
                    op1
                  else
                    base := ext[2*(i-1)];
                    expo := ext[2*(i-1)+1];
                    if expo <> 1 then
                      hold(_power)(base, expo)
                    else
                      base
                    end_if
                  end_if
                else
                  // op1 = 1; ignore it
                  if i > nops(ext) div 2 then
                    FAIL
                  else
                    base := ext[2*i];
                    expo := ext[2*i+1];
                    if expo <> 1 then
                      hold(_power)(base, expo)
                    else
                      base
                    end_if
                  end_if:
                end_if:
              end_proc:

//------------------------------------------------------
// old version
// DOM::set_index:=
// proc(x,i,v)
//   local l;
// begin
//   // warning("in Factored::_index, called by ".expr2text(context(hold(procname)))):
//   l:= extop(x,1);
//   l[i]:= v;
//   return(extsubsop( x,1=l,2="unknown" ))
// end_proc:

DOM::set_index:=
proc(x,i,v)
  local l, i_base, i_expo, base, expo;
begin
  l:= extop(x,1);
  if testtype(v, "_power") then
    base := op(v, 1);
    expo := op(v, 2);
  else
    base := v;
    expo := 1
  end_if;
  if l[1] <> 1 then
    // Indexverschiebung, Konstante wird gezhlt
    i_base := 2*(i-1);
    i_expo := 2*(i-1)+1;
  else
    // konstante 1 wird ignoriert
    if i = 1 then
      i_base := 1;
      if expo <> 1 or not testtype(base, Type::Numeric) then
        error("illegal factor")
      else
        l[1] := base;
        return(extsubsop(x, 1 = l, 2 = "unknown"))
      end_if;
    else
      i_base := 2*i;
      i_expo := 2*i +1:
    end_if;
  end_if;
  l[i_base] := base;
  l[i_expo] := expo;
  return(extsubsop( x,1=l,2="unknown" ))
end_proc:

// ??? is this what you want? and what about split? ???
DOM::select:= x -> select(extop(x,1),args(2..args(0))):

DOM::has:= x -> has(extop(x,1),args(2..args(0))):
    
DOM::hastype:= x -> hastype(extop(x,1),args(2..args(0))):

DOM::hasmsign:= x -> stdlib::hasmsign(DOM::convert_to(x,DOM_EXPR)):

DOM::maprec:= x -> misc::maprec( DOM::convert_to(x,DOM_EXPR),args(2..args(0)) ):

DOM::map:= x -> map( DOM::convert_to(x,DOM_EXPR),args(2..args(0)) ):

// ??? what sense does this make ???
DOM::_concat:= proc()
    local u, a;
begin
    a:= [args()];
    if nops(map( {op(a)}, x -> extop(x,2) )) <> 1 or
       nops(map( {op(a)}, x -> extop(x,3) )) <> 1
    then
        error("incompatible operands")
    else
        a:= map(a, x -> extop(x,1));
        u:= _mult( op(map(a, x -> x[1])) );
        a:= map( a, x -> (_delete(x[1]); x) );
        a:= _concat( [u],op(a) );
        return(extsubsop( args(1),1=a,2="unknown" ))
    end_if
end_proc:

DOM::convert_to:= proc(o,T)
    local i, l, R, pow;
begin
    if T = DOM then 
        return( o )
    elif domtype(T) <> DOM_DOMAIN then 
         T:= domtype(T)
    end_if;

    case T
    of DOM do
        return( o )
    of DOM_LIST do
        return( extop(o,1) )
    of DOM_EXPR do
        // return an expression from the list of factors. That works
        // different to 'expr', which would even convert DOM_POLY's to 
        // expressions, for example.
        l:= extop(o,1); 
        if nops(l) = 1 then 
            return( l[1] )
        else
            pow:= (a,b) -> if b=1 then a else hold(_power)(a,b) end_if;
            if expr(l[1]) = 1 then
                if nops(l) = 3 then 
                    return( pow(l[2],l[3]) )
                else    
                    return( hold(_mult)(pow(l[2*i],l[2*i+1]) $ i=1..nops(l) div 2) )
                end_if
            else 
                return( hold(_mult)(l[1],pow(l[2*i],l[2*i+1]) $ i=1..nops(l) div 2) )
            end_if
        end_if
    otherwise
        l:= extop(o,1);
        R:= extop(o,3);
        o:= _mult( l[1],R::_power(l[2*i],l[2*i+1]) $ i=1..nops(l) div 2 );
        if T = R then
            return( o )
        else
            return( T::convert(o) )
        end_if
    end_case
end_proc:

DOM::convert:=
proc(e,R=Dom::ExpressionField())
  local i, n, o,check_msign;
begin
  case domtype(e)
    of DOM do
      return( e )
    of DOM_LIST do
      n:= nops(e);
      case (n:= nops(e))
        of 0 do
          return( new(DOM,[R::one,R::zero,1],"unknown",R) )
        of 1 do
          if (e[1]:= R::coerce(e[1])) = FAIL then
            return( FAIL )
          else
            return( new(DOM,[R::one,e[1],1],"unknown",R) )
          end_if
        otherwise
          if n mod 2 = 0 then
            return( FAIL )
          else
            if (o:= R::coerce(e[1])) = FAIL then
              return( FAIL )
            end_if;
            e[1]:= o;
            if R::hasProp(Ax::systemRep) and o = -1 then
                    // for system domains, avoid factorizations such as -(b-a)*(b+a):
              check_msign:= TRUE
            else
              check_msign:= FALSE
            end_if;

            for i from 2 to n step 2 do
              if (e[i]:= R::coerce(e[i])) = FAIL then
                return( FAIL )
              elif (n:= R::_power( e[i],e[i+1] )) = FAIL then
                return( FAIL )
              elif (o:= R::_mult( o,n )) = FAIL then
                return( FAIL )
              end_if;
              if check_msign and stdlib::hasmsign(e[i]) then
                e[1]:= 1;
                e[i]:= -e[i];
                check_msign:= FALSE
              end_if
            end_for;

            return( new(DOM,e,"unknown",R) )
          end_if
      end_case
    of R do
      return( new(DOM,[R::one,e,1],"unknown",R) )
    otherwise
      if (e:= R::coerce(e)) = FAIL then
        return( FAIL )
      else
        return( new(DOM,[R::one,e,1],"unknown",R) )
      end_if
  end_case
end_proc:

DOM::text2expr:= o -> text2expr(DOM::convert_to(o,DOM_EXPR)):

// DOM::TeX:= o -> generate::TeX(DOM::convert_to(o,DOM_EXPR)):

DOM::TeX :=
proc(o : DOM)
 local num, den, i;
begin
 num := [];
 den := [];
  case expr(op(o, 1))
    of 1 do /* nothing */ break;
    of -1 do num := ["-"]; break;
    otherwise
      num := [generate::tex(op(o, 1), output::Priority::Mult)];
  end_case;
 for i from 1 to (nops(o)-1)/2 do
  if op(o, 2*i+1) < 0 then
    if op(o, 2*i+1) = -1 then
     den := den . [generate::tex(op(o, 2*i), output::Priority::Mult)];
    else
     den := den . [_power::TeX(_power,
                               hold(_power)(op(o, 2*i),
                                            -op(o, 2*i+1)),
                               output::Priority::Mult)];
    end_if;
  else
    if op(o, 2*i+1) = 1 then
     num := num . [generate::tex(op(o, 2*i), output::Priority::Mult)];
    else
     num := num . [_power::TeX(_power,
                               hold(_power)(op(o, 2*i),
                                            op(o, 2*i+1)),
                               output::Priority::Mult)];
    end_if;
  end_if;
 end_for;

  if num = [] then num := "1"
  else
    if num[1] = "-" then
      if num = ["-"] then
	num := ["-1"]
      else
	num[1..2] := [num[1].num[2]];
      end_if;
    end_if;
    num := num[1] . _concat(("\\cdot ",num[i])$i=2..nops(num));
  end_if;

 if den = [] then
  num
 else
  "\\frac{".num."}{".den[1]._concat(("\\cdot ",den[i])$i=2..nops(den))."}";
 end_if;
end_proc:

DOM::expand:= proc(o)
    local i, t;
begin
    o:= extop(o,1);
    t := _mult(_power(o[2*i],o[2*i+1]) $ i=1..nops(o) div 2 );
    if (domtype(t)=DOM_POLY) then
        t := multcoeffs(t, o[1]);
    else
        t := expand(o[1] * t);
    end_if;
    return(t);
end_proc:

DOM::_power:=
proc(x,n)
  local l, i;
begin
  if n::dom = DOM then
    n:= expr(n);
    if x::dom <> DOM then
      return(x^n)
    end_if
  end_if;
  // now we can be sure that x is a Factored and n is not
  if testtype( n,Type::PosInt ) and
    contains( {"irreducible","squarefree"},extop(x,2) )
    then
    l:= extop(x,1);
    l[1]:= l[1]^n;
    (l[2*i+1]:= l[2*i+1] * n) $ i=1..nops(l) div 2;
    return( extsubsop( x,1=l ) )
  else
    _power(DOM::expand(x),n)
  end_if
end_proc:

DOM::_mult:=
proc(x,y)
  local i, p, R, lx, ly, nx, ny, typ, factors;
begin
  case args(0)
    of 0 do
      return( 1 )
    of 1 do
      return( x )
    of 2 do
      case domtype(x)
        of DOM do
          R:= extop(x,3);
          if domtype(y) = DOM then
            if R <> extop(y,3) then
              return(_mult( DOM::convert_to(x,DOM_EXPR),DOM::convert_to(y,DOM_EXPR) ))
            elif extop(x,2) <> "irreducible" or
              extop(y,2) <> "irreducible" then
              typ:= "unknown"
            else
              typ:= "irreducible"
            end_if;
            lx:= extop(x,1);
            ly:= extop(y,1);
            nx:= nops(lx);
            ny:= nops(ly);
            lx[1]:= R::_mult( lx[1],ly[1] );
            delete ly[1];
            
            factors:= [lx[2*i] $ i=1..nx div 2];

            i:= 1;
            while i<nops(ly) do
              if (p:= contains( factors,ly[i] )) <> 0 then
                lx[2*p+1]:= lx[2*p+1]+ly[i+1];
                // delete ly[i] and its multilplicity ly[i+1]
                delete ly[i];
                // index shift, ly[i+1] is now ly[i]
                delete ly[i];
              else
                // go to next entry
                i:=i+2
              end_if
            end_while;
            return( new(DOM,lx.ly,typ,R) )
            elif (p:= R::coerce(y)) = FAIL then
                // do not return FAIL but work as Factor did, 
                // even in this case!
                return(_mult( DOM::convert_to(x,DOM_EXPR),y ))
            elif R = Dom::ExpressionField() and indets(p) minus Type::ConstantIdents = {} 
                 or R <> Dom::ExpressionField() and R::isUnit(p) = TRUE 
            then
                // Handle special case R = Dom::ExpressionField(), where every object
                // is a unit! In fact, the factorization in this case means factorization
                // over a certain polynomial ring, and therefore only numerical constants 
                // should be handled as units!
                lx:= extop(x,1);
                lx[1]:= R::_mult(lx[1],p);
                return( extsubsop(x,1=lx) )
            else
                lx:= extop(x,1);
                if (i:= contains( [lx[2*i] $ i=1..nops(lx) div 2],p )) <> 0 then
                    lx[2*i+1]:= lx[2*i+1] + 1;
                    typ:= extop(x,2) // same type as x
                else
                    lx:= append( lx,p,1 );
                    typ:= "unknown"
                end_if;
                return( new(DOM,lx,typ,R) )
            end_if
        otherwise
            if domtype(y) = DOM then
                R:= extop(y,3);
                if (p:= R::coerce(x)) = FAIL then
                    // do not return FAIL but work as Factor did, 
                    // even in this case!
                    return(_mult( x,DOM::convert_to(y,DOM_EXPR) ))
                elif R = Dom::ExpressionField() and indets(p) minus Type::ConstantIdents = {} 
                     or R <> Dom::ExpressionField() and R::isUnit(p) = TRUE 
                then
                    // Handle special case R = Dom::ExpressionField(), where every object
                    // is a unit! In fact, the factorization in this case means factorization
                    // over a certain polynomial ring, and therefore only numerical constants 
                    // should be handled as units!
                    ly:= extop(y,1);
                    ly[1]:= R::_mult(ly[1],p);
                    return( extsubsop(y,1=ly) )
                else
                    ly:= extop(y,1);
                    if (i:= contains( [ly[2*i] $ i=1..nops(ly) div 2],p )) <> 0 then
                        ly[2*i+1]:= ly[2*i+1] + 1;
                        typ:= extop(y,2) // same type as y
                    else
                        ly:= append( ly,p,1 );
                        typ:= "unknown"
                    end_if;
                    return( new(DOM,ly,typ,R) )
                end_if
            else
                return( _mult(x,y) )            
            end_if
        end_case
    otherwise
      if domtype(x) = DOM then
        return(DOM::_mult( x,_mult(y,args(3..args(0))) ))
      else
        return(_mult( x,_mult(y,args(3..args(0))) ))
      end_if
    end_case
end_proc:

DOM::iszero:= proc(x)
    local l;
begin
    l:= extop(x,1);
    bool( nops(l)=1 and iszero(l[1]) )
end_proc:

DOM::limit:=
proc(f: DOM, x, lp, dir, options)
begin
  limit(expr(f), x = lp, dir, options)
end_proc:

DOM::unapply:= x -> fp::unapply(DOM::expr(x),args(2..args(0))):

DOM::numeric_rationalize:=
proc(x)
  local i, l;
begin
  l:= extop(x, 1);
  for i from 2 to nops(l) step 2 do
    l[i]:= numeric::rationalize(l[i], args(2..args(0)))
  end_for;
  extsubsop(x, 1 = l)
end_proc:

DOM::zero:= FAIL:
DOM::one:= FAIL:
DOM::func_call:= FAIL:
DOM::Name:= DOM::key:
DOM::domtype:= FAIL:
DOM::type:= FAIL:
DOM::evaluate:= FAIL:
DOM::posteval:= FAIL:
DOM::slot:= FAIL:
DOM::undefinedEntries:= FAIL:
DOM::allEntries:= FAIL:
DOM::allAutoEntries:= FAIL:
DOM::whichEntry:= FAIL:
DOM::new_extelement:= FAIL:
DOM::Content:=FAIL:
DOM::MMLContent := FAIL:
DOM::eval:=FAIL:
DOM::level:=FAIL:
DOM::val:=FAIL:
DOM::hold:= FAIL:
DOM::coerce:= FAIL:
DOM::create_dom_elem:= FAIL:
DOM::bool := FAIL:
DOM::evaluateIndex := FAIL:
DOM::Simplify:= FAIL:
DOM::operandsToSimplify:= FAIL:
DOM::normalGroebner:= FAIL:
DOM::sortSums := x -> new(dom, op(map([extop(x)], generate::sortSums))):
DOM::CF:= loadproc(DOM::CF, pathname("GENERATE"),"CF"):

/*--
make_slot -- create default method

make_slot creates a function from the method index by converting the
index to an expression.

NOTE: This MUST be the last method defined for Factored.
--*/
DOM::convDOM2expr:= proc()
begin
    op(map( [args()], proc(x)
        local l, i;
    begin
        if domtype(x)=DOM then 
            l:= extop(x,1); 
            if nops(l) = 1 then 
                return( l[1] )
            else
                return( _mult(l[1],_power(l[2*i],l[2*i+1]) $ i=1..nops(l) div 2) )
            end_if
         else 
            return( x ) 
        end_if 
    end_proc))
end_proc:

DOM::make_slot:= proc(o,s) begin 
    if domtype(s) = DOM_STRING then
        eval(text2expr(s)) @ DOM::convDOM2expr
    else
        return(FAIL)
    end_if
end_proc:

unalias(DOM):

