/*++
Re.mu

    Re -- computes the real part of
              an expression

    For computing the real part of an expression x,
    the method "Re" of the domain of x will be used respectivly.
    If this method does not exist, 'Re(x)' will be
    returned.

    The basis types DOM_INT, DOM_RAT, DOM_FLOAT, DOM_COMPLEX
    and DOM_EXPR be an exception. These types are treated by
    the procedure 'Re'.
    For expressions of type DOM_EXPR the procedure 'Re' looks
    for a function attribute 'Re' respectivly, which will be
    used for x if such exist.
    Otherwise either, if the function environment of x is one
    of the built-in environments "_plus", "_mult" or "_power",
    x will be treated by the procedure 'Re' or the expression
    'Re(x)' will be returned respectivly.
++*/

Re := 
proc(x)
  name Re;
  local t, fx, multSplit, a, ai;
begin
  if (t:= x::dom::Re) <> FAIL then
    return( t(args()) )
  end_if;
  if args(0) <> 1 then
    error("wrong no of args")
  end_if;
  
  case (t:= domtype(x))
    of DOM_POLY do
      return(mapcoeffs(x, Re));
    of DOM_LIST do
    of DOM_ARRAY do
    of DOM_SET do
    of DOM_TABLE do
      return(map(x, Re));
    of DOM_HFARRAY do
      return(hfa::Re(x));
    of DOM_INT      do
    of DOM_RAT      do
    of DOM_FLOAT    do return(x)
    of DOM_COMPLEX  do return(op(x,1))
  end_case;

  if testtype(x,Type::Constant) then
    // is x real ?
    if traperror((fx := float(x))) = 0 and
      domtype(fx) = DOM_FLOAT then
      return(x)
      // is x an indexed identifier ?
    elif type(x) = "_index" then
      return(procname(x))
    end_if
  end_if;
  
  if t = DOM_EXPR then
    t:= eval(op(x,0));
    if domtype(t) = DOM_FUNC_ENV then
      a:= slot(t,"Re")(op(x));
      if a <> FAIL then
        return( a )
      end_if
    elif domtype(t) = DOM_IDENT then
      return( procname(x) )
    end_if;

    multSplit :=
    proc(z)
      local t,a,b,ci,cr,d, mymult: DOM_PROC;
    begin
      
      // local method mymult; we have to set 0*infinity = 0
      // in order to obtain Re(I*infinity) = 0
      mymult:= (x, y) -> x*y;
      mymult(0, infinity):= 0;
      mymult(infinity, 0):= 0;
      mymult(0, -infinity):= 0:
      mymult(-infinity, 0):= 0:


      // split product into real and imaginary parts if possible
      cr:= 1; ci:= 0;
      for t in op(z) do
        a:= Re(t);
        b:= Im(t);
        d:= cr;
        cr:= mymult(cr, a) - mymult(ci, b);
        ci:= mymult(d, b) + mymult(ci, a)
      end_for;
      [cr,ci]
    end_proc:

    case t
      of _union do
      of _plus do
        return( map(x,Re) )
      of _mult do

        // select constant terms
        t:= split( x,testtype, Type::Real);
        if t[1] <> 1 and t[1] <> 1.0 then
          return( t[1]*Re(t[2]*t[3]) )
        end_if;

        // split product into real and imaginary part if possible
        t := multSplit(x);
        // if t <> FAIL then
        if length(t[1]) <= length(procname(x)) then
          return( t[1] )
        end_if;

        // test if the expression is a symbolic complex number
        if (a:= contains( [op(x)],I )) <> 0 then
          // I*ex = (Re(ex)+Im(ex)*I)*I = Re(ex)*I-Im(ex)
          /* note: using 'contains', 1/I = -I will not be considered where
this rule would lead to a wrong result
*/
          return( -Im(eval(subsop(x, a=1))) )
        else
          return( procname(x) )
        end_if
      of _power do
        if op(x,2) = 1/2 then
          case domtype(op(x,1))
            of DOM_INT      do
            of DOM_RAT      do
              if op(x, 1) >= 0 then
                return(x)
              else
                return(0)
              end_if;
            of DOM_FLOAT    do
              // this can only happen if someone has used hold(...)
              if op(x, 1) >= 0 then
                return(x)
              else
                return(0.0)
              end_if;
            of DOM_COMPLEX  do
              t := op( op(x,1) );
              return( sqrt((op(t,1)+sqrt(op(t,1)^2+op(t,2)^2))/2) )
            otherwise
              if is( op(x,1), Type::Real)=TRUE then
                if is(op(x,1)>=0)=TRUE then
                  return(sqrt(op(x,1)))
                elif is(op(x,1)>=0)=FALSE then
                  return(0)
                end_if
              end_if
          end_case
        elif domtype(op(x,2)) = DOM_INT then
          a := Re(op(x,1));
          ai := Im(op(x,1));
          if iszero(ai) then
            return(a^op(x, 2))
          end_if;
          if has(a,{Re,Im}) or has(ai,{Re,Im}) then
            return( procname(x) )
          end_if;
          t := [a,ai];
          if op(x,2) > 0 and op(x, 2) < Pref::autoExpansionLimit() then
            t := multSplit( [(a+I*ai) $ op(x,2)] );
            if t = FAIL then
              return( procname(x) )
            else
              return( t[1] )
            end_if
          elif op(x,2) <= -1 and
            op(x, 2) > -Pref::autoExpansionLimit() then
            if op(x, 2) <> -1 then
              t := multSplit( [(a+I*ai) $ -op(x,2)] );
              if t = FAIL then
                return( procname(x) )
              end_if;
            end_if;
            return(t[1] /(t[1]^2+t[2]^2) )
          else
            break
          end_if;
        elif testtype( op(x,2),Type::Numeric ) then
          t := Re( hold(exp)(op(x,2)*ln(op(x,1))) );
          if type(t) <> "Re" then return( t ) end_if
        end_if
    end_case
  end_if;

  // uses info from assume
  if is(x,Type::Real)=TRUE then return(x) end_if;
  if is(x,Type::Imaginary)=TRUE then return(0) end_if;
  procname(x)
end_proc:


Re := prog::remember(Re, 
  () -> [property::depends(args()), DIGITS]):

Re:= funcenv
(
 Re,
 NIL,
 table( "type"="Re", "print"="Re",
       "Re"=proc(x) name Re; begin procname(x) end_proc, "Im"=0,
       "float"= proc(x) name floatRe; begin Re(float(x)) end_proc,
       "info"="Re(x) -- the real part of x" )
 ):

// minimal property
Re::minprop := Type::Real:

Re::rectform :=
    loadproc(Re::rectform, pathname("STDLIB","RECTFORM"), "Re"):

Re::series := loadproc(Re::series, pathname("SERIES"), "Re"):

Re::lazy :=
proc(x)
begin
  if indets(x) = {} then
    Re(x)
  else
    procname(args());
  end_if
end_proc:

// for plot...
Re::float := funcenv(Re::float):
Re::float::lazy := Re::lazy:

Re::diff := rf -> Re(diff(op(rf), args(2..args(0)))):

Re::hull := () -> Re(hull(args())):

Re::Content := stdlib::genOutFunc("Creal", 1):
Re::getprop := ( x, options ) -> R_:

// simplify slot
Re::simplify := proc(x)
                  local r;
                begin
                  r := Re(expr(rectform(op(x,1))));
                  if not hastype(r, {"Re", "Im"}) then
                    simplify(r)
                  else
                    r
                  end_if
                end_proc:

// end of file
