/*++
    rectform -- A function for computing the rectangular form of 
                complex expressions

    rectform( x )

    x  -- expression

    rectform(x) tries to split x into its real and imaginary
    parts and return x - if possible - in the form x_r + I*x_i.
    If this is not possible then rectform(x) tries to find for each
    subexpression in x such a complex canonical form.

    The result of rectform is an element of the domain 'rectform'.
    Such an element consists of three operands. The first two operands
    give the real and imaginary parts of the expression. The third
    operand represents all subexpressions of x for which such a
    rectangular form can not be computed (possible 0 if there are
    not such subexpressions). 

    The functions 'Re' and 'Im' are overloaded. If the third operand
    is not 0 then 'Re' returns the first operand and 'Im' the
    second operand of an element of type rectform.
    Otherwise the new expressions which results by 'Re' and 'Im' 
    will be computed to their rectangular form.

    rectform maps onto sets, lists, arrays, hfarrays and also onto polynomials
    and series, where rectform maps onto the coefficients.

    Arithmetical operations between ordinary expressions and elements
    of type rectform are possible, whereby the results will be of
    type 'rectform'.

    The MuPAD function 'expr' is overloaded for arguments of type
    'rectform'. If x is of type 'rectform' then 'expr(x)' will return
    the expression  extop(x,1)+I*extop(x,2)+extop(x,3), i.e. an
    expression of type DOM_EXPR.

    The general assumption of rectform is that unkown variables
    represent complex-valued quantities, but rectform make use
    of assumptions on unknowns (see function assume).

    For expressions, for which the 0-th operand is of type DOM_FUNC_ENV,
    rectform will used the function attribute "rectform" by calling
    the attribute with the operands of x, if it exists.

    The following example shows how the function attribute "rectform"
    should be defined:

    sin := slot( sin, "rectform", proc(xx)
        local a, b, x;
    begin
        x := rectform::new(xx);

        if extop(x,3) <> 0 then
            new(rectform,0,0,sin(x))
        else
            a := extop(x,1);
            b := extop(x,2);
            if domtype(a+I*b) = DOM_COMPLEX
            and domtype(a) = DOM_FLOAT or domtype(b) = DOM_FLOAT
            then
                rectform::new(sin(a+I*b))
            else
                new(rectform,sin(a)*cosh(b),cos(a)*sinh(b),0)
            end_if
        end_if
    end_proc):

    rectform treates expressions of type "_plus", "_mult" and "_power"
    and system functions explicitly.


    Examples:

    >> rectform( tan(x) );

                sin(2 Re(x))                 I sinh(2 Im(x))       
        ---------------------------- + ----------------------------
        cos(2 Re(x)) + cosh(2 Im(x))   cos(2 Re(x)) + cosh(2 Im(x))

    >> Im(%);

                              sinh(2 Im(x))        
                       ----------------------------
                       cos(2 Re(x)) + cosh(2 Im(x))

    >> a := rectform(x);

                              Re(x) + I Im(x)

    >> assume( y,Type::Real ):
    >> b := rectform(y);

                                    y

    >> c := a+b;

                           (y + Re(x)) + Im(x) I

    >> assume( x,Type::Real ): 
    >> c;

                           (y + Re(x)) + Im(x) I

    >> rectform(c);

                                  x + y 
++*/

rectform:= newDomain("rectform"):
rectform::create_dom:=hold(rectform):
rectform::Name:= hold(rectform):
rectform::info := 
    "rectform(x) -- splits x into real and imaginary part [try ?rectform for details]":

rectform::new := proc(x) 
    local t, a;
begin
    if args(0) = 0 then
        error("no arguments specified")
    elif x::dom::rectform <> FAIL then 
        return( x::dom::rectform(args()) ) 
    elif args(0) <> 1 then 
        error("only one argument expected") 
    end_if;

    case (t:= domtype(x))
    of rectform        do 
        return( rectform::new(rectform::expr(x)) )
    of DOM_IDENT       do
        if is(x,Type::Real) = TRUE then
            return( new(rectform,x,0,0) )
        else
            return( new(rectform,Re(x),Im(x),0) )
        end_if
    of DOM_INT         do 
    of DOM_RAT         do
    of DOM_FLOAT       do 
        return( new(rectform,x,0,0) )
    of DOM_COMPLEX     do 
        return( new(rectform,op(x,1),op(x,2),0) )
    of DOM_LIST        do
    of DOM_SET         do
    of DOM_ARRAY       do 
        return( map(x,rectform::new) )
    of DOM_HFARRAY do
        return(x)
    of DOM_POLY        do 
        return( mapcoeffs(x,rectform::new) )
    of Series::Puiseux do 
        return( t::map(x,rectform::new) )
    of DOM_EXPR        do
        t := eval(op(x,0));
        if domtype(t) = DOM_FUNC_ENV then
            a := slot(t,"rectform");
            if a <> FAIL then return( a(op(x)) ) end_if
        end_if;
        if t = _plus or t = _mult then
            return( map(x,rectform::new) )
        elif t = _power then
            return(rectform::_power(
                rectform::new(op(x,1)),rectform::new(op(x,2))
            ) )
        end_if
    end_case;

    if not testtype(x,Type::Arithmetical) then
        error("invalid argument, expecting an arithmetical expression")
    else
        return(new(rectform,0,0,x))
    end_if
end_proc:

rectform::expr := proc(x) 
begin 
    if domtype(args(1)) <> rectform then 
        return(x)
    else
        return(_plus( extop(x,1),extop(x,2)*I,extop(x,3) ))
    end_if
end_proc:

rectform::print :=
proc(x)
  local re, im, rest, imSign;
begin
    re := extop(x,1);
    im := extop(x,2);
    if generate::isNeg(im) then
      im := -im;
      imSign := -1;
    else
      imSign := null();
    end_if;
    rest := extop(x,3);
  
    if iszero(re) = TRUE then
        if iszero(im) = TRUE then 
            return(rest)
        elif rest = 0 then 
            return(hold(_mult)(im, I, imSign))
        else 
            return(
                hold(_plus)(hold(_mult)(im, I, imSign), rest)
            )
        end_if
    elif iszero(im) = TRUE then 
        if rest = 0 then 
            return( re )
        else 
            return(hold(_plus)(re, rest))
        end_if
    elif rest = 0 then
        return(
            hold(_plus)(re, hold(_mult)(im, I, imSign))
        )
    else 
        return(
            hold(_plus)(re, hold(_mult)(im, I, imSign), rest)
        )
    end_if
end_proc:

rectform::float := float@rectform::expr:

rectform::_plus:= loadproc(
    rectform::_plus,pathname("STDLIB","RECTFORM"),"_plus"
):

rectform::_mult := loadproc(
    rectform::_mult,pathname("STDLIB","RECTFORM"),"_mult"
):

rectform::_power := loadproc(
    rectform::_power,pathname("STDLIB","RECTFORM"),"_power"
):

rectform::expand := x -> rectform::new(expand( rectform::expr(x),args(2..args(0)) )):

rectform::simplify := x -> rectform::new(simplify( rectform::expr(x),args(2..args(0)) )):

rectform::rewrite := x -> rectform::new(rewrite( rectform::expr(x),args(2..args(0)) )):

rectform::normal := 
proc(x, options) 
begin
  if args(0) >=2 and (type(options) = DOM_TABLE and options[List] = TRUE) or contains({args()}, List) then 
    map(normal( rectform::expr(x), args(2..args(0))), rectform::new)
  else
    rectform::new(normal( rectform::expr(x) ))
  end_if  
end_proc:
  

rectform::diff:= proc(x) 
begin 
    if args(0) > 1 then
        return(rectform::new( diff(rectform::expr(x),args(2..args(0))) ))
    else 
        // to be compatible when calling this method with more
        // than one argument, call rectform again (to consider
        // changes on the property list of involved identifiers):
        return( rectform::new(x) )
    end_if 
end_proc:

/*--
    the method intmult (for the derivative of poly's)

    May be useful, if rectform is used as the
    coefficient domain for polynomials. Anyway,
    that might be a little bit crucial ...
--*/
rectform::intmult := (x,n) -> rectform::_mult(x,n):

rectform::one:= new(rectform,1,0,0):
rectform::zero:= new(rectform,0,0,0):

rectform::Re := proc(x)
begin 
    if extop(x,3) = 0 then
        return( extop(x,1) )
    else
        return( extop(x,1) + Re(extop(x,3)) )
    end_if
end_proc:

rectform::Im := proc(x)
begin 
    if extop(x,3) = 0 then
        return( extop(x,2) )
    else
        return( extop(x,2) + Im(extop(x,3)) )
    end_if
end_proc:

rectform::conjugate := proc(x)
begin 
    if extop(x,3) = 0 then
        return(new( rectform,extop(x,1),-extop(x,2),0 ))
    else
        return(new( rectform,extop(x,1),-extop(x,2),conjugate(extop(x,3)) ))
    end_if
end_proc :

rectform::abs := proc(x)
begin 
    if extop(x,3) = 0 then
        return(new( rectform,sqrt(extop(x,1)^2+extop(x,2)^2),0,0 ))
    else
        return(new( rectform,abs(rectform::expr(x)),0,0 ))
    end_if
end_proc:

rectform::sign := proc(x)
    local a;
begin 
    if extop(x,3) = 0 then
        a:= sqrt(extop(x,1)^2+extop(x,2)^2);
        return(new( rectform,extop(x,1)/a,extop(x,2)/a,0 ))
    else
        return(new( rectform,0,0,sign(rectform::expr(x)) ))
    end_if
end_proc:

rectform::iszero := proc(x)
begin 
    if iszero(extop(x,1)) = TRUE and iszero(extop(x,2)) = TRUE and iszero(extop(x,3)) = TRUE then
        return( TRUE )
    else
        return( FALSE )
    end_if
end_proc:

rectform::op := proc(x,i)
begin
    if args(0) = 1 then
        return( extop(x,1..3) )
    else
        return( extop(x,i) )
    end_if
end_proc:

rectform::nops := () -> 3:

rectform::expr2text := proc(x)
    local re, im, f;
begin
    re:= extop(x,1); im:= extop(x,2);
    f:= extop(x,3);

    if f = 0 then
      f:= ""
    elif iszero(im) and iszero(re) then
      return("rectform(".expr2text(f).")")
    else
      f := " + ".expr2text(f)
    end;

    if iszero(im) then
        return("rectform(".expr2text(re).f.")")
    elif iszero(re) then
        return("rectform(".expr2text(I*im) . f.")")
    else
        return("rectform(".expr2text(re)." + ".expr2text(I*im) .f.")")
    end
end_proc:

rectform::TeX := proc(x)
    local re, im, f;
begin
    re:= extop(x,1); im:= extop(x,2);
    f:= extop(x,3);

  if f = 0 then f:= ""
  else
    if stdlib::hasmsign(f) then
      f := generate::TeX(f)
    else
      f := " + ".generate::TeX(f)
    end;
  end;
  if iszero(im) then
    return(generate::TeX(re) . f)
  elif iszero(re) then
    return(generate::TeX(I*im) . f)
  else
    if stdlib::hasmsign(im) then
      return(generate::TeX(re)." - ".generate::TeX(-I*im) . f)
    else
      return(generate::TeX(re)." + ".generate::TeX(I*im) . f)
    end_if;
  end
end_proc:

rectform::has := (x,e) -> has( {extop(x)},e ):

rectform::subs := x -> rectform::new(subs( rectform::expr(x),args(2..args(0)) )):

rectform::subsex := x -> rectform::new(subsex( rectform::expr(x),args(2..args(0)) )):

rectform::map := x -> rectform::new(map(rectform::expr(x), args(2..args(0)))):

rectform::length := x -> length(rectform::expr(x)):

rectform::testtype:= 
proc(x,T) 
begin 
    if T = rectform then 
       // the case x::dom = rectform is handled by the kernel
      FALSE  
    elif T = Type::Arithmetical then
      TRUE
    else 
      return( testtype(rectform::expr(x),T) )
    end_if
end_proc:

rectform::convert:= proc(x)
begin
    if domtype(x) = rectform then
        return(x)
    elif testtype( x,Type::Arithmetical ) then
        return(rectform::new(x))
    else
        return( FAIL )
    end_if
end_proc:

rectform::convert_to := (x,T) -> T::convert(rectform::expr(x)):

// define some methods to avoid creation via make_slot 
// -- basic methods, should be the same for other domains,
// -- which overload 'make_slot':
rectform::func_call:= FAIL:
rectform::Name:= rectform::key:
rectform::domtype:= FAIL:
rectform::evaluate:= FAIL:
rectform::posteval:= FAIL:
rectform::slot:= FAIL:
rectform::undefinedEntries:= FAIL:
rectform::allEntries:= FAIL:
rectform::allAutoEntries:= FAIL:
rectform::whichEntry:= FAIL:
rectform::new_extelement:= FAIL:
rectform::Content:=FAIL:
rectform::MMLContent:=FAIL:
rectform::eval:=FAIL:
rectform::level:=FAIL:
rectform::val:=FAIL:
rectform::hold:= FAIL:
rectform::maprec:= FAIL:
// -- special methods for rectform, which should not be
// -- overloaded automatically:
rectform::rectform:= FAIL:
rectform::type:= FAIL:
rectform::isNeg:= FAIL:
rectform::isInverted:= FAIL:
rectform::interface:= {}:
rectform::exported:= {}:
rectform::create_dom_elem:= FAIL:
rectform::bool := FAIL:
rectform::TeXrep := FAIL:
rectform::evaluateIndex := FAIL:
rectform::sortSums := x -> new(dom, op(map([extop(x)], generate::sortSums))):
rectform::CF:= loadproc(rectform::CF, pathname("GENERATE"),"CF"):

// needed by Simplify:
rectform::operandsToSimplify:= FAIL:

/*--
make_slot -- create default method

make_slot creates a function from the method index by converting the
index to an expression; arguments which are 'rectform's are converted
to expressions before the function is called. 

NOTE: This MUST be the last method defined for rectform.
--*/
rectform::args2expr:=
proc(x)
begin
    if args(0) = 1 then
        if domtype(x) = rectform then
            return(rectform::expr(x))
        else
            return(x)
        end_if
    else
        return(
            op( map([args()], x -> if domtype(x) = rectform then rectform::expr(x) else x end) )
        )
    end_if
end_proc:

rectform::make_slot:=
proc(e,s)
begin
    if domtype(s) = DOM_STRING then
        return(eval(text2expr(s)) @ rectform::args2expr)
    else
        return(FAIL)
    end_if
end_proc:

rectform::laplace := rectform::make_slot(FAIL, "transform::laplace"):
rectform::ztrans  := rectform::make_slot(FAIL, "transform::ztrans"):
rectform::fourier := rectform::make_slot(FAIL, "transform::fourier"):
rectform::numeric_rationalize := rectform::make_slot(FAIL, "numeric::rationalize"):

rectform::maprec  := x -> misc::maprec(rectform::args2expr(x), args(2..args(0))):



prog::setcheckglobals(rectform, {_X, _Y}):
