/* stdlib::logic::simplify(e) simplifies the boolean expression e
  Input: e is a boolean expression (with variables)
  Output: a simplified expression
  Examples: logic::simplify(a or a and b) -> a
            logic::simplify(b and c or not c) -> b or not c
            logic::simplify(not a and c or b and not c or not b and not c)
		-> not a or not c
            logic::simplify((a and b) or (a and (not b))) -> a
            logic::simplify((a and b) or (not a and c) or (b and c)) 
		-> a and b or not a and c

May 1996: added logic::evalbool to evaluate constant terms with bool
>> simplify(x and y and (x or y) and 2 < 3, logic);

                                  x and y

  References:
  [1] McCluskey E. J., Introduction to the Theory of Switching Circuits,
      chap. 4, pp 165--179, McGraw-Hill, New York, 1965.
  [2] Williams G. E., Boolean algebra with computer applications,
      chap. 4, pp 78--105, McGraw-Hill, New York, 1970.
  [3] Hill, F. J., Introduction to switching theory and logical design,
      pp 97--137, John Wiley and Sons, Tucson, 1968.

  The function logic::simplify simplifies a given boolean expression into a sum
  of prime implicants, by using the McCluskey algorithm (see [1]).

  This algorithm makes use of the absorption-theorem and the consensus-theorem.	
  The original expression will be converted into a sum of product-terms"...in	
  which the consensus of any two terms is included in some other term and in	
  which no term includes any other."(reference [1], page 171).			
  This sum is called a complete sum. It contains all prime implicants. Some of	
  them are superfluous and can be found with the consensus theorem, but this	
  algorithm does not succeed in finding the minimal sum. For that "a		
  satisfactory general method is not yet available." (reference [1], page 174).	

  Here are two examples where McCluskey algorithm does not find a minimal expression:
  A:=(not a) and (not c) or b and (not c) and d or (not a) and (not b) and d or
     (not a) and b and (not d) or (not b) and (not c) and (not d);
  The algorithm leaves A unchanged, whereas an equivalent simpler expression is
  Asimp:=b and (not c) and d  or  (not a) and (not b) and d  or
         (not a) and b and (not d)  or  (not b) and (not c) and (not d);

  C:=(not a) and c and d   or  b and (not c) and d  or
     (not a) and (not c) and (not d)  or  b and c and (not d);
  The algorithm returns C or (not a and b), which is more complex.
*/ 

alias(logic = stdlib::logic):

logic:=newDomain("logic"):

logic::simplify:=
proc(A) //  main procedure (see comment above)	
begin
  if testtype(A, Type::Boolean) <> TRUE then
    error("Input must be a boolean expression")
  end_if;
  A:=rewrite(A, andor);
  A:=expand(A);
  A:=logic::converttoset(A);
  A:=logic::primeimplic(A);
  A:=logic::reduce(A);
  A:=logic::converttobool(A);
end_proc:

logic::cmp:=proc(Mi,Mj) begin bool(nops(Mi) < nops(Mj)) end_proc:

logic::cover:=proc(M)	// cover reduces a boolean expression 	
local i,L,R,monomset;	// by using the absorption-theorem	
begin
    L:=sort([op(M)] ,logic::cmp);  
    R:={};
    for i from 1 to nops(L) do
	if contains(M,L[i]) then
    	    R:=R union {L[i]};
            M:=M minus {L[i]};
            for monomset in M do
        	if (L[i] minus monomset)={} then 
				M:=M minus {monomset};
		end_if;
            end_for;
        end_if;
    end_for;
    R;
end_proc:

logic::positiv:=proc(monomset)				
local pos,literal;
begin
        pos:={};
	    for literal in monomset do
			pos:=pos union {op(literal)}
	    end_for;
    	pos;
end_proc:

logic::consensus:=proc(Mi,Mj)	// consensus applies the consensus-theorem to two monoms 
local k,Mij,literal,R;  	// and returns the product expression obtained		
begin
    Mij:=logic::positiv(Mi) union logic::positiv(Mj);
    R:={};  k:=0;
    for literal in Mij do 
	if  contains(Mi,literal) and contains(Mj,not literal) or 
		contains(Mi,not literal) and contains(Mj,literal) then
 		k:=k+1;
		if k>1 then return(FALSE) end_if;
	else
		if contains(Mi,literal) or contains(Mj,literal) then
			R:=R union {literal}	
		else 
			R:=R union {not literal}
		end_if;
	end_if;
    end_for;
    if k=0 then return(FALSE) else return(R) end_if;
end_proc:

logic::include:=proc(Mi,Mj)   		
begin 						
	bool((Mi minus Mj)={}) 
end_proc:

logic::generate:=proc(M)			
local i,j,L,monomset;			
begin							 
	L:=[op(M)];
	for i from 1 to nops(L) do 
		for j from i+1 to nops(L) do
			monomset:=logic::consensus(L[i],L[j]);
			if monomset<>FALSE 
			   and not contains({op(map(L,logic::include,monomset))},TRUE) then 
				   return(monomset);
			end_if;
		end_for;
	end_for;
	FALSE;
end_proc:

logic::primeimplic:=proc(M) // primeimplic generates all prime implicants 
local add;							
begin	
    M:=logic::cover(M); 
    add:=logic::generate(M);
    while add<>FALSE do
	M:=M union {add};
	M:=logic::cover(M);
	add:=logic::generate(M);
    end_while;
    M;
end_proc:
	
logic::surplus:=proc(M)	   
local i,j,sp,L;		
begin 
	L:=[op(M)];
	for i from 1 to nops(L) do
		for j from i+1 to nops(L) do
			sp:=logic::consensus(L[i],L[j]);
			if sp<>FALSE and contains({op(L)},sp) then
				return(sp);
			end_if;
		end_for;
	end_for;
	FALSE;
end_proc:

logic::reduce:=proc(M)	// removes the monoms obtained by applying	
local DELETE;		// the consensus-theorem to any two monoms	
begin 					
    DELETE:=logic::surplus(M);
    while DELETE<>FALSE do
	M:=M minus {DELETE};
	DELETE:=logic::surplus(M);
    end_while;
    M;
end_proc:

logic::converttoset:=proc(A)	// converts a sum of products into a set of	
local i,H;			// sets containing the product literals 		
begin
        traperror((A:=misc::maprec(A,
		{"_less","_leequal","_unequal","_equal"}=logic::evalbool)));
	H:={};
        case type(A)
	of "_or" do
        of "_union" do
		for i from 1 to nops(A) do
			if type(op(A,i))="_and" then
            			H:=H union { {op(op(A,i))} };
			else
				H:=H union { {op(A,i)} };
			end_if;
		end_for; break
	of "_and" do
        of "_intersect" do
		H:={ {op(A)} }; break
	of "_minus" do H:={{op(A,1),not op(A,2)}}; break
	otherwise H:={{A}}
        end_case;
	H
end_proc:

// try to evaluate with bool 
logic::evalbool:= 
proc(A)
   local a;
begin
  case type(A)
    of "_equal" do
      if bool(A) then TRUE else A end_if;
      break
    of "_unequal" do
      if not bool(A) then FALSE else A end_if;
      break
    otherwise
      if traperror((a:= bool(A))) = 0 then a else A end_if;
  end_case
end_proc:


logic::converttobool:=proc(A)	// converts a set of sets containing	
local k,monom,B;		// literals into a sum of monoms 	
begin
	B:=FALSE;
	for k from 1 to nops(A) do
		if nops(op(A,k))=1 then  
			monom:=op( op(A,k) );
		else 
			monom:=_and(op( op(A,k) ));
		end_if;
		B:=_or(B,monom);
	end_for;
	B;
end_proc:
				
simplify::logic := logic::simplify:
