/* The simplify attribute of cosh

   Rules:
   1) cosh(x)            -> apply simplify and stdlib::normalizesign to x
   2) cosh(x + k*PI*I/2) -> I^k * cosh/sinh (x + frac(k)*PI*I/2)
                            if domtype(k) = DOM_RAT
   3) cosh(x + k*PI*I/2) -> I^k * cosh/sinh (x) if is(k, Type::Residue(?,2)
                            or is(k, Type::Residue(?,4))
   4) cosh(x)            -> cosh(-x) if stdlib::hasmsign(x) = TRUE
*/

cosh := slot(cosh, "simplify", proc(a)
   name cosh::simplify;
   local x, lnx, c, l, i, e, rest, 
         symbolic_expo, integer_expoI,
         prop, tmp;
begin
   if type(a) <> "cosh" then
         return(a);
   end_if:

   // use cosh(-x) = cosh(x) to normalize the sign of x:
   x := stdlib::normalizesign(op(a, 1))[2]:

   if has(x, ln) then
      if type(x) = "ln" then // cosh(ln(x)) = x/2 + 1/2/x
         return(normal((op(x)^2 + 1)/(2*op(x))));
      elif type(x) = "_mult" then // cosh(c*ln(x)) = (x^(2*c) + 1)/(2*x^c)
          lnx:= select(x, testtype, "ln");
          c:= x/lnx;
          if testtype(c, Type::Constant) and
             type(lnx) = "ln" and
             not has(c, I) then
               x:= op(lnx)^c;
               return(normal((x^2 + 1)/(2*x)));
          end_if;
      end_if:
      // Finally, take care of cosh(c1*ln(x1) + c2*ln(x2) + ...).
      // exp::simplify has (rather complicated) code for this kind
      // of simplification. Call it to let it do the job:
      c:= expand(x, ArithmeticOnly);
      if (not has(c, I)) and type(c) = "_plus" then
         tmp:= misc::maprec(rewrite(cosh(c), exp), {"exp"} = exp::simplify):
          if not hastype(tmp, "ln") then 
            return(tmp);
          end_if;
      end_if:
   end_if:

   // No need to react to arccosh etc., since cosh itself
   // handles this (without simplify)

   if not has(x/I, I) then // cosh(I*y) = cos(y)
      return(cos(x/I))
   end_if;

   //------------------------------------
   // treat shifts by multiples of I*PI/2
   //------------------------------------
   // final symbolic exponent of -1
   symbolic_expo := 0;
   /* final integer exponent of I */
   integer_expoI := 0;
   // finally remaining arguments
   rest := 0;

   // l := list of summands
   if type(x) = "_plus" then
     l := [op(x)]
   else
     l := [x]
   end_if;

   // l := list of pairs [summand, summand*2/I/PI]
   l := map(l, proc(x) begin [x, x*2/I/PI] end_proc);

   /* check whether there is a summand with domtype(summand*2/I/PI) =
      DOM_RAT or DOM_INT (there can be at most one) */
   l := split(l, proc(x) begin
     contains({DOM_RAT, DOM_INT}, domtype(op(x, 2))) end_proc);
   if nops(l[1]) > 0 then
     // reduce the rational number modulo 1
     tmp := op(l, [1,1,2]);
     rest := frac(tmp)*I*PI/2;
     integer_expoI := floor(tmp) mod 4;
   end_if;

   // select those summands with is(summand*2/I/PI, Type::Integer) = TRUE
   l := split(l[2], proc(x) begin is(op(x, 2), Type::Integer) end_proc);

   // rest := unsimplified arguments
   rest := _plus(op(map(l[2].l[3], op, 1)), rest);

   // process arguments of Type::Integer
   l := l[1];
   for i from 1 to nops(l) do
     e := op(l, [i, 2]);

     // simplify exponents that are known to be in some residue class mod 4
     prop := getprop( e mod 4 );
     if type(prop)=DOM_SET then
       if nops(prop)=1 then
         // e is in the residue class of Rem mod 4
         integer_expoI := integer_expoI + op(prop) mod 4;
       elif prop minus {1, 3} = {} then
         // e is odd
         integer_expoI := integer_expoI + 1 mod 4;
         symbolic_expo := symbolic_expo + (e - 1)/2;
       elif prop minus {0, 2} = {} then
         // e is even
         integer_expoI := integer_expoI;
         symbolic_expo := symbolic_expo + e/2;
       else
         rest := rest + op(l, [i, 1]);
       end_if
     else
       rest := rest + op(l, [i, 1]);
     end_if
   end_for;

   /* extract a minus sign from rest if necessary */
   if stdlib::hasmsign(rest) then
     rest := -rest;
     integer_expoI := - integer_expoI mod 4;
   end_if;

   /* determine final result according to the formula
      cosh(x+I*k*PI/2) = cos(-I*x+k*PI/2) */
   case modp(integer_expoI, 4)
   of 0 do
     return((-1)^(symbolic_expo) * cosh(rest))
   of 1 do
     return((-1)^(symbolic_expo) * I*sinh(rest))
   of 2 do
     return((-1)^(symbolic_expo) * -cosh(rest))
   of 3 do
     return((-1)^(symbolic_expo) * -I*sinh(rest))
   end_case;
   error("should never happen")
end_proc):
