/* The simplify attribute of exp

   Rules:
   1) exp(x)            -> apply simplify and stdlib::normalizesign to x
   2) exp(x + k*PI*I/2) -> I^floor(k) * exp(x + frac(k)*PI*I/2)
                           if domtype(k) = DOM_RAT or DOM_INT
   3) exp(x + k*PI*I/2) -> I^k * exp(x) if is(k, Type::Integer)
   4) exp(x + k*ln(y))  -> y^k * exp(x)
*/

exp::simplify:=
proc(a)
  name exp::simplify;
  local x, l, i, e, rest, symbolic_expo, symbolic_expoI, integer_expoI,
  prop, tmp, l1, p, k, li;
begin
  // recursively simplify the argument
  x := op(a);
  x := _mult(op(stdlib::normalizesign(x))):

  // final symbolic exponent of -1
  symbolic_expo := 0;
  // final symbolic exponent of I
  symbolic_expoI := 0;
  // final integer exponent of I
  integer_expoI := 0;
  // finally remaining arguments of exp
  rest := 0;
  // product of factors pulled out of exp
  p := 1;

  // l := list of summands
  if type(x) = "_plus" then
    l := [op(x)]
  else
    l := [x]
  end_if;

  // l := list of pairs [summand, summand*2/I/PI]
  l := map(l, proc(x)
              begin [x, x*2/I/PI] end_proc);

   /* check whether there is a summand with domtype(summand*2/I/PI) =
      DOM_RAT or DOM_INT (there can be at most one) */
  l := split(l, proc(x)
                begin
                  contains({DOM_RAT, DOM_INT}, domtype(op(x, 2)))
                end_proc);
  if nops(l[1]) > 0 then
     // reduce the rational number modulo 1
    tmp := op(l, [1,1,2]);
    rest := frac(tmp)*I*PI/2;
    integer_expoI := floor(tmp) mod 4;
  end_if;

   // select those summands with is(summand*2/I/PI, Type::Integer) = TRUE
  l := split(l[2], proc(x)
                   begin
                     is(op(x, 2), Type::Integer)
                   end_proc);

   // select those summands containing logs or LambertW's
  l1 := split(l[2].l[3],
              s -> has(s, ln) or has(s, lngamma) or has(s, lambertW) or has(s, wrightOmega));

   // rest := unsimplified arguments
  rest := _plus(op(map(l1[2], op, 1)), rest);

   // process arguments of Type::Integer
  l := l[1];
  for i from 1 to nops(l) do
    e := op(l, [i, 2]);

    // simplify exponents that are known to be in some residue class mod 4
    prop := getprop( e mod 4 );
    if type(prop)=DOM_SET then
      if nops(prop)=1 then
        // e is in the residue class of Rem mod 4
        integer_expoI := integer_expoI + op(prop) mod 4;
      elif prop minus {1, 3} = {} then
        // e is odd
        integer_expoI := integer_expoI + 1 mod 4;
        symbolic_expo := symbolic_expo + (e - 1)/2;
      elif prop minus {0, 2} = {} then
        // e is even
        integer_expoI := integer_expoI mod 4;
        symbolic_expo := symbolic_expo + e/2;
      else
        symbolic_expoI := symbolic_expoI + e;
      end_if;
    else
      symbolic_expoI := symbolic_expoI + e;
    end_if
  end_for;

   // process arguments of the form k * ln(foo) or k * lngamma(foo)
  l := l1[1];
  for i from 1 to nops(l) do
    li := op(l, [i, 1]);
    k := 1;
    if type(li) = "_mult" then
      li := split(li, has, ln);
      k := li[2];
      li := li[1];
    end_if;
    if type(li) = "ln" then
      p := p * op(li)^k
    elif type(li) = "lngamma" then
      p := p * gamma(op(li))^k
    else
       // try to recognize a lambertW
       // since exp(W(x)) * W(x) = x,
       // we have that exp(k*W(x)) * W(x)^k = x^k
       // thus we write exp(k*W(x)) = x^k / W(x)^k
      li:= op(l, [i, 1]);
      k:= 1;
      if type(li) = "_mult" then
        li := split(li, testtype, "lambertW");
        k := li[2];
        li := li[1];
      end_if;
      if type(li) = "lambertW" then
        p := p * op(li, 2)^k / li^k
      else
        li:= li*k;
        // try to find wrightOmega's
        if type(li) = "_mult" then 
          li:= split(li, testtype, "wrightOmega");
          k:= li[2];
          li:= li[1]
        else
          k:= 1
        end_if;
        if type(li) = "wrightOmega" then      
          // since omega(x) = lambertW(.., exp(x)), we apply the rule from above
          p:= p * exp(op(li, 1)*k) / li^k
        else
          rest := rest + k*li;
        end_if;
      end_if
    end_if;
  end_for;

   // return final result
  if (symbolic_expoI = 0) then
    //  I^integer_expoI * (-1)^(symbolic_expo) * exp(rest) * p
    (-1)^(integer_expoI/2 + symbolic_expo) * exp(rest) * p
  else
    // I^integer_expoI * I^(symbolic_expoI + 2*symbolic_expo) * exp(rest) * p
    (-1)^(integer_expoI/2 + symbolic_expoI/2 + symbolic_expo) * exp(rest) * p
  end_if
end_proc:
