//    

/*--
	ln/simplify -- the function attribut "simplify" for ln

	Rules (take properties into account):

	1) ln( posint1/posint2 * r )  -->  ln(posint1) - ln(posint2) + ln(r)
	2) ln( exp(real) * r )        -->  real + ln(r)
        3) ln( positive^real * r )    -->  real*ln(positive) + ln(r)
           ln( negative^even * r )    -->  even*ln(-negative) + ln(r)
           ln( negative^odd  * r )    -->   odd*ln(-negative) + ln(-r)
        4) ln( positive * r )         -->  ln(positive) + ln(r)
           ln( negative * r )         -->  ln(-negative) + ln(-r)

        5) ln( y^n )                  -->  n*ln(y) if
                                           -1 < n <= 1 or
                                           n = -1 and is(y <= 0) = FALSE or
                                           -2 <= n <= 2 and Re(y) > 0
        6) ln( exp(z) )               -->  z if -PI < Im(z) <= PI

        //7) ln( x^(p/q)/y )            -->  1/q * ln(x^p/y^(q/p))
        //   -> detour: ln::simplify | combine | ln::simplify

        8) ln(wrightOmega(x))         -->  x - wrightOmega(x)

        The ideology is to have only positive arguments in ln's where
        possible.
--*/

ln::simplify :=
proc(x: "ln", options = simplify::defaultOptions: DOM_TABLE)
  local l, t, r, k, z, y, n;
begin


  l:= op(x);

  // handle rule 8) first
  if type(l) = "wrightOmega" then
    // the functional equation ln(omega(x)) + omega(x) = x does not hold
    // for Re(x) <= -1 and Im(x) = -PI
      return(piecewise(
           [Im(op(l, 1)) = -PI and Re(op(l, 1)) <= -1, op(l, 1) - l + 2*PI*I],
           [Otherwise, op(l, 1) - l]
           ));
   
    /*
    if is(Im(op(l, 1)) = -PI and Re(op(l, 1)) <= -1, Goal = TRUE) then
       return(op(l, 1) - l + 2*PI*I)
    end_if;
    return(op(l, 1) - l)
    */
  end_if;
  
  // convert the result into a list of factors
  if type(l) = "_mult" then
    l := [op(l)];
  else
    l := [l];
  end_if;

  // determine all factors that are recognized as positive or negative by "is"
  if options[IgnoreAnalyticConstraints] then
    r:= 1;
    // assume that everything will be real
  else  
    l := split(l,
               proc(a)
               begin
                 is(a > 0) or is(a < 0)
               end_proc);

    r := op(l[2])*op(l[3]);
    l := l[1];
  end_if;
    
  // now x = ln(_mult(op(l)) * r), and every element of l "is" either >0 or <0

  // handle positive or negative factors
  k := 0;
  for t in l do
    case type(t)

      of DOM_INT do
        if t < 0 then
          t:= -t;
          r:= -r
        end_if;
        y:= numlib::ispower(t);
        if y <> FALSE then
          k:= k + op(y,2)* ln(op(y, 1))
        else
          k:= k + ln(t)
        end_if;
        break
        
      of DOM_RAT do // rule 1)
      // ln(numer/denom) = ln(numer) - ln(denom)
      if t < 0 then
        t := -t;
        r := -r;
      end_if;
      k := k + ln(op(t, 1)) - ln(op(t, 2));
      break;
    of "exp" do // rule 2)
      if options[IgnoreAnalyticConstraints] or is(op(t) in R_, Goal = TRUE) then
	// t "is" positive and ln(t) = op(t)
        k := k + op(t);
      else
        r := r * t;
      end_if;
      break;

    of "_power" do // rule 3)
      [y, n] := [op(t)];
      if options[IgnoreAnalyticConstraints] then
        k := k + n*ln(y)
      elif is(y > 0) = TRUE then
        // ln(y^n * r) = n*ln(y) + ln(r) if y > 0 and n is real
        if is(n, Type::Real) = TRUE then
          k := k + n*ln(y);
        else
          r := r * t;
        end_if;
      elif is(y < 0) = TRUE then
        // ln(y^n * r) = n*ln(-y) + ln(r) if y < 0 and n is even
        // ln(y^n * r) = n*ln(-y) + ln(-r) if y < 0 and n is odd
        if is(n, Type::Even) = TRUE then
          k := k + n*ln(-y);
        elif is(n, Type::Odd) = TRUE then
          k := k + n*ln(-y);
          r := -r;
        else
          r := r * t;
        end_if;
      else
        r := r * t;
      end_if;
      break;
      
    otherwise // rule 4)
      if is(t < 0) = TRUE then
        t := -t;
        r := -r;
      end_if;
      k := k + ln(t);

    end_case;
  end_for;
  // now x = k + ln(r)

  // handle the remainder ln(r)
  case type(r)

  of "_power" do // rule 5
    [y, n] := [op(r)];
    /* ln(y^n)=n*ln(y) when -1< n <=1
       and for n = -1 if y is not real nonpositive
       and for -2 <= n <= 2 if Re(y) > 0
    */
    if options[IgnoreAnalyticConstraints] or
      (is(-1 < n) = TRUE and is(n <= 1) = TRUE) or
       (is(n = -1) = TRUE and is(y <= 0) = FALSE) or
       (is(-2 <= n) = TRUE and is(n <= 2) = TRUE and is(Re(y) > 0) = TRUE) then
       return(k + n*ln(y))
    end_if;
    break;

  of "exp" do // rule 6
    // ln(exp(z)) = z if -PI < Im z <= PI
    z := op(r);
    if options[IgnoreAnalyticConstraints] or
      (is(Im(z) > -PI) = TRUE and is(Im(z) <= PI) = TRUE) or
      contains({-1, 0, 1}, sign(z)) then
      return(k + z)
    end_if;
    break 
    
  of "_mult" do 
    [y, n, t]:= split([op(r)], testtype, "lambertW");
    if nops(y) = 1 and op(y, [1, 1]) = 0 then 
      z:= op(y, [1, 2]);
      n:= _mult(op(n));
      // r = n*lambertW(0, z)
      /*
        We have that 
        ln(lambertW(0, z)/z) = -lambertW(0, z).

        PROOF. By definition, exp(W(z))*W(z) = z; thus W(z)/z = exp(-W(z)). 
               While this holds for every branch, in general we only get 
               ln(W(z)/z) = ln(exp(-W(z)). 
               Since the principal branch always lies between -PI and PI, ln and exp cancel for W=W_0.

        Thus ln(n*lambertW(0, z)) = ln( (n*z) * (lambertW(0, z)/z) ) = ln(n*z) - lambertW(0, z)
        provided that expanding can be justified, e.g. if n*z > 0.
      */
      if is(n*z > 0, Goal = TRUE) then 
        r:= n*z;
        k:= k - lambertW(0, z)
      end_if;
    end_if;  
    break  
  end_case;

  return(k + ln(r))

end_proc:
