/*--
	log/simplify -- the function attribut "simplify" for log
        Analogous to ln::simplify.

	Rules (take properties into account):

	1) log(b, posint1/posint2 * r )  -->  log(b,posint1) - log(b,posint2) + log(b,r)
	2) log(b, exp(real) * r )        -->  real/ln(b) + log(b,r)
        3) log(b, positive^real * r )    -->  real*log(b,positive) + log(b,r)
           log(b, negative^even * r )    -->  even*log(b,-negative) + log(b,r)
           log(b, negative^odd  * r )    -->   odd*log(b,-negative) + log(b,-r)
        4) log(b, positive * r )         -->  log(b,positive) + log(b,r)
           log(b, negative * r )         -->  log(b,-negative) + log(b,-r)

        5) log(b, y^n )                  -->  n*log(b,y) if
                                              y = b and -PI < Im(n) <= PI or
                                              -1 < n <= 1 or
                                              n = -1 and is(y <= 0) = FALSE
                                              or -2 <= n <= 2 and Re(y) > 0
        6) log(b, exp(z) )               -->  z/ln(b) if -PI < Im(z) <= PI

        The ideology is to have only positive arguments in log's where
        possible.
--*/

log::simplify:=
proc(x: "log", options = simplify::defaultOptions: DOM_TABLE)
  local l, t, r, k, y, n, z, b;
begin

  b := op(x, 1);
  l := op(x, 2);

  if type(b) = DOM_INT then
    assert(b>1);
    // log(m^n, l) = 1/n * log(m, l)
    y:= numlib::ispower(b);
    if y <> FALSE then
      return(1/op(y, 2) * log(op(y, 1), l))
    end_if;
  end_if;
  
  // convert the result into a list of factors
  if type(l) = "_mult" then
    l := [op(l)];
  else
    l := [l];
  end_if;

  // determine all factors that are recognized as positive or negative by "is"
  // in IgnoreAnalyticConstraints-mode, we do not care about this

  if options[IgnoreAnalyticConstraints] then
    r:= 1
  else
  
    l := split(l,
               proc(a)
               begin
                 is(a > 0 or a < 0, Goal = TRUE)
               end_proc);
    assert(l[3] = []);
    r := _mult(op(l[2]));
    l := l[1];
  end_if;
    
  // now x = log(b,_mult(op(l)) * r),
  // and every element of l "is" either >0 or <0

  // handle positive or negative factors
  k := 0;
  for t in l do
    case type(t)
        
      of DOM_INT do
        if t < 0 then
          t := -t;
          r := -r;
        end_if;
        // log(b, n^m) = m*log(b, n)
        y:= numlib::ispower(t);
        if y = FALSE then
          k := k + log(b,t)
        else
          k:= k + op(y, 2)*log(b, op(y, 1))
        end_if;
        break
        
      of DOM_RAT do // rule 1
        // log(b,numer/denom) = log(b,numer) - log(b,denom)
        if t < 0 then
          t := -t;
          r := -r;
        end_if;
        k := k + log(b,op(t, 1)) - log(b,op(t, 2));
        break;

      of "exp" do // rule 2
        if options[IgnoreAnalyticConstraints] or is(op(t) in R_, Goal = TRUE) then
          // t "is" positive and ln(t) = op(t)
          k := k + op(t)/ln(b);
        else
          r := r * t;
        end_if;
        break;
        
      of "_power" do // rule 3
        [y, n] := [op(t)];
        if options[IgnoreAnalyticConstraints] then
          k := k + n*log(b,y)
        elif is(y > 0) = TRUE then          
          // log(b,y^n * r) = n*log(b,y) + log(b,r) if y > 0 and n is real
          if is(n, Type::Real) = TRUE then
            k := k + n*log(b,y);
          else
            r := r * t;
          end_if;
        elif is(y < 0) = TRUE then
          // log(b,y^n * r) = n*log(b,-y) + log(b,r) if y < 0 and n is even
          // log(b,y^n * r) = n*log(b,-y) + log(b,-r) if y < 0 and n is odd
          if is(n, Type::Even) = TRUE then
            k := k + n*log(b,-y);
          elif is(n, Type::Odd) = TRUE then
            k := k + n*log(b,-y);
            r := -r;
          else
            r := r * t;
          end_if;
        else
          r := r * t;
        end_if;
        break;
      
      otherwise // rule 4
        if is(t < 0) = TRUE then
          t := -t;
          r := -r;
        end_if;
        k := k + log(b,t);

    end_case;
  end_for;
  // now x = k + log(b,r)

  // handle the remainder log(b,r)
  case type(r)

  of "_power" do // rule 5
    [y, n] := [op(r)];
    /* log(b,y^n)=n*log(b,y) when -1< n <=1
       and for n = -1 if y is not real nonpositive
       and for -2 <= n <= 2 if Re(y) > 0
       and for -PI < Im(n) <= PI if y = b
    */
    if options[IgnoreAnalyticConstraints]
      or
      is(-1 < n and n <= 1, Goal = TRUE)
      or
      is(n = -1, Goal = TRUE) and is(not y <= 0, Goal = TRUE)
      or
      (is(-2 <= n) = TRUE and is(n <= 2) = TRUE and is(Re(y) > 0) = TRUE)
      or
      (y = b and is(-PI < Im(n)) = TRUE and is(Im(n) <= PI) = TRUE)
      then
      return(k + n*log(b,y))
    end_if;
    break;

  of "exp" do // rule 6
    // log(b, exp(z)) = z/ln(b) if -PI < Im z <= PI
    z := op(r);
    if options[IgnoreAnalyticConstraints] or
       (is(Im(z) > -PI) = TRUE and is(Im(z) <= PI) = TRUE) or
       contains({-1, 0, 1}, sign(z)) then
      return(k + z/ln(b))
    end_if;

  end_case;

  return(k + log(b,r))

end_proc:


