/* The simplify attribute of sin

   Rules:
   1) sin(x)            -> apply simplify and stdlib::normalizesign to x
   2) sin(x + k*PI/2)   -> +- sin/cos (x + frac(k)*PI/2)
                           if domtype(k) = DOM_RAT
   3) sin(x + k*PI/2)   -> +- sin/cos (x) if is(k, Type::Residue(?,2))
                           or is(k, Type::Residue(?,4))
   4) sin(x)            -> -sin(-x) if stdlib::hasmsign(x) = TRUE
*/

sin::simplify:=
proc(a)
  local x, lnx, c, l, i, j, success: DOM_BOOL,
  e, rest, symbolic_expo, integer_expoI,
  tmp;
begin
  if type(a) <> "sin" then
     return(a);
  end_if:
  x := op(a);

  if has(x, ln) then 
     if type(x/I) = "ln" then // sin(I*ln(x)) = I/2*x - I/2/x
        x:= op(x/I);
        return(normal(I*(x^2 - 1)/(2*x)));
     elif type(x/I) = "_mult" then // sin(c*I*ln(x)) = I*(x^(2*c) - 1)/x^c)
         lnx:= select(x/I, testtype, "ln");
         c:= (x/I)/lnx;
         if testtype(c, Type::Constant) and
            type(lnx) = "ln" and
            not has(c, I) then
           x:= op(lnx)^c;
           return(normal(I*(x^2 - 1)/(2*x)));
         end_if;
     end_if:
     // Finally, take care of sin(c1*I*ln(x1) + c2*I*ln(x2) + ...).
     // exp::simplify has (rather complicated) code for this kind
     // of simplification. Call it to let it do the job:
     c:= expand(x/I, ArithmeticOnly);
     if (not has(c, I)) and type(c) = "_plus" then
         tmp:= misc::maprec(rewrite(sin(c*I), exp), {"exp"} = exp::simplify):
         if not hastype(tmp, "ln") then
           return(tmp);
         end_if;
     end_if:
  end_if:

  // No need to react to arcsin etc., since sin itself
  // handles this (without simplify)

  if not has(x/I, I) then // sin(I*y) = sinh(y)*I
     return(sinh(x/I)*I);
  end_if;

  //------------------------------------
  // treat shifts by multiples of I*PI/2
  //------------------------------------

  symbolic_expo := 0;
  /* final integer exponent of I; indicates whether the final result
     will be a sin, cos, -sin, or -cos */
  integer_expoI := 0;
  // finally remaining arguments
  rest := 0;

  // l := list of summands
  if type(x) = "_plus" then
    l := [op(x)]
  else
    l := [x]
  end_if;

/* =====================
  l:= split(l, x -> contains({DOM_RAT, DOM_INT}, domtype(x/2/PI))):
  if nops(l[1]) = 1 then // reduce the argument mod 2*PI
    tmp:= frac(_plus(op(l[1]))/2/PI);
    rest:= _plus(op(l[2]), op(l[3]));
  else
    tmp:= 0;
    rest:= x;
  end_if:

  if domtype(16*tmp) = DOM_INT then
     // x =  (16*tmp)*PI/8  + rest  with 16*tmp in [-7 .. 7]
    return(sin(tmp*2*PI)*cos(rest) + cos(2*tmp*PI)*sin(rest));
  else
    return(sin(tmp*2*PI + rest));
  end_if;

===================== */

  // l := list of pairs [summand, summand*2/PI]
  l := map(l, proc(x) begin [x, x*2/PI] end_proc);

   /* check whether there is a summand with domtype(summand*2/PI) =
      DOM_RAT or DOM_INT (there can be at most one) */
   l := split(l, proc(x) begin
     contains({DOM_RAT, DOM_INT}, domtype(op(x, 2))) end_proc);
   if nops(l[1]) > 0 then
     // reduce the rational number modulo 1
     tmp := op(l, [1,1,2]);
     rest := frac(tmp)*PI/2;
     integer_expoI := floor(tmp) mod 4;
   end_if;

   // select those summands with is(summand*2/PI, Type::Integer) = TRUE
   l := split(l[2], proc(x) begin is(op(x, 2), Type::Integer) end_proc);

   // rest := unsimplified arguments
   rest := _plus(op(map(l[2].l[3], op, 1)), rest);
 
   // process arguments of Type::Integer
   l := l[1];
   for i from 1 to nops(l) do
     e := op(l, [i, 2]);

     success:= FALSE;
     for j from 0 to 3 do
       if is((e-j)/4 in Z_) = TRUE then
         integer_expoI := integer_expoI + j;
         success:= TRUE;
         break
       end_if
     end_for;

     if not success then
       for j from 0 to 1 do
         if is((e-j)/2 in Z_) = TRUE then
           integer_expoI := integer_expoI + j;
           symbolic_expo := symbolic_expo + (e - j)/2;
           success:= TRUE;
           break
         end_if
       end_for
     end_if;

     if not success then
       rest := rest + op(l, [i, 1])
     end_if;
   end_for;

   /* extract a minus sign from rest if necessary */
   if stdlib::hasmsign(rest) then
     rest := -rest;
     integer_expoI := 2 - integer_expoI mod 4;
   end_if;

   /* determine final result according to the formula
      cos(x+k*PI/2) + I*sin(x+k*PI/2) = I^k*(cos(x) + I*sin(x)) */
  case modp(integer_expoI, 4)
    of 0 do
      return((-1)^(symbolic_expo) * sin(rest))
    of 1 do
      return((-1)^(symbolic_expo) * cos(rest))
    of 2 do
      return((-1)^(symbolic_expo) * -sin(rest))
    of 3 do
      return((-1)^(symbolic_expo) * -cos(rest))
  end_case;
  // NOT REACHED
  assert(FALSE)
end_proc:
