/* The simplify attribute of tan

   Rules:
   1) tan(x)            -> apply simplify and stdlib::normalizesign to x
   2) tan(x + k*PI/2)   -> (-1)^k * tan/cot (x + frac(k)*PI/2)
                           if domtype(k) = DOM_RAT
   3) tan(x + k*PI/2)   -> (-1)^k * tan/cot (x) if is(k, Type::Residue(?,2))
   4) tan(x)            -> -tan(-x) if stdlib::hasmsign(x) = TRUE
*/

tan::simplify:=
proc(a)
  name tan::simplify;
  local x, l, i, lnx, c,
        e, rest, integer_expo, prop, tmp, sgn;
begin
   if type(a) <> "tan" then
      return(a);
   end_if:
   x := op(a);

   if has(x, ln) then
      if type(x/I) = "ln" then // tan(I*ln(x)) = I*(x - 1/x)/(x + 1/x)
         x:= op(x/I);
         return(normal(I*(x^2-1)/(x^2+1))):
      elif type(x/I) = "_mult" then // tan(c*I*ln(x)) = I*(x^(2*c) - 1)/(x^(2*c) + 1))
         lnx:= select(x/I, testtype, "ln");
         c:= (x/I)/lnx;
         if testtype(c, Type::Constant) and
            type(lnx) = "ln" and
            not has(c, I) then
             x:= op(lnx)^c;
             return(normal(I*(x^2 - 1)/(x^2 + 1)));
         end_if;
      end_if:
      // Finally, take care of tan(c1*I*ln(x1) + c2*I*ln(x2) + ...).
      // exp::simplify has (rather complicated) code for this kind
      // of simplification. Call it to let it do the job:
      c:= expand(x/I, ArithmeticOnly):
      if (not has(c, I)) and type(c) = "_plus" then
         tmp:= misc::maprec(rewrite(tan(c*I), exp), {"exp"} = exp::simplify):
         if not hastype(tmp, "ln") then
           return(tmp);
         end_if;
      end_if:
   end_if:

   // No need to react to arctan etc., since tan itself
   // handles this (without simplify)

   if not has(x/I, I) then // tan(I*y) = tanh(y)*I
      return(tanh(x/I)*I)
   end_if;

   //------------------------------------
   // treat shifts by multiples of I*PI/2
   //------------------------------------
   /* final integer exponent of -1 */
   integer_expo := 0;
   // finally remaining arguments
   rest := 0;

   // l := list of summands
   if type(x) = "_plus" then
     l := [op(x)]
   else
     l := [x]
   end_if;

   // l := list of pairs [summand, summand*2/PI]
   l := map(l, proc(x) begin [x, x*2/PI] end_proc);

   /* check whether there is a summand with domtype(summand*2/PI) =
      DOM_RAT or DOM_INT (there can be at most one) */
   l := split(l, proc(x) begin
     contains({DOM_RAT, DOM_INT}, domtype(op(x, 2))) end_proc);
   if nops(l[1]) > 0 then
     // reduce the rational number modulo 1
     tmp := op(l, [1,1,2]);
     rest := frac(tmp)*PI/2;
     integer_expo := floor(tmp) mod 2;
   end_if;

   // select those summands with is(summand*2/PI, Type::Integer) = TRUE
   l := split(l[2], proc(x) begin is(op(x, 2), Type::Integer) end_proc);

   // rest := unsimplified arguments
   rest := _plus(op(map(l[2].l[3], op, 1)), rest);

   // process arguments of Type::Integer
   l := l[1];
   for i from 1 to nops(l) do
     e := op(l, [i, 2]);

     // simplify exponents that are known to be odd or even
     prop := getprop( e mod 2 );
     if type(prop)=DOM_SET and nops(prop)=1 then
       integer_expo := integer_expo + op(prop) mod 2;
     else
       rest := rest + op(l, [i, 1]);
     end_if
   end_for;

   /* extract a minus sign from rest if necessary */
   sgn := 1;
   if stdlib::hasmsign(rest) then
     rest := -rest;
     sgn := -1
   end_if;

   if integer_expo = 0 then
     return(sgn * tan(rest))
   else // = 1
     return(-sgn * cot(rest))
   end_if
end_proc:
