/*
        asympt  --  computing asymptotic expansions

        asympt(ex,x [=v] [,ord])

        ex : expression in x
        x  : ident
        v  : (optional) expression
        ord: (optional) positive integer

        Compute the asymptotic series of ex of order ord with respect
        to at v. If v is missing then x is assumed to goes to infinity.
        If ord is missing then the value of ORDER is used (6 by default).

        If v is finite, then, by default, an expansion from the right is
        computed. 

        If an expansion could not be computed, then the expression
        ex will be returned.
        Otherwise an element of the domain 'gseries' (defined in
        the library "Series") is returned, that can be manipulated 
        with the standard arithmetic operations.

        See gseries.mu for detailed informations.
*/

asympt:= proc(f,x,ord)
    local r, s, Oterm;
begin 
    if args(0) = 0 then
        error("no arguments given")
    elif f::dom::asympt <> FAIL then 
        // allow overloading of the first argument of asympt:
        return(f::dom::asympt( args() )) 
    elif args(0) < 2 or args(0) > 4 then
        error("wrong no of args")
    elif domtype(x) = DOM_IDENT then
        s:= x = infinity
    elif type(x) <> "_equal" or domtype(op(x,1)) <> DOM_IDENT then
        error("invalid 2nd argument")
    else
        s:= op(x,2);
        if not testtype(s,Type::Numeric) and s <> infinity and s <> -infinity then
            // Note: is(infiniy,Type::Real) gives UNKNOWN!?
            case is(s,Type::Real)
            of TRUE    do break
            of FALSE   do error("expansion point must be a real arithmetical expression")
            of UNKNOWN do error("expansion point cannot be checked on being real")
            end_case;
        end_if;
        s:= x
    end_if;

    //------------------------------------------------------------------
    // Walter 14.2.02: added treatment of int in analogy to Series::Puiseux
    // Warning: this is just a quick hack. In particular, the Oterm needs to
    // be created in a much more sophisticated way! For example, try
    // >> asympt(int(dilog(x), x), x=1)
    case type(f)
    of "int" do
       if ((domtype(x) = DOM_IDENT and x = op(f, 2)))
           // indefinite integral, series variable = integration variable:
           or ((type(x) = "_equal") and op(x, 1) = op(f, 2))
           // definite integral, series variable = integration variable:
           or (not has([op(f, 2..nops(f))], op(x, 1)))
           // the int-boundaries do not contain the series variable
           then // exchange int and series
             r:= asympt(op(f, 1), s, args(3..args(0)));
             if domtype(r) <> Series::gseries then
                r:= Series::gseries::convert(r);
             end_if;
             if domtype(r) = Series::gseries then
               Oterm:= extop(r, 2):
               r:= int(expr(r), op(f, 2..nops(f)));
               if not type(r) = "int" then
                  // successful integration. Convert
                  // this expression to a gseries:
                  Oterm:= int(Oterm, op(f, 2..nops(f)));
                  if type(Oterm) = "_mult" then
                     Oterm:= select(Oterm, _not@testtype, Type::Constant);
                  end_if;
                  if domtype(Oterm) = DOM_NULL then
                     Oterm:= 1;
                  end_if;
                  r:= asympt(r, x, args(3..args(0)));
                  if domtype(r) = Series::gseries then
                    return(extsubsop(r, 2 = Oterm));
                  end_if;
               end_if;
          end_if;
       end_if;
    end_case;
    //------------------------------------------------------------------

    r:= Series::gseries::new(f,s,args(3..args(0)));
    if r = FAIL or
      nops(extop(r, 1)) = 1 and extop(r, 2) = 0 then
      
    
      r := series(f, s, args(3..args(0)),
                if has([args(3..args(0))], {hold(Left), hold(Right)}) then
                  null()
                elif op(s, 2) = infinity then
                  hold(Left)
                else
                  hold(Right)
                  end_if)
    end_if;

    if contains({Series::Puiseux, Series::gseries}, domtype(r)) then
      return(r)
    end_if;

    return(procname(args()))
end_proc:

asympt:= slot(funcenv(asympt), "type", "asympt"):

