/* The following comments are not up-to-date
   and in compliance with the present code!
*/
/*
todo: Ueberarbeiten fuer den series-Fall:

-  numlib::contfrac handelt nur contfracs reeller Zahlen ab.
   Zusaetzliches contfrac, welches sowohl Zahlen als auch
   Ausdruecke akzeptiert. Im Fall einer Zahl dient
   (stdlib::)contfrac nur als Interface zu numlib::contfrac
   (Rueckgabe vom Typ numlib::contfrac). 
   Im series-Fall Rueckgabe vom Typ (stdlib::)contfrac)

-  Fuer rationale Ausdruecke berechnet contfrac eine *exakte*
   Darstellung  (Series::Puiseux(normal(f(x)), x, ord) mit
          ord = Zaehlergrad + Nennergrad + 1
   aufrufen)
   *** reicht nicht, ord kann quadratisch in Zaehlergrad + Nennergrad
   sein, Beispiel x^9+1 / x^10-1
   In diesem Fall sollte contfrac das am besten selbst abhandeln ?

-  interne Datenstruktur: [[a1, b1], [a2, b2], ..., [a.n, b.n]], r, x, x0
   Entspricht

         a1 +  b1
               ------------
               a2 + b2
                    -----------
                     ...   + b.(n-1)
                             ------------
                             a.n +  O(b.n)

   n eine natuerliche Zahl >= 0

   Vereinbarung: O(0) = 0 (exakter Kettenbruch)

   r ist der ausmultiplizierte Kettenbruch ohne den O-Term (i.a. eine
   rationale Funktion)

- Methoden:
    Arithmetik: verwendet Arithmetik der als extop(.., 2)
                gespeicherten rationalen Funktionen und wendet
                contfrac auf das Ergebnis an

    new: folgende Eingabesyntax zulassen:

             contfrac(f(x) <, order>)  (nur falls nops(indets(f(x)) = 1)
             contfrac(f(x), x <, order>)  (default: x0 = 0)
             contfrac(f(x), x = x0 <, order>)

    rational: analog zum numerischen Fall
    series  : liefert series-Approximation
*/
         
/*===========================================================================

contfrac -- continued fractions arithmetic 

contfrac(f <, p>)	 computes the continued fraction for f (with precision p)
                         (default is x if f is a function of only one variable x)

contfrac(f, x <, p>)     computes the continued fraction for f (with precision p)
                         with respect to the variable x (default is x=0); x is not
                         necessarily a variable of f.

contfrac(f, x=x0, <, p>) computes the continued fraction for f (with precision p)
                         with respect to the variable x around the point x=x0

a + b			 adds two continued fractions

a * b			 multiplies two continued fractions
 
a^n                      computes powers of a continued fraction

contfrac::rational(cf,n) computes the rational function represented by the coeffs 
                         1 through n of the continued fraction cf 
                         (default is n=nops(extop(cf, 1)))

contfrac::series(cf, <, p>)     computes the series expansion of a continued
                                fraction
contfrac::series(cf, x <, p>)
contfrac::series(cf, x=x0 <, p>)                       

contfrac::nthcoeff(cf, n) compute the coefficients of a continued fraction.
contfrac::nthterm(cf ,n)  The continued fraction cf=a1 + b1/(a2 + b2/...) is
    represented by the list [[a1, b1], [a2, b2], ..., [am, bm]] :
          nthcoeff(cf, n)=an if n<=m else FAIL
          nthterm(cf, n)=bn if n<m else FAIL

Convention: "..." represents a real between 1 and infinity, thus
            for example the continued fraction of 0 is 1/..., that
            of an integer n is n+1/...


==========================================================================*/

contfrac:=newDomain("contfrac"):
contfrac::create_dom:=hold(contfrac):
contfrac::info:= "domain 'contfrac': arithmetic with continued fractions":
contfrac::interface:= {hold(series), hold(nthcoeff), hold(nthterm), 
          hold(rational), hold(_plus), hold(_mult), hold(_power), 
          hold(function) }:

//------------------------------------------
// The call contfrac(f) returns numlib::contfrac(f) if f is a constant
//  otherwise it returns contfrac::function(f) in the symbolic case.
//
// contfrac(f, x, <m>)
// contfrac(f, x=x0, <m>)
//   f  - an arithmetical expression interpreted as a function of x
//   x  - an identifier
//   x0 - the expansion point. Default is 0
//   m  - a positive integer: the "number of terms". Default is ORDER
//
// Internal representation of a contfrac:
//
//  op(contfrac(f)) = [ [a1,b1], [a2,b2], ...], p, x, x0)
//
// where the continued fraction is given as a1 + b1/(a2 + b2/...) and the
// bi are powers of (x-x0) if x0 is finite and 1/x otherwise,
// p is the rational value of the contfrac expansion.
//------------------------------------------
// O. Cormier 22/11/01
contfrac::new:=
proc(e)
save DIGITS;
begin
  if e::dom::contfrac <> FAIL then
    // the domain of e overloads contfrac
    return(e::dom::contfrac(args()))
  end_if;
  if testtype(args(1), Type::Constant) then
    if args(0)>1 then
      if domtype(args(2)) = DOM_INT then
        return(numlib::contfrac(args()));
      else
        return(contfrac::function(args()));
      end_if;
    else return(numlib::contfrac(args()));
    end_if;
  else
    return(contfrac::function(args()));
  end_if;

end_proc:

//-------------------------------------------------
/* W. Oevel, 4.9.2001:
Interface-Methode zum direkten Erzeugen von contfrac-Objekten.
Wird von Series::Puiseux::contfrac aufgerufen, da nicht mehr
Expressions zurueckgeliefert werden sollen.
*/

contfrac::create := proc()
begin
  return(new(dom, args()));
end;
//-------------------------------------------------

// W. Oevel, 20.7.2010: Migrating from normal(.., Expand = TRUE)
// to normal(.., Expand = FALSE) throughout the library, we
// introduce contfrac::normal = normal(.., Expand = TRUE) to
// make sure that the old normal is used in contfrac.
// Rationale: The normalized rational form is part of the
// internal form. Since the new normal form using 
// normal(.., Expand = FALSE) is not unique (both 1/x/(x+1) 
// and 1/(x^2 + x) are normalized !), we better use the
// unique old form using normal(.., Expand = TRUE).

contfrac::normal:= () -> normal(args(), Expand = TRUE):

//-------------------------------------------------
// W. Oevel, 21.7.01: function wurde noch nicht ueberarbeitet
// continued fraction of a function, for example e = sin(z) 
// O. Cormier, 29.11.01 :
contfrac::function := proc(e)
  local nargs, s, z, ord, arg2;
begin
  nargs:=args(0);
  s:=e;
  z:=indets(s) minus Type::ConstantIdents; // We don't want PI to be a variable!!
  ord:=null();
  if nargs >= 2 then
    if nargs=3 then
      if domtype(args(3)) = DOM_INT and args(3) >= 0 then
        ord:=args(3);
      else error("the 3rd argument (the 'number of terms') must be an integer >= 0");
      end_if;
    end_if;
    arg2:=args(2); 
    if type(arg2) <> "_equal" then
      if domtype(arg2) = DOM_IDENT and not testtype(arg2, Type::Constant) then
        arg2:=args(2)=0;
      elif domtype(args(2)) = DOM_INT and args(2) >= 0 then
        if nops(z)>1 then
          error("1st argument: the expression is not univariate");
        elif nargs > 2 then
          error("unexpected 3rd argument")
        else
          ord:=args(2);
          arg2:=op(z)=0;
        end_if;
      else error("invalid 2nd argument"); 
      end_if;
    else // type(arg2) = "_equal"
      if domtype(op(arg2, 1)) <> DOM_IDENT then
        error("2nd argument: the left hand side of the equation must be an identifier")
      elif not testtype(op(arg2, 2), Type::Arithmetical) then
        error("2nd argument: the right hand side of the equation must be of Type::Arithmetical")
      elif has(op(arg2, 2), op(arg2, 1)) then
        error("2nd argument: the sides of the equation must be independent")
      end_if;
    end_if;
  elif nops(z)>1 then
    error("1st argument: the expression is not univariate");
  else
    arg2:=op(z)=0;
  end_if;

  if testtype(s, Type::RatExpr(op(arg2, 1))) and nops(ord) = 0 then
    // input is a rational function and no precision is specified
    // --> create exact continued fraction
    return(contfrac::ratfunc(s, arg2));
  else 
     e:=series(s,arg2,ord,NoWarning);
     if domtype(e) = Series::Puiseux and not has([coeff(e)], op(arg2, 1)) then
       return(Series::Puiseux::contfrac(e));
     else return(FAIL);
     end_if; 
  end_if;
end_proc:


//-------------------------------------------------
// ratfunc : returns the exact continued fraction of a rational function 
//           without doing any series expansion
// the function is called in contfrac::function
// O. Cormier, 26.11.01
contfrac::ratfunc := proc(r, pt)
  local f,x,l,p,q,c0,ff,pp,num,powerx,x0;
begin
  f:=r;
  x:=op(pt, 1); x0:=op(pt,2);
  if domtype(x0) = stdlib::Infinity then
    x0 := complexInfinity;
  end_if;
  if x0 = complexInfinity then
    f:=subs(f, x=1/x);
  else
    f:=subs(f, x=x+x0);
  end_if;
  l:=[];
  f:=contfrac::normal(f);
  while has(f,x) do
      p:=poly(numer(f),[x]); 
      q:=poly(denom(f),[x]);
      if ground(q)=0 then
          l:=append(l,[0,x^(-ldegree(q))]);
          f:=contfrac::normal(op(q,1)/(op(p,1)*x^(ldegree(q))));
      else
          c0:=contfrac::normal(coeff(p,0)/coeff(q,0));
          ff:=contfrac::normal(f-c0);
          pp:=poly(numer(ff),[x]);
          num:=nterms(pp);
          powerx:=x^(ldegree(pp));
          l:=append(l,[c0,powerx]);
          f:=denom(ff)/(op(pp,1)/powerx);
          f:=contfrac::normal(f);
      end_if;
  end_while;
  if x0 = complexInfinity then
    l:=subs(append(l,[f,0]), x=1/x);
  else
    l:=subs(append(l,[f,0]), x=x-x0);
  end_if;
  return(new(contfrac,l,r,x,x0));
end_proc:


//-------------------------------------------------
// expr: returns the unexpanded contfrac without the O-term 
// (jngerhar, 6.09.01, 17.11.01)
contfrac::expr := proc(c)
  local s, i, l, result;
begin
  [l, s] := [extop(c, 1..2)];
  result := op(l[nops(l)], 1);
  for i from nops(l) - 1 downto 1 do
    result := op(l[i], 1) + op(l[i], 2) / result
  end_for;
  return(result);
end_proc:

//-------------------------------------------------
// print: [[a1, b1], [a2, b2], ...] -->  a1 + b1/(a2 + b2/...),
contfrac::print:=
proc(c) 
  local l, result, i, s, n, mypower, myplus;
begin
  [l, s] := [extop(c, 1..2)];
  n := nops(l);
  // write powers with negative exponents as a^b instead of 1/a^(-b)
  mypower := proc(a, b) begin
    if stdlib::hasmsign(b) then a^stdlib::Exposed(expr2text(b))
    else hold(_power)(a, b)
    end_if
  end_proc;
  // avoids a + 0 or 0 + b
  myplus := proc(a, b) begin
      if iszero(a) then b
      elif iszero(b) then a
      else hold(_plus)(a, b)
      end_if
  end_proc;
  l := subs(l, hold(_power) = mypower, EvalChanges);
  l := map(l, generate::sortSums);
  result := op(l[n], 1);
  if not iszero(op(l[n], 2)) then
      result := myplus(result, hold(O)(op(l[n], 2)));
  end_if;
  for i from n - 1 downto 1 do
      if result = 1 then // avoid x/1
        result := myplus(op(l[i], 1), op(l[i], 2));
      else
        result := myplus(op(l[i], 1), hold(_mult)(op(l[i], 2),
                                                  hold(_power)(result, -1)));
      end_if
  end_for;
  return(result);    
end_proc:

// quick and dirty Content-Slot; modified copy of print
contfrac::Content:=
proc(Out, c) 
  local l, result, i, s, n, mypower, myplus;
  save _domainOutputMode;
begin
  [l, s] := [extop(c, 1..2)];
  n := nops(l);
  // write powers with negative exponents as a^b instead of 1/a^(-b)
  mypower := proc(a, b) begin
    if stdlib::hasmsign(b) then a^hold(``).expr2text(b)
    else hold(_power)(a, b)
    end_if
  end_proc;
  // avoids a + 0 or 0 + b
  myplus := proc(a, b) begin
      if iszero(a) then b
      elif iszero(b) then a
      else hold(_plus)(a, b)
      end_if
  end_proc;
  l := subs(l, hold(_power) = mypower, EvalChanges);
  _domainOutputMode := TRUE;
  l := map(l, generate::sortSums);
  result := op(l[n], 1);
  if not iszero(op(l[n], 2)) then
      result := myplus(result, hold(O)(op(l[n], 2)));
  end_if;
  for i from n - 1 downto 1 do
      if result = 1 then // avoid x/1
        result := myplus(op(l[i], 1), op(l[i], 2));
      else
        result := myplus(op(l[i], 1), hold(_mult)(op(l[i], 2),
                                                  hold(_power)(result, -1)));
      end_if
  end_for;
  return(Out(result));
end_proc:


contfrac::testtype:=
proc(x, T)
begin
if T = Type::Arithmetical then
  TRUE
else
  FAIL
end_if
end_proc:


//-------------------------------------------------
// W. Oevel, 21.7.01: This is the new version of _plus.
// It interprets each contfrac as a rational and does
// exact arithmetic on the rationals. Finally, the
// result is turned into a contfrac
// O. Cormier, 27.11.01: operands need not be necessary
//                       continued fractions
contfrac::_plus:=proc() 
   local s, i, t, p;
begin 
   s:= 0:
   p:= null():
   for i from 1 to args(0) do
       t:= args(i):
       if domtype(t) = numlib::contfrac then
          t := numlib::contfrac::rational(t);
       elif domtype(t) = contfrac then
          if p = null() then
             p:= (extop(t, 3) = extop(t, 4));
          elif p <> (extop(t, 3) = extop(t, 4)) then
             return(FAIL);
          end_if;
          t:= extop(t, 2);
       elif domtype(t) = DOM_POLY then
          t:= extop(t, 1);
       end_if;
       s:= s + t;
   end_for;
   return(contfrac(s, p));
end_proc:

//-------------------------------------------------
// W. Oevel, 21.7.01: This is the new version of _mult.
// It interprets each contfrac as a rational and does
// exact arithmetic on the rationals. Finally, the
// result is turned into a contfrac
// O. Cormier, 27.11.01: operands need not be necessary
//                       continued fractions
contfrac::_mult:=proc() 
   local s, i, t, p;
begin 
   s:= 1;
   p:=null();
   for i from 1 to args(0) do
       t:= args(i):
       if domtype(t) = numlib::contfrac then
            t := numlib::contfrac::rational(t);
       elif domtype(t) = contfrac then 
            if p=null() then
               p := extop(t, 3) = extop(t, 4);
            elif p <> (extop(t, 3) = extop(t, 4)) then
               return(FAIL);
            end_if;
            t:= extop(t, 2);
       elif domtype(t) = DOM_POLY then
          t:= extop(t, 1);
       end_if;
       s:= s * t;
   end_for;
   return(contfrac(s, p));
end_proc:

/*
//-------------------------------------------------
// O. Cormier, 26.11.01: Inverse of a continued fraction of a function
contfrac::_invert:=proc(f) 
   local p;
begin
   p:= extop(f, 2);
   if iszero(p) then
      error("division by zero");
   else
      return(contfrac(1/p, extop(f,3) = extop(f,4)) );
   end_if;
end_proc:
*/


//-------------------------------------------------
// O. Cormier, 26.11.01 
contfrac::_power:=proc(f, n) 
begin
   if n = 0 then return(contfrac(1, extop(f, 3) = extop(f, 4))) end_if;
   if n = 1 then return(f) end_if;
//   if n = - 1 then return(contfrac::_invert(f)) end_if;
   return(contfrac(extop(f, 2)^n, extop(f,3) = extop(f,4)));
end_proc: 



//-------------------------------------------------
// O. Cormier, 27.11.01 : coefficients of a continued fraction cf
//                        of a function
//  the continued fraction cf=a1 + b1/(a2 + b2/...) is represented
//  by the list [[a1, b1], [a2, b2], ..., [am, bm]] :
//  nthcoeff(cf, n)=an if n<=m else FAIL
//  nthterm(cf, n)=bn if n<m else FAIL

contfrac::nthcoeff := proc(cf, n)
  local L;
begin
  if args(0) <> 2 then
     error("expecting only two arguments");
  end_if;
  if domtype(cf) <> contfrac then
     error("expecting an object of type contfrac");
  end_if;
  if domtype(n) <> DOM_INT or n <= 0 then
        error("the 2nd argument must be a positive integer");
  end_if;
  L:=extop(cf, 1);
  if n>nops(L) then return(FAIL); end_if;
  return(L[n][1]);
end_proc:


contfrac::nthterm := proc(cf, n)
  local L;
begin
  if args(0) <> 2 then
     error("expecting only two arguments");
  end_if;
  if domtype(cf) <> contfrac then
     error("expecting an object of type contfrac");
  end_if;
  if domtype(n) <> DOM_INT or n <= 0 then
        error("the 2nd argument must be a positive integer");
  end_if;
  L:=extop(cf, 1);
  if n>=nops(L) then return(FAIL); end_if;
  return(L[n][2]);
end_proc:



//-------------------------------------------------
// O. Cormier, 27.11.01: new method 'rational' implemented
//
//  rational(cf, n)
//     cf - a contfrac
//     n  - a positive integer
//
//  returns the rational function represented by the
//  coeffs 1 through n of cf.
//  Default is n=nops(extop(cf, 1))
contfrac::rational:= proc(cf, n) 
  local L, p, i;
begin
  if args(0) < 1 then
     error("expecting at least one argument");
  end_if;
  if args(0) > 2 then
     error("expecting no more than two arguments");
  end_if;
  if domtype(cf) <> contfrac then
     error("expecting an object of type contfrac");
  end_if;
  if args(0) > 1 then
     n:= args(2);
     if domtype(n) <> DOM_INT or n <= 0 then
        error("the 2nd argument must be a positive integer");
     end_if;
  end_if;

  L:= extop(cf, 1);
  if args(0) = 1 or n > nops(L) then
       n:= nops(L);
  end_if;
  if n = nops(L) then
     return(extop(cf, 2))
  end_if;
  p:= L[n][1];
  for i from n - 1 downto 1 do
    p:= L[i][1] + L[i][2]/p;
  end_for;
  return(contfrac::normal(p));
end_proc:


//-------------------------------------------------
// O. Cormier, 27.11.01: series expansion of a continued fraction
//                       of a function
contfrac::series:= proc(cf)
  local nargs, x, x0, remain, arg2;
begin
  nargs:=args(0);
  x:=extop(cf, 3);
  x0:=extop(cf, 4);
  remain:=null();
  if nargs >= 2 then
    arg2:=args(2); 
    if type(arg2) <> "_equal" then
      if domtype(arg2) = DOM_IDENT then
        if arg2=x then 
          if x0 = 0 then
            arg2:=x=0;
          else
            arg2:= x = 0;
            error("2nd argument: the expansion point ".expr2text(x0).  " of the ".
                  "continued fraction clashes with the requested expansion point ".
                  expr2text(op(arg2, 2)));
          end_if
        else
          arg2:= arg2=0;
        end_if;
        remain := args(3..args(0));
      elif domtype(arg2) = DOM_INT and arg2 >= 0 then
        if nargs > 2 then
          error("invalid 2nd argument")
        else
          remain:=arg2;
          arg2:= x=x0;
        end_if;
      else error("invalid 2nd argument"); 
      end_if;
    else // type(arg2) = "_equal"
      if domtype(op(arg2, 1)) <> DOM_IDENT then
        error("lhs of equation must be an identifier")
      elif not testtype(op(arg2, 2), Type::Arithmetical) then
        error("2nd argument: the right hand side of the equation must be of Type::Arithmetical")
      elif has(op(arg2, 2), op(arg2, 1)) then
        error("2nd argument: the sides of the equation must be independent")
      elif op(arg2, 2) <> x0 and op(arg2, 1) = x then
        error("2nd argument: the expansion point x0 = ".expr2text(x0). 
              " of the continued fraction clashes with the requested expansion point x0 = ".
              expr2text(op(arg2, 2)));
      end_if;
      if domtype(op(arg2, 2)) = stdlib::Infinity and x0 = complexInfinity then
        arg2 := op(arg2, 1) = complexInfinity;
      end_if;
      remain := args(3..args(0));
    end_if;
  else
    arg2:=x=x0;
  end_if;
  return(series(extop(cf, 2), arg2, remain));
end_proc:


