/*
   discont  --  determines the discontinuities of a function


   discont(f,x) returns a set of (possible) discontinuities
   of f, where f is regarded as a function of x, defined on all complex numbers that may be taken on as values by x.

   discont(f,x=a..b) returns the discontinuities in a..b, where a<=b.

   discont looks for an attribute "realDiscont" (or "complexDiscont",
   respectively) of the operator of f in order
   to determine the discontinuities of f viewed as a real
(resp. complex) function.


   For example to say that the only
   discontinuity of dirac is in 0, you write
   dirac:=slot(dirac,"realDiscont",{0})


   discont(f, x, Undefined), or discont(f, x=a..b, Undefined)
   returns the set of all complex numbers (or all numbers
   in the interval [a,b], respectively) for which f is undefined.

   In this case, discont looks for an attribute "undefined" of the operator.

   discont(f, x, Real) assumes that all subexpressions of f are real.



*/


discont :=
proc(f,x)
  local i, s, r, iv, argv, F, options, result, specialoptions,
  border;
  save MAXEFFORT;
begin

  // local method border
  // returns a superset of the border of a closed set of reals
  
  border:=
  proc(S)
  begin
    case type(S)
      of Dom::Interval do
        {S::dom::left(S), S::dom::right(S)} minus {-infinity, infinity};
        break
      of "_union" do
        map(S, border);
        break
      otherwise
        S
    end_case;
  end_proc;
    
  
  //  overloading for domains
  if f::dom::discont <> FAIL then
    return(f::dom::discont(args()))
  end_if;
  
  /* main */

  if testargs() then
    if args(0)<2 then
      error("Two or three arguments expected")
    end_if;
    case type(x)
      of DOM_IDENT do break; // ok
      of "_equal" do
        if map(x,type)<>(DOM_IDENT="_range") then
          error("x=a..b expected as 2nd argument")
        end_if;
        break
      otherwise
        if not testtype(x, Type::Arithmetical) then
          error("invalid second argument")
        end_if
    end_case;
  end_if;

  options:= [args(3..args(0))];
  argv:= {op(options)};

  if argv intersect {Dom::Real, Dom::Complex, C_, R_} <> {} then
    error("Argument Dom::Real/Dom::Complex not supported any longer")
  end_if;

  specialoptions:= argv intersect {Undefined, undefined};

  // non-special options are the same as those of solve
  for i in specialoptions do
    delete options[contains(options, i)];
  end_for;
  options:= solvelib::getOptions(op(options));
  
  if nops(specialoptions) > 0 then
    F:= Undefined
  elif options[Real] then
    F:= Real
  else
    F:= null()
  end_if;
        
  if type(x) = "_equal" then
    if type(op(x, 1)) <> DOM_IDENT or protected(op(x,1)) <> None then
      r:= genident();
      return(subs(discont(subs(f, op(x, 1) = r), r = op(x, 2), F),
                  r = op(x, 1)))
    end_if
  elif type(x) <> DOM_IDENT or protected(op(x, 1)) <> None then
    r:= genident();
    return(subs(discont(subs(f, x = r), r, F), r = x))
  end_if;

  if F= Real then
    // implicit assumption
    assume(op(x, 1), Type::Real, _and)  
  end_if;

  // handle case that a certain range is given

  if type(x)="_equal" then
    r:=op(x,2);
    iv:= Dom::Interval([op(r, 1), op(r, 2)]);
    if iv = {} then
      return({})
    end_if;
    if traperror(assume(op(x, 1) in iv, _and)) <> 0 then
      if getlasterror()[2][1..29] = "Error: assumption impossible " or
        getlasterror()[2][1..40] = "Error: Inconsistent assumptions detected"
        then
        // can't be in this range, therefore no discontinuities here
        return({});
      end;
      lasterror();
    end;
    
    // re-evaluate f under new assumptions
    f:= misc::maprec(f, {"sign", "abs", "signIm"} =
    proc(X)
    begin
      if has(X, op(x, 1)) then
        eval(X)
      else
        X
      end_if  
    end_proc);  
    
    // spend half of the time on recursive discont,
    // half on set operations
    MAXEFFORT:= MAXEFFORT/2;
    s:= discont(f, op(x,1), F);
    if type(s) = "discont" then
      return(procname(f, x, F, solvelib::nonDefaultOptions(options)))
    else
      return(s)
    end_if
  end_if;



  // constant functions may only be constantly undefined or continous

  if not has(f, x) then
    // regard constant expressions as defined
    // (the correct result might be C_, for some paramter values)
    return({})
  end_if;

 // overloading for function environments
  if type((r:=eval(op(f,0))))=DOM_FUNC_ENV then
    case F
      of Real do
        s:= r::realDiscont; break
      of null() do
        if is(op(f, 1) in R_, Goal = TRUE) then
          s:= r::realDiscont
        else
          s:= r::complexDiscont
        end_if;
        break
      of hold(Undefined) do
        s:= r::undefined; break
      otherwise
        // NOT REACHED
        assert(FALSE);
    end_case;

    if s<>FAIL then
      // f(g(x)) is continous unless f is not continous
      // at g(x) or g is not continous at x; let s be the set
      // of exceptional points
      MAXEFFORT:= MAXEFFORT/2;
      if is(op(f, 1) in s) = FALSE then
        iv := {}
      else
        iv:= solvelib::preImage(op(f, 1), x, s, options);
      end_if;
      if type(iv) <> DOM_SET and iv subset R_ = TRUE and
        is(x in R_, Goal = TRUE) then
        // we are always on the branch cut - only border points
        // and real discontinuities matter

        // find the topological border points of iv in R_

        result:= misc::maprec(iv, {Dom::Interval} =
                          proc(J: Dom::Interval)
                          begin
                            {
                             if is(x < J::dom::left(J)) <> FALSE and 
                                is(x = J::dom::left(J)) <> FALSE then
                               J::dom::left(J)
                             else
                               null()
                             end_if,

                             if is(x > J::dom::right(J)) <> FALSE and
                                is(x = J::dom::right(J)) <> FALSE then
                               J::dom::right(J)
                             else
                               null()
                             end_if
                             }
                          end_proc

                          );
        
     
     

        s:= r::realDiscont;
        if s = FAIL then
          return(procname(f, x, F, solvelib::nonDefaultoptions(options)))
        end_if;
        return(solvelib::preImage(op(f, 1), x, s, options) union
               discont(op(f, 1),x, options, F) union
               result
               )
      else
        s:= iv union discont(op(f, 1),x, options, F);
        return(s)
      end_if
    else
      // s = FAIL
      if (s:= r::discont(f, x, options, F) ) <> FAIL then
        return(s)
      end_if
    end_if;
  end_if;


  // do not handle RootOf's
  // (would cause problems in property mechanism)

  if hastype(f, RootOf) then
    return(procname(f, x, F, solvelib::nonDefaultOptions(options)))
  end_if;

  case type(f)
    of DOM_INT do
    of DOM_RAT do
    of DOM_IDENT do
    of "_index" do
    of DOM_COMPLEX do
    of DOM_FLOAT do
    of stdlib::Infinity do
    of stdlib::CInfinity do
      return({})
    of "_plus" do
    of "_mult" do
      f:= select(f, has, x);
      if not contains({"_mult", "_plus"}, type(f)) then
        // only one operand left
        return(discont(f, x, options, F))
      end_if;
      MAXEFFORT:= MAXEFFORT/nops(f);
      return(_union(discont(op(f,i),x, options, F)$i=1..nops(f)))
    of "_power" do

       // special case F=Undefined
       // f(x)^g(x) is undefined iff f(x) or g(x) is undefined
       // or f(x) is zero and g(x) is not >= 0
       // for efficiency reasons, we handle the case g(x)=const separately
      if F=hold(Undefined) then
        if not has(op(f,2), x) then
          return(piecewise([op(f,2)>=0, discont(op(f,1), x, options, F)],
                           [not op(f,2) >= 0, discont(op(f,1), x, options, F) union
                            solve(op(f,1), x, options)]))
        else
          // x occurs in the exponent
          // we have to call solve in oirder to decide
          // exponent >=0 / exponent <0
          MAXEFFORT:= MAXEFFORT/4;
          return(
                 solve(not op(f, 2) >= 0, x, options) intersect
                            solve(op(f,1), x, options)
                 union discont(op(f,2), x, options, F)
                 union discont(op(f,1), x, options, F)
                 )
        end_if
      end_if;

      assert(F <> hold(Undefined));

       // f=a^b
       // special cases for b integer and independent of x
      if not has(op(f,2), x) then
        if is(op(f,2), Type::Integer)=TRUE
        then
         // x -> x^b is everywhere continous for b>0, and has
         // its only discontinuity at zero for b<=0
        return(piecewise([op(f,2)>0, discont(op(f,1), x, options, F)],
                         [op(f,2)<=0, discont(op(f,1), x, options, F) union
                          solve(op(f,1), x, options)]))
        end_if;

        // special case: real base, b independent of x
        // then, as a function of the reals, x -> x^b
        // is everywhere continous for b>0
        // and discontinous only at zero otherwise
        
        // test whether base is real
        if F = Real or is(op(f, 1) in R_, Goal = TRUE) then
          return(piecewise([op(f,2) > 0, discont(op(f,1), x, options, F)],
                         [Otherwise, discont(op(f,1), x, options, F) union
                          solve(op(f,1), x, options)]))
        end_if;
      end_if;

       // = exp(b*ln(a)) if a is not zero
       // discontinuities are those of b and those of ln(a)
      MAXEFFORT:= MAXEFFORT/3;
      if F = Real then
        // the only discontinuity can be at op(f, 1) = 0
        r:= solve(op(f,1), x, options)     
      else
        i:= Dom::Interval(-infinity, 0);
        r:= solvelib::preImage(op(f,1), x, i, options);
        if r <> {} and is(x in R_, Goal = TRUE) then
          if is(op(f, 1) in R_, Goal = TRUE)  then
          // since op(f, 1) takes on real values only, we may see ln(op(f, 1))
          // as a real function of its argument
            r:= solve(op(f,1), x, options)
          elif type(r) = Dom::Interval then
            // it suffices to return the borders (we want to determine the points where we leave the branch cut)
               r:= {if not r::dom::isleftopen(r) then r::dom::left(r) end_if,
                   if not r::dom::isrightopen(r) then r::dom::right(r) end_if
                   }
          end_if;        
        end_if
      end_if;        
    
      return(discont(op(f,2), x, options, F) union
             discont(op(f,1), x, options, F) union r)

    //=================
    // Bessel functions
    //=================
    of "besselI" do
    of "besselJ" do
       // f:=[op(f,2)];
       if has(op(f, 1), x) then
          if has(op(f, 2), x) then
            break // give up
	  else
	    return(discont(op(f, 1), x, options, F));
	  end_if;
       end_if;
       // Note: the besselIJKY(v, x) is an entire function in v for fixed x.
       if F = Real or F = Undefined then
          return(
	      discont(op(f, 2), x, options, F) union
              solvelib::preImage(op(f, 2), x, 
                      piecewise([op(f, 1) in Z_ or Re(op(f, 1)) >=0, {}], 
                                [Otherwise, {0}]),
                  options));
       else
          return(
	      discont(op(f, 2), x, options, F) union
              solvelib::preImage(op(f, 2), x, 
                      piecewise([op(f, 1) in Z_, {}], 
                                [Otherwise, Dom::Interval(-infinity, [0])]),
                  options)) ;
       end_if;

    of "besselY" do
    of "besselK" do
      if has(op(f, 1), x) then
        if has(op(f, 2), x) then
          break // give up
        else
          return(discont(op(f, 1), x, F, options));
        end_if;
      end_if;
      if F = Real or F = Undefined then
        return(
        discont(op(f, 2), x, options, F) union
              solvelib::preImage(op(f, 2), x, {0}, options));
      else
          return(
	      discont(op(f, 2), x, options, F) union
              solvelib::preImage(op(f, 2), x, Dom::Interval(-infinity, [0]), options)) ;
      end_if;

    //====  
    // Ei 
    //====
    of "Ei" do 
      // Ei(n, x) has the same discontinuities as ln(x)
      if nops(f) = 1 then    
        r:= op(f, 1)
      else
        assert(nops(f) = 2);
        r:= op(f, 2)  
      end_if;
      return(discont(hold(ln)(r), x, options, F))
       
    //================================
    // functions without singularities
    //================================
    of "sin" do
    of "cos" do
    of "abs" do
    of "exp" do
    of "sinh" do
    of "cosh" do
    of "erf" do
    of "erfc" do
    of "erfi" do
    of "wrightOmega" do  
    of "airyAi" do
    of "airyBi" do
    of "Re" do
    of "Im" do
      return(discont(op(f, 1), x, options, F))

    //================================
    // functions with a discontinuity in 0
    //================================
    of "dirac" do
    of "heaviside" do
    of "sign" do
      if F=hold(Undefined) then
        return(discont(op(f, 1), x, options, F))
      else
        MAXEFFORT:= MAXEFFORT/2;
        return(discont(op(f, 1),x, options, F) union
               border(solve(op(f, 1),x, options)))
      end_if;
    // functions with a discontinuity in 1
    of "zeta" do
       MAXEFFORT:= MAXEFFORT/2;
       return(discont(op(f),x, options, F) union
              solve(op(f)=1,x, options))
       // functions with discontinuities in all integers
       // to be corrected for F=Dom::Complex !!!
    of "floor" do
    of "frac" do
    of "ceil" do
    of "trunc" do
      if F=hold(Undefined) then
        return(discont(op(f), x, options, F))
      else
        MAXEFFORT:= MAXEFFORT/2;
        return(discont(op(f), x, options, F) union
               border(solvelib::preImage(op(f), x, Z_, options)))
      end_if
    of "round" do
      if F=hold(Undefined) then
        return(discont(op(f), x, options, F))
      else
        i:=genident();
        MAXEFFORT:= MAXEFFORT/2;
        return(discont(op(f), x, options, F) union
              solvelib::preImage(op(f), x, Dom::ImageSet(i+1/2, i, Z_),
                                 options))
      end_if
   // functions with a discontinuity in -n
    of "psi" do
      MAXEFFORT:= MAXEFFORT/2;
      return(discont(op(f, 1), x, options, F) union
             solvelib::preImage(op(f, 1), x,
                                Z_ intersect Dom::Interval(-infinity..[0]),
                                options))

    of "signIm" do
      if F=hold(Undefined) then
        return(discont(op(f), x, options, F))
      elif F=Real then
        return(solve(op(f), x, Real) union discont(op(f), x, options, F))
      else
        return(solvelib::preImage(op(f), x, R_) union discont(op(f), x))
      end_if
    of "binomial" do  
      // binomial(n, k) does not exist if n is a negative integer and k is not an integer
      // everywhere else, binomial is continuous
      return(discont(op(f, 1), x, options, F) union
             discont(op(f, 2), x, options, F) union
             solve(op(f, 1) in Z_ and op(f, 1) < 0 and not op(f, 2) in Z_, x, options)
             )
    of "fact" do
       return(discont(rewrite(f, gamma), x, options, F))
   // logarithm: rewrite by ln
   // assume the base to be positive and not equal 1
   // either the base is constant and positive anyway, or it is an identifier
   // in the latter case, replace with a positive identifier
    of "log" do
      if type(op(f,1))=DOM_IDENT then
        assume(op(f, 1) > 0 and op(f, 1) <> 1);
      end_if;
      return(discont(rewrite(f, ln), x, options, F))
    of "polylog" do
      return(discont(op(f, 1), x, options, F) union
             discont(op(f, 2), x, options, F) union
             solve(op(f, 2)=1, x, options));
    of "lambertW" do
      if op(f, 1) = 0 then 
        return(discont(op(f, 2), x, options, F))
      elif is(op(f, 1) <> 0, Goal = TRUE)  then
        return(solve(op(f, 2) = 0, x) union discont(op(f, 2), x, options, F))
      end_if;
      break
   // integrals
    of "int" do
      if type(op(f,2))="_equal" then
        MAXEFFORT:= MAXEFFORT/3;
        return(discont(op(f,1),x, options, F) union
               discont(op(f,[2,2,1]),x, options, F)
               union discont(op(f,[2,2,2]),x, options, F))
      else
        return(discont(op(f,1),x, options, F))
      end_if
   end_case;

  procname(f, x, F, solvelib::nonDefaultOptions(options))
end_proc:


discont := funcenv(discont):

discont::type := "discont":
discont::print:= "discont":
discont::info := "discont -- discontinuities of a function":

// declare discont to be a set
discont::testtype:=
proc(x, T)
begin
  if T = Type::Set then
     TRUE
  elif T = Type::Arithmetical then
     FALSE
  else
     FAIL
  end_if
end_proc:

discont::avoidAliasProblem:=
proc(S, vars)
  local x, X;
begin
  x:= op(S, 2);
  if type(x) = "_equal" then
    x:= op(x, 1)
  end_if;
  if contains(vars, x) then 
    X:= solvelib::getIdent(C_, indets(S) union vars);
    S:= subs(S, x=X, Unsimplified)
  end_if;  
  S
end_proc:  

discont::freeIndets:=proc(S)
  local x, X;
begin
  x:= op(S, 2);
  if type(x) = "_equal" then
    x:= op(x, 1)
  end_if;
  assert(type(x)=DOM_IDENT);
  X := freeIndets([op(S,1..2)], args(2..args(0))) minus {x};
  if contains({args(2..args(0))}, All) then
    X := X union {hold(discont)};
  end_if;
  return(X);
end_proc:


// overload _plus
discont::_plus := stdlib::set_plus:

// overload _mult
discont::_mult := stdlib::set_mult:

// overload _power
discont::_power := stdlib::set_power:


// end of file
