/*++
gcd -- compute the gcd of polynomials or expressions

gcd(poly,...)
gcd(expr,...)

poly - polynomials
expr - polynomial expressions

gcd computes the gcd of polynomials or expressions.
++*/

gcd:=
proc()
  local a, b, c, p, q, n, T, T2, f, g, h, i, vars, 
        conv, normp, gcdp, dg, expo, subst,
        noFloats, hasfloat, rationalized, 
        useAlgExt, pq, ratsubsts, minpol, inds,
        makeMonic: DOM_PROC;
begin 

  makeMonic:= x -> multcoeffs(x, 1/lcoeff(x));

  if args(0) = 0 then
    return(0)
  end_if;

  // overloading
  for i from 1 to args(0) do
    if ((f:=(args(i))::dom::gcd)) <> FAIL then
      return(f(args()))
    end_if
  end_for;  
    
  a:= {args()};

  // The undocumented optional arguments "NoFloats"
  // and "Rationalized" serve for avoiding costly 
  // functions such as 'rationalize' and the search 
  // for floats

  noFloats:= FALSE;      // default value
  hasfloat:= FALSE;
  rationalized:= FALSE;  // default value
  useAlgExt:= FALSE;  // default value
  if has(a, "NoFloats") then
    a:= a minus {"NoFloats"}; 
    noFloats:= TRUE;
  end_if;
  if has(a, "Rationalized") then
    a:= a minus {"Rationalized"}; 
    rationalized:= TRUE;
    noFloats:= TRUE;
  end_if;
  if has(a, "UseAlgebraicExtension") then
    a:= a minus {"UseAlgebraicExtension"}; 
    useAlgExt:= TRUE;
    noFloats:= TRUE;
  end_if;

  // determine domtypes first to guarantee that gcd(0, poly(x)) is x.
  T:= map(a, domtype);
  hasfloat:= stdlib::hasfloat(a);
  
  a:= select(a, not iszero);
  
  if nops(a) = 0 then 
    // make sure that gcd(0, poly(0,[x])) and gcd(poly(0,[x]),0)
    // produce the same result: 
    if hasfloat then
      return(float(expr(args(1))));
    elif contains(T, DOM_INT) then  // there is an exact 0 in the arguments
      return(expr(args(1)));  // get rid of poly(0, [..])
    else
      return(args(1));
    end_if;
  end_if;
    
  if nops(a) = 1 then
    a:= op(a, 1);
    T2:= domtype(a);
    case T2 
    of DOM_POLY do
      if type(op(a, 3)) = DOM_DOMAIN then
        a:= gcdlib::norm_domp(a)
      elif op(a, 3) <> Expr then // IntMod
        a:= gcdlib::norm_modp(a)
      end_if;
      // it may be that we have removed integer zeroes before
      if T <> {DOM_POLY} then
        a:= expr(a)
      end_if;  
      if hasfloat then
         a:= float(a):
      end_if;
      return(a)
    of DOM_INT do 
    of DOM_RAT do  
      if hasfloat then
         a:= float(a);
      end_if;
      return(abs(a))
    of DOM_COMPLEX do
      if iszero(Re(a)) then
        a:= I*a
      end_if;
      if Re(a) < 0 then
        a:= -a
      end_if;
      if Im(a) < 0 then 
        a:= I*a
      end_if;
      if hasfloat then
         a:= float(a);
      end_if;
      return(a)
    end_case;  
    if hasfloat then
       a:= float(a);
    end_if;
    return(expr(a))
  end_if;  

  // special case: if the arguments contain DOM_FLOATs,
  // rationalize and apply float to the result
  if (not noFloats) and hasfloat then
     a:= numeric::rationalize(a);
     return(float(gcd(op(a), "NoFloats")))
  end_if;   
   
  // only integer or rational args? 
  
  if (T minus {DOM_INT, DOM_RAT, DOM_COMPLEX}) = {} then
    if T = {DOM_INT} then
      return(igcd(op(a)))
    end_if;
    if T minus {DOM_RAT, DOM_INT} = {} then
      return(igcd(op(map(a, numer))) / ilcm(op(map(a, denom))))
    end_if;
  
    // some Gaussian integers among the arguments
    g:= ilcm(op(map(a, denom)));
    
    return(igcd(op(map(a, _mult, g))) / g)
  end_if;
  
  a:= [op(a)];
  n:= nops(a);
  assert(n >= 2); 
  
  // if some arguments are expressions
  if T <> {DOM_POLY} then

    //======================================================
    // Walter, 9.10.09: new specialized code for expressions
    //======================================================
    [a, b, c] := split(a, p -> domtype(p) = DOM_POLY);
    // a is a list of DOM_POLYs, 
    // b is a list of expressions,
    // c is just a dummy
    if nops(b) = 1 then 
       a:= a . b; // just 1 expression. Proceed with the traditional
      // code below that will turn this expression into a DOM_POLY.
    else
       if rationalized then
         // the arguments were rationalized before. 
         // No need to do it again.
         [g, subst]:= [b, []];
       else
         [g, subst]:= subs([rationalize(subs(b, I = #I), FindRelations = ["_power", "exp"])], #I = I):
       end_if;
/*   
       // Experiments with sorting. 
       // Result: This does not seem to be worthwhile.
       if nops(g) > 2 then 
          // sort the polynomials from 'simple' to 'complex':
          g:= sort(g, proc(p1, p2) 
                      local d1, d2;
                      begin
                        vars1:= indets(p1);
                        if indets(p1) = {} then
                           return(TRUE);
                        end_if;
                        if indets(p2) = {} then
                           return(FALSE);
                        end_if;
                        if degree(p1, [vars1], vars1[1]) <
                           degree(p2, [vars2], vars2[1]) then
                           return(TRUE);
                        end_if;
                        return(FALSE);
                      end_proc):
       end_if:
*/
       g:= gcdlib::gcd_of_exprs(op(g));
       if g <> FAIL and a = [] then
          return(subs(g, subst, EvalChanges));
       elif g <> FAIL and a <> [] then
          // use gcd(a, b1, b2) = gcd(a, gcd(b1, b2)) to replace
          // all expressions b1, b2 etc. by their gcd, then 
          // proceed to the traditional code below. It computes 
          // the gcd of DOM_POLYs and a (single) expression.
          a:= a . [subs(g, subst, EvalChanges)];
          n:= nops(a);
       elif g = FAIL and a = [] then
          a:= b;
          // Proceed to the traditional code below. It computes the
          // gcd of several expressions after conversion to DOM_POLYs.
       elif g = FAIL and a <> [] then
          // Proceed to the traditional code below. It computes the
          // gcd of several DOM_POLYs. 
          a:= a . b;
          n:= nops(a);
       end_if;
    end_if;
    //=====================================================
    // end of Walter's new specialized code for expressions
    //=====================================================

    for i from 1 to n do 
      if domtype(a[i]) = DOM_POLY then
        if op(a[i], 3) <> Expr then
          error("Gcd of polynomials and expressions is not possible ".
              "unless coefficient ring of all polynomials is Expr")
        else
           a[i]:= expr(a[i])      
        end_if
      elif testargs() and not testtype(a[i], Type::Arithmetical) then
        error("Argument must be a polynomial or an arithmetical expression")
      end_if;
    end_for;
    
    // rationalize, then do a polynomial gcd
    a:= polylib::makerat(a, 1, TRUE);
    f:= subs(expr(gcd(op(op(a,1)))) / expr(lcm(op(op(a,2)))) , op(op(a,3)));
    return(f)
  end_if; 
  
  // now T = {DOM_POLY}
  T:= op(a[1], 3); // coefficient ring
  vars:= op(a[1], 2);

/*
  b:= map(a, expr);
  if contains(b, 1) > 0 or
     contains(b,-1) > 0 then
     return(poly(1, vars, T));
  end_if:
*/
  
  if testargs() then
    if nops({op(map(a, op, 2))}) > 1 or
       nops({op(map(a, op, 3))}) > 1 then
      error("polynomial types differ")
    end_if;
    
    
    if domtype(T) = DOM_DOMAIN then
      // need gcd method for coeffs 
      if T::gcd = FAIL then error("method 'gcd' missing") end_if;
      if T::_divide = FAIL then error("method '_divide' missing") end_if
    elif T<>hold(Expr)  // T = IntMod(p) 
      // polynomial with coeffs from Zp must have prime modulus 
      and not isprime(op(T,1)) then
      error("modulus must be prime")
    end_if
  end_if;   
  
  // variables that are not identifiers
  if {op(map(vars, domtype))} <> {DOM_IDENT} then
    g:= [ (if domtype(vars[i]) = DOM_IDENT then 
             vars[i]
           else 
             genident() 
           end_if) $ i=1..nops(vars)];
    a:= map(a, poly2list);
    a:= map(a, poly, g, T);
    f:= gcd(op(a));
    return(poly(poly2list(f), vars, T))
  end_if;
  
 
  // handle case n>2 
  // cf. vz Gathen/Gerhard, p.166, algorithm 6.45
  if n>2 then
    c:= map(a, content);
    // make elements of a primitive
    a:= zip(a, map(c, _invert), multcoeffs);
    c:= gcd(op(c));
    expo:= 0;
    while nops(a) > 1 do
      dg:= map(a, degree);
      i:= contains(dg, min(dg));
      f:= a[i];
      delete a[i];
      g:= _plus(multcoeffs(a[i], i^expo + i) $i=1..nops(a));
      h:= gcd(f, g);
      if nops(vars) = 1 then
        a:= map(a, divide, h, Rem);
        a:= select(a, not iszero);
        if nops(a) > 0 and op(f, 3) = Expr then 
          a:= normal(a, Recursive = FALSE)
        end_if;
        a:= select(a, not iszero).[h]   
      else
        a:= select(a, u -> divide(u, h, Exact) = FAIL);
        a:= map(a, groebner::normalf, [h]);
        a:= a.[h]
      end_if;  
      if nops(a) = n then 
        // if nothing has been removed, try different coefficients
        expo:= expo+1;
      else
        n:= nops(a)
      end_if;  
    end_while;
    return(multcoeffs(polylib::primpart(a[1]), c))
  end_if;   

    
  assert(n = 2);
  [p, q]:= a;
  
  conv:=id;
 
  if T = hold(Expr) then
    // polynomial with expressions as coeffs 
    // generate a sequence of types first; only then form a set!
    T2:= coeff(p), coeff(q);
    T2:= map(T2, domtype);
    T:= {T2};
    if T = {DOM_INT} then
      gcdp:= gcdlib::int_gcd;
      normp:= gcdlib::norm_intp;
    elif T minus {DOM_INT,DOM_RAT} = {} then
      gcdp:= gcdlib::int_gcd;
      normp:= F -> F;
      conv:= makeMonic;
      p:= polylib::primpart(p);
      q:= polylib::primpart(q)
    elif T minus {DOM_INT, DOM_RAT, DOM_COMPLEX} = {} then 
      // polynomials over Q(I) are handled separately
      return(gcdlib::gcd_QI(p, q))
    else
      inds := indets(map([p, q], poly2list));
      if useAlgExt=TRUE then
        [pq, ratsubsts, minpol] := [rationalize(map([p, q], poly2list), FindRelations=["_power"], MinimalPolynomials)];
      else
        ratsubsts := {};
      end_if;
      if ratsubsts = {} or
        traperror((T := fp::fold((p, d) -> Dom::AlgebraicExtension(d, poly(p, [op(indets(p) minus inds)], d)),
          Dom::Fraction(Dom::Polynomial(Dom::Rational)))(op(minpol)))) <> 0 or 
        has((pq := map(pq, poly, op(p, 2), T)), [FAIL]) then
        // we try our best... 
        return(poly(gcd(expr(p), expr(q)), op(p, 2..3)))
      end_if;
      [p, q] := pq;
      conv := p -> evalAt(poly(p, Expr), ratsubsts);
      if iszero(p) then return(conv(q)); end_if;
      if iszero(q) then return(conv(p)); end_if;
      gcdp:= gcdlib::dom_gcd;
      normp:= gcdlib::norm_domp;
    end_if
  elif domtype(T) = DOM_DOMAIN then
    gcdp:= gcdlib::dom_gcd;
    normp:= gcdlib::norm_domp;
  else // T = IntMod(p) 
    gcdp:= gcdlib::mod_gcd;
    normp:= gcdlib::norm_modp;
    conv:= makeMonic
  end_if;

  g:= gcdlib::special_cases(p, q, gcdp, normp);
  conv(g)
end_proc:

gcd:= prog::remember(gcd):

//=============================================
//=============================================
// utility gcdlib::gcd_of_exprs(p1, p2, ...)
// accepts polynomial expressions and returns
// their gcd or FAIL. (FAIL indicates that the
// input does not consist of proper polynomials
// in all the indets of p1, p2, .... In this 
// case, the calling function gcd proceeds to 
// the old gcd code that uses DOM_POLYs.)
//=============================================
//=============================================

gcdlib::gcd_of_exprs:= proc()
local a, b, b1, b2, bb, bbb, c, g, i, j, k, m, n, p, 
      vars, varsp, mycontent, exactdivide, extractContent,
      GCD;
begin
  assert(domtype(args(1)) <> DOM_LIST);
  assert(not hastype([args()], DOM_POLY));

  a:= [args()]; // a = list of polynomial expressions
  if has(a, FAIL) then 
     return(FAIL);
  end_if:
  n:= nops(a);
  if n = 1 then 
     return(a[1]);
  elif n = 2 and a[1] = a[2] then
     return(a[1]);
  elif n = 3 then 
    a[1]:= gcdlib::gcd_of_exprs(a[1], a[2]);
    if a[1] = FAIL then return(FAIL); end_if;
    return(gcdlib::gcd_of_exprs(a[1], a[3])); 
  elif n >= 4 then
    a[1]:= gcdlib::gcd_of_exprs(op(a[1 .. n div 2]));
    if a[1] = FAIL then return(FAIL); end_if;
    a[2]:= gcdlib::gcd_of_exprs(op(a[(n div 2)+ 1 .. n])): 
    if a[2] = FAIL then return(FAIL); end_if;
    return(gcdlib::gcd_of_exprs(a[1], a[2]));
  end_if;

/* =================================================================
  // pre-process sums by extracting common factors in all terms.
  // E.g. (x+1)*(x+2) + (x+1)^2*(x+3) should be written as
  // (x+1) * ( x+2 + (x+1)*(x+3) ). 
  // Note: such a process is also used inside stdlib::normalNoExpand 
  // (which is the most important consumer of gcd_of_exprs). Thus, 
  // for normal, it is not reasonable to do this again.
  a:= map(a,
          proc(x)
          name pre_factor_in_gcd;
          local y, t, t1, i, g, f, OK;
          begin
            if type(x) <> "_plus" then
               return(x)
            end_if;
            y:= [op(x)];
            t := table();
            case type(y[1])
              of "_power" do
                t[op(y[1], 1)] := op(y[1], 2);
                break;
              of "_mult" do
                map(y[1], proc(X)
                name expoTableAssign1;
                begin
                  if type(X)="_power" then
                    t[op(X, 1)] := op(X, 2);
                  else
                    t[X] := 1;
                  end_if;
                end_proc);
                break;
              otherwise
                t[y[1]] := 1;
            end_case;

            // heuristic check whether it is worthwhile to look
            // for common factors in the terms of a sum. All the
            // factors f, say, of the 1st term are stored in t. 
            // We substitute f = 0 in the entire expression. If
            // f is a (syntactial) factor in all terms, the 
            // expression vanishes:
            OK:= FALSE;
            for f in map([op(t)], op, 1) do 
              if iszero(subs(x, f = 0)) then
                 OK:= TRUE;
                 break;
              end_if;
            end_for;
            if not OK then
               return(x);
            end_if:
            // There is a common factor. Have a closer look:
            for i from 2 to nops(y) do
              t1 := t; t := table();
              case type(y[i])
                of "_power" do
                  if contains(t1, op(y[i], 1)) then
                    t[op(y[i], 1)] := min(t1[op(y[i], 1)], op(y[i], 2));
                  end_if;
                  break;
                of "_mult" do
                  map(y[i], proc(X)
                  name expoTableAssign2;
                  begin
                    if type(X)="_power" then
                      if contains(t1, op(X,1)) then
                        t[op(X, 1)] := min(t1[op(X, 1)], op(X, 2));
                      end_if;
                    else
                      if contains(t1, X) then
                        t[X] := 1;
                      end_if;
                    end_if;
                  end_proc);
                  break;
                otherwise
                  if contains(t1, y[i]) then
                    t[y[i]] := 1;
                  end_if;
              end_case;
              if nops(t)=0 then return(x); end_if;
            end_for;
            g := _mult(op(map([op(t)], X->op(X,1)^op(X,2))));
            x := g*map(x, _divide, g);
          end_proc);
==================================================================  */
  b:= a;

  //===========================================================
  // [content, y] = extractContent(x) extracts the content from
  // a polynomial expression p that is assumed to be a product
  // of powers of polynomials. The resulting expression q is 
  // again a product of powers of polynomial expressions in 
  // which each factor has the integer content 1. 
  // One has p = content*q.
  //============================================================
  extractContent:= proc(p)
    local c, content_;
    option remember;
  begin
    case type(p)
    of DOM_INT do
    of DOM_RAT do
       return([p, 1]);
    of DOM_FLOAT do
    of DOM_COMPLEX do
    of DOM_IDENT do
    of "_index" do
       return([1, p]);
    of "_mult" do
       content_:= 1:
       p:= map(p, proc(p) local c; begin
                    [c, p]:= extractContent(p);
                    content_:= content_*c;
                    p;
                  end_proc);
       return([content_, p]);;
    of "_power" do
       c:= content(op(p, 1));
       if iszero(c) then 
          // if p is the power of a poly that is not expanded,
          // op(p, 1) may be the zero poly in disguise. Note
          // that the content of the zero poly is zero.
          return([c, 0]);
       end_if;
       content_:= c^op(p, 2);
       p:= (op(p, 1)/c)^op(p, 2);
       return([content_, p]);;
    of "_plus" do
       content_:= content(p);
       if iszero(content_) then 
          return([content_, 0]);
       end_if;
       return([content_, p/content_]);;
    otherwise
       error("forgotten case");
    end_case;
  end_proc;

  // ========================================
  // make use of factorization of the polys
  // ========================================
  // Step 1: search for common factors in all polys

  // Step 1a: extract contents and put poly factors into lists
  bb:= a:
  c:= [1 $ nops(bb)];
  g:= 1;
  for i from 1 to nops(bb) do
    // bb = [p1, p2, p3, ... ]
    // convert all factors of the polys p.i
    // into a list:
    // bb = [p1_1*p1_2*... , p2_1*p2_2*... , ...]
    //    = [[p1_1,p1_2,..], [p2_1,p2_2,..], ...].
    // First, make all polynomials prime and extract
    // the gcd of the content. Store the contents in c:
    [c[i], bb[i]] := extractContent(bb[i]);
    if type(bb[i]) = "_mult" then
       bb[i]:= [op(bb[i], j) $ j = 1 .. nops(bb[i])];
    else 
       bb[i]:= [bb[i]];
    end_if;

    if contains(bb[i], -1) > 0 then
       // the i-th poly has a factor -1. For compatibility
       // with the original gcd code, throw this factor away.
       // (Note that a change of the sign is OK for gcd).
       g:= -g;
    end_if;

    // rewrite powers as [base, exponent], i.e.,
    // bb = [[[b1_1,e1_1], [b1_2,e2_1], ...], ...]
    bb[i]:= map(bb[i], b -> if type(b) = "_power" then
                               [op(b, 1), op(b, 2)]
                            else
                               [b, 1]
                            end_if);
  end_for;
  // compute the gcd of the contents and store it in g
  // (which will become a factor of the final gcd).
  c:= {op(c)}:
  if map(c, domtype) = {DOM_INT} then
     g:= g*igcd(op(c))
  elif map(c, domtype) minus {DOM_INT, DOM_RAT} = {} then 
     g:= g*(igcd(op(map(c, numer))) / ilcm(op(map(c, denom))))
  else
     // return FAIL to indicate that the old DOM_POLY based
     // case should be used.
     return(FAIL);
  end_if:

  // Step 1b: search for common factors in all polynomials
  // We need to look up the position of factor bb[1][k] of
  // the first poly in the list bb[k] representing the k-th
  // poly. For this, we do not need the exponents of the
  // factors. We throw it away to make the search easier:
  bbb:= map(bb, map, op, 1); 
  // Look for a common factor in all polynomials that will
  // enter the gcd. Store it in g (which presently is the
  // gcd of the contents of the polys)
  if iszero(g) then
     g:= 1;
  end_if:
  for j from 1 to nops(bb[1]) do
    n:= bb[1][j][2];
if n < 0 then
 return(FAIL);
end_if:
  
    // There is a factor bb[1][j][1]^n in the 1st polynomial.
    // Is some power of bb[1][j][1] present in all other polys?
    for k from 2 to nops(bb) do
      m:= contains(bbb[k], bb[1][j][1]);
      if m > 0 then 
         n:= min(n, bb[k][m][2]);
      else
         n:= 0;
         break;
      end_if;
    end_for:
    if n = 0 then
      break;
    end_if;
if n < 0 then
  return(FAIL);
end_if;
    // found a factor bb[1][j][1]^n that is present in all polys:
    g:= g*bb[1][j][1]^n;
    bb[1][j][2]:= bb[1][j][2] - n;
    for k from 2 to nops(bb) do
      m:= contains(bbb[k], bb[1][j][1]);
      assert(m > 0);
      bb[k][m][2]:= bb[k][m][2] - n;
    end_for:
  end_for:

  // The common factor is stored in g. 
  if g <> 1 and g <> -1 then
    b:= [_mult(op(map(bb[k], x -> x[1]^x[2]))) $ k = 1..nops(bb)];
    return(g*gcdlib::gcd_of_exprs(op(b))); 
  end_if;

  // ====================================================
  // utility to compute divide(p, q, Quo) for polynomial
  // expressions p, q (with q dividing p). In contrast to
  // divide, this utility tries to avoid the expansion 
  // implied by converting the arguments to poly objects:
  // ====================================================
  exactdivide:= proc(p, q)
    local i, d, OK, m, n, GCD;
    option remember;
  begin
      if q = 1 then
         return(p);
      end_if;
      if q = -1 then
         return(-p);
      end_if;
      if (type(p) = "_mult" or type(p) = "_power") then
         d:= p/q;     // let the kernel do the division
         OK:= TRUE;   
         // check that there are no factors 
         // with negative exponents:
         misc::maprec(d, {"_power"} = proc(x)
                          begin
                            if indets(op(x, 1)) <> {} and 
                               domtype(op(x, 2)) = DOM_INT and
                               op(x, 2) < 0 then
                                 OK:= FALSE;
                                 misc::breakmap();
                            end_if;
                            x;
                          end_proc);
         if OK then
            return(d);
         end_if;
      end_if;
      if type(q) = "_power" then
         [q, n]:= [op(q)];
         if type(p) = "_power" then
            [p, m]:= [op(p)];
         else
            m:= 1;
         end_if:
         if m = n then
            return( exactdivide(p, q)^n );
         elif m > n then
            GCD:= gcd(p, q);
            p:= exactdivide(p, GCD);
            q:= exactdivide(q, GCD);
            // p^m*GCD^m / (q^n*GCD^n)
            return(p^m*exactdivide(GCD^(m-n), q^n))
         else // 1 <= m < n
            return(exactdivide(exactdivide(p, q)^m, q^(n-m) ));
         end_if;
      end_if;
      if type(p) = "_power" then  // type(q) <> "_power"
         [p, m]:= [op(p)];
         GCD:= gcd(p, q);
         if GCD <> 1 and GCD <> -1 then
            p:= exactdivide(p, GCD);
            q:= exactdivide(q, GCD);
            // p^m*GCD^m / (q*GCD)
            return(p^m*exactdivide(GCD^(m-1), q));
         else
            p:= p^m;
         end_if;
      end_if;
      // There were some divisions left. E.g. for
      //   p = expand((x+1)*(x+2)), q = (x+1).
      // In this case, we do a polynomial expansion using divide.
      // However, we do not want to loose factors that are not
      // involved in the exact division! E.g., the division of
      // p=expand((x+1)*(x+2))*(x+3) by q=x+1 should still contain
      // x+3 as a separate factor.
      if type(p) = "_mult" then
         for i from 1 to nops(p) do
             d:= divide(op(p, i), q, Exact);
             if d <> FAIL then
                return(subsop(p, i = d));
             end_if;
         end_for;
      end_if;
      return(divide(p, q, Quo));
  end_proc:

  // Step 2: apply multiplication rules.
  // All 'trivial' gcd factors stemming from factorized
  // input polynomials were extracted above. Now, take
  // further care of factorized input by applying
  // gcd(b1*b2, c) = gcd(b1,c)*gcd(b2, c/gcd(b1,c))): 
  assert(nops(b) >= 2);
  for i from 1 to nops(b) do
    if type(op(b, i)) = "_mult" then
       //   gcd(b1*b2, c1,c2,..)
       // = gcd(b1*b2, gcd(c1,c2,..)    // c = gcd(c1, c2, ...)
       // = gcd(b1,c)*gcd(b2, c/gcd(b1,c)))
       b1:= op(b, [i, 1]);
       b2:= _mult(op(b, [i, j]) $ j = 2..nops(op(b[i])));;
       c:= subsop(b, i = null());
       if nops(c) > 1 then 
          c:= gcdlib::gcd_of_exprs(op(c));
       else
          c:= op(c);
       end_if;
       if c = 1 then 
          next;
       end_if;
       b1:=gcdlib::gcd_of_exprs(b1, c);
       c:= exactdivide(c, b1);
       b2:= gcdlib::gcd_of_exprs(b2, c);
       return(b1*b2);
    elif type(op(b, i)) = "_power" then
       // gcd(b1^n, c2, c3, ..)
       // Important: if there is a non-trivial gcd, it
       // is some power (<= n) of a factor of b1.
       // Return the gcd in factored form!
       n:= op(b, [i, 2]);
       if n < 0 then 
          return(FAIL);
       end_if;
       b1:= op(b, [i, 1]);
       c:= subsop(b, i = null());
       g:= 1;
       while 1 <= n do
          if indets(b1) = {} or
             indets(c) = {} then
             break;
          end_if;
          GCD:= gcdlib::gcd_of_exprs(b1, op(c));
          if GCD <> 1 and GCD <> -1 then
             g:= g*GCD;
             c:= map(c, exactdivide, GCD);
             n:= n - 1;
          else
             break;
          end_if;
       end_while;
       return(g);
    end_if;
  end_for:

  // ==============================
  // utility for fast extraction of the content
  // of expressions. It calls the standard
  // procedure 'content` but is smarter about
  // products and powers.
  // ==============================
  mycontent:= proc(p)
    option remember;
  begin
    case type(p) 
    of "_power" do
       return(mycontent(op(p,1))^op(p, 2));
    of "_mult" do
       return(map(p, mycontent));
    otherwise 
       return(content(p));
    end_case;
  end_proc:

  // ====================================================================
  // Step 3: convert to poly with a minimal number of unknowns.
  // Consider gcd(p(x1, x2), q(x1)). The gcd can only be a function of
  // x1. However, we may not convert p, q to polys in x1, with symbolic
  // parameters x2, since this would lead to enormous expression swell.
  // Instead, we need to write 
  //   poly(p(x1, x2), [x2]) = p0(x1) + p1(x1)*x2 + p2(x1)*x2^2 + ... 
  // and compute
  //   gcd(p(x1, x2),q(x1)) = gcd(p0(x1),p1(x1),p2(x1),...,q(x1)):
  // Since the multivariate gcd is extremely expensive (the costs grow
  // exponentially in the number of unknowns), it pays off to replace a
  // gcd of 2 polynomials in many variables by a gcd of many polynomials
  // in a few variables:
  // ====================================================================

  vars:= _intersect(indets(a) $ a in b); 
  // vars are the common variables. The gcd may depend on these vars.
  if vars = {} then
     b:= map({op(b)}, mycontent);
     if has(b, FAIL) then
        return(FAIL);
     end_if;
     return(gcd(op(b), "Rationalized"));
  end_if;

  for i from 1 to nops(b) do
    p:= op(b, i):
    varsp:= indets(p) minus vars;
    if nops(varsp) = 0 then
       p:= {p};
    else

       p:= poly(p, [op(varsp)]);
       if p = FAIL then
          return(FAIL);
       end_if;
       p:= {coeff(p)};
/*
       p:= {poly(p, [op(indets(p))])};
       for var in varsp do
         p:= map(p,  q -> (coeff(q, var, i) $ i = 0..degree(q, var)));
         if has(p, FAIL) then 
            return(FAIL);
         end_if;
         p:= select(p, _not@iszero);
       end_for;
       p:= {poly(p, [op(vars)])};
*/
    end_if;
    // replace the i-th entry of b by a set of DOM_POLYs:
    b:= subsop(b, i = p);
  end_for:
  // flatten the list of sets:
  assert(domtype(b) = DOM_LIST);
  b:= [op(_union(op(b)))];
  b:= map(b, poly, [op(vars)]);
  if has(b, FAIL) then
    return(FAIL);
  end_if;
  return(expr(gcd(op(b), "Rationalized")))
 end_proc:

gcdlib::gcd_of_exprs:= prog::remember(gcdlib::gcd_of_exprs):
