/*++
infinity -- representing the real positive infinity

An infinite number is represented as an element of the domain
stdlib::Infinity. An element of this domain has one operand:

- The sign.

- Given the sign s the elements represent limes(s*a, a = infinity).

The domain element 'infinity' (which represents the positive real
infinity) is defined here. Other domain elements are only created
via the operations of this domain and NOT via 'new'.
++*/

stdlib::Infinity:= newDomain("stdlib::Infinity"):
stdlib::Infinity::create_dom:=hold(stdlib::Infinity):

infinity:= new(stdlib::Infinity, 1):

stdlib::Infinity::info:= "Domain 'stdlib::Infinity': representing infinity":
stdlib::Infinity::interface:= {}:

stdlib::Infinity::expr:= id:

stdlib::Infinity::print:= 
proc()
begin
  extop(args(1),1) * hold(infinity)
end_proc:

stdlib::Infinity::isNeg:= x->generate::isNeg(extop(x)):

stdlib::Infinity::testtype:=
proc(x, T)
begin
if T = Type::Arithmetical then
  TRUE
else
  FAIL
end_if
end_proc:

stdlib::Infinity::new :=
proc()
begin
  error("Elements of stdlib::Infinity cannot be created directly")
end_proc:

stdlib::Infinity::create :=
proc(dir)
  local signdir;
begin
  if testtype(dir, stdlib::Infinity) then dir
  else
    if type(dir)="_mult" and testtype(op(dir, nops(dir)), Type::Real) then
      if op(dir, nops(dir))>0 then
        dir := _mult(op(dir, 1..nops(dir)-1));
      else
        dir := subsop(dir, nops(dir)=-1);
      end_if;
    end_if;
    if iszero(dir) then
      return(undefined);
    end_if;
    signdir := sign(dir);
    if indets(signdir)={} or length(signdir) <= length(dir) then
      new(dom, signdir);
    else
      new(dom, dir);
    end_if;
  end_if;
end_proc:


// an infinity may evaluate to something else if its operand
// contains an identifier whose value or properties have changed since the
// infinity was generated. We might handle the case that op(inf, 1) contains
// no identifier separately to save running time !?
stdlib::Infinity::evaluate:=
proc(inf)
begin
  dom::create(eval(op(inf, 1)))
end_proc:

stdlib::Infinity::subs:=
proc(inf, subst)
  local unsimp: DOM_BOOL;
begin
  if args(0) = 1 then
    return(inf)
  end_if;
  unsimp:= contains({args()}, Unsimplified) or contains({args()}, EvalChanges) ;
  assert(args(0) = 2 or unsimp);
  // other cases should be caught by the kernel
  if type(subst) = DOM_LIST  then
    return(subs(inf, op(subst), args(3..args(0))))
  end_if;
  assert(type(subst) = "_equal");
  if op(subst, 1) = inf then
    return(op(subst, 2));
  elif op(subst, 1) = infinity then
    return(extop(inf)*op(subst, 2));
  end_if;
  stdlib::Infinity::create(subs(extop(inf), args(2..args(0))));
end_proc:

stdlib::Infinity::TeX:=
proc()
begin
  case extop(args(1),1)
    of 1 do
      "\\infty";
      break;
    of -1 do
      "- \\infty";
      break;
    otherwise
      generate::TeX(extop(args(1), 1))."\\infty";
      break;
    end_case
end_proc:

stdlib::Infinity::_negate:=
proc()
  option noDebug;
begin
  extsubsop(args(1), 1 = -extop(args(1),1))
end_proc:

stdlib::Infinity::_invert:=
proc()
  option noDebug;
begin
  0
end_proc:

stdlib::Infinity::_subtract:=
proc()
  option noDebug;
begin
  args(1)+(-args(2))
end_proc:

stdlib::Infinity::_divide:=
proc(x, y)
  option noDebug;
begin
  x*(1/y)
end_proc:

stdlib::Infinity::_not:=
proc()
begin
  error("invalid operand")
end_proc:

stdlib::Infinity::_power:=
proc(x, pow)
begin
  if expr2text(type(pow)) = "piecewise" then
    return(piecewise::extmap(pow, y -> x^y))
  end_if;
  if domtype(x) = DOM_SET then
    return(map(x, _power, pow))
  end_if;
  case domtype(pow)
    of stdlib::Infinity do
      if // no! x = infinity or // not 'undefined' for infinity^infinity!
         (x::dom = stdlib::Infinity and extop(x)<>1) or
         x = complexInfinity or
         x = undefined
        or (x= infinity and (pow::dom = stdlib::Infinity and extop(pow)<>1))
        then
        return(undefined)
      end_if;

      if not testtype(x, Type::Arithmetical) then
        error("Illegal argument")
      end_if;
      if pow = infinity then
        return(piecewise([x=1, 1],
                         [x>1, infinity],
                         [abs(x) < 1, 0]
                      //   abs(x) >= 1 and not x>0 -> undefined
                         )
               )
      else
        return((x^extop(pow,1))^infinity)
      end_if;
      // NOT REACHED
      assert(FALSE);
      break
    of stdlib::CInfinity do
      // infinity^complexInfinity .-> undefined
      return(undefined)
    of DOM_FLOAT do
      if iszero(pow) then return(1.0) end_if;
    of DOM_INT do
      if pow = 0 then return(1) end_if;
    of DOM_RAT do
      if pow < 0 then return(0) end_if;
      (extop(x,1)^pow) * infinity;
      break
    of DOM_SET do
      return(map(pow, t -> x^t ))
    otherwise
      if not testtype(pow, Type::Arithmetical) then
        error("Illegal argument")
      end_if;
      if x = infinity then
        piecewise([pow > 0, infinity],
                  [Re(pow) < 0, 0])
      elif x = -infinity then
        piecewise([Re(pow) < 0, 0])
      else
        hold(_power)(x, pow);
      end_if
  end_case
end_proc:

stdlib::Infinity::_plus:=
proc()
  local a, b, dummy, Res, Ims, Re_posinf, Re_neginf, Im_posinf,
     Im_neginf, Re_zero, Im_zero, Re_hasposinf, Re_hasneginf, Im_hasposinf, Im_hasneginf;
begin
  // handle sets separately
  [b, a, dummy]:= split([args()], x-> domtype(x) = DOM_SET);
  assert(nops(dummy) = 0);
  if nops(b) > 0 then
    b:= _plus(op(b));
    a:= _plus(op(a));
    return(map(b, _plus, a))
  end_if;
  // handle other Cat::Set separately
  [a, b, dummy]:= split([args()], x-> x::dom::hasProp(Cat::Set) = TRUE);
  assert(nops(dummy) = 0);
  if nops(a) > 0 then
    b:= _plus(op(b));
    a:= _plus(op(a));
    // in this order! uses a::dom::_plus
    return(a+b)
  end_if;
  // handle other sets separately
  [a, b, dummy]:= split([args()], x-> testtype(x, Type::Set) and not testtype(x, Type::Arithmetical));
  assert(nops(dummy) = 0);
  if nops(a) > 0 then
    b:= _plus(op(b));
    a:= _plus(op(a));
    return(Dom::ImageSet(#x + b, #x, a))
  end_if;
  // handle piecewises separately
  [b, a, dummy]:= split([args()], x-> expr2text(domtype(x)) = "piecewise");
  assert(nops(dummy) = 0);
  if nops(b) > 0 then
    b:= _plus(op(b));
    a:= _plus(op(a));
    return(b+a)
  end_if;
  // handle polynomials separately
  [a, b, dummy]:= split([args()], x-> domtype(x) = DOM_POLY);
  assert(nops(dummy) = 0);
  if nops(a) > 0 then
    a:= _plus(op(a));
    b:= _plus(op(b));
    a := subsop(a, 1=op(a, 1)+b);
    if a = FAIL then
      error("Illegal argument");
    end_if;
    return(a);
  end_if;
  // select infinities and others
  [a, b, dummy]:= split([args()], not (stdlib::Infinity)::thisDomain);
  assert(nops(dummy) = 0);
  // skip real part of constants
  if contains(a, complexInfinity) > 0 or
    contains(a, undefined) > 0 then
    return(undefined)
  end_if;  

  b:= map({op(b)}, extop);
  // a*infinity - a*infinity = undefined
  if b intersect -b <> {} then
    return(undefined);
  end_if;
  Res := map(b, Re);
  Ims := map(b, Im);

  // Goal = TRUE to ask for "is one of them known to be positive?"
  Re_hasposinf := is(_or(op(map(Res, x -> x > 0))), Goal=TRUE);
  Re_hasneginf := is(_or(op(map(Res, x -> x < 0))), Goal=TRUE);
  Im_hasposinf := is(_or(op(map(Ims, x -> x > 0))), Goal=TRUE);
  Im_hasneginf := is(_or(op(map(Ims, x -> x < 0))), Goal=TRUE);
  if (Re_hasposinf and Re_hasneginf) or (Im_hasposinf and Im_hasneginf) then
    return(undefined);
  end_if;

  Re_posinf := is(_and(op(map(Res, x -> x > 0))), Goal=TRUE);
  Re_neginf := is(_and(op(map(Res, x -> x < 0))), Goal=TRUE);
  Im_posinf := is(_and(op(map(Ims, x -> x > 0))), Goal=TRUE);
  Im_neginf := is(_and(op(map(Ims, x -> x < 0))), Goal=TRUE);

  // known not to be zero?
  Re_zero := is(_and(op(map(Res, x -> x = 0))));
  Im_zero := is(_and(op(map(Ims, x -> x = 0))));

  // real and imaginary part of finite parts are dropped
  // unless the infinite parts have zero real/imaginary parts.
  if Im_zero = FALSE and Re_zero = FALSE then
    a := [];
  end_if;
  if Im_zero = TRUE then
    a:= map(a, 
    proc(u) 
    begin 
      if testtype(u, Type::Constant) then 
        I*Im(u)
      elif is(u in R_, Goal = TRUE) then
        0
      else
        u
      end_if
    end_proc
    )
  end_if;
  if Re_zero = TRUE then
    a:= map(a, u-> if testtype(u, Type::Constant) then
                     Re(u)
                   else
                     u
                   end_if);
  end_if;
  if Re_posinf = TRUE or Re_neginf = TRUE then
    a := select(a, x -> not is(x in R_, Goal=TRUE));
  end_if;
  if Im_posinf = TRUE or Im_neginf = TRUE then
    a := select(a, x -> not is(I*x in R_, Goal=TRUE));
  end_if;
    
  a:= _plus(op(a));
  if a = undefined or a = complexInfinity then
    return(undefined)
  end_if;

  if Re_posinf then
    if Im_posinf then
      return(hold(_plus)(infinity, I*infinity));
    elif Im_neginf then
      return(hold(_plus)(infinity, -I*infinity));
    end_if;
    b := {1} union (I*{op(Ims)} minus {0});
  elif Re_neginf then
    if Im_posinf then
      return(hold(_plus)(-infinity, I*infinity));
    elif Im_neginf then
      return(hold(_plus)(-infinity, -I*infinity));
    end_if;
    b := {-1} union (I*{op(Ims)} minus {0});
  end;
  if Im_posinf then
    // can't have Re_posinf=TRUE or Re_neginf=TRUE
    b := {I} union (Res minus {0});
  elif Im_neginf then
    b := {-I} union (Res minus {0});
  end_if;

  b := map(sort(b), dom::create);
  if nops(b) > 1 then
    if iszero(a) then
      b := hold(_plus)(op(b));
    elif testtype(a, "_plus") then
      b := hold(_plus)(op(b), op(a));
    else
      b := hold(_plus)(op(b), a);
    end_if;
    return(b);
  elif nops(b)=0 then
    // should not happen!
    return(a)
  else
    b := op(b);
    if iszero(a) then
      b
    elif type(a) = "_plus" then  // flat operands of a
      hold(_plus)(b, op(a))
    else
      hold(_plus)(b, a)
    end_if;
  end_if;
end_proc:

stdlib::Infinity::_mult:=
proc()
  local a, b, c, f, dummy, sgn:DOM_INT;
begin
  // handle sets separately
  [a, b, dummy]:= split([args()], x-> domtype(x) = DOM_SET);
  assert(nops(dummy) = 0);
  if nops(a) > 0 then
    b:= _mult(op(b));
    a:= _mult(op(a));
    return(map(a, _mult, b))
  end_if;
  // handle other Cat::Set separately
  [a, b, dummy]:= split([args()], x-> x::dom::hasProp(Cat::Set) = TRUE);
  assert(nops(dummy) = 0);
  if nops(a) > 0 then
    b:= _mult(op(b));
    a:= _mult(op(a));
    // in this order! uses a::dom::_mult
    return(a*b)
  end_if;
   // handle other sets separately
  [a, b, dummy]:= split([args()], x-> testtype(x, Type::Set) and not testtype(x, Type::Arithmetical));
  assert(nops(dummy) = 0);
  if nops(a) > 0 then
    b:= _mult(op(b));
    a:= _mult(op(a));
    return(Dom::ImageSet(#x * b, #x, a))
  end_if;
  // handle piecewises separately
  [b, a, dummy]:= split([args()], x-> expr2text(domtype(x)) = "piecewise");
  assert(nops(dummy) = 0);
  if nops(b) > 0 then
    b:= _mult(op(b));
    a:= _mult(op(a));
    return(b*a)
  end_if;
  // handle polynomials separately
  [b, a, dummy]:= split([args()], x-> domtype(x) = DOM_POLY);
  assert(nops(dummy) = 0);
  if nops(b) > 0 then
    b:= _mult(op(b));
    a:= _mult(op(a));
    b := multcoeffs(b, a);
    if b = FAIL then
      error("Illegal argument");
    end_if;
    return(b);
  end_if;

  b:= [args()];

  if contains(b, undefined) <> 0 then return(undefined) end_if;
  if contains(b, complexInfinity) <> 0 then return(complexInfinity) end_if;

  // f is only a dummy arg:

  [a,c,f]:=split(b, (stdlib::Infinity)::thisDomain);

  // multiply the signs of all infinities
  sgn:=_mult(op(map(a, extop, 1)));

  for b in c do
    if iszero(b) then
      return(undefined);
    end_if;
    sgn := sgn*b;
  end_for;

  b:=dom::create(sgn);
end_proc:

//------------------------------------------------------------
// the following routine checks that the infinity object
// is semantically a product such as I*infinity or a*infinity: 
//------------------------------------------------------------
stdlib::Infinity::is_mult:= x -> bool(x <> infinity):
    

stdlib::Infinity::abs := inf -> infinity:

stdlib::Infinity::_less:=
proc(x,y)
begin
  if x = infinity or y = -infinity then
    FALSE
  elif x = -infinity then
    bool(y=infinity) or numeric::isreal(y)
  elif y = infinity then
    numeric::isreal(x)
  else
    // throw error 1003, "Can't evaluate to boolean"
    context(hold(bool)(I<0));
  end_if
end_proc:

stdlib::Infinity::_leequal:= (x, y) ->bool(x=y or x<y):

stdlib::Infinity::min:=
proc()
  local b;
begin
  b:= { args() };
  if contains(map(b, domtype), DOM_COMPLEX) or
    nops(select(b, x -> _lazy_and((stdlib::Infinity)::thisDomain(x),
      is(extop(x) in R_, Goal=FALSE)=FALSE))) > 0 then
    error("complex numbers are not allowed as arguments")
  elif contains(b, -infinity) then
    -infinity
  else
    // remove all +infinity
    b:=select(b, _unequal, infinity);
    if b={} then
      infinity
    else
      b := split(b, dom::thisDomain);
      if nops(b[1])>0 then
        if nops(b[2])>0 then
          b[2] := min(op(b[2]));
          hold(min)(b[2], op(b[1]));
        else
          hold(min)(op(b[1]));
        end_if;
      else
        min(op(b[2]));
      end_if;
    end_if
  end_if
end_proc:


stdlib::Infinity::max:=
proc()
  local b;
begin
 b:= { args() };
  if contains(map(b, domtype), DOM_COMPLEX) or
    nops(select(b, x -> _lazy_and((stdlib::Infinity)::thisDomain(x),
      is(extop(x) in R_, Goal=FALSE)=FALSE))) > 0 then
    error("complex numbers are not allowed as arguments")
  elif contains(b, infinity) then
    infinity
  else
    // remove all -infinity
    b:=select(b, _unequal, -infinity);
    if b={} then
      -infinity
    else
      b := split(b, dom::thisDomain);
      if nops(b[1])>0 then
        if nops(b[2])>0 then
          b[2] := max(op(b[2]));
          hold(max)(b[2], op(b[1]));
        else
          hold(max)(op(b[1]));
        end_if;
      else
        max(op(b[2]));
      end_if;
    end_if
  end_if
end_proc:


stdlib::Infinity::intmult:=
proc()
begin
  if iszero(args(2)) then
    undefined
  elif
    args(2) > 0 then
    args(1)
  else
    extsubsop(args(1), 1 = -extop(args(1),1))
  end_if
end_proc:

stdlib::Infinity::thisDomain:=
proc()
begin
  bool(domtype(args(1)) = stdlib::Infinity)
end_proc:


stdlib::Infinity::simplify:=
proc(inf)
begin
  dom::create(simplify(op(inf, 1), args(2..args(0))))
end_proc:



stdlib::Infinity::has:=(x,y) -> _lazy_or(y=infinity, has(extop(x), y)):

stdlib::Infinity::sign:=
inf -> sign(Re(extop(inf,1))) + I*sign(Im(extop(inf,1))):

stdlib::Infinity::round := x -> x:
stdlib::Infinity::ceil := x -> x:
stdlib::Infinity::floor := x -> x:
stdlib::Infinity::trunc := x -> x:
stdlib::Infinity::frac := x -> undefined:

stdlib::Infinity::O:= proc() begin undefined end_proc:

stdlib::Infinity::Re := 
proc(x)
  local rx;
begin
  rx := Re(extop(x));
  // try not to use piecewise, to avoid loading it
  if iszero(rx) then return(rx); end_if;
  piecewise([rx=0, rx], [rx<>0, dom::create(rx)]);
end_proc:

stdlib::Infinity::Im :=
proc(x)
  local ix;
begin
  ix := Im(extop(x));
  // try not to use piecewise, to avoid loading it
  if iszero(ix) then return(ix); end_if;
  piecewise([ix=0, ix], [ix<>0, dom::create(ix)]);
end_proc:

stdlib::Infinity::ceil:= id:
stdlib::Infinity::floor:= id:
stdlib::Infinity::trunc:= id:
stdlib::Infinity::frac:= () -> undefined:

stdlib::Infinity::exp:=
inf -> piecewise([Re(extop(inf)) < 0, 0],
                 [extop(inf) > 0, infinity]
                 ):

stdlib::Infinity::ln :=
inf -> I*arg(extop(inf)) + infinity:

stdlib::Infinity::float:=
inf -> RD_INF*float(extop(inf)):

stdlib::Infinity::hull :=
inf -> hull(RD_INF * extop(inf)):


stdlib::Infinity::hasmsign:= x -> stdlib::hasmsign(extop(x, 1)):


stdlib::Infinity::Content :=
proc(Out, data)
  local res;
begin
  res := Out::Ccn(["type"="constant"], Out::Cinfinity()):
  if extop(data) = 1 then
    res
  elif extop(data) = -1 then
    Out(hold(-infinity))
  else
    Out(hold(_mult)(extop(data, 1), infinity))
  end_if
end_proc:

// end of file