//      
// kg, 11.05.93 

/*++
info -- print short information

info(d)
info(f)

d - domain
f - function

info(d) prints a short information about the domain d. d must have a
method 'info' or a text attribute 'info'. Either the method 'd::info'
is executed or the text 'info' is printed; in the later case, if the
domain has the attibute 'interface', which must be a set of interface
names, these names are printed also.

info(f) prints a short information about the function f. f must have
a string attribute "info". The string is printed unquoted.
++*/

info:= proc(dat) begin
    if args(0) = 0 then 
        info::helpOfTheDay()     
    elif args(0) = 1 then 
        if slot(domtype(dat), "printInfo") = FAIL then
           print(NoNL, "" . expr2text(dat) . " -- of domain type '".
                       expr2text(domtype(dat))."'\n")
        else
           slot(domtype(dat), "printInfo")(dat)
        end_if
    else
        error("wrong no of args") 
    end_if
end_proc:

info := funcenv(info):

info::helpOfTheDay:=
proc()
  local all;
begin
  print(NoNL, "-- Help page of the day:   ");
  all := select(Pref::allComplete(), x -> not x[1] = "_");
  all := [op(all), op(all minus map(stdlib::OPTIONS, expr2text)) $ 10];
  print(NoNL, "?".all[ceil(frandom(CurrentTime)()*nops(all))]."\n");
  null()
end_proc:

DOM_DOMAIN::printInfo:= proc(dm)
    local dom_name, tab, expo, info_str;
begin

    // Allow for overloading by the domain
    if dm::info <> FAIL and domtype(dm::info) <> DOM_STRING then
	dm::info();
	return()
    end_if;
    
    // Fetch the domain name
    dom_name:= (if dm::Name = FAIL then dm::key else dm::Name end_if);
    if dom_name = FAIL then dom_name:="<<unknown>>" end_if;
    if domtype(dom_name) <> DOM_STRING then
	dom_name := expr2text(dom_name);
    end_if;

    if dm::info_str <> FAIL then
	info_str := dm::info_str;
    elif dm::info <> FAIL then
	info_str := dm::info;
    else
	info_str := "".dom_name." -- a domain"
    end_if;
    // Print the info string
    print(NoNL, info_str."\n");

    // Get the interface and the exported entries
    tab:= dm::interface;
    if tab = FAIL then
	return()
    elif not testtype(tab, Type::SetOf(DOM_IDENT)) then
	error("in attribute \"interface\"")
    end_if;

    expo:= dm::exported;
    if domtype(expo) <> DOM_SET then expo:= {} end_if;
    if domtype(dm::_exported) = DOM_SET then
	expo:= expo union dm::_exported
    end_if;

    // Print the interface
    if nops(tab) > 0 then
	tab:= map(select(tab, f -> not contains(expo, f)),
		  f -> dom_name."::".expr2text(f));
	if nops(tab) > 0 then
	    print(NoNL, "\n-- Interface:\n");
	    output::tableForm(tab, Append = ",", hold(Unquoted))
	end_if
    else
	print(NoNL, "\nNo Interface.\n");
    end_if;

    // Print the exported entries
    if nops(expo) <> 0 then
	print(NoNL, "\n-- Exported:\n");
	output::tableForm(expo, hold(Append) = ",", hold(Unquoted));
    end_if;
    null()
end_proc:

DOM_FUNC_ENV::printInfo:=
  proc(f)
    local op0, procName, comment;
  begin
    if slot(f, "info") <> FAIL then
      print(NoNL, slot(f, "info")(f)."\n");
      return();
    end_if;
    op0 := op(f, 1);
    if domtype(op0) = DOM_EXEC then
      if nops(op0) = 5 then
          print(NoNL, "".expr2text(f)." -- a module function of module '".
                expr2text(op(op0,5))."'\n")
      else
          print(NoNL, "".expr2text(f).
                " -- a function of the system kernel".
                if not contains(Pref::noComplete(), expr2text(f)) then
                  " [try ?".op(op0, 3)." for help]"
                else null() end
                ."\n")
      end:
      return();
    end;

    if domtype(op0) = DOM_PROC then
      procName := op(op0, 6);
      if slot(f, "info_str") <> FAIL then
        comment := slot(f, "info_str");
      elif procName = NIL then
        comment := " -- a function environment";
      else
        if domtype(procName) <> DOM_STRING then
          procName := expr2text(procName);
        end;
	comment := " -- a library procedure [try ?".procName." for help]";
      end_if;
      print(NoNL, op0::dom::expr2textInterface(op0, comment) . "\n");
      return():
    end;
    // If nothing else ...
    print(NoNL, "".expr2text(f)." -- a function environment\n");
    null()
  end_proc:

DOM_IDENT::printInfo:=
  proc(ID)
    local noInfo;
  begin
    noInfo := TRUE:
    
    if contains(stdlib::OPTIONS, ID) then
      noInfo := FALSE;
      print(NoNL, "" . ID . " -- an option\n")
    elif  ID = PI then
      noInfo := FALSE;
      print(NoNL, "" . ID . " -- the length of the circumference/diameter ratio of the circle = 3.141..\n")
    elif  ID = EULER then
      noInfo := FALSE;
      print(NoNL, "" . ID . " -- the Euler-Mascheroni constant limit(sum(1/i,i=1..n)-ln(n),n=infinity) = 0.5772..\n")
    elif  ID = CATALAN then
      noInfo := FALSE;
      print(NoNL, "" . ID . " -- the Catalan constant sum((-1)^i/(2*i+1)^2, i=0..infinity) = 0.9159..\n")
    elif protected(ID) <> hold(None) then
      noInfo := FALSE;
      print(NoNL, "" . ID . " -- protected identifier\n")
    end:
    
    if properties(ID) <> FAIL then
      noInfo := FALSE;
      print(NoNL, "" . ID . " -- identifier with property '".expr2text(getprop(ID))."'\n")
    end:
    
    if noInfo then
      print(NoNL, "".ID." -- an unprotected identifier without properties\n")
    end_if:
    null()
  end_proc:

DOM_EXPR::printInfo:=
  proc(ID)
  begin
    print(NoNL, "" . expr2text(ID) . " -- an expression of type ".
          expr2text(type(ID))."\n");
    null()
  end_proc:

// Routine that pretty prints the arguments of a procedure with
// their type and default values
DOM_PROC::expr2textParameters :=
  proc(ID: DOM_PROC) : DOM_STRING
    local parameters, types, defaultValues, i, result;
  begin
    parameters    := op(ID,1);
    if parameters <> NIL then
	parameters := [parameters];
    else
	parameters := [];
    end_if;
    types         := op(ID,7):
    if types <> NIL then
	types := [types];
    else
	types := [NIL $ nops(parameters)];
    end_if;
    defaultValues := op(ID, 10):
    if defaultValues <> NIL then
	defaultValues := [defaultValues];
    else
	defaultValues := [NIL $ nops(parameters)];
    end_if;
    result := "";
    for i from 1 to nops(parameters) do
	if i>1 then
	    result := result . ", ";
	end_if;
	result := result . expr2text(parameters[i]);
	if defaultValues[i] <> NIL then
	    result := result . " = " . expr2text(defaultValues[i]);
	end_if;
	if types[i] <> NIL then
	    result := result . " : " . expr2text(types[i]);
	end_if;
    end_for;
    result;
  end_proc;

DOM_PROC::expr2textInterface :=
  proc(ID: DOM_PROC,
       comment = " -- a procedure of domain type 'DOM_PROC'" : DOM_STRING)
      : DOM_STRING
    local procName, returnType;
  begin
    procName := op(ID, 6);
    if procName = NIL then
      procName := "anonymous";
    end:
    if domtype(procName) <> DOM_STRING then
      procName := expr2text(procName);
    end;
    procName := procName . "(".dom::expr2textParameters(ID).")";
    returnType := op(ID,8):
    if returnType <> NIL then
	procName := procName . " : " . expr2text(returnType);
    end_if;
    if length(comment) + length(procName) > TEXTWIDTH then
	procName := procName . "\n";
    end_if;
    procName.comment;
  end_proc:

DOM_PROC::printInfo:=
  proc(ID)
  begin
    print(NoNL, dom::expr2textInterface(ID)."\n");
  end_proc:

// end of file 
