//   


/*
        limit  -- computing the limit of expressions

        limit(f, x[=a] [,dir] [,options])

        f    -- expression in x
        x    -- limit variable x (an identifier)
        a    -- (optional) limit point: an arithmetical expression
                representing a real number or infinity or -infinity;
                default is 0
        dir  -- (optional) one of 'Left', 'Right', or 'Real'(default)

        options:
        -- Intervals: a set containing all accumulations points is returned
        -- NoWarning: do not output warnings

        The limit function attempts to compute the limit or the set of
        all accumulation points of f as x goes to a, according to the
        following definition:
        y is an accumulation point of f if Re(y) is an
        accumulation point of Re(f) and Im(y) is an accumulation
        point of Im(f).
        For real-valued functions, we use the usual definition.
        
        If no direction is specified, or if it is 'Real', the limit is the real
        bidirectional limit, except in the case where the limit point is
        infinity or -infinity. In this case the limit is from the left to
        infinity and from the right to -infinity, respectively. In any
        case, x represents a real number at all times!
        
        Without option Intervals, limit returns:
        - an arithmetical expression whose real/imaginary part represent either           a real number or +/- infinity, or
        - 'undefined' if the limit is not defined, that is,
          if more than accumulation point exists.
        If the limit could not be computed then limit returns
        unevaluated, i.e. the result is of type "limit".

        With the option Intervals, limit returns
        a superset of the set of all accumulation points instead.
       

*/

// finite/infinite part of a set
alias(FIN(S) = piecewise::extmap(S, extop, 1)):
alias(INFIN(S) = piecewise::extmap(S, extop, 2)):



alias(EPS = 10^(-7)):


limit:=
proc(f)
  option hold;
  save MAXEFFORT;
  local i, t, x, lp, argv,
  dir, pointt,
  options: DOM_TABLE,
  checkpointt: DOM_PROC;
begin

// l o c a l    m e t h o d s

// checkpointt - checks whether variable, limit point and direction are valid
  // uses pointt and dir from outer proc
  checkpointt:=
  proc()
  begin
    if type(pointt) = "_equal" then
      if domtype(op(pointt,1)) <> DOM_IDENT then
        error("invalid limit variable (expecting an identifier)")
      end_if;
      if has( op(pointt,2), op(pointt,1) ) then
        error("expansion variable appears in limit point")
      elif not testtype( op(pointt,2),Type::Arithmetical ) then
        error("limit point must be of 'Type::Arithmetical'")
      end_if
    elif domtype(pointt) <> DOM_IDENT then
      error("invalid limit variable (expecting an identifier)")
    end_if;

    if not contains({infinity, -infinity}, op(pointt,2)) and
      has( op(pointt,2),infinity ) then
      error("invalid limit point")
    end_if;

    if has(op(pointt,2), complexInfinity) then
      error("invalid limit point")
    end_if;

    // no limit from the right at infinity, or from the left at -infinity
    if op(pointt,2) = infinity and dir = Right or
      op(pointt,2) = -infinity and dir = Left
      then
      error("inconsistent limit direction")
    end_if
  end_proc:



  
/**********************************
  argument evaluation and overloading section
  Problem: the limit variable should have the property to be near
           the limit point; f should not be evaluated until this property has
           been set.
           If the arguments are nonsense, there is no limit variable, and thus
           no property can be set.
           But if f::dom overloads limit, we cannot decide whether the
           arguments are nonsense. Thus we also have to evaluate f if argument
           checking fails, in order to find out what f::dom actually is.
*******/


  case args(0)
    of 0 do
      error("limit called without arguments")
    of 1 do
      f:= context(f);
      if type(f) = "_exprseq" then
        return(eval(hold(limit)(f)))
      end_if;
      if f::dom::limit <> FAIL then
        return(f::dom::limit(f))
      else
        error("at least two arguments expected")
      end_if;
    otherwise
      argv:= [context(args(2..args(0)))];
  end_case;

  pointt:= argv[1];
  // get options

  // defaults:
  options:= limit::defaultOptions;  
 
  dir:= Real;
  
  
  for i from 2 to nops(argv) do
    case argv[i]
      of Left do
      of Right do
      of Real do
        dir:= argv[i];
        break
      of TRUE do
      of FALSE do
        warning("wrong option");
        options[Intervals]:= argv[i];
        break  
      of Intervals do
      of NoWarning do
        options[argv[i]]:= TRUE;
        break
      otherwise
        if type(argv[i]) = DOM_TABLE then
           options:= argv[i];
           break
        end_if;
        // illegal option, but perhaps limit has been overloaded
        f:= context(f);
        if f::dom::limit <> FAIL then
          return(f::dom::limit(f, context(args(2..args(0)))))
        end_if;
        error("Illegal option")
    end_case;
  end_for;


  if testargs() then
    if traperror(checkpointt()) <> 0 then
      // illegal argument, but maybe ok in case of overloading
      f:= context(f);
      if f::dom::limit <> FAIL then
        return(f::dom::limit(f, context(args(2..args(0)))))
      else
        lasterror()
      end_if;
    end_if; // traperror <> 0
  end_if; // testargs()



  if type(pointt) = "_equal" then
    x:= op(pointt,1);
    lp:= op(pointt,2)
  else
    x:= pointt;
    lp:= 0
  end_if;

  
  if testargs() then
    if domtype(x) <> DOM_IDENT then
      error("invalid limit variable (expecting an identifier)");
    end_if;
    if lp <>-infinity and lp<>infinity and is(lp in R_) = FALSE then
      error("Only real limit points or +/- infinity allowed")
    end_if;
  end_if;

  if lp = infinity then
    dir:= Left
  end_if;
  if lp = -infinity then
    dir:= Right
  end_if;
  
  // evaluate f in context, taking properties into account
  if traperror
    ((
      case lp
        of infinity do
          assumeAlso(x > 20);
          break
        of -infinity do
          assumeAlso(x < -20);
          break
        otherwise
          assumeAlso(lp in R_);
          if dir = Left then
            assumeAlso(x, Dom::Interval(lp - EPS, lp));
          elif dir = Right then
            assumeAlso(x, Dom::Interval(lp, lp + EPS))
          else
            assumeAlso(x, Dom::Interval(lp - EPS, lp + EPS));
            // TODO: remove one of these as soon as property is fixed!
            assumeAlso(lp <> x and x <> lp)
          end_if
      end_case
      )) <> 0 then
    if not options[NoWarning] then
      warning("Could not attach the property of being close to ".
              "the limit point to limit variable")
    end_if
  end_if;

  if getprop(x) = {} then
    error("Variable cannot be near limit point")
  end_if;

  // for non-interactive use, an additional eval is necessary here
  if traperror((f:= eval(context(f)))) <> 0 then
    error("First argument is not defined under the assumption that ".
          expr2text(x)." has the property ".expr2text(getprop(x)))
  end_if;

  // allow overloading of the first argument of limit:
  if f::dom::limit <> FAIL then
    return(f::dom::limit(f, x, lp, dir, options))
  end_if;


  if type((t:= eval(op(f, 0)))) = DOM_FUNC_ENV and t::limit <> FAIL then
    return(t::limit(f, x, lp, dir, options))
  end_if;
  

/****************************************************
  end of argument evaluation and overloading section
*****************************************************/

// cases where limit maps to the entries of a structure
  if contains({DOM_SET, DOM_LIST, DOM_ARRAY}, domtype(f)) then
    return(map(f, limit::main, x, lp, dir, options))
  end_if;

  // unevaluated cases
  if contains({"limit", "series", Series::Puiseux, Series::gseries,
               "int", "diff"}, type(f)) then
    return(procname(f, op(argv)))
  end_if;

  if testargs() then
    if not testtype(f, Type::Arithmetical) then
      error("First argument must be arithmetical expression")
    end_if;

    // check that f contains no indefinite integrals of the limit variable
    // since the integration constant is unspecified, this would make no sense
    misc::maprec(f, {"int"} =
                 proc(J: "int")
                 begin
                   if op(J, 2) = x then
                     error("Cannot compute limit of indefinite integral when ".
                           "the integration variable equals the limit variable")
                   end_if
                 end_proc
                 )
    
  end_if;


  limit::main(f, x, lp, dir, options)
end_proc:


limit:= funcenv
(limit, NIL,
 table(
       "type" = "limit",
       "info" = "limit -- calculate limits [try ?limit for options]",
       "print"= "limit"
)):


limit::freeIndets:=
proc(L: "limit")
begin
  freeIndets([op(L, 1), op(L, [2, 2])], args(2..args(0))) minus {op(L, [2, 1])}
end_proc:



/*
   table of all options with defaults
*/
limit::defaultOptions:=
table(
      Intervals = FALSE,
      NoWarning = FALSE,
      Recursive = TRUE
      ):


/*
   limit::main(f, x, lp, dir, options)

   returns a superset of the set of all accumulation points of f for
   x -> lp from direction dir

   returns the limit of f for x -> lp from direction dir

   The main part of the work is done in accumulationPoints. This
   procedure does only two things:
   - it handles floats by rationalzing and floating the result in the end
   - it selects the return type of the result, depending on options[Intervals]

*/

limit::main:=
proc(f, x, lp, dir, options: DOM_TABLE)
  local convFloat: DOM_PROC,
  foundFloat: DOM_BOOL,
  super, sub;
  
begin

  // look for floats
  foundFloat := FALSE;
  
  convFloat:=
  proc(g)
    local r;
  begin
    r := numeric::rationalize(g, Minimize);
    if r <> g then
      foundFloat := TRUE;
    end_if;
    r
  end_proc;
    
    // replace all floating point constants by rational approximations
  f := misc::maprec(f, {DOM_FLOAT, DOM_COMPLEX} = convFloat);
  lp:= misc::maprec(lp, {DOM_FLOAT, DOM_COMPLEX} = convFloat);

  if foundFloat then
    return(float(limit::main(f, x, lp, dir, options)))
  end_if;

 
  [super, sub]:= limit::accumulationPoints(f, x, lp, dir, options);

  if options[Intervals] then
    super:= limit::Set::convertToSet(super);
    if super = limit::closure(C_) then
      // do not let the user see this
      return(hold(limit)(f, x=lp, dir, Intervals))
    else
      return(super)
    end_if
  end_if;

  // omit dir in the output if it contains no information
  if lp = infinity or lp=-infinity or dir = Real then
    dir:= null()
  end_if;
  
  piecewise::zip
  (super, sub,
   proc(sper, sb)
   begin
     if sper = limit::Set::emptyset then
       // this is an impossible situation, indicating that
       // the condition was impossible
       undefined
     elif sper = sb then
       return(piecewise([not limit::Set::hasMoreThanOneElement(sper),  limit::Set::theElement(sper)]))

     elif limit::Set::hasMoreThanOneElement(sb) = TRUE then
       undefined
     elif limit::Set::hasMoreThanOneElement(sper) = FALSE then
       // there is at most one accumulation point. However, since we work
       // on a compact set, this *is* an accumulation point
       limit::Set::theElement(sper)
     else
       // return unevaluated
       hold(limit)(f, x=lp, dir)
     end_if
   end_proc)
  
end_proc:


limit::main:= prog::remember(limit::main, property::depends, PreventRecursion,
                       (f, x, lp, dir, options)-> hold(limit)(f, x=lp, dir)):



/*
limit::accumulationPoints(f, x, lp, dir, options)

returns a list of two sets: a superset and a subset of the set
  of all accumulation points of f(x) as x approaches lp from direction
  dir

Both sets are of type limit::Set or piecewise objects consisting of
sets of that type.

The computation is done using several heuristic methods, each of them returning a
superset and a subset of the set of all accumulation points. The
computation succeeds when both are equal. The intersection of any
number of supersets is a superset, the union of any number of subsets
is a subset.

The following heuristic methods are attempted:
- MRV algorithm
- limit computation by recursion
- limit computation by series expansion
- computing the real and imaginary part separately

*/

limit::accumulationPoints:=
proc(f: Type::Arithmetical, x: DOM_IDENT, lp: Type::Arithmetical,
     dir: DOM_IDENT, options: DOM_TABLE): DOM_LIST
  local super: limit::Set, sub: limit::Set,
  newsub, newsuper, newsub2, newsuper2, re, im, t, originalf,
  finish: DOM_PROC;

  save MAXEFFORT;
  
begin

  /*
    local method finish(super, sub)

    given a superset of the set of accumulation points and a subset
      of the set of accumulation points, decides whether this is enough
      information that we may return

      we may stop when:
      - super equals sub; then this is the exact set of accumulation points
      - super contains only one element; then this is the limit
      - we only look for the limit (and not possibly all accumulation points),
        and there are at least two accumulation points; then no limit exists
  */

  finish:=
  proc(super, sub)
  begin
    _lazy_or(super = sub,
             limit::Set::hasMoreThanOneElement(super) = FALSE,
             not options[Intervals] and limit::Set::hasMoreThanOneElement(sub) = TRUE
             )
  end_proc;
    
  
  if not has(f, x) then
    return([limit::Set({f}) $2])
  end_if;

  if type(f) = piecewise then
    if has(piecewise::conditions(f), x) then
      // give up
      return([limit::Set::Cclosure, limit::Set::emptyset])
    else   
      t:= piecewise::extmap(f, limit::accumulationPoints, x, lp, dir, options);
      return([piecewise::extmap(t, op, 1), piecewise::extmap(t, op, 2)])
    end_if  
  end_if;

  // replace protected variables
  if protected(x) <> None then
    t:= genident("x");
    f:= subs(f, x=t);
    x:= t
  end_if;
  
  originalf:= f;

  // extract constants
  case type(f)
    of "_plus"  do
      if (lp=infinity or lp=-infinity) and
        testtype(f, Type::PolyExpr([x])) and
        is(f in R_, Goal = TRUE) and
        is(lcoeff(f, [x]) <>0, Goal = TRUE) then
        return(limit::accumulationPoints
               (lmonomial(f, [x]), x, lp, dir, options)
               )
      end_if;
      [re, im, t]:= split(f, has, x);
      if not iszero(im) then
        re:= limit::accumulationPoints(re, x, lp, dir, options);
        return(map(re, piecewise::extmap, limit::Set::_plusNumber, im))
      end_if;
      break
    of "_mult" do
      [re, im, t]:= split(f, has, x);
      if im <> 1 then
        re:= limit::accumulationPoints(re, x, lp, dir, options);
        if INFIN(re[1]) = {} then
          // simple case
          return(map(re, limit::Set::_multNumber, im))
        end_if;
        return(map(re,
                   proc(limset)
                   begin
                     piecewise([im = 0, limit::Set::zero],
                               [im <> 0,
                                limit::Set::_multNumber(limset, im)]
                               )
                   end_proc
                   )
               )
      end_if;
      break
  end_case;



  f:= misc::maprec(f, {"fact", "binomial"} = (X -> if has(X, x) then rewrite(X, gamma) else X end_if));

  if hastype(f, "gamma") then
     f:= simplify::gamma(f)
  end_if;
  
  /*
  if MAXEFFORT = RD_INF then
    f:= simplify(f, exp);
    f:= expr(factor(f))
  else
    traperror((
               f:= simplify(f, exp);
               f:= expr(factor(f));
               ),
              MaxSteps = ceil(MAXEFFORT/1000)
              )
  end_if;
  */

  if 100*length(f) < MAXEFFORT and 
    (t:= limit::tryMRV(f, x, lp, dir, options)) <> FAIL
    then
    return([limit::Set(t) $2])
  end_if;

  
  // heuristics working for complex expressions
  if MAXEFFORT > 1000 and not has(f, {hold(abs), hold(sign)})
    then
    [super, sub]:= limit::complexSeries(originalf, x, lp, dir, options);
    // by compactness, there must be at least one accumulation point:
    assert(super <> limit::Set::emptyset);
 
    if finish(super, sub) then
      return([super, sub])
    end_if
  else
    super:= limit::Set::Cclosure;
    sub:= limit::Set::emptyset
  end_if;
  

  if options[Recursive] and MAXEFFORT > 500 then
    MAXEFFORT:= MAXEFFORT/3;
    [newsuper, newsub]:= limit::recursive(args());
    // by compactness, there must be at least one accumulation point:
    assert(newsuper <> limit::Set::emptyset);
    if finish(newsuper, newsub) then
      return([newsuper, newsub])
    end_if;
    super:= piecewise::zip(newsuper, super, _intersect);
    sub:= piecewise::zip(newsub, sub, _union);
    if finish(super, sub) then
        return([super, sub])
    end_if;
    
    if dir = Real and MAXEFFORT > 1000 then
      // split into Left and Right
      if is(x>=lp, Goal = TRUE) then
        [newsuper, newsub]:= limit::recursive(f, x, lp, Right, options)
        assumingAlso x<>lp
      elif is(x<=lp, Goal = TRUE) then
        [newsuper, newsub]:= limit::recursive(f, x, lp, Left, options)
        assumingAlso x<>lp
      else  
        [newsuper, newsub]:= limit::recursive(f, x, lp, Left, options)
        assumingAlso x<lp;
        // by compactness, there must be at least one accumulation point:
        assert(newsuper <> limit::Set::emptyset);
        [newsuper2, newsub2]:= limit::recursive(f, x, lp, Right, options)
        assumingAlso x>lp;
        // by compactness, there must be at least one accumulation point:
        assert(newsuper2 <> limit::Set::emptyset);
        // any accumulation point from the right or the left is an
        // accumulation point for the two-sided limit
        newsuper:= piecewise::zip(newsuper, newsuper2, _union);
        newsub:= piecewise::zip(newsub, newsub2, _union)
      end_if;
      if finish(newsuper, newsub) then
        return([newsuper, newsub])
      end_if;
      super:= piecewise::zip(newsuper, super, _intersect);
      sub:= piecewise::zip(newsub, sub, _union);
      if finish(super, sub) then
        return([super, sub])
      end_if;
    end_if;
  else
    MAXEFFORT:= MAXEFFORT/2;
    newsuper:= limit::Set::Cclosure;
    newsub:= limit::Set::emptyset
  end_if;
  
  


  // heuristics that work for real expressions only
  // first try is, then try rectform
  if is(f in R_, Goal = TRUE) then
    re:= [f, 0, 0]
  elif MAXEFFORT = RD_INF then
    re:= rectform(f)
  elif MAXEFFORT > 1000 then
    case traperror((re:= rectform(f)), MaxSteps= floor(MAXEFFORT/1000))
      of 1321 do
        // too slow
        return([super, sub])
      of 0 do
        break
      otherwise
        lasterror()
    end_case
  else
    return([super, sub])
  end_if;

  // working on re = rectform(f)
  
  if op(re, 3) = 0 then
    im:= op(re, 2);
    re:= op(re, 1);

    // do some simplifications
    re:= subsex(re, Re(x) + I*Im(x) = x);
    im:= subsex(im, Re(x) + I*Im(x) = x);

    // too slow:
    // re:= rewrite(re, piecewise);
    // im:= rewrite(im, piecewise);
      
    if iszero(re) then
      im:= limit::tryMRV(im, x, lp, dir, options);
      if im = FAIL then
        return([super, sub])
      end_if;
      re:= [limit::Set::zero $2];
      im:= [limit::Set(im) $2]
    elif iszero(im) then
      if re = f then
        // no reason to try MRV again
        return( [super, sub])
      end_if;
      re:= limit::tryMRV(re, x, lp, dir, options);
      if re = FAIL then
        return( [super, sub])
      end_if;
      im:= [limit::Set::zero $2];
      re:= [limit::Set(re) $2]
    else
      // provide the information that re and im are reals to the property
      // mechanism?!
      // assumeAlso(re in R_);
      // assumeAlso(im in R_);
      MAXEFFORT:= MAXEFFORT/2;
      re:= limit::accumulationPoints(re, x, lp, dir, options);
      if not options[Intervals] and
        limit::Set::hasMoreThanOneElement(re[2]) = TRUE then
        // no limit exists; we may return anything that produces undefined
        // in limit::main, and re will do so
        return(re)
      end_if;
      if re[1] = limit::Set::Cclosure then
        // give up
        return([super, sub])
      end_if;
      im:= limit::accumulationPoints(im, x, lp, dir, options)
    end_if;
    newsuper:= limit::Set::superSetFromReIm(re[1], im[1]);
    // by compactness, there must be at least one accumulation point:
    assert(newsuper <> limit::Set::emptyset);
    newsub:= limit::Set::subSetFromReIm(re[2], im[2]);
    // combine with the known accumulation points
    super:= piecewise::zip(super, newsuper, _intersect);
    assert(super <> limit::Set::emptyset);
    sub:= piecewise::zip(sub, newsub, _union);
  end_if;

  // by compactness, there must be at least one accumulation point:
  assert(super <> limit::Set::emptyset);
  
  [super, sub]
end_proc:


/*
   limit::tryMRV(f, x, lp, dir, options)

   returns a superset of the set of possible accumulation points
   of f when x -> lp
   returns FAIL if the MRV algorithm did not succeed

*/

limit::tryMRV:=
proc(f, x, lp, dir, options): Type::Union(DOM_FAIL, DOM_SET)
  local re, t;
begin
  if dir = Real then
    re:= limit::transformToInfinity(f, x, lp, Left);
    property::subsIdent(x, limit::transformToInfinity(x, x, lp, Left));
    t:= limit::limitMRV(re, x);
    if t <> FAIL then
      // need not transform the variable, it still goes to infinity
      // assume(x>=20);
      re:= limit::transformToInfinity(f, x, lp, Right);
      re:= limit::limitMRV(re, x);
      if re <> FAIL then
        return({re, t})
      end_if
    end_if
  else
    if lp <> infinity then
      // assume(x>=20)
      property::subsIdent(x, limit::transformToInfinity(x, x, lp, dir))
    end_if;
    re:= limit::transformToInfinity(f, x, lp, dir);
    re:= limit::limitMRV(re, x);
    if re <> FAIL then
      return({re})
    end_if
  end_if;
  FAIL
end_proc:


/*
  limit::recursive(f, x, lp, dir, options)

  returns a list consisting of a superset and a subset of the set of
  accumulation points for f when x -> lp

  This is a collection of heuristics, using that:
  - when x0 is an accumulation point of g(x) and f is continuous at
  x0, it follows that f(x0) is an accumulation point of f(g(x))
  - the limit of a sum/product is the sum/product of the limits of the
  operands, provided that the latter exist
  - l'Hospital can sometimes be used to resolve infinity/infinity and
  infinity-infinity


*/

limit::recursive:=
proc(f, x, lp, dir, options): DOM_LIST
  local s, t, u, maxeffort, result, i, inf, a, z, n,
  infinpart, finpart, zeroes, nonzeroes, infinities, minusInfinities, boundednonzero,
  otherbounded, other,
  infinproduct, nonzeroproduct, boundedproduct, boundedaccumpoints,
  nonreal: DOM_BOOL,
  handleInfinityMinusInfinity: DOM_PROC,
  recursiveExp: DOM_PROC;
  save MAXEFFORT;
begin

  // handle f = exp(y)

  recursiveExp:=
  proc(y): DOM_LIST
    local s, S,
    handleInfinity: DOM_PROC;
  begin

    // handleInfinity(inf) - return all accumulation points exp(y) must have
    // on any subsequencfe of y's tending to inf
    handleInfinity:=
    proc(inf)
      local s,
      detectunitcircle: DOM_PROC;
    begin

      detectunitcircle:=
      proc()
      begin
        if limit::isSurjectiveAtInfinity(y/op(inf, 1), x, lp, dir, options)
            then
          limit::Set::unitcircle
        else
          limit::Set::emptyset
        end_if
      end_proc;
          
      if type(inf) = stdlib::Infinity then
        s:= op(inf, 1)
      else
        s:= sign(inf);
        if has(s, infinity) then
          s:= sign(Re(inf)) + I*sign(Im(inf))
        end_if;

        if has(s, infinity) then
          return(limit::Set::emptyset)
        end_if;
      end_if;

      
      // s <> 0 as inf is an infinity
      piecewise([Re(s) < 0, limit::Set::zero],
                [Re(s) = 0 and Im(s) <> 0, detectunitcircle()], 
                [s > 0, limit::Set::infinity],
                [Otherwise, limit::Set::emptyset]
                )
    end_proc;
    
    s:= limit::accumulationPoints(y, x, lp, dir, options);

    S:= piecewise::extmap(INFIN(s[2]),
                          proc(infset)
                          begin
                            if type(infset) = DOM_SET and nops(infset) > 0 then
                              _union(op(map(infset, handleInfinity)))
                            else
                              limit::Set::emptyset
                            end_if;
                          end_proc
                            );
    
      
    s[2]:= piecewise::zip(S, 
                          limit::Set(exp(FIN(s[2])), {}),
                          limit::Set::_union);
    s[1]:= exp(s[1]);
    return(s)
  end_proc;


  // local method to handle infinity - infinity
  // infinities: expression that goes to infinity
  // minusInfinities: expression that goes to -infinity
  
  handleInfinityMinusInfinity:=
  proc(infinities, minusInfinities)
    local t;
  begin
    /*
        write
        infinities + minusInfinities =
        infinities * (1+ minusInfinities/infinities)

        apply l'Hospital, possibly twice

    */

    t:= limit::accumulationPoints(minusInfinities/infinities, x, lp, dir,
                         options);

    if limit::Set::hasMoreThanOneElement(t[1]) = FALSE then
      t:= limit::Set::theElement(t[1]);
      t:= 1 + t;
      case is(t=0)
        of FALSE do
          return(t*infinity)
        of TRUE do
          // infinity*0 -> we have to apply another L'hospital
          t:= limit::lhospital([1+ minusInfinities/infinities],
                               [infinities], x, lp, dir,
                               options);
          if t<> FAIL then
            return(t)
          end_if
          // cannot properly react to UNKNOWN yet
      end_case;
    end_if; // t <> FAIL
    
    FAIL
  end_proc;


  
  if not has(f, x) then
     return([limit::Set({f}) $2])
  end_if;
  
  case type(f)
    of DOM_IDENT do
      assert(f = x);
      return([limit::Set({lp}) $2])
    of "_plus" do

      options[Intervals]:= TRUE;     

      u:= [op(f)];
      MAXEFFORT:= MAXEFFORT/nops(select(u, has, x));
      t:= [NIL $nops(u)];
      result:= infinities:= minusInfinities:= 0;
      other:= FAIL;
      
      for i from 1 to nops(u) do
        t:= limit::accumulationPoints(u[i], args(2..args(0)));
        if limit::Set::oneFinitePoint(t[1]) then
          result:= result + limit::Set::theElement(t[1])
        elif t[1] = limit::Set::infinity then
          infinities:= infinities + u[i]
        elif t[1] = limit::Set::minusInfinity then
          minusInfinities:= minusInfinities + u[i]
        elif INFIN(t[1]) <> {} then
          // unbounded: give up
          return([limit::Set::Cclosure, limit::Set::emptyset])
        elif other = FAIL then
          other:= t
        else
          // all accumulation points must be contained in the sum of sets
          other[1]:= limit::Set(FIN(other[1]) + FIN(t[1]), {});
          // but we do not know any accumulation point for sure
          other[2]:= limit::Set::emptyset
        end_if;
      end_for;

      
  
      if infinities = 0 then
        if minusInfinities = 0 then
          // each summand tends to a limit
          userinfo(20, "No summands have infinite limits")
        else
          // some summands go to -infinity, all others remain bounded
          result:= result - infinity
        end_if
      elif minusInfinities = 0 then
        // some summands go to infinity, all others remain bounded
        result:= result + infinity
      else
        // now infinities -> infinity and minusInfinities -> -infinity
        t:= handleInfinityMinusInfinity(infinities, minusInfinities);
        if t = FAIL then
          break
        end_if;
        result:= result + t
      end_if;

      // now result is the limit of the sum,
      // with possibly the "other" summand left out

      if other = FAIL then
        return([limit::Set({result}) $2])
      else  
        // there is one other summand
        if has(result, infinity)
          and limit::Set::oneFinitePoint((t:= Im(other[1]))) 
          then
          result:= result + limit::Set::theElement(t);
          // the infinite object dominates if the others are real
          return([limit::Set({}, {result}) $2])
        elif type(FIN(other[1])) <> DOM_SET and has(result, infinity) then
          return([limit::Set::Cclosure, limit::Set::emptyset])
        else
          return([limit::Set(FIN(other[1]) + {result}, {}),
                  limit::Set(FIN(other[2]) + {result}, {})
                  ])
        end_if
      end_if
    of "_mult" do
      assert(nops(f) >= 2);
      u:= [op(f)];
     
      // TODO: rewrite the following such that piecewise results  are handled

      
      // find the accumulation points of all factors
      MAXEFFORT:= MAXEFFORT/nops(u);
      // initialize some lists
      t:= [NIL $nops(u)];
      infinities:= zeroes:= nonzeroes:= boundednonzero:= otherbounded:= [];
    
      infinproduct:= {1};
      boundedproduct:= {1};
      boundedaccumpoints:= {1}; // accum. points that the bounded factors
                                // in boundednonzero and otherbounded
                                // surely have (if at least one such factor exists)
      nonzeroproduct:= 1;

      nonreal:= FALSE; // are there non-real bounded factors?      

      for i from 1 to nops(t) do

        // special case: negative powers (= factors of the denominator)
        if type(u[i]) = "_power" and not has(op(u[i], 2), x) and
          is(op(u[i], 2) < 0, Goal = TRUE) then
          t[i]:= limit::accumulationPoints(1/u[i], args(2..args(0)));
          if t[i][1] = limit::Set::Cclosure then
            // give up
            return([limit::Set::Cclosure, limit::Set::emptyset])
          end_if;
          
          /* add the factor 1/u[i] to one of the lists described below;
             replace t[i] by 1/t[i] */

          if FIN(t[i][1]) = {} then
            t[i]:= [limit::Set::zero $2]
          elif t[i][1] = limit::Set::zero then
            // only infinite accumulation points
            if is(u[i] >= 0, Goal = TRUE) then
              t[i]:= [limit::Set::infinity, limit::Set::emptyset]
            elif is(u[i] <= 0, Goal = TRUE) then
              t[i]:= [limit::Set::minusInfinity, limit::Set::emptyset]
            elif is(u[i] in R_, Goal = TRUE) then
              t[i]:= [limit::Set({}, {-infinity, infinity}),
                      limit::Set::emptyset]
            else  
              t[i]:= [limit::Set({}, universe), limit::Set::emptyset]
            end_if
          else
            if is(not 0 in FIN(t[i][1]), Goal = TRUE) then
              infinpart:= {}
            elif is(u[i] in R_, Goal = TRUE) then
              infinpart:= {-infinity, infinity}
            else
              // give up
              return([limit::Set::Cclosure, limit::Set::emptyset])
            end_if;
              
            // determine the finite accumulation points
            
            if not iszero(Im(u[i])) and not is(u[i] in R_, Goal =TRUE) then
               nonreal:= TRUE
            end_if;
            if INFIN(t[i][1]) = {} then
              t[i][1]:= limit::Set(1/FIN(t[i][1]), infinpart);
              t[i][2]:= limit::Set(1/FIN(t[i][2]), {})
            else
              t[i][1]:= limit::Set(1/FIN(t[i][1]) union {0}, infinpart);
              t[i][2]:= limit::Set(1/FIN(t[i][2]) union {0}, {})
            end_if;
            
          end_if;

        else
          // not a negative power
          t[i]:= limit::accumulationPoints(u[i], args(2..args(0)));
          if t[i][1] = limit::Set::Cclosure then
            // give up
            return([limit::Set::Cclosure, limit::Set::emptyset])
          end_if;
        end_if;
          
        
     
     
        /* add the factor u[i] to one of the following lists:
           - infinities: factors with only infinite accumulation points
           - nonzeroes: factors with finite, nonzero limit
           - zeroes: factors tending to zero
           (todo: - factors with finite limit which may be zero or not; these
             are currently included in otherbounded)
           - boundednonzero: bounded (e.g. oscillating), but nonzero 
           - otherbounded: bounded, and (possibly) zero
           - all other factors, i.e. those with both finite and infinite
             accumulation points; if one of these occurs, we give up immediately
        */
        
          
        if FIN(t[i][1]) = {} then
          infinities:= infinities.[u[i]];
          if INFIN(t[i][1]) = universe then
            
            infinproduct:= {undefined}
          else
            infinproduct:= infinproduct * INFIN(t[i][1])
          end_if;
        elif INFIN(t[i][1]) = {} then
          finpart:= FIN(t[i][1]);
          if type(finpart) = DOM_SET and nops(finpart) = 1 then
            case is(op(finpart, 1) = 0)
              of TRUE do
                zeroes:= zeroes.[u[i]];
                break
              of FALSE do
                if not is(u[i] in R_, Goal =TRUE) then
                   nonreal:= TRUE
                end_if;
                nonzeroes:= nonzeroes.[u[i]];
                nonzeroproduct:= nonzeroproduct * op(finpart, 1);
                break
              of UNKNOWN do
                if not is(u[i] in R_, Goal =TRUE) then
                   nonreal:= TRUE
                end_if;
                otherbounded:= otherbounded.[u[i]];
                boundedproduct:= boundedproduct*finpart;
                boundedaccumpoints:= boundedaccumpoints*op(finpart, 1);
                break
            end_case;
          elif is(not 0 in finpart, Goal = TRUE) then
            boundednonzero:= boundednonzero.[u[i]];
            boundedproduct:= boundedproduct*finpart;
            if type(boundedaccumpoints) = DOM_SET and
              nops(boundedaccumpoints) = 1 then
              boundedaccumpoints:= boundedaccumpoints*FIN(t[i][2])
            else
              boundedaccumpoints:= {}
            end_if
          else
            otherbounded:= otherbounded.[u[i]];
            boundedproduct:= boundedproduct*finpart;
            if type(boundedaccumpoints) = DOM_SET and
              nops(boundedaccumpoints) = 1 then
              boundedaccumpoints:= boundedaccumpoints*FIN(t[i][2])
            else
              boundedaccumpoints:= {}
            end_if
          end_if
        else
          // both finite and infinite accumulation points may occur ->
          // we give up
          return([limit::Set::Cclosure, limit::Set::emptyset])
        end_if;        
      end_for;


      // now handle the different cases depending on which of the
      // lists are empty
      
      // only finite accumulation points involved?
      if nops(infinities) = 0 then
        if nops(zeroes) > 0 then
          return([limit::Set::zero $2])
        end_if;
        s:= [NIL, NIL];
        s[1]:= limit::Set(_mult(FIN(t[i][1]) $i=1..nops(t)), {});
        if _lazy_and((type(FIN(t[i][1])) = DOM_SET and
                     nops(FIN(t[i][1])) = 1)  $i=1..nops(t)) then
          s[2]:= s[1]
        else
          s[2]:= limit::Set::emptyset
        end_if;
        return(s)
      end_if;

      // from here, some infinite factors exist
      
      // eliminate all factors tending to zero using l'Hospital, or give up
      // if this fails

      maxeffort:= MAXEFFORT;
      
      if nops(zeroes) > 0 then

        // first, we have to check for factors that could be *identically*
        // zero
        t:= map(zeroes, testeq, 0, Steps = 5, KernelSteps = 2);
        if contains(t, TRUE) > 0 then
          return([limit::Set::zero $2])
        end_if;

        if contains(t, UNKNOWN) > 0 then
          // give up
          return([limit::Set::Cclosure, limit::Set::emptyset])
        end_if;
        
        MAXEFFORT:= MAXEFFORT/2;
        t:= limit::lhospital(zeroes, infinities, x, lp, dir, options);
        if has(t, infinity) then
          infinities:= [_mult(op(zeroes), op(infinities))];
          zeroes:= [];
          infinproduct:= {t};
        elif t <> FAIL then  
          boundedproduct:= boundedproduct * nonzeroproduct;
          if type(boundedproduct) = DOM_SET and nops(boundedproduct) = 1 then
            return([limit::Set(boundedproduct*t, {}) $2])
          else
            return([limit::Set(boundedproduct*t, {}), limit::Set::emptyset])
          end_if;
        else
          // to do: discrete l'Hospital: if a_n, b_n are sequences -> infinity,
          // then lim(a_n/b_n) = lim((a_(n+1)-a_n)/(b_(n+1)-b_n))
          
          break
        end_if;
      end_if;

      // by now, we should have eliminated all zeroes
      assert(nops(zeroes) = 0);
      assert(nops(infinities) >= 1);
      MAXEFFORT:= maxeffort;
      
      // we cannot handle finite factors with non-real limit
      // for one can construct a, b such that a*b has constant imaginary part,
      // a -> 1 and b -> infinity
      if nonreal then 
        break 
      end_if;

      // *only* infinities and finite factors with nonzero limit?
      // We can deal with this if the limits of the factors
      // are all purely real or purely imaginary, but not e.g. for a*b where
      // a goes to infinity+3*I and b goes to I*infinity+3.

      
      if nops(otherbounded) = 0 and nops(boundednonzero) = 0 then
        if infinproduct subset
          {infinity, -infinity, I*infinity, -I*infinity} then
          if nops(infinproduct) = 1 then
            return([limit::Set({}, nonzeroproduct*infinproduct) $2])
          else
            return([limit::Set({}, nonzeroproduct*infinproduct),
                    limit::Set::emptyset])
          end_if  
        else
          break
        end_if
      end_if;

      

      if has(infinproduct, undefined) then
        break
      end_if;

      s:= [NIL, NIL]:

      // if no oscillating factors have accumulation point zero,
      // we may simply multiply the infinities
      // and the finite accumulation points
     
      
      if nops(otherbounded) = 0 then
        s[1]:= limit::Set({}, infinproduct*nonzeroproduct*boundedproduct):
        s[2]:= limit::Set::emptyset;
        return(s)
      end_if;

      if type(infinproduct) = DOM_SET and nops(infinproduct) = 1 then
        s[1]:= limit::Set::Cclosure;
        s[2]:= limit::Set({}, nonzeroproduct *
                          (sign(boundedaccumpoints) minus {0}) *
                          infinproduct);
        return(s)
      end_if;
      break
    of "_power" do
      // first case: base constant, exponent does not change sign
      if not has(op(f, 1), x) and
        _lazy_or(is(op(f, 1) <> 0) = TRUE,
         is((s:= limit(sign(op(f, 2)), x = lp)) > 0) = TRUE,
         is(s<0, Goal = TRUE)
         )
        then
        // compute the accumul. points of C^h(x)
        s:= limit::accumulationPoints(op(f,2), args(2..args(0)));
        t:= op(f, 1);
        if limit::Set::oneFinitePoint(s[1]) then
          t:= op(f, 1);
          u:= limit::Set::theElement(s[1]);
          if (is(t<>0 or u>0, Goal = TRUE)) then
            return([limit::Set({t^u}, {}) $2])
          end_if;
          break
        end_if;
        
        case s[1]
          of limit::Set::infinity do
            // special case
            if t = -1 then
              if limit::isSurjectiveAtInfinity(op(f, 2), x, lp, dir, options)
                then
                // all of the unit circle must be accumulation points
                return([limit::Set::unitcircle $2])
              elif is(op(f, 2) in 2*Z_, Goal = TRUE) then
                return([limit::Set({1}, {}) $2])
              elif is(op(f, 2) in Z_, Goal = TRUE) then
                return([limit::Set({-1, 1}, {}),
                        limit::Set::emptyset
                        ])
              else
                return([limit::Set::unitcircle,
                        limit::Set::emptyset
                        ])
              end_if
            end_if;
            if is(abs(t) < 1, Goal = TRUE) then
              return([limit::Set::zero $2])
            end_if;
            if is(t > 1, Goal = TRUE) then
              return([limit::Set({}, {infinity}) $2])
            end_if;
            if is(t >= 0, Goal = TRUE) then
              s:= piecewise([t < 1, limit::Set::zero],
                            [t = 1, limit::Set({1}, {})],
                            [t > 1, limit::Set::infinity]
                            );
              return([s $2])
            end_if;
            break
          of limit::Set::minusInfinity do
            if is(abs(t) > 1, Goal = TRUE) then
              return([limit::Set::zero $2])
            end_if;
            if is(t < 1, Goal = TRUE) then
              return([limit::Set::infinity $2])
            end_if;
            break
        end_case;
          
        // two cases where the exponent is constant. First, we handle
        // integer exponents
      elif domtype(op(f, 2)) = DOM_INT then
        s:= limit::accumulationPoints(op(f, 1), args(2..args(0)));
        if op(f,2) > 0 then
          return([limit::Set::_powerPosIntSuper(op(s, 1), op(f, 2)),
                  limit::Set::_powerPosIntSub(op(s, 2), op(f, 2))])
        else  
          // negative exponent
         
          
          if s[1] = limit::Set::infinity or
            s[1] = limit::Set::minusInfinity then
            return([limit::Set::zero $2])
          
          elif limit::Set::oneFinitePoint(s[1]) then
            u:= limit::Set::theElement(s[1]);
            if op(f,2) mod 2 = 0 then
              return([piecewise([u=0, limit::Set::infinity $2],
                               [u<>0, limit::Set({(u^op(f, 2))}, {}) $2]
                               )
                      $2]
                     )
            else
              // op(f, 2) is odd
              if testeq(u, 0, Steps = 5) <> FALSE then
                break
              end_if;
              if is(u<>0, Goal = TRUE) then
                return([limit::Set({u^op(f, 2)}, {}) $2])
              end_if;
              if is(op(f, 1) > 0, Goal = TRUE) then
                return([piecewise
                       ([u=0, limit::Set::infinity $2],
                        [u<>0, limit::Set({u^op(f, 2)}, {}) $2]
                        )
                        $2]
                       )
              end_if;
              if is(op(f, 1) < 0, Goal = TRUE) then
                return([piecewise
                       ([u=0, limit::Set::minusInfinity $2],
                        [u<>0, limit::Set({u^op(f, 2)}, {}) $2]
                        )
                        $2]
                       )
              end_if;
              // no heuristic applied
            end_if; // op(f,2) mod 2 = 0
          end_if; // s[1] = infinity or one point
            
          // try inversion
          // we can reduce op(f,1)^op(f, 2) to
          // handling (op(f, 1)^(-op(f, 2)))^(-1) but cannot handle zero then
          if op(f, 2) = -1 then
            t:= op(s, 1)
          else      
            t:= limit::Set::_powerPosIntSuper(op(s, 1), -op(f, 2))
          end_if;
          if is(not 0 in FIN(t), Goal = TRUE) then
            return(map([t,
                        limit::Set::_powerPosIntSub(op(s, 2), -op(f, 2))],
                       _invert
                       )
                   )
          else
            return([limit::Set::Cclosure,
                    _invert(limit::Set::_powerPosIntSub(op(s, 2), -op(f, 2)))
                    ])
          end_if // 0 in FIN(t)
        end_if // op(f, 2) > 0

        // second case where exponent is constant: non-integers
      elif not has(op(f, 2), x) then
        // f(x)^C
        // check whether f(x) tends to a limit
        s:= limit::accumulationPoints(op(f, 1), args(2..args(0)));
        if limit::Set::oneFinitePoint(s[1]) then
          s:= limit::Set::theElement(s[1]);
          if is(Re(op(f, 2)) > 0, Goal = TRUE) then
            return([piecewise([s=0, limit::Set::zero],
                             [s<>0, limit::Set({s^op(f, 2)}, {})]
                             )
                    $2]
                   )
          end_if;
          
          if is(s<>0, Goal = TRUE) then
            return([limit::Set({s^op(f, 2)}, {}) $2])
          end_if
        end_if;
      end_if;
      // try heuristic a^b -> exp(b*ln(a)) if a <> 0
    
      if not iszero(op(f, 1)) then
        return(recursiveExp(op(f, 2)*ln(op(f, 1))))
      end_if;

      // op(f, 1) may be zero: we have no other heuristic
      break
    of "ln" do
      s:= limit::accumulationPoints(op(f, 1), args(2..args(0)));
      if s[1] = limit::Set::zero then
        if is(op(f, 1)>0, Goal = TRUE) then
          return([limit::Set::minusInfinity $2])
        elif is(op(f, 1) < 0, Goal = TRUE) then
          return([limit::Set({}, {-infinity + I*PI}) $2])
        end_if;
        // try the limit of arg
        t:= limit::accumulationPoints(arg(op(f, 1)), args(2..args(0)));
        finpart:= FIN(t[1]);
        if type(finpart) = DOM_SET and nops(finpart) = 1 then
           t:= op(finpart, 1);
           return([limit::Set({}, {-infinity + t*I}) $2])
        end_if;

        // try solve
        if solve(op(f, 1) >= 0, x) = getprop(x) then
          return([limit::Set::minusInfinity $2])
        elif solve(op(f, 1) <= 0, x) = getprop(x) then
          return([limit::Set({}, {-infinity + I*PI}) $2])
        end_if;
      end_if;
      if s[1] = limit::Set::infinity then
        return([limit::Set::infinity $2])
      end_if;
      if s[1] = limit::Set::minusInfinity then 
        // if op(f, 1) -> -infinity, we need to find out whether it stays on the same side of the branch cut 
        // all the time. Currently, we only handle the case that op(f, 1) is real
        if is(op(f, 1) in R_, Goal = TRUE) then
          return([limit::Set({}, {I*PI + infinity}) $2])
        end_if;  
      elif INFIN(s[1]) = {} then
        finpart:= FIN(s[1]);
        if finpart intersect ln::complexDiscont = {} or
         is(op(f, 1) in R_, Goal = TRUE) and
         is(not 0 in finpart, Goal = TRUE)
          then
        // the ln is one-to-one
        return([limit::Set(ln(FIN(s[1])), {}),
                limit::Set(ln(FIN(s[2])), {})]
               )
        end_if;
        
        if s[1] = limit::Set::unitcircle then
          return([limit::Set(I*Dom::Interval(-PI, [PI]), {}),
                  limit::Set(ln(FIN(s[2])), {})]
                 )
        end_if;

        if type(finpart) = DOM_SET and nops(finpart) = 1 then
          s:= op(finpart, 1);
          if is(op(f, 1) >= 0, Goal = TRUE) then
            return([piecewise([s = 0, limit::Set::minusInfinity],
                             [s <> 0, limit::Set({ln(s)}, {})]
                             )
                    $2]
                   )
          end_if
        end_if;
        
      end_if;
      break

    of "exp" do
      return(recursiveExp(op(f, 1)))     
    of "sin" do
    of "cos" do
      // handle integer case (e.g. sin(n*PI)) separately
      if is(x in Z_, Goal = TRUE) then
        t:= slot(eval(op(f, 0)), "simplify")(f);
        if t <> f then
          return(limit::recursive(t, args(2..args(0))))
        end_if
      end_if;
      
      
      s:= limit::accumulationPoints(op(f, 1), args(2..args(0)));

      s[1]:= piecewise::extmap
      (s[1],
       proc(S)
       begin
         if type(INFIN(S)) <> DOM_SET or
           map(INFIN(S), Im) minus {0} <> {} then
           limit::Set::Cclosure
         elif INFIN(S) = {} then
           limit::Set(eval(op(f, 0))(FIN(S)), {})
         else
           limit::Set(eval(op(f, 0))(FIN(S)) union
                          Dom::Interval([-1, 1]),
                          {})
         end_if
       end_proc
       );
                               
     
      
      s[2]:= piecewise::extmap
      (s[2],
       proc(S)
         local finpart;
       begin
         finpart:= eval(op(f, 0))(FIN(s[2]));
         if type(INFIN(S)) = DOM_SET and
           is(op(f, 1) in R_, Goal =TRUE) and
           _lazy_or(
                    limit::isSurjectiveAtInfinity(op(f, 1)/extop(inf, 1),
                                          x, lp, dir, options)
                    $inf in select(INFIN(S), u -> type(u) = stdlib::Infinity)
                    ) then
            limit::Set(Dom::Interval([-1, 1]) union finpart, {})
          else
            limit::Set(finpart, {})
          end_if
       end_proc);
      return(s)
    of "tan" do
      s:= limit::accumulationPoints(op(f, 1), args(2..args(0)));
      if limit::Set::oneFinitePoint(s[1]) then
        s:= limit::Set::theElement(s[1]);
        if is(not (s/PI)-1/2 in Z_, Goal = TRUE) then
          return([limit::Set({tan(s)}, {}) $2])
        end_if;
        break
      end_if;

      if s[1] = limit::Set::infinity and
        limit::isSurjectiveAtInfinity(op(f, 1), args(2..args(0))) then
        return([limit::Set(R_, {-infinity, infinity}) $2])
      end_if;

      if s[1] = limit::Set::minusInfinity and
        limit::isSurjectiveAtInfinity(-op(f, 1), args(2..args(0))) then
        return([limit::Set(R_, {-infinity, infinity}) $2])
      end_if;
      break
      
    of "sinh" do
      s:= limit::accumulationPoints(op(f, 1), args(2..args(0)));
      s[1]:= piecewise::extmap
      (s[1],
       proc(S: limit::Set)
         local finpart, infinpart;
       begin
         if is(op(f, 1) in R_, Goal = TRUE) or INFIN(S) = {} then
           // argument of sinh is either real or bounded
           // use that sinh is finite-to-one on R_ and on every bounded
           // subset of C_
           finpart:= sinh(FIN(S))
         else
           finpart:= C_
         end_if;
         if INFIN(S) = {} then
           infinpart:= {}
         elif type(INFIN(S)) <> DOM_SET or map(INFIN(S), Im) <> {0} then
           infinpart:= universe
         else // sinh(inf) = inf for both -infinity and infinity
           infinpart:= INFIN(S)
         end_if;
         limit::Set(finpart, infinpart)
       end_proc
       );

      s[2]:= piecewise::extmap
      (s[2],
       proc(S)
         local finpart, infinpart;
       begin
         finpart:= sinh(FIN(S));
         if INFIN(S) = {} then
           infinpart:= {}
         elif type(INFIN(S)) <> DOM_SET or map(INFIN(S), Im) <> {0} then
           infinpart:= {}
         else // sinh(inf) = inf for both -infinity and infinity
           infinpart:= INFIN(S)
         end_if;
         limit::Set(finpart, infinpart)
       end_proc
       );

      return(s)
      
    of "cosh" do
      s:= limit::accumulationPoints(op(f, 1), args(2..args(0)));

      s[1]:= piecewise::extmap
      (s[1],
       proc(S: limit::Set)
         local finpart, infinpart;
       begin
         if is(op(f, 1) in R_, Goal = TRUE) or INFIN(S) = {} then
           // argument of cosh is either real or bounded
           // use that cosh is finite-to-one on R_ and on every bounded
           // subset of C_
           finpart:= cosh(FIN(S))
         else
           finpart:= C_
         end_if;
         if INFIN(S) = {} then
           infinpart:= {}
         elif type(INFIN(S)) <> DOM_SET or map(INFIN(S), Im) <> {0} then
           infinpart:= universe
         else // cosh(inf) = infinity 
           infinpart:= {infinity}
         end_if;
         limit::Set(finpart, infinpart)
       end_proc
       );

      s[2]:= piecewise::extmap
      (s[2],
       proc(S)
         local finpart, infinpart;
       begin
         finpart:= cosh(FIN(S));
         if INFIN(S) = {} then
           infinpart:= {}
         elif type(INFIN(S)) <> DOM_SET or map(INFIN(S), Im) <> {0} then
           infinpart:= {}
         else 
           infinpart:= {infinity}
         end_if;
         limit::Set(finpart, infinpart)
       end_proc
       );

      return(s)

      
// TODO : tanh, arcsin, arccos, arctan, arccosh

    of "arcsinh" do
      s:= limit::accumulationPoints(op(f, 1), args(2..args(0)));
      if limit::Set::oneFinitePoint(s[1]) then
        s:= limit::Set::theElement(s[1]);
        if is(Re(s) <> 0 or abs(Im(s)) < 1, Goal = TRUE) then
          return([limit::Set({arcsinh(s)}, {}) $2])
        end_if
      elif s[1] = limit::Set::infinity then
        return([limit::Set::infinity $2])
      elif s[1] = limit::Set::minusInfinity then
        return([limit::Set::minusInfinity $2])
      end_if;
      break

    of "arctanh" do
      if not is(op(f, 1) in R_, Goal = TRUE) then
        // do not handle this here as arctanh has a branch cut on the real axis
        // so the limit found recursively would be useless unless it is
        // a non-real complex number
        break
      end_if;

      s:= limit::accumulationPoints(op(f, 1), args(2..args(0)));
      if limit::Set::oneFinitePoint(s[1]) then
        s:= limit::Set::theElement(s[1]);
        if is(s<>1 and s<>-1, Goal = TRUE) then
          return([limit::Set({arctanh(s)}, {}) $2])
        end_if
      // todo: s[1] = limit::Set::infinity or ...minusInfinity  
      end_if;
      break

    of "gamma" do
      s:= limit::accumulationPoints(op(f, 1), args(2..args(0)));
      if s[1] = limit::Set::infinity then
        return([limit::Set::infinity $2])
      end_if;
      break
    of "igamma" do
      assert(nops(f) = 2);
      [a, z]:= [op(f)];
      if has(a, x) then
        // not yet implemented
        break
      end_if;  
      
      s:= limit::accumulationPoints(op(f, 2), args(2..args(0)));
      // igamma(a, x) -> 0 for x-> infinity, whatever a is
      
      if s[1] = limit::Set::infinity then
        return([limit::Set::zero $2])
      end_if;
      
      if limit::Set::oneFinitePoint(s[1]) then
        s:= limit::Set::theElement(s[1]);
        return([limit::Set({igamma(a, s)}, {}) $2])
      end_if;
      break
    of "Ei" do
      if nops(f) > 1 then
        if has(op(f, 1),  x) then 
          // not implemented
          break     
        end_if;
        
        n:= op(f, 1);
        s:= limit::accumulationPoints(op(f, 2), args(2..args(0)));
        // Ei(n, x) -> 0 for x-> infinity, whatever n is
        if s[1] = limit::Set::infinity then
          return([limit::Set::zero $2])
        end_if;
        if s[1] = limit::Set::minusInfinity then
          if n=1 then
            return([limit::Set({}, {infinity - I*PI}) $2]) 
          elif is(n in Z_ and n<=0, Goal = TRUE)  then
            return([limit::Set::infinity $2])
          end_if;
        elif limit::Set::oneFinitePoint(s[1]) then
          s:= limit::Set::theElement(s[1]);
          if is(not s <= 0, Goal = TRUE) then
            return([limit::Set({Ei(n, s)}, {}) $2])
          end_if;
          if n=1 and iszero(s) then
            // the left-hand limit of Ei(1, z) as z -> 0 is infinity - I*PI
            // from the right, the limit is infinity
            if is(op(f, 2) >= 0, Goal = TRUE) then
              return([limit::Set::infinity $2])
            end_if;
            if is(op(f, 2) <= 0, Goal = TRUE) then
              return([limit::Set({}, {infinity - I*PI}) $2]) 
            end_if;
          end_if;  
        end_if;
        
        break
        
      end_if; // nops(f) > 1
      
      

      


      // one-argument Ei
      // is *not* equal to any two argument Ei and must be handled separately
      s:= limit::accumulationPoints(op(f, 1), args(2..args(0)));
      if s[1] = limit::Set::infinity then
        return([limit::Set::infinity $2])
      end_if;
      if s[1] = limit::Set::minusInfinity and
        is(op(f, 1) in R_, Goal = TRUE) then
        return([limit::Set::zero $2])
      end_if;
      if limit::Set::oneFinitePoint(s[1]) then
        s:= limit::Set::theElement(s[1]);
        if is(s <> 0, Goal = TRUE) then
          return([limit::Set({Ei(s)}, {}) $2])
        end_if
      end_if;
      break
    of "Li" do
      if is(op(f, 1) in R_, Goal = TRUE) then
        s:= limit::accumulationPoints(op(f, 1), args(2..args(0)));
        if limit::Set::oneFinitePoint(s[1]) then
          s:= limit::Set::theElement(s[1]);
          if is(s<>1, Goal = TRUE) then
            return([limit::Set({Li(s)}, {}) $2])
          end_if
        end_if
      end_if;
      break
    of "Si" do   
      s:= limit::accumulationPoints(op(f, 1), args(2..args(0)));
      if s[1] = limit::Set::infinity then
        return([limit::Set({PI/2}, {}) $2])
      end_if;
      if s[1] = limit::Set::minusInfinity then
        return([limit::Set({-PI/2}, {}) $2])
      end_if;
      if limit::Set::oneFinitePoint(s[1]) then
        return([limit::Set({Si(limit::Set::theElement(s[1]))}, {})  $2])
      end_if;
      
      if is(op(f, 1) in R_, Goal = TRUE) and FIN(s[1]) = {} and
        nops(INFIN(s[1])) = 1 then
        assert(has(op(INFIN(s[1]), 1), infinity));
        return([{sign(op(s[1], 1)) * PI/2} $2])
      end_if;
      break
    of "Shi" do
      s:= limit::accumulationPoints(op(f, 1), args(2..args(0)));
      if s[1]= limit::Set::minusInfinity or
        s[1]= limit::Set::infinity then
        return([s[1] $2])
      end_if;
      if limit::Set::oneFinitePoint(s[1]) then
        return([limit::Set({Shi(limit::Set::theElement(s[1]))}, {}) $2])
      end_if;
      break
    of "zeta" do
      s:= limit::accumulationPoints(op(f, 1), args(2..args(0)));
      if s[1] = limit::Set::infinity then
        return([limit::Set({1}, {}) $2])
      end_if;
      if limit::Set::oneFinitePoint(s[1]) then
        s:= limit::Set::theElement(s[1]);
        if is(s<>1, Goal = TRUE) then
          return([limit::Set({zeta(s)}, {}) $2])
        end_if
      end_if;
      break

    of "erf" do
      
      s:= limit::accumulationPoints(op(f, 1), args(2..args(0)));
      if limit::Set::oneFinitePoint(s[1]) then
        s:= limit::Set::theElement(s[1]);
        return([limit::Set({erf(s)}, {}) $2])
      end_if;

      s[1]:= erf(s[1]);
      s[2]:= limit::Set::erfSub(s[2]);
      return(s)
    
    of "erfc" do
      if nops(f) = 1 then
        // erfc = 1 - erf
        s:= limit::recursive(erf(op(f, 1)), args(2..args(0)));
        s:= map(s, limit::Set::_multNumber, -1);
        return(map(s, limit::Set::_plusNumber, 1))
      end_if:

      break;
   
    of "dilog" do
      s:= limit::accumulationPoints(op(f, 1), args(2..args(0)));
      if limit::Set::oneFinitePoint(s[1]) then
        s:= limit::Set::theElement(s[1]);
        if is(not s<0, Goal = TRUE) or is(op(f, 1) in R_, Goal = TRUE) then
          return([limit::Set({dilog(s)}, {}) $2])
        else
          break
        end_if
      end_if;

      if s[1] = limit::Set::infinity then
        return([limit::Set::minusInfinity $2])
      end_if;

      break;

    of "polylog" do
      if has(op(f,1), x) then
        break
      end_if;

      s:= limit::accumulationPoints(op(f, 2), args(2..args(0)));

      if limit::Set::oneFinitePoint(s[1]) then
        s:= limit::Set::theElement(s[1]);
        if is(not s>=1, Goal = TRUE) then
          return([limit::Set({polylog(op(f, 1), s)}, {}) $2])
        else
          break
        end_if
      end_if;
      break

    of "wrightOmega" do
      s:= limit::accumulationPoints(op(f, 1), args(2..args(0)));
      if limit::Set::oneFinitePoint(s[1]) then
        s:= limit::Set::theElement(s[1]);
        return([limit::Set({wrightOmega(s)}, {}) $2])
      end_if;

      if s[1] = limit::Set::infinity then
        return([limit::Set::infinity $2])
      end_if;
      if s[1] = limit::Set::minusInfinity then
        return([limit::Set::zero $2])
      end_if;
      break
      
    of "Re" do
    of "Im" do  
      s:= limit::accumulationPoints(op(f, 1), args(2..args(0)));
      // use overloaded slots Re and Im of limit::Set
      return(map(s, eval(op(f, 0))))

    of "abs" do
      s:= limit::accumulationPoints(op(f, 1), args(2..args(0)));
      s[1]:= abs(s[1]);
      s[2]:= limit::Set::emptyset;
      return(s)
      
    of "sign" do
 
      s:= limit::accumulationPoints(op(f, 1), args(2..args(0)));
      if type(INFIN(s[1])) <> DOM_SET or
        map(INFIN(s[1]), type) minus {stdlib::Infinity} <> {} then
        break
      end_if;

      
      s[1]:= limit::Set(sign(FIN(s[1])) union map(INFIN(s[1]), sign),
                        {});

      // if FIN(s[1]) is not a subset of R_, it might be better to use
      // limit::Set({0} union FIN(limit::Set::unitcircle), {})
      // instead of sign(C_)

      finpart:= sign(FIN(s[2]));
      // it may happen that 0 is an accumulation point of g(x)
      // but not of sign(g(x)) (e.g. , if g converges from above)
        
      if type(finpart) <> DOM_SET or
        {-1, 1} minus finpart <> {} or
        not limit::isSurjectiveAtInfinity(op(f, 1), x, lp, dir, options)
        then
        finpart:= finpart minus {0}
      end_if;
                    
      s[2]:= limit::Set(finpart union map(INFIN(s[2]), sign), {});
      
      return(s)
     
      
    
      // TODO: floor, round, ceil, trunc
      
  end_case;

  // default: no information available
  [limit::Set::Cclosure, limit::Set::emptyset]
end_proc:


/*

limit::lhospital(zeroes, infinities, x, lp, dir, options)

   compute the limit of _mult(op(zeroes)) * _mult(op(infinities))
   as x goes to lp

   returns that limit or FAIL if it could not be computed

   zeroes must be a list of expressions going to zero
   infinities must be a list of expressions without finite
   accumulation point

*/

limit::lhospital:=
proc(zeroes: DOM_LIST, infinities: DOM_LIST, x, lp, dir, options):
  Type::Union(Type::Arithmetical, DOM_FAIL)

  local L, numerator, denominator, n, u,
  expnestingdepth: DOM_PROC, minabs, g;
  
begin

  // local method

  expnestingdepth:=
  proc(f)
  begin
    if not has(f, x) then
      return(0)
    end_if;

    case type(f)
      of "_plus" do
      of "_mult" do
        max(op(map([op(f)], expnestingdepth)));
        break
      of "_power" do
        // a^b = exp(b*ln(a))
        if has(op(f, 1), x) then
          f:= [op(f, 2)*ln(op(f, 1))]
        else
          f:= [op(f, 2)]
        end_if
        // fall through to the exp-branch , and access only op(f, 1)
      of "exp" do
        expnestingdepth(op(f, 1)) + 1;
        break
      of "ln" do
        expnestingdepth(op(f, 1)) - 1;
        break
      of "gamma" do
        // ln(gamma(n)) is roughly n*ln(n), i.e., harmless
        expnestingdepth(op(f, 1)) + 1;
        break
      otherwise
        0
    end_case;
  end_proc;
    
    
  
  
  if MAXEFFORT < 1000 then
    return(FAIL)
  end_if;

  options[Intervals]:= TRUE;

  numerator:= _mult(op(infinities));
  denominator:= _mult(op(zeroes));
  case testeq(denominator = 0, Steps = 10)
    of TRUE do
      userinfo(10, "Found that ".expr2text(denominator = 0)." identically");
      // the product of all factors going to zero is identically zero
      // thus our function in question is identically zero
      return(0)
    of UNKNOWN do
      userinfo(10, "Could not decide zero equivalence of ".
               expr2text(denominator));
      // be careful, although this means almost certainly 0
      return(FAIL)
  end_case;
  // now we are sure that the inverse of the factors going to zero exists
  // remaining problem: they could be identically zero only locally,
  // that is, in some environment of the limit point
  denominator:= 1/denominator;
  userinfo(10, "Both ".expr2text(numerator)." and ".expr2text(denominator).
           " tend to infinity, using l'Hospital for their quotient");

  // preprocessing step
  // like in MRV, the problem is that functions may grow at least as fast as
  // exp, such that their derivatives grow to infinity faster and faster ...

  minabs:= (x, y) -> if x<0 and y<0 then max(x, y) else min(x, y) end;
  n:= minabs(expnestingdepth(numerator), expnestingdepth(denominator));

  // we want to substitute x-> (ln@@n)(u)
  // then as x-> lp, u -> (exp@@n)(lp)
  // problem: we need to substitute by a *real* variable, so we just do this
  // for infinite x at the moment

  if n > 0 then
    if lp=infinity then
      u:= genident("u");
      assume(u>20);
      numerator:= evalAt(numerator, x=(ln@@n)(u));
      denominator:= evalAt(denominator, x=(ln@@n)(u));
      x:= u;
    elif lp=-infinity then
      // substitute -x by (ln@@n)(u) 
      u:=genident("u");
      assume(u>20);
      numerator:= evalAt(numerator, x=-(ln@@n)(u));
      denominator:= evalAt(denominator, x=-(ln@@n)(u));
      lp:= infinity;
      dir:= Left;
      x:= u
    end_if
  end_if;

 
  
  numerator:= diff(numerator, x);
  denominator:= diff(denominator, x);
  if iszero(denominator) then
    // this can happen in rare cases, for functions like sign
    return(FAIL)
  end_if;
  userinfo(10, "Derivative of numerator is ".expr2text(numerator));
  userinfo(10, "Derivative of denominator is ".expr2text(denominator));

  // TODO: problem: take care that we do not cancel any common factor
  // with infinitely many zeroes near the limit point! (see Gruntz p. 22)

  g:= gcd(numerator, denominator);
  if g<>1 then
    numerator:= normal(numerator/g);
    denominator:= normal(denominator/g);
  end_if;
  g:= numerator/denominator;
  
  L:= limit::accumulationPoints(g, x, lp, dir, options);

  if limit::Set::hasMoreThanOneElement(L[1]) = FALSE then
    userinfo(10, "L'Hospital succeeds");
    u:= limit::Set::theElement(L[1]);
    if has(u, hold(ln)) then // we have created some logarithms ...
      u:= misc::maprec(u, {"ln"} = ln::simplify);
      u:= simplify(u, ln);
    end_if;
    normal(u)
  else
    userinfo(10, "L'Hospital fails");
    FAIL
  end_if
end_proc:

  

  
limit::complexSeries:=
proc(f, x, lp, dir, options)
  local signlc, s, S, ld, re, im, infs, a,
  order: DOM_INT, i: DOM_INT;

begin
  userinfo(3,"Try series computation");

  f:= rewrite(f, gamma); // to eliminate fact etc.
  
  s:= FAIL;
  S:= FAIL;
  for order from 1 to ORDER do
    // an error occurs if the requested order is too small!
    if traperror((s:= series(f, x = lp, order, dir,
                             hold(NoWarning), UseGseries = TRUE))) = 0 then
      if domtype(s) <> Series::Puiseux and
        domtype(s) <> Series::gseries then
        if order < ORDER/2 then
          next;
        else
          break
        end_if;
      end_if;
      S := FAIL;
      if traperror((S := ldegree(s))) <> 0 then
        // ldegree produces an error, so it does not make
        // much sense to continue with higher order
        break;
      end_if;
      if S = FAIL then
        if not contains({-infinity, infinity}, lp) then
          if type(s) = Series::Puiseux and Series::Puiseux::order(s) > 0 then
            // if f = O(z^epsilon), then f->0 as z->0
            return([limit::Set::zero $2])
          end_if
        end_if
      else
        // success
        break
      end_if
    end_if;
  end_for;
  if (domtype(s) = Series::Puiseux or domtype(s) = Series::gseries)
    and S <> FAIL then
    case sign(S)
      of  0 do
        ld:= lmonomial(s);
        if hastype(ld, {"sin", "cos", "exp", "ln", "signIm"}) then
          // rewrites, e.g., sin(x)^2 + cos(x)^2 --> 1
          ld := simplify(ld)
        end_if;  
        if has(ld,x) then
          if Simplify::defaultValuation(ld) >= Simplify::defaultValuation(f)
            then
            // avoid infinite loop
            break
          else
            MAXEFFORT:= MAXEFFORT/2;
            s:= limit::accumulationPoints(ld, x, lp, dir, options);
            infs:= INFIN(s[1]);
            if type(infs) = DOM_SET and infs <> {} then
              if map(Im(infs), X -> is(X<>0, Goal = TRUE)) <> {TRUE} and
                testeq(Im(f), 0) = FALSE then
                // there might be a small imaginary part to add
                ld:= ld + I*(Im(f) - Im(ld))
              elif map(Re(infs), X -> is(X<>0, Goal = TRUE)) <> {TRUE} and
                testeq(Re(f), 0) = FALSE then
                ld:= ld + Re(f) - Re(ld)
              else
                return(s)
              end_if
            else
              return(s)
            end_if;

            if Simplify::defaultValuation(ld) >=
              Simplify::defaultValuation(f) then
              break
            end_if;
            
            return(limit::accumulationPoints(ld, x, lp, dir, options))
          end_if
        else
          return([limit::Set({ld}, {}) $2])
        end_if
      of  1 do
        // if the leading coefficient contains x, then we do not know in
        // general whether it is bounded.
        // We assume that the leading coefficient is continuous and therefore
        // bounded near 0 if evaluation at zero does not fail
        s:= lcoeff(s);
        assert(s <> FAIL);
        // transform to zero
        if not has(lp, infinity) then
          s:= evalAt(s, x = x+lp)
        else
          s:= evalAt(s, x = 1/x)
        end_if;
        
        if limit::logBounded(s, x) = TRUE then
          re:= [limit::Set::zero $2];
          // we still may have to care about singularities of the leading coefficient
          if not has(s, x) and nops((a:= freeIndets(s))) = 1 then
             a:= op(a, 1);
             if traperror((ld:= discont(s, a, Undefined)), MaxSteps = 10) = 0 and
             type(ld) = DOM_SET then 
               // first, we throw away those elements of ld where f itself is undefined (we assume that the user's input makes sense)
               ld:= select(ld, X -> traperror((evalAt(f, a = X))) = 0);
               re:= [[not a in ld, re],
                     [a = op(ld, i), limit::accumulationPoints(evalAt(f, a=op(ld, i)),  x, lp, dir, options)] $i=1..nops(ld)
                     ];
               re:= piecewise(op(re));
               return([piecewise::extmap(re, op, i) $i=1..2])
             else
               // cannot plug in elements of arbitrary sets
               // not quite correct, we are missing some cases !!
               return(re)
             end_if;
           else   
             return(re)  
           end_if  
        end_if;
        break
      of -1 do
        // NOTE: the leading coefficient could contain x!
        // Example: limit(ln(x)/x,x=0,Left).
        // However, it is guaranteed that the lcoeff does not grow
        // stronger than the rest of the leading monomial. So the
        // limit is infinity*sign(lcoeff).
        ld:= lcoeff(s);
        signlc:= sign(ld);
        if not has(signlc,x) then
          if type(S) = DOM_RAT then
            break
          elif type(S) = DOM_INT then
            if S mod 2 = 1 and lp <> infinity then
              if dir = Left or lp = -infinity then
                s:= {-signlc}
              elif dir = Real then
                s:= {signlc, -signlc}
              else
                s:= {signlc}
              end_if
            else
              s:= {signlc}
            end_if;
          else
            // symbolic sign or complex number
            break
          end_if;
        else
          if not is(signlc in R_, Goal = TRUE) then
            // do not handle complex sign
            break
          end_if;
          s:= limit::accumulationPoints(signlc, x, lp, dir, options);
          s[1]:= FIN(s[1]);
          if type(s[1]) <> DOM_SET then
            break
          end_if;
          s[2]:= FIN(s[2]);
          if s[1] <> s[2] and nops(s[1]) > 1 then
            break
          end_if;
          s:= s[1]
        end_if;
        // the leading term of the series is s/x. However, it is still
        // not clear what the imaginary part of the limit is.
        if map(s, Im) = {0} then
          re:= {};
          for i from 1 to nops(s) do
            re:= re union
                 piecewise([s[i]>0, {infinity}],
                           [s[i]<0, {-infinity}],
                           [s[i] = 0, {0}]
                           )
          end_for;
          
          re:= piecewise::extmap
          (re,
           proc(u)
             local l;
           begin
             l:= split(u, has, infinity);
             limit::Set(l[2], l[1])
           end_proc
           );

           // determine the imaginary part
           // it can happen that Im knows more than is 
           if not is(f in R_, Goal = TRUE) and not iszero((im:= Im(f))) then
             
             
             if has(im, hold(Im)) then          
               im:= rectform(f);
               if op(im, 3) = 0 then
                 im:= op(im, 2)
               else
                 im:= FAIL
               end_if;
             end_if;
             
             if im <> FAIL then
               im:= limit::accumulationPoints(im, x, lp, dir, options);
               re:= 
               [
               limit::Set::superSetFromReIm(re, im[1]),
               limit::Set::subSetFromReIm(re, im[2])
               ]
             else
               break           
             end_if;
             
           else
             re:= [re $2];
             userinfo(5, "Imaginary part is zero")
           end_if;
           // we must not immediately return re as the leading coefficient could be zero
           // catch only simple cases of this now
           if not has(ld, x) and nops((a:= freeIndets(ld))) = 1 and testtype(ld, Type::PolyExpr((a:= op(a, 1)))) then
             s:= solve(ld, a);
             if type(s) = DOM_SET then 
               re:= [[not a in s, re],
                     [a = op(s, i), limit::accumulationPoints(evalAt(f, a=op(s, i)),  x, lp, dir, options)] $i=1..nops(s)
                     ];
               re:= piecewise(op(re));
               return([piecewise::extmap(re, op, i) $i=1..2])
             else
               // cannot plug in RootOf's
               // not quite correct, we are missing some cases !!
               return(re)
             end_if;
           else
             // not quite correct, we are missing some cases !!
             return(re)
           end_if  
         end_if;  
         break
      otherwise
        userinfo(10, "cannot determine sign of ".expr2text(S) );
        break
    end_case;
  end_if;

  userinfo(3,"Direct series computation fails");

  return([limit::Set::Cclosure, limit::Set::emptyset])
end_proc:



/*
  limit::transformToInfinity(ex, x, lp, dir)

  returns an expression f such that the accumulation points of f for
  x->infinity are the same as the accumulation points of ex for  x->lp
  from dir

*/
limit::transformToInfinity:=
proc(ex, x, lp, dir)
begin
  if lp = -infinity then
    evalAt(ex, x = -x)
  elif lp = infinity then
    ex
  elif dir = hold(Right) then
    evalAt(ex, x = lp + 1/x)
  elif dir = hold(Left) then
    evalAt(ex, x = lp - 1/x)
  else
    error("cannot do transform to infinity")
  end_if
end_proc:

/*
limit::isSurjectiveAtInfinity(ex, x, lp, dir)

given that ex tends to infinity as x tends to lp, find out whether
ex takes on every large enough real value

returns TRUE if ex is provably surjective, and FALSE otherwise
*/
limit::isSurjectiveAtInfinity:=
proc(ex, x, lp, dir)
  local c, isEnvironment: DOM_PROC;
begin

  // returns TRUE if the set of all x satisfying cond is an environment of
  // lp (possibly restricting the domain of all x under consideration to some
  // interval to the left/right of lp); we assume that cond is satisfied by at
  // least *one* x in every environment
  // returns FALSE also if we don't know 

  // this is not 100% reliable at the moment: if something(x) has been assumed
  // to lie in an interval, we hope that the component of the preimage that has limit point lp is connected
  isEnvironment:=
  proc(cond)
  begin
    case type(cond)
      of "_less" do
      of "_leequal" do
        return(TRUE)
      of "_in" do
        if type(op(cond, 2)) = Dom::Interval or
          op(cond, 2) = R_ then
          return(TRUE)
        end_if;
        break
    end_case;
    FALSE
  end_proc;
    
  case type(ex)
    of DOM_IDENT do
      assert(ex = x);
      // x is defined in an environment of lp?
      if _lazy_and(isEnvironment(c) $c in showprop(x)) then
        return(TRUE)
      else
        return(FALSE)
      end_if
    of "exp" do
    of "ln" do
      return(limit::isSurjectiveAtInfinity(op(ex, 1), x, lp, dir))
  end_case;

  // if ex is defined in an environemnt of lp and continuous, then
  // it must be surjective by the intermediate value theorem since
  // every environment contains a connected environment
  if _lazy_and(isEnvironment(c) $c in showprop(x)) and
    discont(ex, x) = {} then
    TRUE
  else
    // don't know whether ex is surjective
    FALSE
  end_if
end_proc:



/***************************************************************

The datatype limit::Set represents subsets of the closure of C_
(possible sets of accumulation points)
A set S in the closure of C_ consists of two operands:
- S intersect C_, the finite part of S: any object of Type::Set
- the infinite objects of S: a DOM_SET, or universe to indicate that there
  are infinitely many infinite points (we usually cannopt handle this)
****************************************************************/

limit::Set:= newDomain("limitSet"):

limit::Set::interface:= {}:


limit::closure:= funcenv( S -> procname(S)):

// *****
// particular subsets of the closure of C_
// *****

limit::Set::Cclosure:= new(limit::Set, C_, universe):

limit::Set::emptyset:= new(limit::Set, {}, {}):

limit::Set::zero:= new(limit::Set, {0}, {}):

limit::Set::infinity:= new(limit::Set, {}, {infinity}):
limit::Set::minusInfinity:= new(limit::Set, {}, {-infinity}):


limit::unitcircle:= Dom::ImageSet(cos(#x*PI) + I*sin(#x*PI), #x,
                                           Dom::Interval([0], 2)):

limit::Set::unitcircle:= new(limit::Set, 
                             limit::unitcircle,
                             {}):

/*
  methods for handling limit::Set
*/

limit::Set::iszero:= S -> bool(FIN(S) = {0} and INFIN(S) = {}):

limit::Set::new:=
proc(finite, infinite)
  local new_intern;
begin
  case args(0)
    of 1 do
      return(limit::Set::convert(finite))
    of 2 do
      assert(_lazy_or(type(infinite) <> DOM_SET,
                      not contains(map(infinite, has, infinity), FALSE)));
      
      new_intern:=
      proc(fin, inf)
      begin
        if type(inf) <> DOM_SET then
          return(limit::Set::Cclosure)
        else
          return(new(limit::Set, fin, inf))
        end_if
      end_proc;

      return(piecewise::zip(finite, infinite, new_intern))
            
  end_case;

  error("Wrong number of arguments")
end_proc:

limit::Set::convert:=
proc(S)
  local l;
begin
  if type(S) = DOM_SET then
    l:= split(S, has, infinity);
    new(limit::Set, l[2], l[1])
  elif type(S) = piecewise then
    piecewise::extmap(S, limit::Set::convert)
  elif S = universe then
    new(limit::Set, C_, universe)
  else
    new(limit::Set, S, {})
  end_if
end_proc:

limit::Set::convertToSet:=
proc(S: Type::Union(limit::Set, piecewise))
begin
  if type(S) = piecewise then
    piecewise::extmap(S, limit::Set::convertToSet)
  elif extop(S, 2) = {} then
    extop(S, 1)
  elif extop(S, 1) = {} then
    extop(S, 2)
  elif type(extop(S, 2)) <> DOM_SET then
    limit::closure(C_)
  else
    hold(_union)(extop(S))
  end_if
end_proc:

limit::Set::oneFinitePoint:=
proc(S: Type::Union(limit::Set, piecewise)): DOM_BOOL
begin
  bool(type(S) = limit::Set and
       type(FIN(S)) = DOM_SET and
       nops(FIN(S)) = 1 and
       INFIN(S) = {}
       )
end_proc:

/*
  hasMoreThanOneElement(S)

  returns TRUE provided that S has provably more than one element
  returns FALSE if not
  returns a boolean expression equivalent to nops(S) = 1 otherwise

*/
limit::Set::hasMoreThanOneElement:=
proc(S: Type::Union(limit::Set, piecewise)): Type::Boolean 
  local i: DOM_INT, fin, infin, unequal;
begin

  // local method unequal(a, b)
  // returns TRUE if the (infinite) points a and b are mathematically
  // different, and FALSE if they are not 
  // returns an expression equivalent to a<>b otherwise
  // from the way the function is called, we know that a<>b syntactically
   
  unequal:=
  proc(a, b)
     local ra, rb, ia, ib, rainf, rbinf, iainf, ibinf;
  begin
    if indets(a) = {} and indets(b) = {} then
      // they are syntactically unequal, so:
      return(TRUE)
    end_if;
    if type(a) = stdlib::Infinity and type(b) = stdlib::Infinity then
      return(sign(op(a, 1)) <> sign(op(b, 1)))
    end_if;

    ra:= Re(a);
    rb:= Re(b);
    ia:= Im(a);
    ib:= Im(b);
    [rainf, rbinf, iainf, ibinf]:= map([ra, rb, ia, ib], has, infinity);

    if rainf <> rbinf or iainf<>ibinf then
       FALSE
    elif ra = rb then
      if not iainf then
         ia <> ib
      else
         unequal(ia, ib)
      end_if
    elif ia = ib then
      if not rainf then 
        ra <> rb
      elif ra <> a or rb <> b then
        unequal(ra, rb)
      else
        // to avoid infinite recursion 
        a <> b
      end_if
    else
      a <> b
    end_if
  end_proc;
  
  if type(S) = piecewise then
    return(_or((piecewise::condition(S, i) and limit::Set::hasMoreThanOneElement(piecewise::expression(S, i))) $i=1..nops(S)))
  end_if;

  fin:= extop(S, 1);
  infin:= extop(S, 2);
  
  if type(fin) <> DOM_SET or type(infin) <> DOM_SET then
    return(TRUE)
  end_if;

  case nops(fin)
    of 0 do
      if nops(infin) <= 1 then
        return(FALSE)
      end_if;
      // there are more than one infinite accumulation points;
      // but are two of them distinct?
      return(simplify::simplifyCondition(_or(unequal(op(infin, i), op(infin, 1)) $i=2..nops(infin))))
        
    of 1 do
      return(bool(nops(infin) > 0))
    otherwise
      if nops(infin) > 0 then
        return(TRUE)
      end_if;
      // there are several finite accumulation points;
       // but are two of them distinct?
      return(simplify::simplifyCondition(_or(op(fin, i) <> op(fin, 1) $i=2..nops(fin)))
             )
  end_case;

  // we should have returned before
  assert(FALSE);
  UNKNOWN
  
end_proc:

limit::Set::theElement:=
proc(S: Type::Union(limit::Set, piecewise))
begin
  assert(limit::Set::hasMoreThanOneElement(S) <> TRUE);
  if type(S) = piecewise then
    piecewise::extmap(S, limit::Set::theElement)
  elif FIN(S) = {} then
    if INFIN(S) = {} then
      FAIL
    else
      op(INFIN(S), 1)
    end_if
  else
    op(FIN(S), 1)
  end_if
end_proc:


limit::Set::testtype:=
proc(S, typ)
begin
  if op(typ, 2) = "PolyExpr" then
    return(FALSE)
  end_if;
  FAIL
end_proc:



limit::Set::_union:=
proc()
  local argv, i;
begin
  argv:= [args()];
  if (i:= contains(map(argv, type), piecewise)) > 0 then
    piecewise::extmap(argv[i], _union, op(argv, 1..i-1),
                      op(argv, i+1..nops(argv)))
  else
    new(limit::Set,
        _union(op(map([args()], extop, 1))),
        _union(op(map([args()], extop, 2))))
  end_if
end_proc:


limit::Set::_intersect:=
proc()
  local argv, i;
begin
  argv:= [args()];
  if (i:= contains(map(argv, type), piecewise)) > 0 then
    piecewise::extmap(argv[i], _intersect, op(argv, 1..i-1),
                      op(argv, i+1..nops(argv)))
  else
    limit::Set(
               solvelib::solve_intersect(op(map([args()], extop, 1))),
               limit::intersectInfinities(op(map([args()], extop, 2)))
               )
  end_if
end_proc:


// return a superset of the intersection of sets consisting of infinite objects
// only
limit::intersectInfinities:=
proc()
  local argv, result: DOM_SET, equal: DOM_PROC,
  i: DOM_INT, j: DOM_INT, k: DOM_INT,
  found: DOM_BOOL;
begin
  // let us hope that is(..) can also handle infinities; otherwise change this
  equal:=
  proc(inf1, inf2)
  begin
    is(Re(inf1) = Re(inf2) and Im(inf1) = Im(inf2))
  end_proc;
    
  
  argv:= select([args()], testtype, DOM_SET);
  if nops(argv) = 0 then
    return(universe)
  end_if;
  // smallest set first
  argv:= prog::sort(argv, nops);
  // extract those elements that are obviously common to all sets in the input
  result:= _intersect(op(argv));
  argv:= map(argv, _minus, result);
  
  for i from 1 to nops(argv[1]) do
    found:= TRUE; // so far, op(argv[1], i) has been in all members of argv
    for j from 2 to nops(argv) do
      // try to find op(argv[1], i) in argv[j]
      found:= FALSE; 
      for k from 1 to nops(argv[j]) do
        if equal(op(argv[1], i), op(argv[j], k)) <> FALSE then
          found:= TRUE;
          break
        end_if
      end_for;
      if not found then
      // not in the intersection; need not look at the other members of argv
        break
      end_if
    end_for;
    if found then
      result:= result union {op(argv[1], i)}
    end_if
  end_for;

  result
end_proc:

  

limit::Set::_plusNumber:=
proc(S, a)
begin
  if type(S) = piecewise then
    // remove conditions for which a is not defined
    S:= piecewise::selectConditions
    (S, cond -> piecewise::evalAssuming(a, cond) <> undefined);
    piecewise::extmap(S, limit::Set::_plusNumber, a)
  elif type(INFIN(S)) <> DOM_SET then
    S
  else
    extsubsop(S, 1= FIN(S) + a, 2 = INFIN(S) + a)
  end_if
end_proc:


limit::Set::_multNumber:=
proc(S, a)
begin
  if type(S) = piecewise then
    // remove conditions for which a is not defined
    S:= piecewise::selectConditions
    (S, cond -> piecewise::evalAssuming(a, cond) <> undefined);
    piecewise::extmap(S, limit::Set::_multNumber, a)
  elif S = limit::Set::Cclosure then
    S
  else
    extsubsop(S, 1= FIN(S) * a, 2= INFIN(S)*a)
  end_if
end_proc:


limit::Set::_invert:=
proc(S)
begin
  if INFIN(S) = {} then
    limit::Set(1/FIN(S), {})
  else
    limit::Set(1/FIN(S) union {0}, {})
  end_if
end_proc:
  

limit::Set::exp:=
proc(S)
  local finite, infinite;
begin
  if type(INFIN(S)) <> DOM_SET then
    return(limit::Set::Cclosure)
  end_if;

  finite:= exp(FIN(S));
  infinite:= {};
  
  map(INFIN(S),
      proc(x)
        local s;
      begin
        if type(x) = stdlib::Infinity then
          s:= op(x, 1)
        else      
          s:= sign(x);
          if has(s, infinity) then
            s:= sign(Re(x)) + I*sign(Im(x))
          end_if;

          if has(s, infinity) then
            infinite:= universe;
            return(infinite)
          end_if;
        end_if;
      
        finite:= piecewise([Re(s) = 0, finite union limit::unitcircle],
                           [Re(s) < 0, finite union {0}],
                           [Re(s) > 0, finite]
                           );

        infinite:= piecewise([Re(s) <= 0, infinite],
                             [s > 0, infinite union {infinity}],
                             [Otherwise, INFIN(limit::Set::Cclosure)]
                             )
      end_proc
      );
  
  limit::Set(finite, infinite)
end_proc:

// compute a superset/subset of {x^n; x in S}
limit::Set::_powerPosIntSuper:=
proc(S: Type::Union(limit::Set, piecewise), n: DOM_INT)
begin
  if n=1 then
    return(S)
  end_if;

  if type(S) = piecewise then
    return(piecewise::extmap(S, limit::Set::_powerPosIntSuper, n))
  end_if;
  
  if type(INFIN(S)) <> DOM_SET then
    return(limit::Set::Cclosure)
  end_if;

  if map(INFIN(S), type) minus {stdlib::Infinity} <> {} then
    // infinities like infinity + I*infinity where one cannot
    // determine the polar angle
    return(limit::Set::Cclosure)
  end_if;

  limit::Set::new(FIN(S)^n, INFIN(S)^n)
end_proc:

limit::Set::_powerPosIntSub:=
proc(S: Type::Union(limit::Set, piecewise), n: DOM_INT)
  local inf;
begin
  if n=1 then
    return(S)
  end_if;

  if type(S) = piecewise then
    return(piecewise::extmap(S, limit::Set::_powerPosIntSub, n))
  end_if;
  
  if type(INFIN(S)) <> DOM_SET then
    inf:= {}
  end_if;

  limit::Set::new(FIN(S)^n,
                  (select(INFIN(S), x -> type(x) = stdlib::Infinity))^n)
end_proc:


/*
  given subsets of {Re(x); x in S} and {Im(x); x in S},
  compute a subset of S
*/
limit::Set::subSetFromReIm:=
proc(realpart: Type::Union(limit::Set, piecewise),
     imagpart: Type::Union(limit::Set, piecewise))
begin
  if type(realpart) = piecewise then
    piecewise::extmap(realpart, limit::Set::subSetFromReIm, imagpart)
  elif type(imagpart) = piecewise then
    piecewise::extmap(imagpart, u-> limit::Set::subSetFromReIm(realpart, u))
  elif iszero(imagpart) then
    realpart
  elif limit::Set::hasMoreThanOneElement(realpart) = FALSE and
    limit::Set::hasMoreThanOneElement(imagpart) = FALSE then
    limit::Set(FIN(realpart) + I*FIN(imagpart),
               INFIN(realpart) + I*INFIN(imagpart))
  else
    limit::Set::emptyset
  end_if;
end_proc:


limit::Set::Re:=
proc(S)
begin
  if S = limit::Set::Cclosure then
    limit::Set(R_, {-infinity, infinity})
  else
    limit::Set(Re(FIN(S)) union Re(INFIN(S)))
  end_if
end_proc:

limit::Set::Im:=
proc(S)
  begin
  if S = limit::Set::Cclosure then
    limit::Set(R_, {-infinity, infinity})
  else
    limit::Set(Im(FIN(S)) union Im(INFIN(S)))
  end_if
end_proc:

limit::Set::abs:=
proc(S)
begin
  limit::Set(
             if type(FIN(S)) in {Dom::Interval, DOM_SET, solvelib::BasicSet} then
               abs(FIN(S))
             else
               Dom::Interval([0], infinity)
             end_if
               ,
             if INFIN(S) = {} then
               {}
             else
               {infinity}
             end_if
             )
end_proc:



/*
 given supersets of {Re(x); x in S} and {Im(x); x in S},
  compute a superset of S
*/
limit::Set::superSetFromReIm:=
proc(realpart: Type::Union(limit::Set, piecewise),
     imagpart: Type::Union(limit::Set, piecewise))
begin
  if type(realpart) = piecewise then
    piecewise::extmap(realpart, limit::Set::superSetFromReIm, imagpart)
  elif type(imagpart) = piecewise then
    piecewise::extmap(imagpart, u-> limit::Set::superSetFromReIm(realpart, u)) 
  elif iszero(imagpart) then
    // by compactness, there must be at least one accumulation point:
    assert(realpart <> limit::Set::emptyset);
    realpart
  elif type(INFIN(realpart)) = DOM_SET and
    type(INFIN(imagpart)) = DOM_SET then
    limit::Set(FIN(realpart) + I*FIN(imagpart),
               FIN(realpart) + I*INFIN(imagpart)
               union
               INFIN(realpart) + I*FIN(imagpart)
               union
               INFIN(realpart) + I*INFIN(imagpart)
               )
  else
    limit::Set::Cclosure
  end_if
end_proc:


limit::Set::erf:=
proc(S: limit::Set)
  local infinpart, resultinfin, resultfin, a, b, c, sgn;
begin
  infinpart:= INFIN(S);
  if type(infinpart) <> DOM_SET then
    return(limit::Set::Cclosure)
  end_if;

  resultinfin:= {};
  resultfin:= {};
  for a in infinpart do
    if type(a) = stdlib::Infinity then
      b:= op(a, 1);
      if stdlib::hasmsign(b) then
        b:= -b;
        sgn:= -1
      else
        sgn:= 1
      end_if;
      if type(b) = "_power" and
        (op(b, 2) = 1/2 or op(b, 2) = -1/2) then
        c:= op(b, 1);
        if op(b, 2) = -1/2 then c:= 1/c end;
        resultinfin:= resultinfin union
                      piecewise([c < 0, {sgn*I*infinity}],
                                [Re(c) >=0 and c<>0, {}],
                                [Otherwise, universe]
                                );

        resultfin:= resultfin union
                    piecewise([Re(c) >= 0 and c<>0, {sgn}],
                              [Otherwise, {}]
                              ):
        
      else  
      
        resultinfin:= resultinfin union
                      piecewise([sign(b) = I or sign(b) = -I, {a}],
                                [Re(b) <> 0 and abs(Re(b)) >= abs(Im(b)), {}],
                                [Otherwise, universe]
                                );

        resultfin:= resultfin union
                    piecewise([abs(Re(b)) >= abs(Im(b)), {sgn*sign(Re(b))}],
                              [Otherwise, {}]
                              ):
      end_if
                     
    else

      if not has(Im(a), infinity) then
       resultfin:= resultfin union {sign(Re(a))}
      else
        return(limit::Set::Cclosure)
      end_if
    end_if;
  end_for;
  
  resultfin:= resultfin union erf(FIN(S));

  limit::Set(resultfin, resultinfin)

end_proc:


limit::Set::erfSub:=
proc(S)
  local infinpart, resultinfin, resultfin, a, b, c, sgn;
begin
  if type(S) = piecewise then
    return(piecewise::extmap(S, limit::Set::erfSub))
  end_if;

  infinpart:= INFIN(S);
  resultfin:= {};
  resultinfin:= {};
  if type(infinpart) = DOM_SET then
    for a in infinpart do
      if type(a) = stdlib::Infinity then
        b:= op(a, 1);
        if stdlib::hasmsign(b) then
          // apply erf(x) = -erf(-x)
          b:= -b;
          sgn:= -1
        else
          sgn:= 1
        end_if;
        if type(b) = "_power" and
          (op(b, 2) = 1/2 or op(b, 2) = -1/2) then
          c:= op(b, 1);
          if op(b, 2) = -1/2 then c:= 1/c end;
          resultinfin:= resultinfin union
                        piecewise([c < 0, {sgn*I*infinity}],
                                  [Re(c) >=0 and c<>0, {}],
                                  [Otherwise, universe]
                                  );

          resultfin:= resultfin union
                      piecewise([Re(c)>=0 and c<>0, {sgn}],
                                [Otherwise, {}]
                                ):

        else  
        
          resultinfin:= resultinfin union
                        piecewise([sign(b) = I or sign(b) = -I, {a}],
                                  [Re(b) <> 0 and abs(Re(b)) >= abs(Im(b)), {}],
                                  [Otherwise, universe]);
          
          resultfin:= resultfin union
                      piecewise([abs(Re(b)) >= abs(Im(b)), {sgn*sign(Re(b))}],
                                [Otherwise, {}])

        end_if
      end_if;
    end_for;
  end_if;

  resultfin:= resultfin union erf(FIN(S));
  
  return(limit::Set(resultfin, resultinfin))

end_proc:



/**********************************************************************
 end of limit::Set section
***********************************************************************/

// determine that Simplify(limit(f, x=lp)) only simplifies f
limit::operandsToSimplify:= [1]:


limit::Content :=
proc(Out, data)
  local direction, condition, tendsto, bvar;
begin
  if nops(data) < 2 or nops(data) > 3 then
    return(Out::stdFunc(data));
  end_if;
  if type(op(data, 2)) = "_equal" then
    tendsto := Out(op(data, [2,2]));
    bvar := Out(op(data, [2,1]));
  else
    tendsto := Out(0);
    bvar := Out(op(data, 2));
  end_if;

  if nops(data) = 3 then
    if op(data, 3) = hold(Right) then
      direction := Out::Ctendsto(["type" = "above"]);
    elif op(data, 3) = hold(Left) then
      direction := Out::Ctendsto(["type" = "below"]);
    else
      direction := Out::Ctendsto(["type" ="two-sided"]);
    end_if;
    condition := Out::Ccondition(
                                 Out::Capply(direction, bvar, tendsto)
                                 );
  else
    condition := Out::Clowlimit(tendsto);
  end_if;

  Out::Capply(Out::Climit, Out::Cbvar(bvar), condition, Out(op(data,1)));
end_proc:

limit::TeX :=
proc(l, data, prio)
  local direction, tendsto, bvar;
begin
  if type(op(data, 2)) = "_equal" then
    tendsto := "{".
               generate::tex(op(data, [2,2]), output::Priority::Stmtseq).
    "}";
    bvar := generate::tex(op(data, [2,1]), output::Priority::Stmtseq);
  else
    tendsto := "0";
    bvar := generate::tex(op(data, 2), output::Priority::Stmtseq);
  end_if;

  direction := "";
  if nops(data) = 3 then
    if op(data, 3) = hold(Right) then
      direction := "^+";
    elif op(data, 3) = hold(Left) then
      direction := "^-";
    else
      direction := "^{\\pm}";
    end_if;
  end_if;

  "\\lim_{".bvar."\\to ".tendsto.direction."}".
  generate::tex(op(data,1), output::Priority::Mult);
end_proc:

limit::diff := (lim, x) -> limit(diff(op(lim, 1), x), op(lim, 2..nops(lim))):




/*
limit::logBounded(f, x)

checks whether f remains bounded by some power of log(x) when x is in a small interval [0, epsilon]

returns TRUE if f is provably bounded, and FALSE otherwise

*/

limit::logBounded:=
proc(f, x)
  local s, i;
begin
  if not has(f, x) then
    return(TRUE)
  end_if;
  if traperror(evalAt(f, x=0)) = 0 then
    userinfo(10, "Function is defined at zero and assumed to be continuous there");
    return(TRUE)
  end_if;
  // try interval arithmetic
  s:= evalAt(f, x=0...0.1);
  if type(s) = DOM_INTERVAL and not has(s, {RD_NINF, RD_INF}) then
    userinfo(10, "Function proved to be bounded by interval arithmetics");
    return(TRUE)
  end_if;

  // recursive descent
  case type(f)
    of DOM_IDENT do
      assert(f = x);
      return(TRUE)
    of "_mult" do
    of "_plus" do
      return(_and(limit::logBounded(op(f, i), x) $i=1..nops(f)))
    of "sin" do
    of "cos" do
      return(TRUE)
    of "ln" do
      return(limit::logBounded(op(f, 1), x))
    of "_power" do
      if is(op(f, 2) >= 0) = TRUE then
        return(limit::logBounded(op(f, 1), x))
      end_if;
      break
  end_case;

  return(FALSE)
end_proc:



/***************************************************************************
methods for generalized power series (at zero, from the right).
A generalized power series is a series
\sum a_alpha x^alpha  where the alpha are real numbers.

It is represented as a domain element with the following operands:
- a list of pairs [coefficient, exponent]
- the order of the error term: a real number or infinity
- the series variable

Cf. Gruntz, Section 3.3.3, p.48/49

The slots limit::powerSeries::f for creating a power series for f(g(x))
have the following syntax:
limit::powerSeries::f(gs, x, order, lnx, xarg, param)

gs - powers series for g(x)
 x - variable
order - order of the error term
lnx  - identifier representing ln(x)
xarg - the number of the argument of g in which x occurs. This is usually 1, but may be 1 or 2 for functions like igamma. 
       We cannot handle the case that x occurs in both arguments (then limit::powerSeries::new returns FAIL without overloading).
param - the other argument not containing x, or null() if g is a function of one argument
****************************************************************************/

limit::powerSeries:= newDomain("limit::powerSeries"):

limit::powerSeries::interface:= {}: // only internal domain at the moment

limit::powerSeries::errorterm:= S -> extop(S, 2):

limit::powerSeries::variable:= S -> extop(S, 3):

limit::powerSeries::expr:=
proc(S: limit::powerSeries): Type::Arithmetical
  local
  x,
  l: DOM_LIST,
  i: DOM_INT;
begin
  x:= limit::powerSeries::variable(S);
  l:= extop(S, 1);
  _plus(l[i][1]* x^(l[i][2]) $i=1..nops(l))
end_proc:

limit::powerSeries::print:=
proc(S: limit::powerSeries)
  local x, err;
begin
  x:= limit::powerSeries::variable(S);
  err:= limit::powerSeries::errorterm(S);
  if err = infinity then
    expr(S)
  else
    expr(S) + hold(O)(x^err)
  end_if
end_proc:


// normalize: adds entries with equal exponent, and sorts by
  // exponent in increasing order
  // deletes terms that are smaller or equal to the error term
limit::powerSeries::normalize:=
proc(l: DOM_LIST, err)
  local i: DOM_INT;
begin
  l:= sort(l, proc(x, y)
              begin
                _lazy_and(x[2] <> infinity,
                          _lazy_or(
                                   y[2] = infinity,
                                   numeric::isless(x[2], y[2]) = TRUE
                                   )
                          )
              end_proc
           );
  i:= 1;
  while i <= nops(l) do
    // it may happen that the first two entries have been removed, 
    // and i has become zero therefore
    if i=0 then
      i:= i+1;
      next
    end_if;
    if err <> infinity and numeric::isless(l[i][2], err) = FALSE then
      // skip this and all of the following terms
      return([op(l, 1..i-1)])
    end_if;
    // do l[i] and l[i+1] have the same exponent?
    if i < nops(l) and numeric::isnonzero(l[i][2] - l[i+1][2]) <> TRUE then
      l[i][1]:= l[i][1] + l[i+1][1];
      delete l[i+1];
      // if is(l[i][1] = 0, Goal = TRUE) then
      if iszero(l[i][1]) then
        delete l[i];
        i:= i-1
      end_if
    else
      if iszero(l[i][1]) then
        delete l[i]
      else
        i:= i+1
      end_if
    end_if
  end_while;
  l
end_proc:


// we define the ldegree of a power series to be the *minimal*
// exponent, or the exponent of the error term if the series is zero
limit::powerSeries::ldegree:=
proc(S)
begin
  if extop(S, 1) = [] then
    limit::powerSeries::errorterm(S)
  else
    extop(S, 1)[1][2]
  end_if
end_proc:

limit::powerSeries::lcoeff:=
proc(S)
begin
  if extop(S, 1) = [] then
    0
  else
    extop(S, 1)[1][1]
  end_if
end_proc:

limit::powerSeries::nterms:=
proc(S: limit::powerSeries)
begin
  nops(extop(S, 1))
end_proc:


limit::powerSeries::mapcoeffs:=
proc(S, f)
  local argv;
begin
  argv:= args(3..args(0));
  extsubsop(S, 1= map(extop(S, 1), pair -> [f(pair[1], argv), pair[2]]))
end_proc:

limit::powerSeries::multcoeffs:=
proc(S, a)
begin
  mapcoeffs(S, _mult, a)
end_proc:


limit::powerSeries::leaveOutLterm:=
proc(S)
  local l;
begin
  l:= extop(S, 1);
  assert(nops(l) > 0);
  delete l[1];
  extsubsop(S, 1=l)
end_proc:


// create a new power series for f(x) for x->0+  
// with error term O(x^order+e0) where e0 = ldegree of the series
// f - arithmetical expression
// x - series variable
// order - real number
// lnx - ln(x), expressed as a constant independent of x 


limit::powerSeries::new:=
proc(f, x, order, lnx)
  local l, create, normalize, result, c0, e0, n, k: DOM_INT,
  numberOfTerms, Phi, Phik: DOM_LIST, precision, ord, WW, ff, param, xarg;
begin
  assert(args(0) >= 2);

  userinfo(15, "Creating power series of ".expr2text(f).
           " at ".expr2text(x)." = 0");
  
  if args(0) <= 3 then
    lnx:= ln(x)
  end_if;
  
  create:=
  proc(l: DOM_LIST, errterm=infinity)
  begin
    new(limit::powerSeries, l, errterm, x)
  end_proc;

  normalize:= limit::powerSeries::normalize;
  
  if iszero(f) then
    return(create([]))
  elif not has(f, x) then
    if traperror((ff:= is(f=0, Goal = TRUE)), MaxSteps = 1) = 0 and ff then
      return(create([]))
    else
      return(create([[f, 0]]))
    end_if
  end_if;

  assert(args(0) >= 3);

  case type(f)
    of DOM_IDENT do
      assert(f=x);
      return(create([[1, 1]]))
    of "_plus" do

      ord:= float(order);
      k:= 0;
      while TRUE do
        // this loop can only be left by returning FAIL or by returning
        // a series when the
        // desired precision has been reached
        k:= k+1;
        l:= map([op(f)], limit::powerSeries::new, x, ord, lnx);
        if contains(l, FAIL) > 0 then
          return(FAIL)
        end_if;
        result:= _plus(op(l));
        precision:= limit::powerSeries::errorterm(result) - ldegree(result);
        if precision = undefined then // infinity - infinity -> exact 0
          return(result)
        end_if;
        if precision >= order then // precise enough
          return(result)
        end_if;
        if k=2 then
          // failed second attempt
           // is the sum identically zero?
          if protected(x) <> None then
            WW:= genident();
            ff:= subs(f, x = WW)
          else
            WW:= x;
            ff:= f
          end_if;
          case testeq(ff, 0) assuming WW in Dom::Interval(0, EPS)
            of TRUE do
              return(limit::powerSeries(0, x))
            of UNKNOWN do
              if traperror((ff:= (ff | WW=0))) <> 0 then
                // give up
                return(FAIL)
              end_if;
              if is(ff = 0, Goal = TRUE) then
                return(limit::powerSeries(0, x))
              end_if;
              // fall through
            of FALSE do 
              userinfo(10, "Increasing order of series expansion")
          end_case
        end_if;
        // increase required order in recursive calls.
        if nterms(result) > 0 then
          e0:= min(op(map(l, ldegree)));
          e0:=  float(ldegree(result) - e0);
          if e0 <= 0 then
            return(FAIL)
          end_if;
          ord:= e0 + ord
        else
          // result is O(...)
          // how much should we add ?!
          ord:= ord + 1
        end_if
      end_while;
      assert(FALSE);
      return(FAIL)
    of "_mult" do  
      l:= map([op(f)], limit::powerSeries::new, x, order, lnx);
      if contains(l, FAIL) > 0 then
        return(FAIL)
      end_if;
      return(_mult(op(l)))

    of "_power" do

      if type(op(f, 2)) = DOM_COMPLEX then
        return(FAIL)
      end_if;
        
      if not testtype(op(f, 2), Type::Constant) then
        // should not happen
        return(FAIL)
      end_if;

      if op(f, 1) = x then
        return(create([[1, op(f, 2)]]))
      end_if;
      if op(f, 1) = 1/x then
        return(create([[1, -op(f, 2)]]))
      end_if;
      if op(f, 2) = -1 then
        result:= limit::powerSeries::new(op(f, 1), x, order, lnx);
        if result = FAIL then
          return(FAIL)
        end_if;
        return(_invert(result, x, order))
      end_if;

      /*
       Other exponents, where the base has to expanded first 
       Let g = \sum c_i x^{e_i} .
       Then g^a = exp(a * ln(g))
                = exp(a*ln(c0*x^e0) + a*\sum_{k=1}^{\infty} (-1)^(k-1)/k Phi^k) 
                = c0^a*x^(a*e0) * exp(a*\sum_{k=1}^{\infty} (-1)^(k-1)/k Phi^k)
      
      */

      
      result:= limit::powerSeries::new(op(f, 1), x, order, lnx);
      if result = FAIL then
        return(FAIL)
      end_if;

      c0:= FAIL;
      while nterms(result) > 0 and
        is((c0:= lcoeff(result)) = 0, Goal =TRUE) do
        result:= limit::powerSeries::leaveOutLterm(result)
      end_while;


      if c0 <> FAIL and not is(c0 > 0, Goal = TRUE) and
        not is(op(f, 2) in Z_, Goal = TRUE) then
        // would give a complex lcoeff c0^op(f, 2)
        userinfo(20, "Possibly complex lcoeff");
        return(FAIL)
      end_if;
      
      e0:= ldegree(result);
      if e0 = infinity then // exact zero!
        return(result)
      end_if;
      
      Phi:= limit::powerSeries::Phi(result, x, order);
      if Phi = FAIL then
        return(FAIL)
      end_if;
  
      n:= ldegree(Phi);
      assert(n>0);
      numberOfTerms:= ceil(order/n);
      Phik:= [NIL $ numberOfTerms];

      // let Phik[k] = Phi^k
      if numberOfTerms>=1 then
        Phik[1]:= Phi;
        for k from 2 to numberOfTerms do
          Phik[k]:= limit::powerSeries::bin_mult(Phik[k-1], Phik[1]);
        end_for;
        result:= _plus(multcoeffs(Phik[k], (-1)^(k-1)/k) $k=1..numberOfTerms);
        result:= multcoeffs(result, op(f, 2));
        // Phi^(numberOfTerms) is the first summand missing
        result:= limit::powerSeries::plusO(result, n*(numberOfTerms+1))
      else
        result:= limit::powerSeries(0, x, order, lnx);
        result:= limit::powerSeries::plusO(result, order)
      end_if;

     
      result:= exp(result, x, order, lnx);
      return(extsubsop(result, 1= map(extop(result, 1),
                                      pair -> [pair[1]*c0^op(f, 2),
                                               pair[2] + op(f, 2)*e0]),
                       2 = extop(result, 2) + e0*op(f, 2)))
      

      /*

      Note. Reducing _power to exp and ln, i.e., 
      proceeding like this:
      
      result:= limit::powerSeries::new(ln(op(f, 1)) * op(f, 2), x, order, lnx);
      if result = FAIL then
        return(FAIL)
      else
        return(limit::powerSeries::exp(result, x, order, lnx))
      end_if;
      
      would *not* work as the MRV forbides transforming f into exp(ln(f))
      because the predefined value ln(x) := lnx would be used when computing
      ln(f)

      */

    otherwise
      if nops(f) <=2 and (l:= slot(limit::powerSeries, type(f))) <> FAIL then
        if nops(f) = 1 then
          xarg:= 1;
          param:= null()
        else
          assert(nops(f) = 2);
          case map([op(f)], has, x)
          of [TRUE, FALSE] do
            xarg:= 1;
            param:= op(f, 2);
            break
          of [FALSE, FALSE] do
            // this should not happen as we should have caught constants before
            assert(FALSE);
            return(FAIL);
          of [TRUE, TRUE] do
            // cannot handle this
            return(FAIL)
          of [FALSE, TRUE] do
            xarg:= 2;
            param:= op(f, 1);
            break
          end_case;
        end_if;  
        result:= limit::powerSeries::new(op(f, xarg), x, order, lnx);
        if result = FAIL then
          return(FAIL)
        else  
          if type(f) = "ln" and ldegree(result) = 0 and lcoeff(result) = 1
            then
            // we need to increase the precision
            result:= limit::powerSeries(op(f, xarg) - 1, x, order, lnx);
            if result = FAIL then
              // should not happen!
              assert(FALSE);
              return(FAIL)
            end_if;
            result:= result + limit::powerSeries(1, x)
          end_if;
          result:= l(result, x, order, lnx, xarg, param);
          return(result)
        end_if
      end_if
      
  end_case;

  FAIL
end_proc:


limit::powerSeries::_negate:= S -> multcoeffs(S, -1):

limit::powerSeries::_plus:=
proc()
  local l, err, x;
begin
  l:= [args()];
  // we cannot add power series and other things yet
  // in particular, no automatic conversion of expressions into
  // series is done
  assert(map({op(l)}, domtype) = {limit::powerSeries});
  x:= limit::powerSeries::variable(l[1]);
  err:= min(op(map(l, limit::powerSeries::errorterm)));
  // make one large list of all terms to add
  l:= map(l, S -> op(extop(S, 1)));
  return(new(limit::powerSeries, limit::powerSeries::normalize(l, err), err, x))
end_proc:

limit::powerSeries::_subtract:= (S, T) -> S + _negate(T):

// add S + O(x^alpha)
// this gives S if the error of S was x^alpha or larger; otherwise, it
// sets the error term of the result to x^alpha
limit::powerSeries::plusO:=
proc(S, alpha)
begin
  if extop(S, 2) = infinity or
    numeric::isless(alpha, extop(S, 2)) = TRUE then
    extsubsop(S,
              1 = select(extop(S, 1),
                         pair -> numeric::isless(pair[2], alpha) = TRUE),
              2 = alpha)
  else
    S
  end_if
end_proc:
  

limit::powerSeries::_mult:=
proc()
  local result, i;
begin
  result:= args(1);

  for i from 2 to args(0) do
    // multiply result by args(i)
    result:= limit::powerSeries::bin_mult(result, args(i))
  end_for;
  result
end_proc:


limit::powerSeries::bin_mult:=
proc(S1, S2)
  local l1, l2, l, err1, err2, err, x, i: DOM_INT, j: DOM_INT;
begin
  if S1::dom <> dom then
    if S2::dom <> dom then
      return(S1 * S2);
    end_if;
    // swap
    [S1, S2] := [S2, S1];
  end_if;
  if S2::dom <> dom then
    return(multcoeffs(S1, S2));
  end_if;
  
  x:= limit::powerSeries::variable(S1);
  if x <> limit::powerSeries::variable(S2) then
    error("Variables do not match")
  end_if;
  l1:= extop(S1, 1);
  l2:= extop(S2, 1);
  err1:= limit::powerSeries::errorterm(S1);
  err2:= limit::powerSeries::errorterm(S2);

  if l1 = [] or l2 = [] then // zero + errorterm
    return(new(limit::powerSeries, [], ldegree(S1) + ldegree(S2), x))
  end_if;

  // the error term of (c1*x^a1 + .. + x^err1)*(c2*x^a2 + ... + x^err2) is
  // min(a1 + err2, a2 + err1), as multiplying out shows
  // the following complicated code is faster than the simple
  // err:= min(ldegree(S1) + err2, ldegree(S2) + err1)
  if err1 = infinity then
    if err2 = infinity then
      err:= infinity
    else
      err:= ldegree(S1) + err2
    end_if
  else
    if err2 = infinity then
      err:= ldegree(S2) + err1
    else
      err:= min(ldegree(S1) + err2, ldegree(S2) + err1)
    end_if
  end_if;

  l:= [[l1[i][1] * l2[j][1], l1[i][2] + l2[j][2]] $i=1..nops(l1)
       $j=1..nops(l2)];

  l:= limit::powerSeries::normalize(l, err);
  
  new(limit::powerSeries, l, err, x)
end_proc:


limit::powerSeries::_invert:=
proc(S, x: DOM_IDENT, order): Type::Union(limit::powerSeries, DOM_FAIL)
  local c0, e0, Phi, n, k, l, numberOfTerms, Phik, result;
begin
  // if S = \sum c_i x^{e_i}, then
  // 1/S = 1/c0 x^(-e0) \sum_{k=0}^\infty (-1)^k Phi^k


  
  while nops(extop(S, 1)) > 0 and limit::zerotest((c0:= lcoeff(S))) do
    S:= limit::powerSeries::leaveOutLterm(S)
  end_while;

  if extop(S, 1) = [] then
    return(FAIL)
  end_if;

  
  // c0 should be nonzero now; nope for the best
  e0:= ldegree(S);
  Phi:= limit::powerSeries::Phi(S, x, order);
  if Phi = FAIL then
    return(FAIL)
  end_if;
  
  n:= ldegree(Phi);
  numberOfTerms:= ceil((order/n) + 1);
  Phik:= [NIL $ numberOfTerms];

  // let Phik[k] = (-1)^k Phi^k = (-Phi)^k
  if numberOfTerms>=1 then
    Phik[1]:= multcoeffs(Phi, -1);
    for k from 2 to numberOfTerms do
      Phik[k]:= limit::powerSeries::bin_mult(Phik[k-1], Phik[1]);
    end_for;
  end_if;

  // let result = \sum_{k=1}^... (-1)^k Phi^k
  if numberOfTerms >= 1 then
    result:= _plus(Phik[k] $k=1..numberOfTerms);
    // add the summand k=0, i.e., 1
    result:= result + limit::powerSeries(1, x, infinity)
  else
    result:= limit::powerSeries(1, x, infinity)
  end_if;
  // multiply by 1/c0 * x^(-e0)
  l:= map(extop(result, 1), pair -> [pair[1]/c0, pair[2] - e0]);
  result:= extsubsop(result, 1=l);
  // the first omitted term is of order Phi^(numberOfTerms), i.e.,
  // of order n*(numberOfTerms+1)
  limit::powerSeries::plusO(result, n*(numberOfTerms+1)-e0)
end_proc:
  

/*
  Slots used for overloading. If S is a power series, such a slot f is
  called in the form f(S, x, order, lnx)
  and may return either a power series or FAIL

*/

// fourth argument lnx is not needed/used
limit::powerSeries::exp:=
proc(S, x, order)
  local Sk, n, k: DOM_INT, numberOfTerms, c0;
begin

  while nops(extop(S, 1)) > 0 and limit::zerotest((c0:= lcoeff(S))) do
    S:= limit::powerSeries::leaveOutLterm(S)
  end_while;
  
  if iszero(limit::powerSeries::errorterm(S)) then
    assert(iszero(expr(S)));
    // exp(O(1)) = O(1)
    return(S)
  end_if;
    
  n:= ldegree(S);
  case sign(n)
    of -1 do
      if iszero(expr(S)) then
        // result is too inaccurate
        return(FAIL)
      end_if;
      // should not happen
      // warning("cannot compute exponential");
      return(FAIL)
    of 1 do
      // the result is \sum S^k/k!
      c0:= 0;
      break
    of 0 do
      // the result is exp(c0) * \sum_{k=0}^{\infty} psi(k)/k!
      // where psi = S - c0
      c0:= lcoeff(S);
      S:= limit::powerSeries::leaveOutLterm(S);
      // new ldegree
      n:= ldegree(S);
      assert(n<>0);
      break
    otherwise
      // warning("Cannot determine sign");
      return(FAIL)
  end_case;
  
  numberOfTerms:= ceil(order/n);
  Sk:= [NIL $numberOfTerms];

  if numberOfTerms >= 1 then
    Sk[1]:= S;
    for k from 2 to numberOfTerms do
      Sk[k]:= limit::powerSeries::bin_mult(S, Sk[k-1]);
      Sk[k]:= multcoeffs(Sk[k], 1/k);
    end_for;
  end_if;
  
  S:= _plus(limit::powerSeries(1, x, infinity), op(Sk));
  if not is(c0 = 0, Goal = TRUE) then
    c0:= exp(c0);
    if type(c0) = "exp" then
      c0:= exp::simplify(c0)
    end_if;
    S:= multcoeffs(S, c0)
  end_if;
  // the first omitted term is S^(numberOfTerms+1), i.e., (numberOfTerms+1)*n
  limit::powerSeries::plusO(S, n*(numberOfTerms+1))
end_proc:


/*
  Phi(S) is defined as (S / (c0*x^e0)) - 1
  where c0*x^e0 is the leading monomial of S
  It is, of course, essential that really c0 <> 0 here
*/
limit::powerSeries::Phi:=
proc(S, x)
  local c0, e0, l;
begin
  l:= extop(S, 1);
  if nops(l) = 0 then
    if extop(S, 2) = infinity then
      error("division by zero")
    else
      return(FAIL)
    end_if
  end_if;
  c0:= l[1][1];
  e0:= l[1][2];
  delete l[1];
  l:= map(l, pair -> [pair[1]/c0, pair[2] - e0]);
  extsubsop(S, 1=l, 2=extop(S, 2) - e0)
end_proc:
  

limit::powerSeries::ln:=
proc(S, x, order, lnx)
  local Phi, Phik, n, k, numberOfTerms, result;
begin
  while nterms(S) >= 1 and limit::zerotest(lcoeff(S)) do
    S:= limit::powerSeries::leaveOutLterm(S)
  end_while;
  
  Phi:= limit::powerSeries::Phi(S, x);
  if Phi = FAIL then
    return(FAIL)
  end_if;

  if limit::asymptoticSign(lcoeff(S), x) <> 1 then
    // cannot handle ln of negative numbers!
    return(FAIL)
  end_if;

  
  // the result is
  // ln(c0) + e0*ln(x) +  \sum_{k=1}^infinity (-1)^(k-1)*Phi^k/k
  if nops(extop(Phi, 1)) = 0 then
    // Phi = zero + error term
    result:= limit::powerSeries(ln(lcoeff(S)) + ldegree(S)*lnx,
                                    x);
    return(limit::powerSeries::plusO(result,
                                     limit::powerSeries::errorterm(Phi))
           )
  end_if;
  
  n:= ldegree(Phi);
  assert(n>0);
  numberOfTerms:= ceil(order/n);
  
  Phik:= [NIL $ numberOfTerms];

  if numberOfTerms >= 1 then
    Phik[1]:= Phi;
    for k from 2 to numberOfTerms do
      Phik[k]:= limit::powerSeries::bin_mult(Phik[k-1], Phi);
    end_for;
  end_if;
  
  // the first omitted term is of order Phi^(numberOfTerms), i.e.,
  // of order n*(numberOfTerms+1)
  result:= _plus(limit::powerSeries(ln(lcoeff(S)) + ldegree(S)*lnx,
                                    x, infinity),
                 multcoeffs(Phik[k], (-1)^(k-1)/k) $k=1..numberOfTerms);
  limit::powerSeries::plusO(result, n*(numberOfTerms+1))
end_proc:


/***
    extension to abs and sign:
    these are locally identical with exp-log functions

***/

// given a power series S for f(x), the power series of abs(f(x)) is
// S if the first nonzero coefficient of S is positive, and -S if the first
// nonzero coefficient of S is negative
limit::powerSeries::abs:=
proc(S, x, order, lnx)
  local l, i: DOM_INT;
begin
 
  
  // coeff list
  l:= extop(S, 1);
  
  while nops(l) > 1 and
    testeq(l[1][1], 0, Steps = 5, KernelSteps = 10) = TRUE do
    delete l[1]
  end_while;
    
  if nops(l) = 0 then
    if limit::powerSeries::errorterm(S) = infinity then
      return(limit::powerSeries(0, x))
    else  
      return(FAIL)
    end_if
  end_if;

  for i from 1 to nops(l) do
    l[i][1]:= sign(l[1][1])*l[i][1]
  end_for;

  extsubsop(S, 1=l)
end_proc:


// sign is locally constant
limit::powerSeries::sign:=
proc(S, x, order, lnx)
  local l: DOM_LIST;
begin
   // coeff list
  l:= extop(S, 1);
  
  while nops(l) > 1 and
    testeq(l[1][1], 0, Steps = 5, KernelSteps = 10) = TRUE do
    delete l[1]
  end_while;
    
  if nops(l) = 0 then
    if limit::powerSeries::errorterm(S) = infinity then
      return(limit::powerSeries(0, x))
    else  
      return(FAIL)
    end_if
  end_if;

  l:= [[sign(l[1][1]), 0]];

  extsubsop(S, 1=l, 2=infinity)

end_proc:



/********************************************
extension to "tractable" functions
*********************************************/

/*
   byTaylor(g, ispole, evalg)

   returns a function F(S, x, order, lnx) that computes
   the power series of g(f(x)) at x=0+ to the given order, given
   the power series of f

   returns FAIL if S is not a power series (ldegree < 0) or
   if f->c0 and g has a pole at c0; ispole is used to check this

   to evaluate g at a constant, evalg is used

   Cf. Gruntz, p. 76


*/
limit::powerSeries::byTaylor:=
proc(g, ispole = FALSE, evalg)
  option escape;
begin

  if args(0) <= 2 then
    evalg:= g
  end_if;
  
  // return value
  proc(S: limit::powerSeries, x: DOM_IDENT, order, lnx)
    name limit::powerSeries::byTaylor;
    local h, c0, result, n, k: DOM_INT, gk, gk0, hk, ldeg;
  begin
      
    // write S = c0 + h where ldegree(h) > 0
    case sign((n:= ldegree(S)))
      of -1 do
        // cannot expand at infinity
        return(FAIL)
      of 0 do
        if nterms(S) = 0 then
          // S = O(1)
          return(S)
        else  
          c0:= lcoeff(S);
          h:= limit::powerSeries::leaveOutLterm(S);
          n:= ldegree(h);
        end_if;
        break
      of 1 do
        c0:= 0;
        h:= S;
        break
      otherwise
        // should not happen
        // warning("Cannot determine sign of ldegree");
        return(FAIL)
    end_case;

    if ispole(c0) then
      return(FAIL)
    end_if;

    gk:= evalg(x);
    k:= 1;
    result:= limit::powerSeries(evalg(c0), x, order);
    ldeg:= ldegree(result);
    if ldeg = infinity then
      ldeg:= ldegree(h)
    end_if;
    if ldeg = infinity then
      ldeg = 0
    end_if;
    
    while n * k < order + ldeg do
      // h^k is still relevant for the result
      // let hk:= h^k
     
      if k=1 then
        hk:= h
      else
        hk:= limit::powerSeries::bin_mult(hk, h)
      end_if;

      // let gk be the k-th derivative of g
      gk:= diff(gk, x);
      // it may happen (e.g., in case of dilog) that the
      // derivative is defined at c0 but only via an expression of the form
      // 0/0
      if traperror((gk0:= (gk | x=c0))) <> 0 then
        gk0:= limit(gk, x=c0)
      end_if;
      result:= result + multcoeffs(hk, gk0/k!);
      
      k:= k+1
    end_while;

    limit::powerSeries::plusO(result, n*k)
  end_proc

end_proc:

// functions tractable at finite arguments:

limit::powerSeries::sin:= limit::powerSeries::byTaylor(sin):
limit::powerSeries::cos:= limit::powerSeries::byTaylor(cos):
limit::powerSeries::dilog:= limit::powerSeries::byTaylor(dilog):


// functions that are also tractable at infinity

limit::powerSeries::arctanFinite:= limit::powerSeries::byTaylor(arctan):

limit::powerSeries::arctan:=
proc(S, x, order, lnx)
  local invS, invS2, Sk, k, n, result, numberOfTerms;
begin

  case sign((n:= ldegree(S)))
    of 0 do
    of 1 do
      return(limit::powerSeries::arctanFinite(S, x, order, lnx))
    of -1 do
      // plug S into the power series of arctan at infinity
      // arctan(z) = PI/2 - 1/z + 1/3/z^3 - 1/5/z^5 + ...
      result:= limit::powerSeries(PI/2, x);
      if order >= -n then
        invS:= _invert(S, x, order, lnx);
        invS2:= invS*invS;

        numberOfTerms:= ceil(order/2/(-n));
        Sk:= [NIL $ numberOfTerms];
        Sk[1]:= invS;
        for k from 2 to numberOfTerms do
          Sk[k]:= Sk[k-1]*invS2
        end_for;

        result:= result + _plus(multcoeffs(Sk[k], (-1)^k/(2*k-1))
                                $k=1..numberOfTerms
                                );
        result:= limit::powerSeries::plusO(result, -n*(2*numberOfTerms)-1)
      else
        result:= limit::powerSeries::plusO(result, -n)
      end_if;
      // arctan(-x) = -arctan(x) such that we may reduce the case
      // S -> -infinity to the case S -> infinity
      Sk:= simplify(sign(lcoeff(S)));
      return(multcoeffs(result, Sk))
  end_case;
  
  // warning("Cannot determine sign of ldegree");
  FAIL
end_proc:


// functions with poles

limit::powerSeries::psiFinite:=
limit::powerSeries::byTaylor(X-> psi(X),
                             x -> not is(x>0 or not x in Z_, Goal = TRUE)):

limit::powerSeries::psi:=
proc(S: limit::powerSeries, x: DOM_IDENT, order, lnx)
  local n, lnS, result, numberOfTerms, Sk, k;
  
begin
  
  case sign((n:= ldegree(S)))
    of 0 do
      return(limit::powerSeries::psiFinite(S, x, order, lnx))
    of -1 do
      if not is(lcoeff(S) > 0, Goal = TRUE) then
        // psi(z) for z -> -infinity: cannot handle this 
        break
      end_if;

      // use Abramowitz/Stegun, formula 6.3.18
      // psi(z) = ln(z) - 1/2/z -
      //          \sum_{n=1}^{\infty} bernoulli(2*n)/(2*n*z^(2*n))

      lnS:= limit::powerSeries::ln(S, x, order, lnx);
      if lnS = FAIL then
        return(FAIL)
      end_if;

      // we do not need S anymore, but its inverse
      S:= _invert(S, x, order, lnx);
      if S = FAIL then
        return(FAIL)
      end_if;

      result:= lnS + multcoeffs(S, -1/2);
      numberOfTerms:= ceil(order/2/(-n) -1);
      
      if numberOfTerms >= 1 then
        Sk:= [NIL $ numberOfTerms];
        // let Sk = S^(2*k)
        Sk[1]:= S*S;
        for k from 2 to numberOfTerms do
          Sk[k]:= Sk[k-1]*Sk[1]
        end_for;

        result:= result- _plus(multcoeffs(Sk[k], bernoulli(2*k)/2/k)
                               $k=1..numberOfTerms);

        result:= limit::powerSeries::plusO(result, -n*(numberOfTerms+1))
        
      else
        result:= limit::powerSeries::plusO(result, order)
      end_if;

      return(result)


    of 1 do
      // psi(z) for z->0
      // combining Abramowitz/Stegun 6.3.5 and 6.3.14, we have
      // psi(z) = -1/z + psi(z+1)
      //        = -1/z - EULER + \sum_{n=2}^{\infty} (-1)^n zeta(n) * z^(n-1)

      result:= _invert(S, x, order, lnx);
      result:= multcoeffs(result, -1);
      result:= result + limit::powerSeries(-EULER, x);

      numberOfTerms:= ceil(order/n -1);
      if numberOfTerms >= 1 then
        Sk:= [NIL $ numberOfTerms];
        // let Sk = S^k
        Sk[1]:= S;
        for k from 2 to numberOfTerms do
          Sk[k]:= Sk[k-1]*Sk[1]
        end_for;
        result:= result+ _plus(multcoeffs(Sk[k], (-1)^(k+1)* zeta(k+1))
                               $k=1..numberOfTerms);
      else
        result:= limit::powerSeries::plusO(result, order)
      end_if;
      return(result)
  end_case;

  // should not happen ...
  FAIL
end_proc:



limit::zerotest:= 
proc(a)
  save DIGITS;
begin
  a:= limit::rewriteBack(a);
  DIGITS:= max(40, DIGITS);
  testeq(a, 0, 
    Steps = 0,                  // steps in Simplify
    Seconds = infinity,         // time limit in Simplify
    hold(NumberOfRandomRatTests) = 4, // insertions identifier <-> rational numbers
    NumberOfRandomTests = 15,    //  different insertions identifier <-> numbers
    IgnoreProperties = FALSE,    // relevant for random tests + Simplify
                                // (in the random tests, use only numbers consistent 
                                //  with the properties of the identifiers?)
    hold(ZeroTestMode) = TRUE
  )
end_proc:




/***************************************************************************
   Methods related to the MRV algorithm
See: Gaston H. Gonnet and Dominik Gruntz:         
"Limit Computation in Computer Algebra", Nov. 1992.          
See also the PhD thesis by Gruntz:
"Limit computation in a symbolic manipuilation system"
We refer to the latter just in the form 'Gruntz, pp. n'.

General remarks. 
The MRV algorithm deals with real-valued functions ex of a real
variable x tending to infinity. This is to be understood in the
following way: each subexpression of ex must be real for large enough
values of x. Note for example that ln causes problems here! Therefore,
every element of an MRV set must be positive for large x. This is the
case for the original exp-log algorithm (see lemma 3.14 (2) in Gruntz)
and must be observed for any extension. This is currently done by
limit::powerSeries which returns FAIL if it detects something non-real.


*****************************************************************************/


/*
   rewriting:
   1) express special functions in terms of others such that we need few
      attributes of limit::powerSeries
   2) express functions with essential singularities in terms of "tractable"
     functions. Cf. Gruntz, Section 5.2. To this end, we introduce
   auxiliary functions limit::erfs and limit::Eis below
*/

limit::rewriteExpLog :=
proc(ex, x: DOM_IDENT)
begin
  // handle binomial etc.
  ex:= rewrite(ex, gamma);
  subs(ex,
            [hold(_power) =
             proc(b, e)
             begin
               if not has({b, e}, x) or
                 contains({DOM_INT, DOM_RAT, DOM_FLOAT}, domtype(e))
                 then
                 b^e 
               elif iszero(b) then
                 1-abs(sign(e))
               else
                 exp(ln(b)*e)
               end_if
             end_proc,
             hold(sinh) =
             (X ->
             if not has(X, x) then
               hold(sinh)(X)
             else
               (exp(X)-exp(-X))/2
              end_if),
             hold(cosh) =
             (X ->
              if not has(X, x) then
                hold(cosh)(X)
              else
                (exp(X)+ exp(-X))/2
              end_if),
             hold(tanh) =
             (X ->
              if not has(X, x) then
                hold(tanh)(X)
              else
                (exp(X)-exp(-X))/
                (exp(X)+exp(-X))
              end_if
              ),
              hold(coth) =
             (X ->
              if not has(X, x) then
                hold(coth)(X)
              else
                (exp(X)+exp(-X))/
                (exp(X)-exp(-X))
              end_if
              ),
             hold(log) =
             ((b, X) ->
              if not has(X, x) then
                hold(log)(b, X)
              else
                ln(X)/ln(b)
              end_if
              ),
             hold(tan) =
             (X ->
              if not has(X, x) then
                hold(tan)(X)
              else
                sin(X)/cos(X)
              end_if
              ),
             hold(cot) =
             (X ->
              if not has(X, x) then
                hold(cot)(X)
              else
                cos(X)/sin(X)
              end_if
              ),
             hold(igamma) =
             proc(a, z)
             begin
             if not has(a, x) then
               hold(igamma)(a, z)
             else
               exp(lngamma(a)) - exp(a*ln(z)) * limit::igammas(a, z)  
             end_if  
             end_proc,
             hold(Ei)  =
             proc(n, a)
               local u;
               begin
                 if not has([args()], x) then
                   hold(Ei)(args())
                 elif args(0) = 1 then
                   limit::Eis(n)*exp(n)
                 else
                   assert(args(0) = 2);
                   u:= igamma(1 - n, a);
                   if type(u) = "igamma" then
                     u:= limit::rewrite(u, x)
                   end_if;  
                   a^(n - 1)*u
                 end_if
               end_proc,
               // Li(X) = Ei(ln(X))
               //       = Eis(ln(X)) * exp(ln(x))
              hold(Li) =
               proc(X)
               begin
                 if not has(X, x) then
                   hold(Li)(X)
                 else
                   limit::Eis(ln(X))*X
                 end_if
               end_proc,
               hold(erfc) = proc(X, n)
               begin
                  if not has([args()], x) then
                    hold(erfc)(args())
                  elif args(0) = 1 then
                    exp(-X^2)*limit::erfs(X) + 1 - sign(X)
                  elif domtype(n) = DOM_INT then // 2 argument version erfc(x, n)
                    hold(erfc)(args())   // ToDo: can be improved!
                  else // cannot do anything
                    hold(erfc)(args())
                  end_if
               end_proc,
               hold(erf) =
               (X ->
                if not has(X, x) then
                  hold(erf)(X)
                else
                  sign(X) - exp(-X^2)*limit::erfs(X)
                end_if
                ),
               hold(fact) =
               (X ->
                if not has(X, x) then
                  hold(fact)(X)
                else
                  exp(lngamma
                            (X + 1))
                end_if
                ),
               hold(gamma) =
               (X ->
                if not has(X, x) then
                  hold(gamma)(X)
                else
                  exp(lngamma(X))
                end_if
                )
               ], 
               EvalChanges
            )       
end_proc:


/*
   rewriteBack: take care that none of the auxiliary functions enters the final
   result
*/
limit::rewriteBack:=
proc(ex)
begin
  subs(ex,
            [hold(limit::Eis) = (X -> Ei(X)/exp(X)),
             hold(limit::erfs) = (x -> exp(x^2)*(sign(x)-erf(x))),
             hold(limit::igammas) = ((a, z) -> (igamma(a, z) - gamma(a))/z^a),
             hold(exp) =
             proc(X)
               begin
                 if type(X) = "lngamma" then
                   // exp(lngamma(x)) -> gamma(x)
                   gamma(op(X, 1))
                 else
                   hold(exp)(X)
                 end_if
             end_proc
             ], 
             EvalChanges
       )
end_proc:


/*
        limitMRV(ex,x)

        ex - expression
        x  - limit variable

        Calls lterm and analyzes the leading
        coefficient of the asymptotic expansion (in w), i.e.
        returns the limit of ex for x -> infinity, if possible.

        Return Values:
        l       - the limit (arithmetical expression).
        FAIL    - limit cannot be computed.
       
*/
limit::limitMRV:=
proc(ex, x: DOM_IDENT, options = limit::defaultOptions: DOM_TABLE)
  local lc, s;
begin
  userinfo(2,"MRV: compute limit of ",ex);
  if not has(ex, x) then
    return(ex);
  end_if;

  if type(ex) = piecewise then
    if has(piecewise::conditions(ex), FAIL) then
      return(FAIL)
    else
      return(piecewise::extmap(ex, limit::limitMRV, x))
    end_if
  end_if;

  ex:= limit::rewriteExpLog(ex, x);
  
  lc:= limit::lterm(ex, x, options);
  if lc = FAIL then
    return(FAIL)
  else
    // (d) Analyze the leading degree
    case sign(lc[3])
      of  0 do
        return( limit::rewriteBack(lc[1]) )
      of  1 do
        return( 0 )
      of -1 do
        s:= limit::asymptoticSign(lc[1],x);
        if s = FAIL or iszero(s) then
          // Happens if series returns zero as the leading coefficient
          // of the series expansion!
          return( FAIL )
        else
          return( s*infinity )
        end_if
      otherwise
        if not options[NoWarning] then
          warning("cannot determine sign of ".expr2text(lc[3]));
        end_if;
        return( FAIL )
    end_case
  end_if
end_proc:








/*
        lterm(ex, x <, S>)

        ex -- expression
        x  -- limit variable
        S  -- (optional) mrv set

m        Compute the list [c0,w,e0], where c0*w^e0 is the leading
        term of an asymptotic expansion of ex for x -> infinity.

        Return Values:
        [c0,w,e0] - c0*w^e0 as the leading term
        FAIL      - limit cannot be computed.
        
       
*/
limit::lterm:=
proc(ex, x: DOM_IDENT, options = limit::defaultOptions: DOM_TABLE)
  local S, s, e, lc, le, t, w, W, order;
    
begin
  if not has(ex, x) then
    return( [ex, 1, 0] )
  end_if;


// rewrite to exp-log form
  ex:= limit::rewriteExpLog(ex, x);
  userinfo(5, "Rewriting to exp-log form gives ".expr2text(ex));

  
  // Find the most rapidly varying subexpressions
  
  S:= limit::getMRV(ex, x, options);
  if S = FAIL then
    userinfo(3,"Determining MRV set of ".expr2text(ex)." fails");
    return(FAIL)
  elif nops(S) = 0 then
    return([ex, 1, 0])
  end_if;

  userinfo(5, "MRV set is ".expr2text(S));

  if contains(S, x) then // Upward Movement (Gruntz p. 50)
    userinfo(5, "Substituting ".expr2text(x)." by ".expr2text(exp(x)));
    e:= limit::lterm(subs(ex, [ln(x) = x, x = exp(x)], EvalChanges),
                             x, options);
    if domtype(e) <> DOM_LIST then
      // lterm fails:
      userinfo(5, "Could not determine lterm");
      return(FAIL)
    else
      // moving down
      return(subs(e,[exp(x)=x, x=ln(x)], EvalChanges))
    end_if
  end_if;


  //  Choose one expression in this set and call it w. Rewrite the
  //  other expressions in that set as g(x)=w^p h(x) and consider all
  //  expressions  independent of w as constants.
  //  See Gruntz, p.43
  

  if has(ex, `#W`) then
    W:= genident("W")
  else
    W:= `#W`
  end_if;
  s:= limit::substitutions(S, x, W);
  if s = FAIL then
    return(FAIL)
  end_if;

  w:= subs(W, map(s, x -> op(x,2) = op(x,1)));
  assert(type(w) = "exp");

  userinfo(5, "Let W:= ".expr2text(w));
  
  le:= op(w);
  t:= limit::asymptoticSign(le, x);
  if t = 1 then
    // we have that le -> infinity and w -> infinity but want that
    // le -> -infinity and thus w -> 0 
    s:= subs(s, W = 1/W);
    w:= 1/w;
    le:= -le
  elif t <> -1 then // FAIL, I, and the like ..
    return(FAIL)
  end_if;

  userinfo(5,"Rewriting the MRV set gives ".expr2text(s));


  // Combine products of powers, as for example w*w^(-c) must be rewritten
  // to w^(1-c), because w^(1-c) = exp((1-c)*ln(w)) = exp((1-c)*le)!!
  // Otherwise the result would only be w*exp(-c*le).
  // An example for that this step is necessary:
  //   e3:=x -> exp(exp(exp(x))): limit(e3(n+exp(-n))/e3(n),n=infinity) = infinity
  
  ex:= subs(ex, op(s), EvalChanges);
  userinfo(5, "Substituting the expression gives ".expr2text(ex));
  userinfo(5, "where ".expr2text(W)." goes to zero from the right");
  

  // Compute the leading term of the Puiseux series in w around w = 0+
  // See Gruntz, p.48
  
  userinfo(5, "Computing series at zero from the right");
  
  if not has(ex, W) then

    lc:= ex; le:= 0;
  else
      
    // Series computation:
    order:= 0.01;
    repeat
      s:= limit::powerSeries(ex, W, order, le) assuming x > 2;
      order:= order + 1
    until order > 2 or s<> FAIL and not iszero(expr(s)) end_repeat;

  
    if s=FAIL then
      userinfo(5, "Series computation fails");
      return(FAIL)
    end_if;
  
    if iszero(expr(s)) then
      if limit::powerSeries::errorterm(s) = infinity then
        // ex equals zero
        userinfo(5, "Expression is zero");
        return([0, 1, 0])
      else
        userinfo(5, "leading term of series has too high order")
      end_if;
      return(FAIL)
    end_if;

    le:= ldegree(s);
    lc:= lcoeff(s);

  end_if;
    
       
  if type(w) = "exp" then
    s:= op(w);
    if type(s) = "_mult" and type((t:= op(s, nops(s)))) = DOM_INT then
      w:= exp(s/abs(t));
      le:= le*abs(t)
    end_if;
  end_if;
    


  userinfo(4,"MRV: lcoeff is ",lc, "; ldegree is ",le);
  
  s:= sign(le);
  if domtype(s) = DOM_COMPLEX then
    // we cannot handle this within the MRV algorithm
    return(FAIL)
  end_if;
  if domtype(s) <> DOM_INT then
    if is(le=0, Goal = TRUE) then
      s:= 0
    else      
      userinfo(5, "MRV cannot handle leading coefficient ".expr2text(le));
      return(FAIL)
    end_if
  end_if;
  
  if s <> 0 then
    return([lc,w,le])
  elif not has(lc, x) then
    return([lc, 1, 0])
  else   
    return( limit::lterm(lc, x, options) )
  end_if
end_proc:

/*
        getMRV(f,x)

        f -- expression
        x -- limit variable

        Find the most rapidly varying subexpressions at x=infinity
        (see section 3.1).

        Even with the algorithm extended, the return value
        must only contain exp-log functions g with
        limit(sign(g), x=infinity) = 1!


        Return Values:
        S       - set of MRV subexpressions (possibly empty).
        FAIL    - limit cannot be computed.
       
*/
limit::getMRV:=
proc(f, x: DOM_IDENT, options = limit::defaultOptions: DOM_TABLE):
  Type::Union(DOM_FAIL, DOM_SET)
  local s, t, g, i: DOM_INT;
  save MAXEFFORT;
begin

  if MAXEFFORT < 10 then
    userinfo(10, "MAXEFFORT too small");
    return(FAIL)
  end_if;


  // exp-log-functions (Gruntz PhD, p.40) 
  
  if not has(f, x) then
    if is(f in R_, Goal = TRUE) then
      return( {} )
    else
      return(FAIL)
    end_if
  end_if;
    
  
  case type(f)
    of DOM_IDENT do
      assert(f = x);
      return( {x} )
    of "_mult" do
    of "_plus" do
      s:= {};
      
      i:= nops(select({op(f)}, hastype, "exp"));
      if i>1 then
        MAXEFFORT:= MAXEFFORT/i
      end_if;
      
      for i from 1 to nops(f) do
        t:= limit::getMRV(op(f,i), x, options);
        if t = FAIL then
          return(t)
        end_if;
        s:= limit::maxMRV(s, t, x);
        if s = FAIL then
          return(FAIL)
        end_if
      end_for;
      return(s)
    of "_power" do
      if not has(op(f,2),x) then
        return( limit::getMRV(op(f,1),x, options) )
      elif iszero(op(f,1)) then
        // This should not happen. However, 0^g(x) is not rapidly varying.
        return({})
      else
        return(limit::getMRV(exp(ln(op(f,1))*op(f,2)),x, options) )
      end_if
        
    of "ln" do
      return(limit::getMRV(op(f,1), x, options))

    of "exp" do
      // if op(f,1) tends to +/- infinity, then
      // return maxMRV({exp(op(f, 1))}, MRV(op(f, 1)))
      // else return(mrv(op(f, 1)))  
      if f = exp(x) or f = exp(-x) then
        return({f})
      end_if;
        
      t:= limit::getMRV(op(f), x, options);
      if t = FAIL then
        return(FAIL)
      end_if;
      s:= limit::lterm(op(f), x, options);
      if domtype(s) <> DOM_LIST then
        // lterm fails:
        return( s )
      elif sign(s[3]) = -1 then
        g:= limit::compare(f, op(t,1), x);
        case g
          of ">" do
            return({f})
          of "<" do
            return(t)
          of "=" do
            return({f} union t)
          otherwise
            // compare fails
            return(FAIL)
        end_case
      else
        return(t)
      end_if;

      // extension to so-called tractable functions
      // A function g is tractable at y0 if it has a generalized
      // power series expansion at y0. We can then compute the limit of
      // g(f(x)) at x0 provided f(x) goes to y0 as x goes to x0.
      // See Gruntz, pp. 77

    
      
    of "sin"  do
    of "cos"  do
      // won't give a limit if op(f, 1) goes to infinity
    of "sign" do
    of "abs"  do
    of "dilog" do
    of "lngamma" do  
    of "arctan" do
      return( limit::getMRV(op(f,1), x, options) )

    of "psi" do
      if nops(f) = 1 then
        return( limit::getMRV(op(f,1), x, options) )
      else
        return(FAIL)
      end_if  
      
   /* not yet handled :
    of "arcsin" do
    of "arccos" do
   
    of "arcsinh" do
    of "arccosh" do
  
    of "arctanh" do
    of "Si" do   
       
     */ 
  
   
    /*
     // two-argument functions not yet handled by the MRV:
      of "polylog" do
      of "besselJ" do
      return( limit::getMRV(op(f,2),x, options) )
    */

      // -------------------------------------------------------
      // functions that have been created by the pre-processing step
    of "erfs"   do
    of "Eis" do    
      return( limit::getMRV(op(f,1), x, options) )
    of "igammas" do
      if has(op(f, 2), x) then
        // not implemented
        return(FAIL)
      else
        return( limit::getMRV(op(f,1), x, options) )  
      end_if; 
    of "erfc" do
    of "Ei" do
      // may occur as igamma sometimes evaluates to Ei
      assert(nops(f) = 2);
      return(FAIL)
      // these should not show up due to rewriting:
    of "Li" do  
    of "tan"  do
    of "cot"  do
    of "sinh" do
    of "cosh" do  
    of "tanh" do
    of "gamma" do
    of "erf" do  
      assert(FALSE);
      break
  end_case;

  FAIL 
end_proc:

/*
        substitutions(S,x, W)

        S -- mrv set of f in respect to x
        x -- limit variable
        W -- new variable

        Returns [w,S,l] where w represents W and l is
        the list of elements of S rewritten in W.
        The i.th element of l is the i.th element of S rewritten
        in W (i=1..|S|).

        Return Values:
        [w,S,l] - see above.
        FAIL    - limit cannot be computed.
        
*/
limit::substitutions:=
proc(S: DOM_SET, x: DOM_IDENT, W: DOM_IDENT): Type::Union(DOM_LIST, DOM_FAIL)
  local g, i, c, cc, f, A, S0, subspat;
begin

  // very frequent special case that must be dealt with
  // very quickly
  if nops(S) = 1 then
    return([op(S, 1) = W])
  end_if;
  
  g:= op(S,1);
  for i from 2 to nops(S) do
    // it is essential that the *largest* element of the MRV set
    // is taken as a new variable. Cf. Bug 1713.

    f:= op(S, i);
    if length(f) < length(g) then
      g:= f
    end_if
    
  end_for;


  
  S0:= prog::sort([op(S)],
                  a -> nops(limit::getMRV(a, x)),
                  Reverse
                  );

  subspat:= null();
  assume(x >= 20);
  for f in S0 do
  
    c:= limit(op(f)/op(g), x = infinity);
    if c = undefined then 
      return(FAIL)    
    end_if;

    A:= exp(op(f) - c*op(g));
    // Be sure not to generate terms of the form a^b 
    if testtype(c, Type::Constant) then
      cc:= numeric::rationalize(float(c));
      if domtype(cc) = DOM_INT then
        subspat:= subspat, f = A*W^cc
      else
        subspat:= subspat, f = A*W^c
      end_if;
    else
      subspat:= subspat, f = A*W^c
    end_if
      
  end_for;
  [subspat]
end_proc:

/*
        compare(f,g,x)

        determines R with gamma(f) R gamma(g)
        where R is one of the relations "<", ">", "=".
        Uses Gruntz, Lemma 3.6 (p.34).

        f, g -- expressions in x, representing functions with
                limit(sign(f), x = infinity) = limit(sign(g),
                x=infinity) = 1
        x -- limit variable

        Returns either "<", ">", or "=".
*/
limit::compare:=
proc(f, g, x)
  local lnf, lng;
begin
  if type(f) = "exp" then
    lnf:= op(f)
  else
    lnf:= ln(f)
  end_if;
  if type(g) = "exp" then
    lng:= op(g)
  else
    lng:= ln(g)
  end_if;


  // gseries::dominates refers to limits of x-> +0
  property::subsIdent(x, 1/x); 
//  assume(_and(op(subs(showprop(x), x=1/x))));
  lnf:= evalAt(lnf, x = 1/x);
  lng:= evalAt(lng, x = 1/x);
  
  
  case Series::gseries::dominates(lnf, lng, x)
    of 1 do
      return(">")
    of 0 do
      return("=")
    of -1 do
      return("<")
  end_case;
  
  
  userinfo(1, "cannot compare Mrv sets");
  FAIL
end_proc:

/*
        maxMRV(s,x)

        s -- set containing sets of rapidly varying terms in x
        x -- limit variable

        Find the more rapidly varying set (see section 3.2).

        Return Values:
        S       - the more rapidly varying set.
        FAIL    - limit cannot be computed.
       
*/
limit::maxMRV:=
proc(s1: DOM_SET, s2: DOM_SET, x): Type::Union(DOM_SET, DOM_FAIL)
  local f, g, c;
begin
  if s1 = {} then
    return(s2)
  elif s2 = {} then
    return(s1)
  elif s1 intersect s2 <> {} then
    return(s1 union s2)
  elif contains(s1, x) then
    return(s2)
  elif contains(s2, x) then
    return(s1)
  else
    f:= op(s1,1); g:= op(s2,1);
    
    c:= limit::compare(f, g, x);
    if c = ">" then
      return(s1)
    elif c = "<" then
      return(s2)
    elif c = "=" then
      return(s1 union s2)
    else
      assert(c = FAIL);
      return(FAIL)
    end_if
  end_if
end_proc:

/*
        asymptoticSign(f,x)

        f - arithmetical expression
        x - identifier

        Compute the limit of sign(ex) when x goes to infinity.
        Returns FAIL if this cannot be determined.
*/
limit::asymptoticSign:=
proc(ex, x)
  local s, t, S;
begin

  userinfo(4,"MRV: compute the sign of ",ex);
  if ex = x then
    return( 1 )
  end_if;

  if not has(ex, x) then
    s:= sign(ex);
    if not has(s,sign) then
      return( s )
    else
      return( FAIL )
    end_if
  end_if;

  t:= eval(op(ex,0));

  if (t:= t::asymptoticSign) <> FAIL then
    return(t(ex, x))
  end_if;
    
  case type(ex)
    of "_mult"  do
      S:= 1;
      for t in ex do
        s:= limit::asymptoticSign(t, x);
        if s = FAIL then
          return( FAIL )
        else
          S:= S * s
        end_if
      end_for;
      return( S )
    of "_power" do
      if domtype(op(ex,2)) = DOM_INT and modp(op(ex,2),2) = 0 then
        return( 1 )
      elif op(ex,1) = x then
        return( 1 )
      elif limit::asymptoticSign(op(ex,1), x) = 1 then
        return( 1 )
      else
        t:= limit::lterm(ex, x);
        if t = FAIL then
          return( FAIL )
        else
          return( limit::asymptoticSign( t[1], x ) )
        end_if
      end_if
    of "exp" do
      return( 1 )
    otherwise
      t:= limit::lterm(ex, x);
      if t = FAIL then
        return( FAIL )
      else
        return( limit::asymptoticSign(t[1], x ))
      end_if
  end_case
end_proc:






/*******************************************************************************/

/*
Auxiliary special functions needed for limit only
Cf. Gruntz, Section 5.2, p.79

These are created by limit::rewriteExpLog, and later rewritten back
by limit::rewriteBack

We currently have:
Eis(x) = Ei(x)/exp(x)
erfs(x) = exp(x^2)*(sign(x)-erf(x))

igammas(a, z) = (igamma(a, z) - gamma(a))/z^a  (only used if a depends on x)

*/

/*
  Eis(x) := Ei(x) / exp(x)
*/

limit::Eis:= funcenv( proc() begin procname(args()) end ):

limit::Eis::print:= "Eis":
limit::Eis::type:= "Eis":

limit::powerSeries::EisFinite:=
limit::powerSeries::byTaylor
               (limit::Eis, FALSE,
                proc(X)
                  local result;
                begin
                  result:= Ei(X)/exp(X);
                  if hastype(result, "Ei") then
                    limit::Eis(X)
                  else
                    result
                  end_if
                end_proc
                ):


limit::powerSeries::Eis:=
proc(S: limit::powerSeries, x: DOM_IDENT, order, lnx)
  local n, result, lnS,
  numberOfTerms: DOM_INT,
  Sk: DOM_LIST,
  k: DOM_INT;
  
begin
  while nterms(S) >= 1 and limit::zerotest(lcoeff(S)) do
    S:= limit::powerSeries::leaveOutLterm(S)
  end_while;

  case sign((n:= ldegree(S)))
    of 0 do
      // finite nonzero value, use taylor expansion
      return(limit::powerSeries::EisFinite(S, x, order, lnx))
    of 1 do
      // Ei(x)/ exp(x) where x->0
      // we have that Ei(x) = EULER + ln(x) + \sum_{n=1} x^n/n/n!
      // we just multiply this by the series for exp(-x)
      result:= limit::powerSeries(EULER, x);
      lnS:=  limit::powerSeries::ln(S, x, order, lnx);
      // ln may fail
      if lnS = FAIL then
        return(FAIL)
      end_if;
      result:= result + lnS;
      
      
      
      numberOfTerms:= ceil(order/n);

      if numberOfTerms >= 1 then
        Sk:= [NIL $numberOfTerms];
        Sk[1]:= S;
        for k from 2 to numberOfTerms do
          Sk[k]:= Sk[k-1]*S
        end_for;
        result:= result+ _plus(multcoeffs(Sk[k], 1/k/k!) $k=1..numberOfTerms)
      end_if;

      result:= limit::powerSeries::plusO(result, n*(numberOfTerms+1));
      return(result* limit::powerSeries::exp(-S, x, order, lnx))

    of -1 do
      // Ei(x)/exp(x) = \sum_{n=1} (n-1)!/x^n for x-> +/- infinity
      S:= _invert(S, x, order, lnx);
      numberOfTerms:= ceil(order/(-n)) + 1;
      Sk:= [NIL $numberOfTerms];
      Sk[1]:= S;
      for k from 2 to numberOfTerms do
        Sk[k]:= Sk[k-1]*S
      end_for;
      result:= _plus(multcoeffs(Sk[k], (k-1)!) $k=1..numberOfTerms);
      result:= limit::powerSeries::plusO(result, (-n)*(numberOfTerms+1));
      return(result)
      
  end_case;
      
  FAIL
end_proc:

limit::Eis::diff:=
proc(f: "Eis", x)
begin
  case args(0)
    of 1 do
      return( f )
    of 2 do
      f:= op(f);
      if has(f, x) then
        // diff(Ei(x) / exp(x), x) = 1/x - Ei(x)/exp(x) = 1/x - Eis(x)
        // chain rule:
        (1/f - limit::Eis(f))*diff(f, x)
      else
        0
      end_if;
      break
    otherwise
      return( diff( diff(f,x),args(3..args(0)) ) )
  end_case
end_proc:


// as exp(x) is positive, sign(Eis(x)) = sign(Ei(x)) = sign(x) 
limit::Eis::asymptoticSign:=
proc(f,x)
begin
  limit::asymptoticSign(op(f, 1), x)
end_proc:


//------------------------------------------
//        The function limit::erfs(x)
//------------------------------------------

/*
  erfs(x) := exp(x^2)*(sign(x) - 1 + erfc(x)) = exp(x^2)*(sign(x)-erf(x)) 
  at positive x, this becomes
  erfs(x) = exp(x^2) * erfc(x) = exp(x^2) * (1-erf(x))

*/

limit::erfs:= funcenv( proc() begin procname(args()) end ):

limit::erfs::print:= "erfs":


limit::powerSeries::erfs:=
proc(S: limit::powerSeries, x: DOM_IDENT, order, lnx)
  local Sk, n, numberOfTerms, k, result;
begin
  while nterms(S) >= 1 and limit::zerotest(lcoeff(S)) do
    S:= limit::powerSeries::leaveOutLterm(S)
  end_while;

  case sign((n:=ldegree(S)))
    of -1 do
      /* as z -> infinity
         erfs(z) = 2/sqrt(PI)/(2*z) *
                   \sum_{m=0}^{\infty} (-1)^m (2*m)!/m!/(2*z)^(2*m)
         (Abramowitz/Stegun, formula 7.2.14 with n=0)
      */
      // compute 2*S
      S:= multcoeffs(S, 2);
      // and apply the above formula to 2*z = S

      result:= limit::powerSeries::_invert(S, x, order, lnx);
      if result = FAIL then
        return(FAIL)
      end_if;
      numberOfTerms:= ceil(order/2/(-n));
      
      if numberOfTerms >= 1 then
         // for all k, let Sk[k] = 1/S^(2*k)
        Sk:= [NIL $numberOfTerms];
        Sk[1]:= result*result;
        for k from 2 to numberOfTerms do
          Sk[k]:= Sk[k-1]*Sk[1]
        end_for;
        result:= result +
                 _plus(multcoeffs(Sk[k], (-1)^k *(2*k)!/k!)
                       $k=1..numberOfTerms);
     
      end_if;
      
      result:= limit::powerSeries::plusO(result, 2*(numberOfTerms+1)*(-n));
      result:= multcoeffs(result, 2/sqrt(PI));
      return(result);
      
    of 0 do
    of 1 do  
      // we have to evaluate exp(x^2) * erfc(x) at a finite point
      // this is no problem and can be done by Taylor expansion
      return(limit::powerSeries::erfsFinite(S, x, order, lnx))
  end_case;
  
  FAIL
end_proc:

limit::powerSeries::erfsFinite:=
limit::powerSeries::byTaylor
               (limit::erfs, FALSE,
                proc(X)
                  local result;
                begin
                  if X = 0 then return(1) end_if;
                  result:= exp(X^2)*(erfc(X) + 1 - sign(X));
                  if hastype(result, {"erf", "erfc"}) then
                    limit::erfs(X)
                  else
                    result
                  end_if
                end_proc
                ):



limit::erfs::type:= "erfs":

limit::erfs::diff :=
    proc(f,x)
    begin
        case args(0)
        of 1 do
            return( f )
        of 2 do
            f:= op(f);
            if has(f,x) then
                return( (-2/sqrt(PI)+2*limit::erfs(f)*f)*diff(f,x) )
            else
                return( 0 )
            end_if
        otherwise
            return( diff( diff(f,x),args(3..args(0)) ) )
        end_case
    end_proc:



limit::erfs::asymptoticSign:=
proc(f,x)
  local s;
begin
  s:= limit::limitMRV(f,x);
  if contains( {FAIL},s ) then
    return( FAIL )
  else
    s:= sign(s);
    if s = 0 or s = 1 then return( 1 ) else return( FAIL ) end_if
  end_if
end_proc:


// new power series attribute for new specfunc lngamma
// replaces the old limit::gammas


/* given the power series for f(x) at x=0+, compute the
   power series of ln(gamma(f(x))) at x=0+

 We have to distinguish the following cases.
 (a) f(x) -> a finite positive number.

 (b) f(x) -> 0


 (c) f(x) -> infinity


 (d) f(x) -> something negative or -infinity
*/

limit::powerSeries::lngamma:=
proc(S: limit::powerSeries, x: DOM_IDENT, order, lnx)
  local n, k, lnS, Sk, numberOfTerms, result, `1/S^2`;
begin
  while nterms(S) >= 1 and limit::zerotest(lcoeff(S)) do
    S:= limit::powerSeries::leaveOutLterm(S)
  end_while;

  if nterms(S) = 0 then
    // cannot compute ln(gamma(0))
    return(FAIL)
  end_if;
  
  case sign((n:=ldegree(S)))
    of 1 do
      // f(x) -> 0
      // use Abramowitz/Stegun, formula 6.1.33
      // ln(gamma(z+1)) = -ln(z+1) + z*(1-EULER)
      //  + \sum_{n=2}^{\infty} (-1)^n * (zeta(n)-1)* z^n/n  
      // thus, using gamma(z+1) = z*gamma(z), we get
      // ln(gamma(z)) = ln(gamma(z+1)) - ln(z)
      // with z->0, this gives
      // ln(gamma(z)) = -ln(z) - z + \sum_{n=2} (-1)^n* z^n/n +
      // z*(1-EULER) + \sum_{n=2}^{\infty} (-1)^n * (zeta(n)-1)* z^n/n
      // = -ln(z) - EULER*z + \sum_{n=2} (-1)^n *zeta(n) *z^n/n

      lnS:= limit::powerSeries::ln(S, x, order, lnx);
      if lnS = FAIL then
        return(FAIL)
      end_if;
      result:= multcoeffs(lnS, -1); // result := -ln(S)
      result:= result + multcoeffs(S, -EULER);
      
             
      numberOfTerms:= ceil(order/n -1);
      Sk:= [NIL $ numberOfTerms];

      // let Sk[k] = S^k
      if numberOfTerms>=2 then
        Sk[1]:= S;
        for k from 2 to numberOfTerms do
          Sk[k]:= limit::powerSeries::bin_mult(Sk[k-1], Sk[1]);
        end_for;
        result:= result+
                 _plus(multcoeffs(Sk[k], zeta(k)*(-1)^k/k) $k=2..numberOfTerms);
      else
        result:= limit::powerSeries::plusO(result, order)
      end_if;

      return(result)

    of 0 do
      if is(lcoeff(S) > 0, Goal = TRUE) then
        // this is simply a taylor series
        return(limit::powerSeries::lngammaPositive(S, x, order, lnx))
      end_if;

      // else:
      // what to do for ln(gamma(f(x))) if f(x) -> something negative?
      // at the moment, we just give up
      
      break
    of -1 do
      // ln(gamma(f(x))) where f(x) -> infinity or -infinity
      if not is(lcoeff(S) > 0, Goal = TRUE) then
        // f(x) -> -infinity: cannot handle this 
        break
      end_if;

      // use Abramowitz/Stegun, formula 6.1.40
      // lngamma(z) = (z-1/2)*ln(z) - z + 1/2*ln(2*PI) +
      // \sum_{m=1} bernoulli(2*m)/2/m/(2*m-1)/z^(2*m-1)
      lnS:= limit::powerSeries::ln(S, x, order, lnx);
      if lnS = FAIL then
        return(FAIL)
      end_if;
      result:= (S+limit::powerSeries(-1/2, x)) * lnS;
      result:= result - S;
      result:= result + limit::powerSeries(ln(2*PI)/2, x);
      numberOfTerms:= ceil(order/2/(-n) -1);
      if numberOfTerms>=1 then
        Sk:= [NIL $ numberOfTerms];
        Sk[1]:= _invert(S, x, order, lnx);
        `1/S^2`:= Sk[1]*Sk[1];
        for k from 2 to numberOfTerms do
          Sk[k]:= Sk[k-1]*`1/S^2` // let Sk[k] := 1/S^(2*k-1)
        end_for;
        if numberOfTerms >= 2 then
          result:= result+
                   _plus(multcoeffs(Sk[k], bernoulli(2*k)/2/k/(2*k-1))
                         $k=2..numberOfTerms)
        end_if;
        result:= limit::powerSeries::plusO(result, (2*numberOfTerms+1)*(-n))
      else
        result:= limit::powerSeries::plusO(result, order)
      end_if;

      return(result)
        
  end_case;

  FAIL
end_proc:

limit::powerSeries::lngammaPositive:=
limit::powerSeries::byTaylor(lngamma, FALSE):


/**************************************
The function limit::igammas(a, z)

is defined by igamma(a, z) = gamma(a) - z^a*igammas(a, z)
(Abramowitz/Stegun, p.260, formula 6.5.3, \gamma(a, z) = z^a*igammas(a, z) in our notation)

***************************************/

limit::igammas:= funcenv( proc() begin procname(args()) end ):

limit::igammas::print:= "igammas":

limit::igammas::type:= "igammas":

limit::powerSeries::igammas:=
proc(S: limit::powerSeries, x: DOM_IDENT, order, lnx, xarg, z)
  local invS, Sk, n, numberOfTerms, k, m, u, result;
begin
  while nterms(S) >= 1 and limit::zerotest(lcoeff(S)) do
    S:= limit::powerSeries::leaveOutLterm(S)
  end_while;

  // we use igammas only if the first argument depends on x and the other does not
  
  if xarg = 2 then
    return(FAIL)
  end_if;  
  assert(xarg = 1);
  
  
  
  case sign((n:=ldegree(S)))
  of -1 do
      // we do not know what to do if a -> -infinity //
      if not is(lcoeff(S) > 0, Goal = TRUE) then
        return(FAIL)
      end_if;  
        
    
      /* as a -> infinity
         igammas(a, z) = \sum_{m=0}^{\infty} (-1)^m z^m/m!/(a+m)
         (Abramowitz/Stegun, formula 6.5.33)
         Writing 1/(a+m) as a series 
         1/(a+m) = \sum_{k=1}^{\infty} (1/a)^k * (-m)^(k-1), 
         we may substitute a by its series representation S.
         Thus 
         igammas(a, z) = \sum_{m=0}^{\infty} (-1)^m z^m/m! * \sum_{k=1}^{\infty} (1/S)^k * (-m)^(k-1)
                       = \sum_{k=1}^{\infty} (1/S)^k \sum_{m=0}^{\infty} (-m)^(k-1) * (-1)^m * z^m / m!
                       = (1/S)* exp(-z) + (1/S)^2 * z*exp(-z) + ....

         To understand the k-th coefficient of this sum, let us convolute it with exp(z):
         coeff(.., k) * exp(z) = \sum_{m=0}^{\infty} (-m)^(k-1) * (-1)^m * z^m / m! * sum_{t=0}^{\infty} z^t / t!
                               = \sum_{u=0}^{\infty} z^u * \sum_{m=0}^u (-m)^(k-1) * (-1)^m / m! / (u-m)!

         It turns out that the inner sum is zero for u=0 (unless k=1) and u>k, such that the outerv sum runs from u=0 to k only.

      */
      
      
      result:= invS:= limit::powerSeries::_invert(S, x, order, lnx);
      if invS = FAIL then
        return(FAIL)
      end_if;
      numberOfTerms:= ceil(order/(-n));
      result:= multcoeffs(result, exp(-z));
      // now result = (1/S)*exp(-z)
      
      if numberOfTerms >= 1 then
         // for all k, let Sk[k] = 1/S^k
        Sk:= [NIL $numberOfTerms];
        Sk[1]:= invS;
        for k from 2 to numberOfTerms do
          Sk[k]:= Sk[k-1]*Sk[1]
        end_for;
        result:= result +
                 _plus(multcoeffs(Sk[k], 1/exp(z) * _plus(z^u * _plus((-m)^(k-1) * (-1)^m / m! / (u-m)! $m=1..u) $u=1..k))
                       $k=1..numberOfTerms);
     
      end_if;
      
      result:= limit::powerSeries::plusO(result, numberOfTerms+1);
      return(result);
      
    of 0 do
      // this is no problem and can be done by Taylor expansion
      return(limit::powerSeries::igammasFinite(z)(S, x, order, lnx))
     
    of 1 do 
      // this power series does not exist, as igammas has a pole here
      
  end_case;
  
  FAIL
end_proc:

limit::powerSeries::igammasFinite:=
proc(z)
  option escape;
begin  
  limit::powerSeries::byTaylor
               (limit::igammas, FALSE,
                proc(a)
                  local result;
                begin
                  result:= (igamma(a, z) - gamma(a))/z^a;
                  if hastype(result, {"igamma", "gamma"}) then
                    limit::igammas(a, z)
                  else
                    result
                  end_if
                end_proc
                ):
end_proc:                





limit::igammas::diff :=
    proc(f,x)
      local a, z;
    begin
        case args(0)
        of 1 do
            return( f )
        of 2 do
            [a, z]:= [op(f)];
            if has(f,x) then
                diff((igamma(a, z) - gamma(a))/z^a, x)
            else
                return( 0 )
            end_if
        otherwise
            return( diff( diff(f,x),args(3..args(0)) ) )
        end_case
    end_proc:


    

limit::igammas::asymptoticSign:=
proc(f,x)
  local s;
begin
  s:= limit::limitMRV(f,x);
  if contains( {FAIL},s ) then
    return( FAIL )
  else
    s:= sign(s);
    if s = 0 or s = 1 then return( 1 ) else return( FAIL ) end_if
  end_if
end_proc:


// end of file


