/*++
normal -- normal form of a rational expression

normal(x)

x - expression

The normal form of x is numer(x)/denom(x), where the numerator and
denominator are expanded polynomial expressions with gcd 1. 
++*/

normal:=
proc(x)
  local t, result, oldnum, oldden, num, den,
  substitutions,
  options: DOM_TABLE,
  options2: DOM_TABLE;
  
begin
  if args(0) = 0 then
    error("normal called without args")
  end_if;

  // overloading
  t:= x::dom;
  if t::normal <> FAIL then
    return(t::normal(args()))
  end_if;

  if args(0) = 2 and type(args(2)) = DOM_TABLE then
    options:= args(2)
  else            
    options:= prog::getOptions(2,                     // start scanning for options with the 2nd argument
                               [args()],              // all arguments of the call
                               table(                 // table of default values
                                 normal::defaultOptions,
                                 "Squarefree" = FALSE // additional option for rnormal
                               ),
                               TRUE,                  // react to unexpected arguments with an error?
                               table(
                                 normal::optiontypes, // table of expected option types
                                 "Squarefree" = DOM_BOOL
                               )
                              )[1];
    // option ToCancel implies NoGcd
    if options[ToCancel] <> FAIL then
      options[NoGcd]:= TRUE
    end_if
  end_if;
  
  case type(x)
    of DOM_FLOAT do
      if options[List] then
        return([x, 1.0])
      else
        return(x)
      end_if
    of DOM_COMPLEX do
    of DOM_INT do
    of DOM_RAT do
    of DOM_IDENT do
      if options[List] then
        return([x, 1])
      else
        return(x)
      end_if
    of "_index" do
    of "_concat" do
      if options[List] then
        return([map(x, normal), 1])
      else
        return(map(x, normal))
      end_if
  end_case;

  if options[List] and contains({DOM_LIST, DOM_SET, DOM_ARRAY, DOM_HFARRAY,
                              DOM_TABLE, DOM_POLY, "_equal", "_unequal", 
                              "_less", "_leequal", "_range", "_in", 
                              "_union", "_intersect", "_minus"},
                             type(x)) then
    error("Option List is not allowed for input of this type")
  end_if;

  if t = DOM_HFARRAY then
     return(x);
  end_if:

  if contains({DOM_LIST, DOM_SET, DOM_ARRAY, DOM_TABLE}, t) then
    return(map(x, normal, options))
  end_if;

  if t = DOM_POLY then
    return(mapcoeffs(x, normal, options));
  elif contains({"_equal", "_unequal", "_less", "_leequal",
                 "_range", "_in",
                 "_union", "_intersect", "_minus"}, type(x)) then
    return(map(x, normal, options))
  elif not testtype(x, Type::Arithmetical) then
    error("expecting an arithmetical expression")
  end_if;

  // After type checking, we are ready to call the normal version 
  // that makes use of factorized input, if requested by Expand = FALSE:
  if not options[Expand] then
     return(stdlib::normalNoExpand(x, options));
  end_if;

  if options["Use_normalNoExpand"] then
     // this is the compatibility mode between Expand = FALSE and
     // Expand = TRUE: Do the normalization with Expand = FALSE for
     // cheaper gcd computations. However, post-process with the
     // classical mode (Expand = TRUE) with the option NoGcd to
     // produce the same results as in the classical way.
     
     options2:= options:
     options2[Expand]:= FALSE;
     options2[List]:= TRUE;
     [num, den]:= normal(x, options2);

     if options[Expand] then
       [num, substitutions]:= [rationalize(num)]:
       num:= subs(expand(num, ArithmeticOnly), substitutions, EvalChanges);
       [den, substitutions]:= [rationalize(den)]:
       den:= subs(expand(den, ArithmeticOnly), substitutions, EvalChanges);
     end_if;

     if options[List] then
        return([num, den]);
     else
        return(num/den);
     end_if;

/*
     // Post process in the classical mode with NoGcd:
     options[Expand]:= TRUE;
     options["Use_normalNoExpand"]:= FALSE;
     options[NoGcd]:= TRUE;
//   options[NoGcd]:= FALSE;
     return(normal(num/den, options));
*/
  end_if;

  // With Expand = TRUE, we do the 'classical' normal that uses poly.
  // First, remove all floats and approximate them by rationals

  if stdlib::hasfloat(x) then
    return(float(normal(numeric::rationalize(x), options)))
  end_if;

  result:= stdlib::normal(x, options);

  // cancel all elements of ToCancel
  if options[ToCancel] <> FAIL then
    oldnum:= result[1];
    oldden:= result[2];
    for t in options[ToCancel] do
      while traperror((den:=divide(oldden, t, Exact))) = 0 and den <> FAIL and
         traperror((num:=divide(oldnum, t, Exact))) = 0 and num <> FAIL do
        // t was succesfully cancelled both from numerator and denominator
        oldnum:= num;
        oldden:= den
      end_while
    end_for;
    result:= [oldnum, oldden]
  end_if;
  
  if options[List] then
    result
  else
    result[1] / result[2]
  end_if
end_proc:

normal:= funcenv(normal,
                 NIL,
                 table("print" = "normal",
                       "info" = "normal(x) -- normalizes expression x",
                       "type" = "normal")
                 ):


normal::rationalizeOptions:= rationalize::defaultOptions:
normal::rationalizeOptions[FindRelations]:= ["_power", "exp"]:
normal::rationalizeOptions[ReplaceHardToEval]:= TRUE:

normal::rationalize:= x -> rationalize(x,  normal::rationalizeOptions):
normal::rationalizeNone:= x -> (x, {}, {}):

normal::optiontypes:= table(Expand = DOM_BOOL,
                            Iterations = Type::NonNegInt,
                            List = DOM_BOOL,
                            NoGcd = DOM_BOOL,
                            Rationalize = Type::AnyType,
                            Recursive = DOM_BOOL,
                            ToCancel = Type::Union(DOM_SET, DOM_FAIL)
                            ):

normal::defaultOptions:=
table(
      Expand = TRUE,
      Iterations = 5,
      List = FALSE,
      NoGcd = FALSE,
      Rationalize = normal::rationalize,
      Recursive = TRUE,
      "Use_normalNoExpand" = FALSE,
      ToCancel = FAIL
      ):



/*--
stdlib::normal -- normal form of a rational expression

stdlib::normal(x)

x - expression

The normal form of x is numer(x)/denom(x), where the numerator and
denominator are expanded polynomial expressions. stdlib::normal
returns the list [numer(x), denom(x)].
--*/

stdlib::normal:=
proc(x, options = normal::defaultOptions: DOM_TABLE)
  local ex2ind, r, n, d, _f, fn, fd, i, rnormal, normal_newind, tryd,
  substitutions, oldsubstitutions, oldresults, minpolys: DOM_SET,
  counter: DOM_INT,
  gcdf, ratf;
begin

  if options[NoGcd] then
    gcdf:= gcdlib::greatestCommonMonomial
  else
    gcdf:= gcd
  end_if;

  // option List is handled on top level
  options[List]:= FALSE;
  
  ratf:= options[Rationalize];
  
  if type(ratf) = DOM_IDENT then
    // some procedures have default names
    case ratf
    of None do
      ratf:= normal::rationalizeNone; break
    otherwise
      error("Unknown method in option Rationalize")
    end_case
  end_if;
  
/*--
normal_newind -- create new ident for non-rational expression

normal_newind(x)

x - non-rational expression

Returns an ident which replaces x. The new ident and x are inserted
into ex2ind !
--*/

  normal_newind:=
  proc(x)
    local y;
  begin
    if contains(ex2ind, x) then
      ex2ind[x]
    else
      y:= genident("_normal_");
      ex2ind[x]:= y;
      y
    end_if
  end_proc:

/*--
rnormal -- normal form of a rational expression

rnormal(x)

x   - expression

rnormal returns a list [n,d,u,s] where n is the numerator,
d the denominator of x, u the set of unknowns of u and d and s is
a list of substitutions for non-rational subexpressions.
--*/

  rnormal:=
  proc(x)
    local y, d, g, u, ind, S, xx;
    name recursive_normal;
  begin
    // floats should have been removed before
    assert(type(x) <> DOM_FLOAT);
    case type(x)
      of DOM_IDENT do
        return([ x, 1, {x}, {} ]);

      of DOM_INT do
        return([ x, 1, {}, {} ]);

      of DOM_RAT do
        return([ op(x,1), op(x,2), {}, {} ]);

      of "_plus" do
        d:= nops(x);
        xx:= x;
        g:= d div 2;
        y:= rnormal(_plus(op(x, 1..g)));
        x:= rnormal(_plus(op(x, (g+1)..d)));
        u:= x[3] union y[3];
        S:= x[4] union y[4];

        if nops(x[3] intersect y[3]) = 0 then
          return([ x[1]*y[2] + y[1]*x[2], x[2]*y[2], u, S ])
        end_if;

        if domtype(x[2]) = DOM_INT then
          if domtype(y[2]) = DOM_INT then
            return([ x[1]*y[2] + y[1]*x[2], x[2]*y[2], u, S ])
          end_if
        end_if;

        ind:= [ op(u) ];
        x[1]:= poly(x[1], ind);
        x[2]:= poly(x[2], ind);
        y[1]:= poly(y[1], ind);
        y[2]:= poly(y[2], ind);

        if contains({x[1], x[2], y[1], y[2]}, FAIL) then
          u:= normal_newind(xx);
          return([ u, 1, {u}, {u=xx}])
        end_if;


        g:= gcdf(x[2], y[2]);
        if iszero(expr(g)-1) then
          // no cancellation possible
          return([ expr(x[1]*y[2] + y[1]*x[2]), expr(x[2]*y[2]), u, S ])
        end_if;
        d:= divide(x[2], g, Exact);
        x:= x[1] * divide(y[2], g, Exact) + y[1] * d;
        y:= y[2] * d;
        g:= gcdf(x, y);
        d:= expr(divide(y, g, Exact));
        assert(not iszero(d));
        return([ expr(divide(x, g, Exact)), d, u, S ]);

      of "_mult" do
        d:= nops(x);
        xx:= x;
        g:= d div 2;
        y:= rnormal(_mult(op(x, 1..g)));
        x:= rnormal(_mult(op(x, (g+1)..d)));
        u:= x[3] union y[3];
        S:= x[4] union y[4];

        if nops(x[3] intersect y[3]) = 0 then
          return([ x[1]*y[1], x[2]*y[2], u, S ])
        end_if;

        if domtype(x[2]) = DOM_INT then
          if domtype(y[2]) = DOM_INT then
            return([ x[1]*y[1], x[2]*y[2], u, S ])
          end_if
        end_if;

        ind:= [ op(u) ];

        if traperror((
        x[1]:= poly(x[1], ind);
        x[2]:= poly(x[2], ind);
        y[1]:= poly(y[1], ind);
        y[2]:= poly(y[2], ind);
                      ),
                     MaxSteps = 4) <> 0
          or contains({x[1], x[2], y[1], y[2]}, FAIL) then
          u:= normal_newind(xx);
          return([ u, 1, {u}, {u=xx}])
        end_if;

        g:= gcdf(x[1], y[2]);
        d:= gcdf(y[1], x[2]);
        return([ expr(divide(x[1], g, Exact) * divide(y[1], d, Exact)),
                expr(divide(x[2], d, Exact) * divide(y[2], g, Exact)), u, S ]);

      of "_power" do
        d:= op(x,2);

        // there should be no more floating point numbers now
        assert(domtype(d) <> DOM_FLOAT);


        // special case

        if op(x,1) = -1 and type(op(x,2)) = DOM_RAT then
          x:= simplify(x);
          if type(x) <> "_power" then
            return(rnormal(x))
          end_if
        end_if;

        // should we treat our subexpression as base^d or as 1/ (base^(-d)) ?
        // We do the latter if d is "negative" in the sense of stdlib::hasmsign

        if stdlib::hasmsign(d) then
          g:= rnormal(op(x,1)^(-d));
          if iszero(g[1]) then
            userinfo(1, "Zero denominator detected in normal");
            // raise a kernel error 260:
            1/0.0
          end_if;
          return([ g[2], g[1], g[3], g[4] ])
        end_if;

        case type(d)
          of DOM_INT do
            assert(d > 0);
            g:= rnormal(op(x,1));
            return([ g[1]^d, g[2]^d, g[3], g[4] ])
          of DOM_RAT do
            assert(d > 0);
            break;
          of "_plus" do
          of "_mult" do
            d:= expand(x, MaxExponent = 255, ArithmeticOnly);
            if d <> x then
              return(rnormal(d))
            end_if;
        end_case;
        break;

      of DOM_COMPLEX do
        u:= normal_newind(I);
        x:= rnormal(op(x,1) + u*op(x,2));
        return([ x[1], x[2], x[3], x[4] union {u=I} ]);

      of "_and" do
      of "_or" do
      of "_not" do
      of "_minus" do
      of "_intersect" do
      of "_union" do
        error("not an arithmetical expression");

      otherwise
        if x::dom::normal <> FAIL then
          x:= x::dom::normal(x);
          u:= normal_newind(x);
          return([ u, 1, {u}, {u=x} ]);
        end_if;
    end_case;

    if domtype(x) = DOM_EXPR and options[Recursive] then
      d:= op(x,0);
      // avoid applying normal to options!
      // also avoid normalizing x $ m in diff(f(x), x$m)
      y:= map(x, u -> if type(u) = "_equal" or type(u) = "_seqgen" then u else normal(u, options) end);
      if y <> x and d <> hold(_concat) then
        x:= eval(y)
      end_if;
      if op(x, 0) <> d then
        return(rnormal(x))
      end_if;
    end_if;

    u:= normal_newind(x);
    [ u, 1, {u}, {u=x} ]
  end_proc:


    ////////////
    // m a i n  p r o g r a m   o f   s t d l i b :: n o r m a l
    ///////////


 
  oldsubstitutions := {};
  oldresults:= {};


  // if the following fails due to counter = 5, this indicates an endless loop
  // in rationalize and should be investigated
  
  counter:= 0;
  repeat
    counter:= counter + 1;
    oldresults:= oldresults union {x};
    substitutions:= ratf(x);
    if nops(substitutions) < 2 or nops(substitutions) > 4 or not testtype(op(substitutions, 2), Type::SetOf("_equal")) then
      error("The function used as value of the option Rationalize has wrong return type")
    end_if;
    if nops(substitutions) = 3 then
       minpolys:= op(substitutions, 3)
    else 
       minpolys:= {}
    end_if;
    _f:= op(substitutions,1);
    substitutions:= op(substitutions, 2);
    // it may happen that the right hand side contains newly generated
    // variables, which would prevent termination
    substitutions:= map(substitutions,
                        X -> (op(X, 1) = subs(op(X, 2), op(substitutions)))
                        );
    r:= map(substitutions, op, 2); 
    if contains(map(r, type), stdlib::Infinity) then
      // do not normalize infinities
      return([x, 1])
    end_if;
    if r <> {} and
      r <> oldsubstitutions and
      domtype(_f) <> DOM_IDENT then
      // new irrational subexpressions have occurred
      oldsubstitutions:= r;
      _f := normal(_f, options);
      if nops(minpolys) > 0 then
        _f := property::normalGroebner(_f, [op(minpolys)])
      end_if;
      x:= subs(_f, substitutions)    
    end_if;

    /*
    if counter = 5 then
      warning("Possibly endless loop")
    end_if;
    */
  until contains(oldresults, x) or counter = options[Iterations] end_repeat;

    
  ex2ind:= table();

    // call to rnormal. Passing an additional parameter ex2ind
    // is unnecessary because of the new scoping rules
   r:= rnormal(x);
  delete rnormal, normal_newind;


  if nops(r[3]) = 0 then
    // no irrational subexpressions had to be substituted
    assert(nops(r[4]) = 0);
    return([r[1], r[2]])
  end_if;


  _f:= [ op(r[3]) ];
  if traperror( ( n:= poly(r[1], _f) ), MaxSteps = 2 ) <> 0 or
     traperror( ( d:= poly(r[2], _f) ), MaxSteps = 2 ) <> 0 then
    // this may happen for large exponents
    return(subs([r[1], r[2]], r[4]))
  end_if;
  assert(domtype(n) = DOM_POLY and domtype(d) = DOM_POLY);


  if iszero(n) then
    return([ 0, 1 ])
  end_if;

  // Nenner reell machen

  if contains(map(r[4], op, 2), I) then
  // make denominator real
    i:= op(select(r[4], proc() begin op(args(1),2) = I end_proc ), [1,1]);
    if degree(d, i) > 0 then
      fd:= evalp(d, i=-I);
      tryd:= evalp(d, i=I) * fd;
      // tryd will be the new denominator
      // however, we have to check whether it is zero
      if not testeq(subs(expr(tryd), r[4]), 0, hold(ZeroTestMode)) then
        n:= evalp(n, i=I) * fd;
        if iszero(n) then return([ 0, 1 ]) end_if;
        n:= poly(mapcoeffs(n, subs, I=i), _f);
        d:= tryd;
        d:= poly(mapcoeffs(d, subs, I=i), _f);
        _f:= gcdf(n, d);
        n:= divide(n, _f, Exact);
        d:= divide(d, _f, Exact);
      end_if
    end_if;
  end_if;



// make polynomials primitive
  fn:= icontent(n);
  fd:= icontent(d);
  if lcoeff(d) < 0 then fn:= -fn; fd:= -fd end_if;
  _f:= fn/fd;
  if domtype(_f) = DOM_INT then
    n:= expr(multcoeffs(n, _f/fn));
    d:= expr(multcoeffs(d, 1/fd));
  else
    n:= expr(multcoeffs(n, op(_f,1)/fn));
    d:= expr(multcoeffs(d, op(_f,2)/fd));
  end_if;

  assert(d <> 0);

    // un-freeze non-rational subexpressions
  if nops(r[4]) = 0 then
    [ n, d ]
  else
    _f:= [op(r[4])];
    n:= subs(n, _f);
    d:= subs(d, _f);
    // emergency break if d=0
    if iszero(d) then
      return([args(1), 1])
    end_if;
    r:= n/d;
    [ n, d ]
  end_if // nops(r[4]) = 0
end_proc:


stdlib::normal:= prog::remember(stdlib::normal, property::depends):
