//   

// piecewise - the domain of piecewise defined functions


// an object of type piecewise has one operand:

// 1 - a list of branches


// the value of the piecewise defined function is the first expression
// the condition of which is satisfied
// the expression may be any object; however, the user is responsible
// not to use any operations that do not make sense







// branch - the domain of branches
// each branch has two or more operands
// 1. a condition
// 2...n  an expression (flattening of expression sequences
//                       cannot be prevented, hence n>=3 may happen)





alias(BRANCH=stdlib::branch):

BRANCH:=newDomain("stdlib::branch"):
BRANCH::create_dom:=hold(stdlib::branch):
BRANCH::info:="branch - the domain of branches":


BRANCH::new:=
proc(cond)
  option hold;
  local expression, badOtherwise;
begin
  if args(0)=0 then
    return(new(BRANCH, FALSE, undefined))
  end_if;
  
  cond:= context(cond);

  if type(cond) = "_exprseq" then
    return(context(hold(BRANCH::new)(op(cond), context(args(2..args(0))))))
  end_if;
  
  if args(0) >= 2 then
    expression:= args(2..args(0));
    case cond
      of TRUE do
        return(new(BRANCH, TRUE, context(expression)))
      of FALSE do
        return(new(BRANCH, FALSE, undefined))
      of Otherwise do
        return(new(BRANCH, Otherwise, context(expression)))
      otherwise
        if has(cond, Otherwise) then
          badOtherwise := FALSE;
          misc::maprec(cond,
                       proc(X)
                       begin
                         bool( type(X)=piecewise or X=Otherwise);
                       end_proc =
                       proc(X)
                       begin
                         if X=Otherwise
                           then
                           badOtherwise := TRUE;
                           X;
                         end_if;
                       end_proc, PostMap, Unsimplified, NoOperators);
          if badOtherwise then
            error("Condition contains Otherwise in subexpression");
          end_if;
        end_if;
        expression:= op(piecewise::evalAssuming([context(expression)], cond));
        return(new(BRANCH, cond, expression))
    end_case;
  end_if;


  if args(0)<>1 then
    error("Wrong number of arguments")
  end_if;




  
  cond := BRANCH::convert(cond);
  if cond=FAIL then
    error("Cannot convert to type branch")
  else
    cond
  end_if
end_proc:

BRANCH::create:=
proc(x: DOM_LIST)
begin
  new(BRANCH, op(x))
end_proc:


BRANCH::convert:=
proc(x)
begin
  if args(0)=0 then
    return(new(BRANCH, FALSE, undefined))
  end_if;

  case type(x)
    of BRANCH do
      return(x)
    of piecewise do
      if nops(x)<>1 then
        return(FAIL)
      else
        return(extop(x,1))
      end_if
    of DOM_LIST do
      if nops(x)<2 then
        return(FAIL)
      else
        return(BRANCH::new
               (piecewise::simplifyCondition(op(x, 1)), op(x,2..nops(x)))
               )
      end_if;
    otherwise
      FAIL
  end_case
end_proc:


BRANCH::condition:= br -> extop(br, 1);

BRANCH::expression:= br -> extop(br, 2..extnops(br));

BRANCH::setCondition:= (br, cond) ->
                       BRANCH::new(cond, extop(br, 2..extnops(br)));

BRANCH::setExpression:=
proc(br)
begin
  // the truth of the condition need not be checked again
  new(BRANCH, extop(br, 1), args(2..args(0)))
end_proc;

BRANCH::map := (br) -> BRANCH::new(op(map([BRANCH::condition(br), BRANCH::expression(br)], args(2..args(0))))):

BRANCH::apply:=
proc(br)
  local errlevel, result;
begin

  errlevel:=traperror(
                      (result:=BRANCH::new(extop(br,1),
                                   args(2)(extop(br,2), args(3..args(0)))))
                      );


  case errlevel
    of 0 do
      return(result)
    of 1010 do
      // throw the error
      lasterror()
    otherwise
      userinfo(1, "Leaving out branch ".expr2text(br).
               " which produced an error");
      return(new(BRANCH, FALSE, undefined))
  end_case
end_proc:

BRANCH::applyToCondition:=
proc(br: BRANCH, f)
  local cond, newcond;
begin
  cond:= extop(br,1);
  newcond:= f(cond, args(3..args(0)));
  if cond = newcond then
    br
  else  
    BRANCH::new(piecewise::simplifyCondition(newcond),
                BRANCH::expression(br))
  end_if
end_proc:

/*
  applies cond = f(cond) until no more change happens

*/
BRANCH::applyToConditionRepeatedly:=
proc(br: BRANCH, f)
  local cond, newcond;
begin
  newcond:= extop(br,1);
  repeat 
    cond:= newcond;
    newcond:= f(cond, args(3..args(0)));
    if cond <> newcond then
      newcond:= piecewise::simplifyCondition(newcond)
    end_if;  
  until newcond = cond end_repeat;  
  if cond = extop(br, 1) then
    br
  else  
    BRANCH::new(cond, BRANCH::expression(br))
  end_if
end_proc:

BRANCH::has:= (br, obj) -> has([extop(br)], [obj]):

BRANCH::hastype:=
proc(br: BRANCH)
begin
  hastype(extop(br, 1), args(2..args(0))) or
  hastype(extop(br, 2), args(2..args(0)))
  or bool(args(2) = BRANCH
  or _lazy_and(type(args(2)) = DOM_SET, contains(args(2), BRANCH)))
end_proc:

//------------------------------------------------
// float method added by Walter, 24.2.03:
// - The float method needs to map float both to
//   the expressions as well as to the conditions
// - Note that float can have a second argument
//   (the precision). For speed, do not bother
//   with type checking
BRANCH::float:=
proc(br)
begin
  new(BRANCH, op(map([extop(br)], float, args(2..args(0))))):
end_proc:
//------------------------------------------------

BRANCH::avoidAliasProblem:=
proc(br)
begin
  new(BRANCH, op(map([extop(br)],
                     solvelib::avoidAliasProblem,
                     args(2..args(0)))))
end_proc:

BRANCH::print:=
proc(br)
  local L1, L2;
begin
  if PRETTYPRINT = FALSE then
    [extop(br)]
  else
    L1 := expr2text(generate::sortSums(BRANCH::expression(br)));
    L2 := expr2text(generate::sortSums(BRANCH::condition(br)));
    if L2 = "Otherwise" then
      L1." else"
    else
      L1."  if ".L2."\n"
    end_if
  end_if
end_proc:

BRANCH::enableMaprec := TRUE:

piecewise:=newDomain("piecewise"):
piecewise::create_dom:=hold(piecewise):

piecewise::info:= "piecewise -- the domain of piecewise defined functions":

piecewise::interface:={
                       hold(branch),
                       hold(condition),
                       hold(disregardPoints),
                       hold(expression),
                       hold(extmap),
                       hold(insert),
                       hold(mapConditions),
                       hold(numberOfBranches),
                       hold(remove),
                       hold(restrict),
                       hold(selectConditions),
                       hold(selectExpressions),
                       hold(setCondition),
                       hold(setExpression),
                       hold(solveConditions),
                       hold(splitBranch),
                       hold(splitConditions)
                       }:


piecewise::new:=
proc()
  option hold;
  local argv: DOM_LIST, 
  i, j, k, result, cond, newbranch, br, argument, expression,
  removedConditions,
  beenthere: DOM_TABLE, otherwiseBranch, badOtherwise;
  save MAXEFFORT;

begin
  if args(0) = 0 then
    return(undefined)
  end_if;

  argv:= [args()];
  
  
  if args(0)<>1
  or type(context(hold(level)(args(1), 0))) = DOM_LIST
    then
    // convert each argument to branch
    otherwiseBranch := undefined;
    removedConditions:= FALSE;
    result:=[]; // list of all branches created so far
    beenthere:= table(); // table of all expressions obtained so far
                         // beenthere[object]=i means that object is in the
                         // i-th branch
    MAXEFFORT:= MAXEFFORT/args(0);
    i:= 1;
    while i <= nops(argv) do 
      br:= context(hold(val)(argv[i]));
      if type(br) = DOM_LIST then
        if nops(br) = 0 then
          error("Missing condition in ".output::ordinal(i)." branch")
        end_if;
        cond:= context((argv[i][1]));
        case (cond:=piecewise::simplifyCondition(cond))
          of TRUE do
            // we are sure that we may invest all of our effort here
            MAXEFFORT:= MAXEFFORT*(args(0) + 1 - i);
            return(context(argv[i][2]))
          of FALSE do
            break
          of Otherwise do
            if otherwiseBranch = undefined then 
              otherwiseBranch := context(argv[i][2]);
            else
              error("Otherwise may occur in at most one branch")
            end_if;
            break
          otherwise
            expression:= op(piecewise::evalAssuming([context(argv[i][2])], cond));
            if expression = undefined then
              removedConditions:= removedConditions or cond;
              break
            end_if;
            if testargs() then
              if has( [expression], Otherwise ) then
                badOtherwise := FALSE;
                misc::maprec( [expression],
                             proc(X)
                             begin
                               bool( type(X)=piecewise or X=Otherwise);
                             end_proc =
                             proc(X)
                             begin
                               if X=Otherwise then
                                 badOtherwise := TRUE;
                                 X;
                               end_if;
                             end_proc, PostMap, Unsimplified, NoOperators );
                if badOtherwise then
                  error( "Expression containing Otherwise" );
                end_if;
              end_if;
             
             end_if;
                         
              newbranch:= BRANCH::new(cond, expression);
              newbranch:= [piecewise::flattenBranch(newbranch)];
              for br in newbranch do
                expression:= BRANCH::expression(br);
                if type((j:= beenthere[expression])) = "_index" then
                  // we have never met that object
                  result:=result.[br];
                  beenthere[expression] := nops(result)
                else
                  assert(type(j) = DOM_INT);
                  // we have already a branch with that object
                  // hence we just have to combine the conditions with _or
                  // unless one of them is Otherwise, which is absorbing
                  if BRANCH::condition(br) = Otherwise then
                    result[j] := BRANCH::setCondition(result[j], Otherwise);
                  elif BRANCH::condition(result[j]) <> Otherwise then
                    result[j]:= BRANCH::applyToCondition(result[j], _or,
                                                        BRANCH::condition(br));
                  end_if;
                  if op(result[j], 1)=TRUE then
                    return(BRANCH::expression(result[j]))
                  end_if;
                  if op(result[j], 1)=FALSE then
                    delete beenthere[expression]
                  end_if
                end_if;
              end_for;
         end_case;
         i:= i+1   
      elif type(br) = "_seqgen" then 
         argv[i]:= context(br)
         // continue with the same i   
      else 
        // br is not a list or sequence
        newbranch:=BRANCH(context(argv[i]));
        if BRANCH::condition(newbranch) = Otherwise then
             if otherwiseBranch = undefined then
                 otherwiseBranch := BRANCH::expression(newbranch)
             else
                 error("Otherwise may occur in at most one branch")
             end_if;
        else        
       
          newbranch:= [piecewise::flattenBranch(newbranch)];
          newbranch:=  select(newbranch,
                                proc(br)
                                begin
                                  bool(
                                  BRANCH::condition(br) <> FALSE 
                                      )
                                end_proc
                                );
          for k from 1 to nops(newbranch) do
            expression:= BRANCH::expression(newbranch[k]);
            if BRANCH::condition(newbranch[k]) = TRUE then
              return(expression)
            end_if;
            if BRANCH::expression(newbranch[k]) = undefined then
              removedConditions:= removedConditions or BRANCH::condition(newbranch[k])
            elif type((j:= beenthere[expression])) = "_index" then
              result:= result.[newbranch[k]];
              beenthere[expression] := nops(result)
            else
              assert(type(j) = DOM_INT);
              if removedConditions <> FALSE and BRANCH::condition(newbranch[k]) = Otherwise then
                newbranch[k]:= BRANCH::setCondition(newbranch[k], piecewise::simplifyCondition(not _or(removedConditions, BRANCH::condition(newbranch[j]) $j=1..k-1)))
              end_if;  
              if BRANCH::condition(newbranch[k]) = Otherwise then
                result[j] := BRANCH::setCondition(result[j], Otherwise)
              elif BRANCH::condition(result[j]) <> Otherwise then
                result[j]:= BRANCH::applyToCondition
                (result[j], _or,
                BRANCH::condition(newbranch[k]));
                if op(result[j], 1) = TRUE then
                  return(BRANCH::expression(result[j]))
                end_if;
                if op(result[j], 1)=FALSE then
                  delete beenthere[expression]
                end_if
              end_if;
            end_if
          end_for  // k from 1 to nops(newbranch)
        end_if; // condition = Otherwise
        i:= i+1
      end_if; // type(br)
    end_while;

    if otherwiseBranch <> undefined then
      expression := otherwiseBranch;
      if type(expression) = piecewise then
      cond := [BRANCH::condition(result[k]) $ k=1..extnops(result)];
      cond := _or(op( cond ));
      cond := piecewise::simplifyCondition(  not eval(cond));
      newbranch:= BRANCH::new(cond, expression);
      newbranch:= [piecewise::flattenBranch(newbranch)];
      for br in newbranch do
        expression:= BRANCH::expression(br);
        if type((j:= beenthere[expression])) = "_index" then
          // we have never met that object
          result:=result.[br];
          beenthere[expression] := nops(result)
        else
          assert(type(j) = DOM_INT);
          // we have already a branch with that object
          // hence we just have to combine the conditions with _or
          // unless one of them is Otherwise, which is absorbing
          if BRANCH::condition(br) = Otherwise then
            result[j] := BRANCH::setCondition(result[j], Otherwise);
          elif BRANCH::condition(result[j]) <> Otherwise then
            result[j]:= BRANCH::applyToCondition(result[j], _or,
                                                BRANCH::condition(br));
          end_if;
          if op(result[j], 1)=TRUE then
            return(BRANCH::expression(result[j]))
          end_if;
          if op(result[j], 1)=FALSE then
            delete beenthere[expression]
          end_if
        end_if;
      end_for;
      elif removedConditions <> FALSE then 
         cond:= piecewise::simplifyCondition(not (removedConditions or _or(op([BRANCH::condition(result[k]) $ k=1..extnops(result)]))));
         result:= result.[BRANCH(cond, expression)]
      else
         result:= result.[BRANCH(Otherwise, expression)]
      end_if 
    end_if;

    result:= piecewise::sortBranches(result);
    if nops(result) = 0 then
      return(undefined)
    elif nops(result) = 1 and BRANCH::condition(result[1]) = Otherwise then
      return(BRANCH::expression(result[1]));
    else
      return(new(piecewise, op(result)))
    end_if;
  end_if;


  argument:=context(args(1));

  case type(argument)
  of "_exprseq" do 
    // force flattening
    // if this is ever changed, look at the piecewise and piecewise2 back
    // translation code in the Content library and remove the hold(hold)(.).
    return(context(hold(piecewise::new)(argument)))
  of DOM_NULL do
    return(undefined)
  end_case;

  result:= piecewise::convert(argument);
  if result = FAIL then
    error("Cannot convert to type piecewise")
  else
    result
  end_if
end_proc:


piecewise::create:=
proc()
  local i: DOM_INT;
begin
  new(piecewise, BRANCH::create(args(i)) $i=1..args(0))
end_proc:

piecewise::convert:=
proc(x)
  local br:BRANCH;
begin
  if args(0)=0 then
    error("Wrong number of arguments")
  end_if;

  case type(x)
    of piecewise do
      return(x)
    of BRANCH do
      br:=x;
      break
    otherwise
      br:=BRANCH::convert(x);
      if br = FAIL then
        return(FAIL)
      end_if;
  end_case;
  // here br is of type branch
  case BRANCH::condition(br)
    of Otherwise do
    of TRUE do
      return(BRANCH::expression(br))
    of FALSE do
      return(undefined)
    otherwise
      br:= piecewise::flattenBranch(br);
      if br = null() then
        return(undefined)
      end_if;
      return(new(piecewise, br))
      //return(new(piecewise, br))
  end_case;
end_proc:

piecewise::convert_to:=
proc(x:piecewise, T:DOM_DOMAIN)

begin
  case T
    of piecewise do
      return(x)
    of DOM_LIST do
      return([extop(x)])
    otherwise
      FAIL
  end_case
end_proc:

piecewise::sortBranches:=
proc(brs: DOM_LIST): DOM_LIST
  local cmp: DOM_PROC;
begin
  cmp:=
  proc(br1, br2): DOM_BOOL
  begin
    case sign(Simplify::defaultValuation(op(br1, 1)) -
              Simplify::defaultValuation(op(br2, 1)))
      of -1 do
        return(TRUE)
      of 1 do
        return(FALSE)
      of 0 do
        // should rarely happen
        return(sysorder(op(br1, 1), op(br2, 1)))
    end_case;
    assert(FALSE)
  end_proc;

  brs := split(brs, br -> op(br, 1) = Otherwise);
  sort(brs[2], cmp).brs[1];
end_proc:

piecewise::expr:= pw -> hold(piecewise)(op(map([op(pw)], expr))):

piecewise::testtype:=
proc(x, T)
  local i;
begin
  if domtype(x) <> dom then
    return(FALSE)
  end_if;
  if T="piecewise" then return(TRUE); end_if;
  if T=dom then
    return(TRUE)
  end_if;

  if T=Type::Arithmetical or T=Type::Set then
    // test whether every object is of the correct type
    for i from 1 to extnops(x) do
      if testtype(piecewise::expression(x, i), T) = FALSE then
        return(FALSE)
      end_if
    end_for;
    return(TRUE)
  end_if;

  FAIL
end_proc:

piecewise::branch:=
proc(pw:piecewise, n:Type::PosInt)
begin
  if n > extnops(pw) then
    FAIL
  else
    [extop(extop(pw, n))]
  end_if
end_proc:

piecewise::numberOfBranches:= x -> if type(x) <> piecewise then
                                     1
                                   else
                                     extnops(x)
                                   end_if:

piecewise::condition:=
proc(pw:piecewise, n:DOM_INT)
// returns the n-th condition
begin
  if n<1 or n>extnops(pw) then
    error("Index out of range")
  end_if;
  BRANCH::condition(extop(pw,n))
end_proc:

piecewise::expression:=
proc(pw:piecewise, n:DOM_INT)
// returns the expression that is valid if the n-th condition holds
begin
  if n<1 or n>extnops(pw) then
    error("Index out of range")
  end_if;
  BRANCH::expression(extop(pw,n))
end_proc:

piecewise::conditions:= pw -> map([extop(pw)], op, 1):

piecewise::expressions:= pw -> map([extop(pw)], op, 2):


piecewise::setBranch:=
proc(pw: piecewise, n: DOM_INT, br)
begin
  if n < 1 or n > extnops(pw) then
    error("Index out of range: no such branch")
  end_if;
  extsubsop(pw, n = BRANCH(br))
end_proc:

piecewise::setCondition:=
proc(pw: piecewise, n: DOM_INT, cond)
begin
  if n < 1 or n > extnops(pw) then
    error("Index out of range: no such branch")
  end_if;
  extsubsop(pw, n = BRANCH::setCondition(extop(pw, n), cond))
end_proc:

piecewise::setExpression:=
proc(pw: piecewise, n: DOM_INT)
begin
  if n < 1 or n > extnops(pw) then
    error("Index out of range: no such branch")
  end_if;
  extsubsop(pw, n = BRANCH::setExpression(extop(pw, n), args(3..args(0))))
end_proc:


// flattenBranch
//
// input: a branch [cond, pw]
//
// output: the input if pw is not piecewise
//         a sequence of branches [cond and c1, xpr1], ...,[cond and cn, xprn]
//                            if pw = piecewise([c1, xpr1],...,[cn, xprn])
//


piecewise::flattenBranch:=
proc(branch)
  local pw;
begin
  if type((pw := BRANCH::expression(branch))) <> piecewise then
    return(branch)
  end_if;

  if testargs() then
    if BRANCH::condition(branch) = Otherwise then
      error("Cannot flatten a branch with condition Otherwise")
    end_if
  end_if;
  assert(type(pw) = piecewise);
  pw:=(piecewise::mapConditions(pw, _and, BRANCH::condition(branch)));
  if type(pw)=piecewise then
    extop(pw)
  elif pw = undefined then
    // [FALSE, undefined]
    null()
  else
    [TRUE, pw]
  end_if;
end_proc:


piecewise::removePiecewiseFromCondition:=
proc(cond)
  local i, l, r, j;
begin
  case type(cond)
    of "_and" do
    of "_or" do
    of "_not" do
      return(map(cond, piecewise::removePiecewiseFromCondition))
    of "_less" do
    of "_leequal" do
    of "_equal" do
    of "_unequal" do
      [l, r]:= [op(cond)];
      if type(l)= piecewise then
        if type(r)=piecewise then
          _or(((dom::condition(l,i) and dom::condition(r, j) and
                op(cond, 0)(dom::expression(l, i), dom::expression(r, j)))
               $i=1..extnops(l)) $j=1..extnops(r))
        else
          _or((dom::condition(l,i) and op(cond, 0)(dom::expression(l, i), r))
              $i=1..extnops(l))
        end_if
      else
        if type(r)=piecewise then
          _or((dom::condition(r, i) and op(cond, 0)(l, dom::expression(r, i)))
              $i=1..extnops(r))
        else
          cond
        end_if
      end_if;
      break
    otherwise
      return(cond)
  end_case

end_proc:




piecewise::simplifyCondition:=
x -> simplify::simplifyCondition(piecewise::removePiecewiseFromCondition(x)):
// for purpose of testing, also try these:
// piecewise::simplifyCondition:= Simplify:


piecewise::replaceOtherwise:=
proc(pw: piecewise)
  local br, cond, k;
begin
  if piecewise::condition( pw, nops(pw) )=Otherwise then
    br := extop(pw, nops(pw));
    cond := [ piecewise::condition(pw, k) $ k=1..extnops(pw)-1];
    cond := _or(op((cond)));
    cond := piecewise::simplifyCondition(not cond);
    br := BRANCH::setCondition( br, cond );
    extsubsop(pw, nops(pw) = br)
  else
    pw
  end_if
end_proc:


piecewise::op:=
proc(pw:piecewise, n)
  local k;
begin
  if args(0) = 1 then
    piecewise::branch(pw, k) $k=1..extnops(pw)
  elif n=0 then
    piecewise
  else
    piecewise::branch(pw, n)
  end_if
end_proc:



// _index - return the i-th object
// this is equivalent to piecewise::expression
piecewise::_index:= piecewise::expression:




/*++

	extmap - application of a function to all expressions

++*/


piecewise::extmap:=
proc(pw)
  local i: DOM_INT;
  save MAXEFFORT;
begin
  if pw=undefined then
	return(undefined)
  end_if;
  if type(pw) <> piecewise then
    args(2)(pw, args(3..args(0)))
  else
  // piecewise::flatten
    MAXEFFORT:= MAXEFFORT/extnops(pw);
    piecewise::new(
                   BRANCH::apply(extop(pw, i), args(2..args(0)))
                   $i=1..extnops(pw))
  end_if
end_proc:

/*

   map - map a "map" command to all branches
            but simply map to objects that are not piecewise

*/


piecewise::map:=
proc(pw)
begin
  if type(pw)<>piecewise then
    map(pw, args(2..args(0)))
  else
    piecewise::extmap(pw, map, args(2..args(0)))
  end_if
end_proc:



/*++

	mapConditions - application of a function to all
			conditions

  makeMapConditions(Apply) - creates a mapConditions function
  such that Apply(cond, args(2..args(0))) is applied to all conditions

++*/

piecewise::makeMapConditions:= 
proc(Apply = BRANCH::applyToCondition)
  option escape;
begin  
  
proc(pw)
  local i, k, result, br, cond, change: DOM_BOOL;
  save MAXEFFORT;
  name piecewise::mapConditions;
begin
  if domtype(pw) <> piecewise then
    return(pw)
  end_if;
  pw:= piecewise::replaceOtherwise(pw);
  result:=null();
  MAXEFFORT:= MAXEFFORT/extnops(pw);
  change:= FALSE;
  for i from 1 to extnops(pw) do
    if piecewise::condition( pw, i )=Otherwise then
      br := extop(pw,i);
      cond := [ piecewise::condition(pw, k) $ k=1..extnops(pw)];
      cond := _or(op((select(cond, _not@_equal, Otherwise))));
      cond := piecewise::simplifyCondition(  not eval(cond) );
      br := BRANCH::setCondition( br, cond );
      br:= Apply(br, args(2..args(0)));
      if not change and BRANCH::condition(br) = cond then
        // no change -> express this as Otherwise
        br:= extop(pw, i)
      end_if
    else
      br:= Apply(extop(pw,i), args(2..args(0)));
      if br <> extop(pw, i) then
        change:= TRUE
      end_if
    end_if;
    case BRANCH::condition(br)
      of TRUE do
        return(BRANCH::expression(br))
      of FALSE do
        break;
      otherwise
        result:=result, br
    end_case;
  end_for;
  if result = null() then
    undefined
  else
    piecewise::new(result)
  end_if
end_proc
end_proc:

piecewise::mapConditions:= piecewise::makeMapConditions():


// piecewise::mapConditionsRepeatedly - replaces every condition cond by f(cond) until f(cond) = cond. Thus, this won't terminate unless
// f is a simplification function that eventually does not change its argument any more
piecewise::mapConditionsRepeatedly:= piecewise::makeMapConditions(BRANCH::applyToConditionRepeatedly):



piecewise::unfreeze :=
pw -> piecewise::extmap(piecewise::mapConditions(pw, unfreeze), unfreeze):

piecewise::unblock :=
(pw, blockdomain, Recurse, do_eval) ->
piecewise::extmap(piecewise::mapConditions(pw, unblock, args(2..args(0))),
	       unblock, args(2..args(0))):



/* piecewise::selectConditions(pw, f)

keep only those conditions for which f(condition)=TRUE

*/

piecewise::selectConditions:=
proc(pw,f)
local argv, branches: DOM_LIST, i: DOM_INT;
begin
  argv:=args();
  branches:= [extop(pw)];
  if op(branches[nops(branches)], 1) = Otherwise then
    branches[nops(branches)]:= subsop(branches[nops(branches)],
                                      1 = _and(not op(branches[i], 1)
                                               $i=1..nops(branches)-1
                                               )
                                      )
  end_if;
  branches:= select(branches, x-> f(op(x,1), op(argv, 3..nops(argv))));
  if nops(branches)=0 then
    undefined
  else
    piecewise::new(op(branches))
  end_if;
end_proc:



piecewise::selectExpressions:=
proc(pw, f)
  local argv;
begin
  argv:=args();
  select([extop(pw)], x-> f(op(x, 2), op(argv, 3..nops(argv))));
  if nops(%)=0 then
    undefined
  else
    piecewise::new(op(%))
  end_if
end_proc:


/* piecewise::splitConditions(pw, f)

split pw into a list of three piecewise objects depending on whether
f(condition) gives TRUE, FALSE, or UNKNOWN

*/

piecewise::splitConditions:=
proc(pw, f)
local argv;
begin
  argv:=args();
  split([extop(pw)], x-> f(op(x,1), op(argv, 3..nops(argv))));
  map(%, u-> if nops(u)>0 then
               piecewise::new(op(u))
             else
               undefined
             end_if)
end_proc:

/*++
	zip - argumentwise application of a law of composition
	for two piecewise defined functions
++*/

piecewise::zip:=
proc(pwf, pwg, f)
  local i,j,k, result,cond,g,conds_so_far, c1, c2, res;
  save MAXEFFORT;
begin
  if type(pwf)<>piecewise then
    g:= x -> f(pwf, x);
    return(piecewise::extmap(pwg,g))
  elif type(pwg)<>piecewise then
    return(piecewise::extmap(pwf, f, pwg))
  end_if;
  result:=[];
  conds_so_far:=[];
  MAXEFFORT:= MAXEFFORT/nops(pwf)/nops(pwg)/2;
  for i from 1 to extnops(pwf) do
    for j from 1 to extnops(pwg) do
      if ( c1 := piecewise::condition(pwf,i) ) = Otherwise then
        c1 := [piecewise::condition(pwf, k) $ k=1..extnops(pwf)];
        c1 := _or(op((select(c1, _not@_equal, Otherwise))));
        c1 := piecewise::simplifyCondition(  not eval(c1));
      end_if;
      if ( c2 := piecewise::condition(pwg,j) ) = Otherwise then
        c2 := [piecewise::condition(pwg, k) $ k=1..extnops(pwg)];
        c2 := _or(op((select(c2, _not@_equal, Otherwise))));
        c2 := piecewise::simplifyCondition(  not eval(c2));
      end_if;
      cond:=piecewise::simplifyCondition(c1 and c2);
      // possibility check
      case cond
        of TRUE do
          return(f(piecewise::expression(pwf,i),
                          piecewise::expression(pwg,j)))
        of FALSE do
          break
        otherwise
          if contains(conds_so_far, cond)=0 then
            if traperror((res:=  f(piecewise::expression(pwf,i),
                     piecewise::expression(pwg,j)))) = 0 and
              res <> undefined then
              result:=append(result, BRANCH(cond, res))
            end_if;
            conds_so_far:=append(conds_so_far, cond)
          end_if
      end_case;
	end_for
  end_for;
  if nops(result)= 0 then
    undefined
  else
    piecewise::new(op(result))
  end_if
end_proc:


// apply a law of composition to n objects (not all necessarily piecewise)
piecewise::zipnary:=
proc(argv: DOM_LIST, f, holdf, neutral)
  local pws, nonpws, dummy, zipnary_internal, i: DOM_INT;
begin
  argv:= select(argv, _unequal, neutral);
  if nops(argv) = 0 then
    return(neutral)
  elif nops(argv) = 1 then
    return(argv[1])
  end_if;
  // estimate maximum number of branches of the result
  if _mult(piecewise::numberOfBranches(argv[i]) $i=1..nops(argv)) >
    MAXEFFORT then
    return(holdf(op(argv)))
  end_if;


  zipnary_internal:=
  proc(argv)
    local result1, result2;
    save MAXEFFORT;
  begin

    case nops(argv)
      of 0 do
        return(neutral)
      of 1 do
        return(argv[1])
      of 2 do
        return(piecewise::zip(argv[1], argv[2], f))
      otherwise
        MAXEFFORT:= MAXEFFORT / 3;
        result1:=
        zipnary_internal([op(argv, 1..nops(argv) div 2)]);
        result2:=
        zipnary_internal([op(argv, (nops(argv) div 2)+1 .. nops(argv))]);
        assert(type(result1) <> DOM_LIST);
        assert(type(result2) <> DOM_LIST);
        if type(result1) <> piecewise and type(result2) <> piecewise then
          holdf(if op(result1, 0) = holdf then op(result1) else result1 end,
                if op(result2, 0) = holdf then op(result2) else result2 end)
        else
          piecewise::zip(result1, result2, f)
        end_if
    end_case
  end_proc;

  [pws, nonpws, dummy]:= split(argv, x -> type(x) = piecewise);
  assert(dummy = []);
  pws:= zipnary_internal(pws);
  if nops(nonpws) = 0 or ((nonpws:= f(op(nonpws)))) = neutral then
    pws
  else
    if op(pws, 0) = holdf then
      holdf(op(pws), nonpws)
    else
      piecewise::extmap(pws, f, nonpws)
    end_if
  end_if
end_proc:


piecewise::evaluate:=
proc(pw)
  local i, j, result, cond, br, brlist, beenthere: DOM_TABLE, xpr,
  newbr, otherconds, Err;
  save MAXEFFORT;
begin
  result:=[];
  beenthere:= table();
  // pw:=piecewise::flatten(pw);
  if nops(pw) = 0 then return(undefined) end_if;
  MAXEFFORT:= MAXEFFORT/nops(pw);
  for i from 1 to extnops(pw) do
    Err:= traperror
      ((cond:=
          piecewise::simplifyCondition
          (

            piecewise::evalCondition(piecewise::condition(pw,i))
           )
          ));
    if Err = 1010 then
      lasterror()
    elif Err <> 0 then
      next
    end_if;
    case cond
      of TRUE do
        return(eval(piecewise::expression(pw, i)))
      of FALSE do
        break
      otherwise
        if cond = Otherwise then
          otherconds := FAIL;
          traperror
          ((
            otherconds := [piecewise::condition(pw, i) $ i=1..extnops(pw)];
            otherconds := _or(op((select(otherconds, _not@_equal, Otherwise)))
                              );
            otherconds := piecewise::simplifyCondition(eval(otherconds));
            ));
          if otherconds = TRUE then
            break
          end_if;
          if otherconds = FALSE then
            return(eval(piecewise::expression(pw, i)));
          end_if;
          if domtype((xpr:= eval(piecewise::expression(pw, i)) assumingAlso not otherconds)) = piecewise
            then
            // can only flatten this if I know what otherwise means
            cond:= not otherconds;
            brlist:= [piecewise::flattenBranch
                      (BRANCH(cond, xpr))]
          else
            brlist:= [ BRANCH(cond, xpr)]
          end_if;
        else
          brlist:= [piecewise::flattenBranch
                    (BRANCH
                     (cond,
                      eval(piecewise::expression(pw, i)) assumingAlso cond))
                    ];
        end_if;
        for br in brlist do
          xpr:= BRANCH::expression(br);
          if BRANCH::condition(br) <> FALSE and
            xpr <> undefined then
            if type((j:= beenthere[xpr])) = "_index" then
              result:=result.[br];
              beenthere[xpr] := nops(result)
            else
              if BRANCH::condition(br) = Otherwise then
                newbr := BRANCH::setCondition(result[j], Otherwise);
              elif BRANCH::condition(result[j]) <> Otherwise then
                newbr:= BRANCH::applyToCondition
                                (result[j], _or, BRANCH::condition(br));
                if BRANCH::condition(newbr) = TRUE then
                  return(xpr)
                end_if;
              end_if;
              result[j]:= newbr;
            end_if;
          end_if;
        end_for;
    end_case;
  end_for;
  result:= [op(piecewise::sortBranches(result))];
  if nops(result)=0 then
    return(undefined)
  elif nops(result)=1 and BRANCH::condition(result[1]) = Otherwise then
    return(eval(BRANCH::expression(result[1])))
  else
    new(piecewise, op(result))
  end_if
end_proc:


piecewise::subs:=
proc(pw:piecewise)
begin
  if contains({args()}, hold(Unsimplified)) then
    piecewise::new(subs(extop(pw), args(2..args(0))))
  else
    eval(piecewise::new(subs(extop(pw), args(2..args(0)))))
  end_if
end_proc:

piecewise::evalAt:=
proc(pw:piecewise)
begin
  pw:= piecewise::mapConditions(pw, piecewise::evalAtCondition, args(2..args(0)));
  piecewise::extmap(pw, evalAt, args(2..args(0)))
end_proc:

// evaluate a condition at a point such that all leaves that produce an error
// become FALSE
// this is necessary as "0=0 or 1/0=nonsense"" should become TRUE
// cf. piecewise::evalCondition below
piecewise::evalAtCondition:=
proc(cond)
  local c;
begin
  case type(cond)
    of "_and" do
    of "_or" do
    of "_not" do
      map(cond, piecewise::evalAtCondition, args(2..args(0)));
      break
    otherwise
      if traperror((c:= evalAt(cond, args(2..args(0))))) <> 0 then
        return(FALSE)
      else
        return(c)
      end_if
  end_case
end_proc:




piecewise::insert:=
proc(pwf, l)
  local i, pwb, ex;
begin
  l:= BRANCH::new(l);
  ex := BRANCH::expression(l);
  if ex = undefined or extop(l,1) = FALSE then
    return(pwf)
  end_if;
  if pwf = undefined then
    return(piecewise::new(l))
  end_if;
  if type(ex) = piecewise then
    ex:= piecewise::mapConditions(ex, _and, BRANCH::condition(l));
    return(pwf.ex)
  end_if;
  
  /* check if the piecewise already contains the expression */
  i := contains( map( [extop(pwf)], X->bool(BRANCH::expression(X)=ex) ), TRUE );
  if i<>0 then
    pwb := extop(pwf,i);
    if BRANCH::condition(l) = Otherwise then
      extsubsop( pwf, i=l );
    elif BRANCH::condition(pwb) <> Otherwise then
      pwb:= BRANCH::applyToCondition(pwb, _or, BRANCH::condition(l));
      case BRANCH::condition(pwb)
        of TRUE do
          return(BRANCH::expression(pwb))
        of FALSE do
          // should rarely happen!
          // We are able to decide that A or B is FALSE
          // although we think that A may be TRUE
          return(piecewise::remove(pwf, i))
        otherwise
          extsubsop( pwf, i=pwb )
      end_case
    end_if    
  else
    new(piecewise, op(piecewise::sortBranches([extop(pwf), l])))
  end;
end_proc:

piecewise::remove:=
proc(pwf, no: Type::PosInt)
  local i;
begin
  if extnops(pwf)<no then
    error("Illegal position")
  end_if;
  if extnops(pwf) = 1 then
    assert(no = 1);
    return(undefined)
  end_if;
  new(piecewise, extop(pwf, i) $i=1..no-1, extop(pwf, i) $i=no+1..extnops(pwf))
end_proc:


// fill cases: produce a nowhere undefined
// piecewise by filling pw with a where it was previously undefined
piecewise::fillCases:=
proc(pw, a)
  local br, i;
begin
  if pw = undefined then
    return(a)
  end_if;
  if type(pw) <> piecewise then
    return(pw)
  end_if;
  if piecewise::condition(pw, nops(pw)) = Otherwise then
    return(pw);
  end_if;

  /* don't add Otherwise if all cases are masked */
  br :=piecewise::simplifyCondition( _or( piecewise::condition(pw, i) $i=1..nops(pw)) );
  if br=TRUE or a=undefined then
    pw;
  else
    br:= BRANCH::new([ Otherwise, a ]);
    new(piecewise, extop(pw), br)
  end_if
end_proc:


/*
splitBranch(pw, n, cond)

split the nth branch [c, x] into two branches
[c and cond, x], [c and not cond, x]
*/


piecewise::splitBranch:=
proc(pwf: piecewise, no: Type::PosInt, cond): piecewise
  local i;
begin
  if extnops(pwf) < no then
    error("Not that many branches exist")
  end_if;
  new(piecewise,
      extop(pwf, i) $i=1..no-1,
      [piecewise::condition(pwf, no) and cond, piecewise::expression(pwf, no)],
      [piecewise::condition(pwf, no) and not cond,
       piecewise::expression(pwf, no)],
      extop(pwf, i) $i=no+1..extnops(pwf) )
end_proc:



/*
disregardPoints: assume every condition of type x=C to be FALSE
and every condition of type x<>C to be TRUE
if a second argument (set of variables) is given, this is only done
if x or C contains one of the variables
*/

piecewise::disregardPoints:=
proc(pw, vars)
  local f: DOM_PROC;
begin
  if domtype(pw) <> piecewise then
    return(pw)
  end_if;

  if args(0) = 2 then
    f:= cond -> has(cond, vars)
  else
    f:= TRUE
  end_if;  
  
  piecewise::mapConditionsRepeatedly(pw, solvelib::specialToFalse, f)
end_proc:


// assumeIdentsTRUE(pw, identset) - assume that all conditions in which
// one of the identifiers in identset appears is TRUE


piecewise::assumeIdentsTRUE:=
proc(pw, identset:DOM_SET)
  local subsCond, result;
begin
  subsCond:=
  proc(cond)

  begin
    case type(cond)
      of "_or" do
      of "_and" do
        return(map(cond, subsCond))
      of "_not" do
        // would have been distributed by the kernel
        assert(not contains({"_or", "_and"}, type(op(cond, 1))));
        // fall through
      otherwise
        if indets(cond) intersect identset <> {} then
          return(TRUE)
        else
          return(cond)
        end_if
    end_case
  end_proc;

  result:=piecewise::mapConditions(pw, subsCond);
  delete subsCond;
  result
end_proc:

// assumeIdentsFALSE(pw, identset) - assume that all conditions in which
// one of the identifiers in identset appears is FALSE


piecewise::assumeIdentsFALSE:=
proc(pw, identset:DOM_SET)
  local subsCond, result;
begin
  subsCond:=
  proc(cond)

  begin
    case type(cond)
      of "_or" do
      of "_and" do
        return(map(cond, subsCond))
      of "_not" do
        // would have been distributed by the kernel
        assert(not contains({"_or", "_and"}, type(op(cond, 1))));
        // fall through
      otherwise
        if indets(cond) intersect identset <> {} then
          return(FALSE)
        else
          return(cond)
        end_if
    end_case
  end_proc;

  result:=piecewise::mapConditions(pw, subsCond);
  delete subsCond;
  result
end_proc:


//  operations on conditions //


// innerOfCondition: restrict condition to interior points w.r.t. inds

piecewise::innerOfCondition:=
proc(c, inds)
  local S, interior;
begin
  interior:=
  proc(S)
    local l;
  begin

    if S::dom::interior <> FAIL then
      return(S::dom::interior(S))
    end_if;

    case type(S)
      of solvelib::BasicSet do
        case S
          of R_ do
          of C_ do
            return(S)
          of Z_ do
          of Q_ do
            return({})
        end_case;
      of DOM_SET do
      of RootOf do
        return({})
      of "_union" do
        // may produce a proper subset !!
      of "_intersect" do
        return(map(S, interior))
      of Dom::ImageSet do
        l:= map(extop(S, 3), interior);
        if l = extop(S, 3) then
          return(S)
        else
          return(eval(extsubsop(S, 3 = l)))
        end_if
    end_case;

    // default
    warning("Cannot determine interior of ".expr2text(S));
    S
  end_proc;

  if args(0) > 1 and not has(c, inds) then
    return(c)
  end_if;

  case type(c)
    of "_equal" do
      return(FALSE)
    of "_unequal" do
    of "_less" do
      return(c)
    of "_leequal" do
      return(_less(op(c)))
    of "_and" do
    of "_or" do
    of "_not" do
      return(map(c, piecewise::innerOfCondition))
    of "_in" do
      S:= interior(op(c, 2));
      if S = {} then
        return(FALSE)
      else
        return(op(c, 1) in interior(op(c, 2)))
      end_if
    otherwise
      error("Cannot handle condition of type ".expr2text(type(c)))
  end_case
end_proc:

piecewise::borderOfCondition:=
proc(c, inds)
  local border;
begin
  border:=
  proc(S)
  begin
    if S::dom::border <> FAIL then
      return(S::dom::border(S))
    end_if;

    case type(S)
      of DOM_SET do
        return(S)
      of "_union" do
        // may produce a proper superset!!
      of "_intersect" do
        return(map(S, border))
      of Dom::Interval do
        return({op(Dom::Interval::borders(S))})
      of solvelib::BasicSet do
        if S=Z_ or S=Q_ then
          return(S)
        else
          return({})
        end_if
    end_case;
    // default
    warning("Cannot determine border of ".expr2text(S));
    return({})
  end_proc;

  if args(0) > 1 and not has(c, inds) then
    // empty border
    return(FALSE)
  end_if;

  case type(c)
    of "_equal" do
      return(c)
    of "_unequal" do
    of "_leequal" do
    of "_less" do
      return(_equal(op(c)))
    of "_or" do
      return(map(c, piecewise::borderOfCondition))
    of "_and" do
      return(piecewise::borderOfCondition(op(c,1)) or
             piecewise::borderOfCondition(op(c,2))
             )
    of "_not" do
      return(piecewise::borderOfCondition(op(c,1)))
    of "_in" do
      return(op(c,1) in border(op(c, 2)))
    otherwise
      error("Cannot handle condition of type ".expr2text(type(c)))
  end_case
end_proc:

piecewise::outerOfCondition:=
proc(c)
begin
  piecewise::innerOfCondition(not c, args(2..args(0)))
end_proc:

piecewise::satisfiesCondition:=
proc(x, value, c)
begin
  is(subs(c, x=value))
end_proc:


piecewise::avoidAliasProblem:=
proc(pw)
  local i: DOM_INT;
begin
  new(piecewise, BRANCH::avoidAliasProblem(extop(pw, i), args(2..args(0)))
      $i=1..extnops(pw))
end_proc:


//  arithmetical/ set theoretic operations (argumentwise)

piecewise::_plus:= () -> piecewise::zipnary([args()], _plus, hold(_plus), 0):
piecewise::_mult:= () -> piecewise::zipnary([args()], _mult, hold(_mult), 1):
piecewise::_union:= () -> piecewise::zipnary([op({args()})], _union, hold(_union), {}):
piecewise::_intersect:=
proc()
begin
  piecewise::zipnary([op({args()})],
                     solvelib::solve_intersect, hold(_intersect), C_)
end_proc:


// overload solvelib::isEmpty
piecewise::isEmpty:=
proc(pw)
  local i: DOM_INT;
begin
  _or((piecewise::condition(pw, i) and
      solvelib::isEmpty(piecewise::expression(pw, i)))
      $i=1..nops(pw)
      )
end_proc:



piecewise::_negate:=proc() begin piecewise::extmap(args(1), _negate) end_proc :

piecewise::_invert:=proc() begin piecewise::extmap(args(1), _invert) end_proc :

piecewise::_subtract:=proc() begin piecewise::zip(args(),_subtract) end_proc :

piecewise::_divide:=proc() begin piecewise::zip(args(),_divide) end_proc :

piecewise::_power:=proc() begin piecewise::zip(args(),_power) end_proc :

piecewise::_div:=proc() begin piecewise::zip(args(),_div) end_proc :

piecewise::_mod:=proc() begin piecewise::zip(args(),_mod) end_proc :

piecewise::_minus:=
proc()
begin
  piecewise::zip(args(),solvelib::solve_minus)
end_proc:





piecewise::cmp:=
proc(cmp)
  option escape;
begin
  proc(x, y)
    local pw, i;
  begin
    pw:= piecewise::zip(x,y, cmp):
    if type(pw) <> piecewise then
      return(pw)
    end_if;
    _or((piecewise::condition(pw, i) and piecewise::expression(pw, i))
        $i=1..nops(pw))
  end_proc

end_proc:

piecewise::_less:= piecewise::cmp(hold(_less)):

piecewise::_leequal:= piecewise::cmp(hold(_leequal)):

piecewise::_unequal:= piecewise::cmp(hold(_unequal)):

piecewise::_equal:= piecewise::cmp(hold(_equal)):

// overload solvelib::Union
/* piecewise::Union(pw, x, xset)

   returns the union of all pw(x), x varying over xset

   There are two kinds of branches:
   a) the condition in the branch does not depend on x, i.e. the
      branch is applicable for all or none x \in xset. The Union may be
      computed separately for every such branch

   b) the condition depends on x. We assume that pw is consistent
      (branches agree where they domains of definition overlap).
      Then \bigcup_{x in xset} (S1(x) if cond1(x), S2(x) if cond2(x))
      equals \bigcup{x in {y \in xset; cond1(y)}} S1(x)
       \cup \bigcup{x in {y \in xset; cond2(y)}} S2(x)
*/

piecewise::Union:=
proc(pw:piecewise, x:DOM_IDENT, xset)
  local dep, indep, dummy, iU, dU, i;
begin

  // simple case

  if xset={} then
    return({})
  end_if;

  if MAXEFFORT < 5*nops(pw) or nops(pw) > 50 then
	return(hold(solvelib::Union)(pw, x, xset))
  end_if;

  [dep, indep, dummy] := piecewise::splitConditions(pw, has, x);
  assert(dummy = undefined);

  // simpler part first: conditions do not depend on the variable
  if indep = undefined then
    iU:= undefined;
  else
    iU:= piecewise::indepUnion(indep, x, xset)
  end_if;


  if dep = undefined then
    return(iU)
  elif indep <> undefined then
    dep:= piecewise::mapConditions(dep, _and,
                                   _and (not piecewise::condition(indep, i)
                                         $i=1..nops(indep)));
    if dep = undefined then
      return(iU)
    end_if  
  end_if;

  if type(dep) = piecewise then
    userinfo(20, expr2text(extnops(dep))."conditions depend on variable");
    dU:= piecewise::depUnion(dep, x, xset)
  else
    dU:= solvelib::Union(dep, x, xset)
  end_if;

  if iU = undefined then
    dU
  else
    iU:= piecewise::fillCases(iU, {}):
    iU union dU
  end_if;
end_proc:

piecewise::indepUnion:= (pw, x, xset) -> piecewise::extmap(pw, solvelib::Union, x, xset):

piecewise::depUnion:=
proc(pw: piecewise, x, xset)
  local mysolve, i;
  save MAXEFFORT;

begin

  mysolve:=
  proc(cond)
    local S, T;
    save MAXEFFORT;
  begin
    /* save x;
    assume(x in xset);
    cond:= eval(cond);
    unassume(x); */

    MAXEFFORT:= MAXEFFORT/2;
    S:= solve(cond, x);
    T:= solvelib::solve_intersect(S, xset);
    if type(T) = piecewise and
      type(S) <> piecewise and
      type(xset) <> piecewise then
      hold(_intersect)(S, xset)
    else
      T
    end_if
  end_proc;

  // first step: delete empty sets
  pw:= piecewise::selectExpressions(pw, _unequal, {});
  if pw=undefined then
    return({})
  end_if;

  if MAXEFFORT < 100*nops(pw) then
     return(hold(solvelib::Union)(pw, x, xset))
  end_if;

  // Todo: be careful with determining x in {y \in xset; condi(y)}
  // solve condi for y only if it is rather simple !?

  MAXEFFORT:= MAXEFFORT/3/nops(pw);
  _union(solvelib::Union
         (piecewise::expression(pw, i), x,
          mysolve(piecewise::condition(pw, i)))
         $i=1..nops(pw))
end_proc:


// _subset: overload both S subset pw and pw subset S for all types of S
// if both S and T are piecewise, the function calls itself recursively
// (may need optimization if this occurs really often)

piecewise::_subset:=
proc(S, T)
  local i: DOM_INT;
begin
  if S::dom = piecewise then
    _or((piecewise::expression(S, i) subset T and piecewise::condition(S, i))
        $i=1..nops(S)
        )
  else
    assert(T::dom = piecewise);
    _or((S subset piecewise::expression(T, i) and piecewise::condition(T, i))
        $i=1..nops(T)
        )
  end_if
end_proc:



// special functions

piecewise::sin:= x -> piecewise::extmap(x, sin):
piecewise::cos:= x -> piecewise::extmap(x, cos):
piecewise::tan:= x -> piecewise::extmap(x, tan):
piecewise::exp:= x -> piecewise::extmap(x, exp):
piecewise::ln := x -> piecewise::extmap(x, ln):
piecewise::arcsin:= x-> piecewise::extmap(x, arcsin):
piecewise::arccos:= x -> piecewise::extmap(x, arccos):
piecewise::arctan:= x -> piecewise::extmap(x, arctan):
piecewise::sinh := x -> piecewise::extmap(x, sinh):
piecewise::cosh := x-> piecewise::extmap(x, cosh):
piecewise::tanh := x -> piecewise::extmap(x, tanh):
piecewise::coth := x-> piecewise::extmap(x, coth):
piecewise::arcsinh:= x -> piecewise::extmap(x, arcsinh):
piecewise::arccosh:= x -> piecewise::extmap(x, arccosh):
piecewise::arctanh:= x -> piecewise::extmap(x, arctanh):
piecewise::arccoth:= x -> piecewise::extmap(x, arccoth):
piecewise::sech:= x-> piecewise::extmap(x, sech):
piecewise::csch:= x-> piecewise::extmap(x, csch):
piecewise::sign := x -> piecewise::extmap(x, sign):
piecewise::abs  := x-> piecewise::extmap(x, abs):
piecewise::heaviside:= x -> piecewise::extmap(x, heaviside):
piecewise::arg:= x-> piecewise::extmap(x, arg):
piecewise::surd:= (x, n) -> piecewise::extmap(x, surd, n):
piecewise::erf:= x -> piecewise::extmap(x, erf):
piecewise::erfc:= x-> piecewise::extmap(x, erfc):

piecewise::ceil := x -> piecewise::extmap(x, ceil):
piecewise::floor := x -> piecewise::extmap(x, floor):
piecewise::trunc := x -> piecewise::extmap(x, trunc):
piecewise::round := x -> piecewise::extmap(x, round):

piecewise::Re := x -> piecewise::extmap(x, Re):
piecewise::Im := x -> piecewise::extmap(x, Im):

piecewise::numer := x -> piecewise::extmap(x, numer):
piecewise::denom := x -> piecewise::extmap(x, denom):
piecewise::ldegree:= x-> piecewise::extmap(x, ldegree, args(2..args(0))):
piecewise::coeff := x -> piecewise::extmap(x, coeff, args(2..args(0))):
piecewise::collect:= x -> piecewise::extmap(x, collect, args(2..args(0))):

// special functions with more than one argument

piecewise::Ei:=
proc(n, x)
begin
  if type(n) = piecewise then
    piecewise::extmap(n, Ei, args(2..args(0)))
  else
    assert(type(x) = piecewise);
    piecewise::extmap(x, u -> Ei(n, u))
  end_if
end_proc:



/*++
	restrict - impose an additional restriction
		   on the piecewise function
++*/

piecewise::restrict:=
proc(pw, cond)

begin
  if type(pw) <> piecewise then
    piecewise(BRANCH(cond, pw))
  else
    piecewise::mapConditions(pw, _and, cond)
  end_if
end_proc:

piecewise::_concat:=
proc()
  local argv;
begin
  argv:=select([args()], _unequal, undefined);
  // the everywhere undefined function is neutral
  // element w.r.t. concatenation
  if map({op(argv)}, type)<>{piecewise} then
    error("All arguments must be of type piecewise")
  end_if;
  new(piecewise,op(map(argv, extop))) // flatten !
end_proc:

/*++
	simplify - try to decide the truth of the condition and
	remove unnecessary conditions
++*/

piecewise::simplify:=
proc(pw: piecewise)
  local i, j, result, cond, xpr, otherconds;
  save MAXEFFORT;
begin
  if domtype(pw) <> piecewise then
    return(pw);
  end_if;
  result:=[];
  MAXEFFORT:= MAXEFFORT/extnops(pw);
  for i from 1 to extnops(pw) do
    cond:=piecewise::simplifyCondition(simplify(piecewise::condition(pw, i)));
    case cond
      of FALSE do
        break; // do not enter this branch into result
      of TRUE do
        return(simplify(piecewise::expression(pw,i)))
      otherwise
        // test whether the same object already exists

        if cond = Otherwise then
          assert(i = extnops(pw));
          otherconds := FALSE;
          traperror
          ((
            otherconds := [piecewise::condition(pw, i) $ i=1..extnops(pw)-1];
            otherconds := _or(op(otherconds));
            otherconds := piecewise::simplifyCondition(eval(otherconds));
            ));
          if otherconds = TRUE then
            break
          end_if;
          if otherconds = FALSE then
            return(eval(piecewise::expression(pw, i)));
          end_if;

          otherconds:= not otherconds
        else
          otherconds:= cond
          
        end_if;


        
        if traperror
          ((xpr:= piecewise::evalAssuming
            (simplify(piecewise::expression(pw, i)), otherconds)
            ))
          <>0 then
          // regard this as FALSE
          userinfo(1, "simplification of ".expr2text(op(pw, i))." failed");
          break
        end_if;

        j:=contains(map(result, op, 2), xpr);
        if j>0 then
          // combine old and new condition
          if piecewise::condition(pw, i)=Otherwise or
            result[j][1] = Otherwise then
            cond := Otherwise;
          else
            cond:=piecewise::simplifyCondition(result[j][1] or piecewise::condition(pw, i));
          end_if;
          if cond=TRUE then
            return(xpr)
          elif cond=FALSE then
            // should not happen, since result[j][1] was not FALSE
            delete result[j]
          else
            result[j][1]:=cond;
            result[j][2]:= piecewise::evalAssuming(result[j][2], cond)
          end_if
        else  // cond has still its original value!
          result:=append(result, [cond, xpr])
        end_if;
    end_case;
  end_for;

  piecewise::combineSpecialCases(result);
end_proc:

  /* This merges branches with conditions like IDENT=constant into matching other branches.
   * For example the following piecewise
   * piecewise(0 if c = 0, c if c <> 0)
   * will be simplified to just "c".
  */
piecewise::combineSpecialCases :=
proc(pw : Type::Union(piecewise, DOM_LIST))
  local result, i, j, cond, equations, success, equ, inds, xpr;
begin
  if pw::dom = piecewise then
    result := [op(pw)];
  else
    result := pw;
  end_if;

  i := 1;
  while i<= nops(result) do
    cond := op( result, [i,1] );
    if type(cond) =  "_equal"  then
      equations:= [cond]
    elif type(cond) = "_and" then
      equations:= select([op(cond)], testtype, "_equal")
    else
      equations:= []
    end_if;

    success:= FALSE;
    for equ in equations do
      inds := indets(equ);
      for j from 1 to nops(result) do
        if indets( op( result, [j,2] ) ) intersect inds={} then
          next;
        end_if;
        if i<>j then
          if traperror(( xpr := property::normalGroebner( op( result, [j,2] ),
                                                         equ ) )) = 0
            and xpr=op( result, [i,2] ) then
            // expression would simplify to the same
            if op(result, [j,1])=Otherwise then
              cond := Otherwise
            else
              cond := piecewise::simplifyCondition( (op(result, [j,1]) or cond) );
            end_if;
            if cond = TRUE then
              return( op( result, [j,2] ) );
            end_if;
            result := subsop( result, [j,1] = cond );
            result[i] := null();
            i := i-1;
            success:= TRUE;
            break;
          end_if;
        end_if; // i <> j
      end_for; // j
      if success then
        break
      end_if;
    end_for; // equ in equations
    i := i+1;
  end_while;

  piecewise::new(op(result))
end_proc:

// needed for Simplify
piecewise::operandsToSimplify:=
proc(pw: piecewise)
begin
  [$1..nops(pw)]
end_proc:


// piecewise::subsEqualities - simplifies a given expression
// taking into account the given condition

piecewise::subsEqualities:=
proc(xpr, cond)
local i, result; // j is not used anymore

begin
  case type(cond)
    of "_equal" do
      if traperror((result:= (xpr | cond)))<>0 then
        userinfo(1,"Error in subsEqualities");
        return(undefined)
      else
        return(result)
      end_if;
    // case _less etc.: use assume
  /*  of "_less" do
    of "_leequal" do
      Lhs:= op(cond, 1);
      Rhs:= op(cond, 2);
      if not has(xpr,Lhs) and not has(xpr, Rhs) then
        return(xpr)
      end_if;
      i:=genident(); j:=genident();
      newexpr:=xpr;
      if testtype(Lhs, Type::Numeric)<>TRUE then
        newexpr:=subs(newexpr, Lhs=i);
        newcond:=subsop(cond, 1=i)
      else
        newcond:=cond
      end_if;
      if testtype(Rhs, Type::Numeric)<>TRUE then
        newexpr:=subs(newexpr, Rhs=j);
        newcond:=subsop(newcond, 2=j)
      end_if;
        // substitute a new variable for the left/right hand side
        // in both the expression and the condition
        // then assume the new condition generated in this way
        // to be true, and simplify the newly generated expression

      /* since assume does not evaluate its arguments */
      eval(subs(hold(assume(dummy)), hold(dummy)=newcond));
      eval(subs(simplify(newexpr), i=Lhs, j=Rhs));
      break
    of "_in" do
      if domtype(op(cond, 1)) = DOM_IDENT and
        not contains(Type::ConstantIdents, op(cond, 1))
        and has(xpr, op(cond, 1))
        then
        return(eval(xpr) assuming cond)
      else
        xpr
      end_if
      */
    of "_and" do
      result:=xpr;
      for i in cond do
        result:= piecewise::subsEqualities(result, i)
      end_for;
      result;
      break
    otherwise
      xpr
  end_case
end_proc:


piecewise::evalAssuming:=
proc(a, cond)
  option hold;
begin
  if MAXEFFORT<length(cond) then
    return(context(a))
  end_if;
  cond:= context(cond);
  if cond <> Otherwise then
    /* we ignore "incosistent assumptions"-errors */
    traperror(assume(cond, _and));
  end_if;
  case traperror((a:= property::normalGroebner(context(a))))
    of 0 do
      return(a)
    of 1010 do
      lasterror()
    otherwise
      undefined
  end_case
end_proc:


piecewise::normalGroebner:=
proc(pw)
begin
  pw:= piecewise::mapConditions(pw, property::normalGroebner, args(2..args(0)));
  piecewise::extmap(pw, property::normalGroebner, args(2..args(0)))
end_proc:


// evaluate a condition such that all leaves that produce an error
// become FALSE
// this is necessary as "0=0 or 1/0=nonsense"" should become TRUE
piecewise::evalCondition:=
proc(cond)
  local c;
begin
  case type(cond)
    of "_and" do
    of "_or" do
    of "_not" do
      map(cond, piecewise::evalCondition);
      break
    otherwise
      if traperror((c:= eval(cond))) <> 0 then
        return(FALSE)
      else
        return(c)
      end_if
  end_case
end_proc:
  

// some methods for expression manipulation, realized by mapping

piecewise::expand:=     pw -> piecewise::extmap(pw, expand, args(2..args(0))):
piecewise::combine:=    pw -> piecewise::extmap(pw, combine, args(2..args(0))):
piecewise::rewrite:=    pw -> piecewise::extmap(pw, rewrite, args(2..args(0))):
piecewise::factor:=     pw -> piecewise::extmap(pw, factor):
piecewise::partfrac:=   pw -> piecewise::extmap(pw, partfrac):


piecewise::normal:=
proc(pw: piecewise, options)
  local res;
begin
  if contains({args()}, List) or (args(0)>=2 and type(options) = DOM_TABLE and options[List] = TRUE) then
     res:= piecewise::extmap(pw, normal, args(2..args(0)));
     [piecewise::extmap(res, op, 1), piecewise::extmap(res, op, 2)]
  else
    piecewise::extmap(pw, normal, args(2..args(0)))
  end_if;
end_proc:


/* solve(pw, x, options) - solves the equation pw=0 for x */

piecewise::solve:=
proc(pw:piecewise, x)
  local otherargs;
  save MAXEFFORT;
begin
  if args(0) = 1 then
    return(solvelib::avoidAliasProblem
           (piecewise::extmap(pw, solve), freeIndets(pw))
           )
  end_if;
  otherargs:=args(3..args(0));
  MAXEFFORT:= MAXEFFORT/2/nops(pw);
  if MAXEFFORT < 4000 then
    if type(otherargs) = DOM_TABLE then
       otherargs:= solvelib::nonDefaultOptions(otherargs)
    end_if;
    return(hold(solve)(pw, x, otherargs))
  end_if;
  if not has(map([extop(pw)], op, 1), x) then
    return(
           piecewise::fillCases
           (
            solvelib::avoidAliasProblem
           (piecewise::extmap(pw, solve, x, otherargs),
            freeIndets(pw)),
            {})
           )
  end_if;
  _union(op(map
            ([extop(pw)],
             /* f. every branch, find the set of x that both satisfy the
                given condition and solve the given expression */
             proc(branch)
               name solveBranch;
               save MAXEFFORT;
               local result, cond;
             begin
               userinfo(30, "Solving case ".
                        expr2text(BRANCH::condition(branch)));
               cond:= BRANCH::condition(branch);
               if not has(cond, x) then
                 piecewise([cond, solve(BRANCH::expression(branch), x,
                                        otherargs)],
                           [not cond, {}])
               else
                 MAXEFFORT:= MAXEFFORT/3;
                 result:=
                 solvelib::solve_intersect
                 (
                  solve(BRANCH::condition(branch), x, otherargs),
                  solve(BRANCH::expression(branch), x, otherargs)
                  );
                 piecewise::fillCases(result, {})
               end_if
             end_proc
             )))
end_proc:


/* piecewise::solveConditions

   solve all conditions for the given variable x and reconvert
   the sets of solutions into expressions
*/

piecewise::solveConditions:=
proc(pw:piecewise, x:DOM_IDENT)
begin
  piecewise::mapConditions(pw, cond -> (x in solve(cond,x)))
end_proc:

/*
	discont - determine the discontinuities (the points for which
	          the function is defined on both sides of the point,
		  but the limits from the left/right are not the same
		  or do not agree with the function value at the point
*/

piecewise::discont:=
proc(pw:piecewise, x)
  local discontlist:DOM_SET,
  i:DOM_INT,
  j:DOM_INT,
  k: DOM_INT,
  cond, disc, disc2;

begin
  if args(0)<2 then
    error("At least two arguments expected")
  end_if;

  if type(x)="_equal" then
    if type(op(x, 2))<> "_range" then
      error("Right hand side of second argument must be range")
    else
      return(piecewise::discont(pw, op(x, 1), args(3..args(0)))
             intersect Dom::Interval(op(x, 2)))
    end_if
  elif type(x) <> DOM_IDENT then
    error("Second argument must be identifier or equation")
  end_if;

  discontlist:={};
  for i from 1 to extnops(pw) do
    cond:=piecewise::condition(pw, i);
    // disc: border points of the condition
    disc:= solve(piecewise::borderOfCondition(cond, x),x);
    userinfo(10, "Found border points ".expr2text(disc));
    // disc2: discontinuities of the expression
    disc2:= discont(piecewise::expression(pw, i), x, args(3..args(0)));
    if type(disc2) = DOM_SET then 
      disc2:= select(disc2, u -> piecewise::satisfiesCondition( x, u, cond)=TRUE )
    end_if;  
    discontlist:= discontlist union disc2;
    if type(disc) <> DOM_SET and
      (type(disc) <> piecewise or _lazy_or(type(piecewise::expression(disc, k)) = DOM_SET $k=1..nops(disc))) then
      userinfo(10, "Complicated set of border points, cannot check for overlappings");                             
    else  
      // check for overlappings with other branches: we need not return border points of one condition that are inner points of another
      for j from 1 to extnops(pw) do
        if j<>i then
          userinfo(10, "Checking whether border points are inner points of ".
          output::ordinal(j)." condition");        
          disc:=piecewise::extmap(disc, select,
          u -> not piecewise::satisfiesCondition
          (x, u, piecewise::innerOfCondition
          (piecewise::condition(pw,j))) )
        end_if;
      end_for;
    end_if;  
    discontlist:= discontlist union disc;
  end_for;
  userinfo(5, "Discontinuities are ".expr2text(discontlist));
  discontlist
end_proc:



// isDefined(pw) - return a boolean expression that is equivalent to
// "at least one of the conditions is satisfied"
piecewise::isDefined:=
proc(pw: piecewise)
  local condlist: DOM_LIST;
begin
  condlist:= piecewise::conditions(pw);
  if contains(condlist, Otherwise) > 0 then
    TRUE
  else
    simplify::simplifyCondition(_or(op(condlist)))
  end_if
end_proc:


/*
limit - compute the limit of a piecewise defined object

limit(f, x, lp, dir, options)

the arguments have the same meaning as in limit::main and
limit::accumulationPoints




*/
piecewise::limit :=
proc(f: piecewise, x: DOM_IDENT, lp, dir, options)
  local res, i: DOM_INT, j: DOM_INT, xpr, L,
  cond, maybes,
  FIN: DOM_PROC,
  simplifyInfinities: DOM_PROC,
  limitCondition: DOM_PROC,
  unevaluated: DOM_PROC;
begin
  
  //
  // l o c a l  m e t h o d s
  //
  
  // extract the finite part of a limit::Set or a piecewise consisting
  // of limit::Set
  FIN:= S ->  piecewise::extmap(S, extop, 1);


  // return an unevaluated limit call
  unevaluated:=
  proc()
  begin
    if lp = infinity or lp = -infinity or dir = Real then
      return(hold(limit)(f, x=lp))
    else
      return(hold(limit)(f, x=lp, dir))
    end_if;
  end_proc;

  // simplify conditions that contain objects of type stdlib::Infinity
  // this method can be removed once simplifyCondition has been improved
  simplifyInfinities:=
  proc(cond)
    local l, r;
  begin
    // define abbrevs for equalities/inequalities
    if nops(cond) = 2 then
      [l, r]:= [op(cond)]
    end_if;
    if not has(cond, infinity) then
      return(cond)
    end_if;
    case type(cond)
      of "_and" do
      of "_or" do
      of "_not" do
        return(map(cond, simplifyInfinities))
      of "_less" do
        if has(l, infinity) then
          if has(r, infinity) then
            if type(l) = stdlib::Infinity and type(r) = stdlib::Infinity then
              return(op(l,1) < op(r, 1))
            end_if;
            return(cond)
          else
            return(simplifyInfinities(l = -infinity and r in R_))
          end_if
        else
          assert(has(r, infinity));
          return(simplifyInfinities(r = infinity and l in R_))
        end_if;
      of "_leequal" do
        if has(l, infinity) then
          if has(r, infinity) then
            if type(l) = stdlib::Infinity and type(r) = stdlib::Infinity then
              return(op(l,1) <= op(r, 1))
            end_if;
            return(cond)
          else
            return(simplifyInfinities(l = -infinity and r in R_))
          end_if
        else
          assert(has(r, infinity));
          return(simplifyInfinities(r = infinity and l in R_))
        end_if;
      of "_equal" do
        if not has(l, infinity) or not has(r, infinity) then
          // all of our infinite objects contain infinity syntactically
          // (is this correct?!)
          return(FALSE)
        end_if;
        if type(l) = stdlib::Infinity and type(r) = stdlib::Infinity then
          return(sign(op(l, 1)) = sign(op(r, 1)))
        end_if;
        break
      of "_unequal" do
        return(not simplifyInfinities(l=r))
    end_case;
    cond
  end_proc;
  
  
  // limitCondition(cond) - return a boolean expression cond2
  //  such that for x close enough to lp, cond and cond2 are equivalent
  // cond = cond2 does this,
  // but we try to find a cond2 that is independent of x
  
  limitCondition:=
  proc(cond)
    local lim, lim2, isReal; 
  begin
    // constant condition
    if not has(cond, x) then
      return(simplifyInfinities(cond))
    end_if;

    case type(cond)
      of "_equal" do
        // if zero is not an accumulation point of the difference,
        // then the operands of cond are unequal near the limit point
        // on the other hand, we have no method to detect that the difference
        // has no zeroes if zero is an accumulation point (use solve?)
        lim:= limit::accumulationPoints(op(cond, 1) - op(cond, 2),
                                        x, lp, dir, options);
        if is(not 0 in FIN(lim[1]), Goal = TRUE) then
          return(FALSE)
        else
          return(cond)
        end_if
      of "_unequal" do
        return(not limitCondition(_equal(op(cond))))
      of "_leequal" do
        // simple heuristic: compute both limits
        // if they exist and are unequal, then it suffices to
        // compare them
        // if they may be equal, we do not yet know anything
        lim:= limit(op(cond, 1), x=lp, dir);
        lim2:= limit(op(cond, 2), x=lp, dir);
        if lim <> undefined and lim2 <> undefined and not
          hastype({lim, lim2}, "limit") and
          (map({lim, lim2}, has, infinity) = {TRUE, FALSE} or
           is(lim <> lim2, Goal = TRUE)) then
          cond:= simplify::simplifyCondition(piecewise::_leequal(lim, lim2));
          // as simplifyCondition does not enough at the moment, we need
          // another call to simplifyInfinities
          cond:= simplifyInfinities(cond);
          return(cond)
        end_if;

        // if the simple heuristic fails, then we check whether
        // both sides are real and apply limit to the sign of the difference
        isReal:= op(cond, 1) in R_ and op(cond, 2) in R_;
        lim:= limit::accumulationPoints
        (sign(op(cond, 1) - op(cond, 2)),
         x, lp, dir, options) assumingAlso isReal;
        case is(1 in FIN(lim[1]))
          of TRUE do
            // sometimes op(cond, 1) > op(cond, 2) near the limit point
            if is(not -1 in FIN(lim[1]), Goal = TRUE) then
              if is(not 0 in FIN(lim[1]), Goal = TRUE) then
                // always op(cond, 1) > op(cond, 2) 
                return(FALSE)
              else
                // both sides may be equal arbitrarily near to the limit point
                return(limitCondition(op(cond, 1) = op(cond, 2)) and isReal)
              end_if
            else
              break
            end_if
          of FALSE do
            // the difference op(cond, 1) - op(cond, 2) is always <= 0.
            // check that the operands are real
            return(limitCondition(isReal))
          otherwise
            break
        end_case;
        break
      of "_less" do
        return(limitCondition(op(cond, 1) <= op(cond, 2) and
                              op(cond, 1) <> op(cond, 2)))
      of "_in" do
        return(cond)
      of "_and" do
      of "_or" do
      of "_not" do
        return(map(cond, limitCondition))
    end_case;

    cond
  end_proc;

  //
  // m a i n   p r o c e d u r e
  // 
  
  // cases handled so far:
  res:= undefined;

  // cases that apply for maybe some but not all x so far:
  maybes:= undefined;
  
  for i from 1 to extnops(f) do
    cond:= piecewise::condition(f, i);
    if cond = Otherwise then
      assert(i = extnops(f));
      cond:= not (_or(piecewise::condition(f, j) $j=1..i-1))
    end_if;
    xpr:= piecewise::expression(f, i);

    if traperror((
                  L:= limit::main(xpr, x, lp, dir, options) assumingAlso cond
                  )) <> 0 then
      next
    end_if;
    
    cond:= limitCondition(cond);
    if not has(cond, x) then
      // the condition does not depend on x;
      // if it is true, L is the correct result
      res:= piecewise::insert
      (res, [simplify::simplifyCondition(cond), L]);
      if type(res) <> piecewise and res <> undefined then
        return(res)
      end_if;
    else
      if options[Intervals] then
        // add all accumulation points, even if they occur only under
        // some condition on the free parameters
        res:= res union L
      else
        // add this branch to those branches that may be true somewhere
        // near the limit point
        maybes:= piecewise::insert(maybes,
                                   [cond, L]
                                   )
      end_if;
      if type(maybes) <> piecewise and maybes <> undefined then
        // the cases that may happen cover everything, and give a result
        return(maybes)
      end_if
    end_if
  
  end_for;

  if maybes = undefined then
    res
  else
    unevaluated()
  end_if
end_proc:



piecewise::indefint :=
proc(pw:piecewise, x:DOM_IDENT)
begin
  piecewise::extmap(pw, int, x, args(3..args(0)))
end_proc:

piecewise::defint :=
proc(pw:piecewise, eq: "_equal")
  local x, a,b, i: DOM_INT, S, J, cond, res,
  newpw: piecewise, // the rest which cannot be integrated
  covered,         // the subset of [a,b] already covered;
  integral,        // int_{ x \in covered} pw(x) dx
  defint_b_gr_a: DOM_PROC, // local procedure
  options;         // pass through to int

begin

  options := args(3..args(0));

  // local method
  // performs the operation assuming in addition that a <= b

  defint_b_gr_a:=
  proc(a, b)
    local j, isFinite;
    begin

      J:= Dom::Interval([a,b]);
      // we just have to compute intOverSet(pw, x, [a, b])
      // since a <= b


      // and the subset already integrated is empty
      // the integral is zero

      covered:={};
      integral:=0;
      newpw:=[];

      for i from 1 to extnops(pw) do
        cond:= piecewise::condition(pw,i);
        if cond = hold(Otherwise) then
          assert(i = extnops(pw));
          cond:= not _or(piecewise::condition(pw, j) $j=1..i-1)
        end_if;
        S:=Simplify(solve(cond, x) intersect
            (J minus covered));
        userinfo(20, "Condition ".expr2text(cond)." satisfied over ".
                 expr2text(S));

        res:=intlib::intOverSet(piecewise::expression(pw, i), x, S, options)
          assuming cond;

        if type(res)<> "intOverSet" then
          userinfo(20, "Integral over ".expr2text(S)." is ".
                   expr2text(res));
          covered:=covered union S;
          integral:=integral+ res;
        else
          newpw:=append(newpw, extop(pw, i))
        end_if;

        isFinite:= solvelib::isFinite(J minus covered);
        if isFinite = TRUE then
          return(integral)
        end_if;
      end_for;
    if extnops(newpw)>0 then
      newpw:= new(piecewise, extop(newpw));
      integral + hold(intlib::intOverSet)(newpw, x, J minus covered, options)
    else
      assert(isFinite <> TRUE); // otherwise, we would have returned integral above
      if isFinite = FALSE then  
        // parts of [a,b] were not covered because func is not defined there
        // we could also error here
        userinfo(10, "Integral not everywhere defined on the integration interval");
        undefined
      else 
        // we do not know whether we have covered everything
        warning("Could not check whether the integrand is defined everywhere on the integration interval");
        integral
      end_if;
    end_if;

  end_proc;


  // end of local method


  x:=op(eq,1);
  if type(x) <> DOM_IDENT then
    error("First argument must be identifier")
  end_if;
  if type(op(eq,2)) <> "_range" then
    error("Right hand side of second argument must be range")
  end_if;
  a:=op(eq,[2,1]);
  b:=op(eq,[2,2]);

  if a=b then
    return(0)
  end_if;

  return(piecewise([a <= b, defint_b_gr_a(a, b)],
                   [b <= a, -defint_b_gr_a(b, a)]))
end_proc:

piecewise::int :=
proc(pw:piecewise, x)
begin
  if type(x)="_equal" then
    piecewise::defint(args())
  elif type(x)=DOM_IDENT then
    piecewise::indefint(args())
  end_if
end_proc:

piecewise::intOverSet :=
proc(xpr, x, pw:piecewise)
begin
  // the conditions of pw must *not* depend on x
  piecewise::extmap(pw, S -> intlib::intOverSet(xpr, x, S, args(4..args(0))))
end_proc:


piecewise::laplace:=
proc(pw)
local i;
begin
  if has([piecewise::condition(pw, i) $ i = 1..extnops(pw)], args(2)) then
     return(hold(transform::laplace)(pw, args(2..args(0))));
  end_if:;
  piecewise::extmap(pw, transform::laplace, args(2..args(0)))
end_proc:

piecewise::invlaplace:=
proc(pw)
local i;
begin
  if has([piecewise::condition(pw, i) $ i = 1..extnops(pw)], args(2)) then
     return(hold(transform::invlaplace)(pw, args(2..args(0))));
  end_if:;
  piecewise::extmap(pw, transform::invlaplace, args(2..args(0)))
end_proc:


piecewise::fourier:=
proc(pw)
local i;
begin
  if has([piecewise::condition(pw, i) $ i = 1..extnops(pw)], args(2)) then
     return(hold(transform::fourier)(pw, args(2..args(0))));
  end_if:;
  piecewise::extmap(pw, transform::fourier, args(2..args(0)))
end_proc:


piecewise::invfourier:=
proc(pw)
local i;
begin
  if has([piecewise::condition(pw, i) $ i = 1..extnops(pw)], args(2)) then
     return(1/(2*PI)*hold(transform::fourier)(pw, args(2), -args(3), args(4..args(0))));
  end_if:;
  piecewise::extmap(pw, transform::invfourier, args(2..args(0)))
end_proc:


piecewise::ztrans:=
proc(pw)
local i;
begin
  if has([piecewise::condition(pw, i) $ i = 1..extnops(pw)], args(2)) then
     return(hold(transform::ztrans)(pw, args(2..args(0))));
  end_if:;
  piecewise::extmap(pw, transform::ztrans, args(2..args(0)))
end_proc:


piecewise::invztrans:=
proc(pw)
local i;
begin
  if has([piecewise::condition(pw, i) $ i = 1..extnops(pw)], args(2)) then
     return(hold(transform::invztrans)(pw, args(2..args(0))));
  end_if:;
  piecewise::extmap(pw, transform::invztrans, args(2..args(0)))
end_proc:

piecewise::diff :=
proc(pw: piecewise, x)
begin
case args(0)
  of 1 do return(pw)
  of 2 do
	return(piecewise::extmap(piecewise::mapConditions
                         (pw, piecewise::innerOfCondition, {x}), diff, x))
  otherwise
    diff(piecewise::diff(pw, x), args(3..args(0)) )
end_case
end_proc:

//------------------------------------------------
// float method added by Walter, 24.2.03:
// - The float method needs to map float both to
//   the expressions as well as to the conditions
// - Note that float can have a second argument
//   (the precision). For speed, do not bother
//   with type checking
piecewise::float :=
proc(pw)
  local i;
begin
  piecewise::new(float(op(pw, i), args(2..args(0))) $i=1..nops(pw))
end_proc:
//------------------------------------------------

piecewise::contains:=
proc(pw:piecewise, x)
begin
  if args(0)<>2 then
    error("Wrong number of arguments")
  end_if;
  piecewise::extmap(pw, contains, x)
end_proc:

piecewise::has:=
proc(pw:piecewise, x)
begin
  has([extop(pw)], x)
end_proc:


piecewise::hastype:=
proc(pw: piecewise)
  local i: DOM_INT;
begin
  _lazy_or(hastype(extop(pw, i), args(2..args(0))) $i=1..extnops(pw)) or
  bool(args(2) = piecewise
  or _lazy_and(type(args(2)) = DOM_SET, contains(args(2), piecewise)))
end_proc:

// overload _in::expand

piecewise::_in:=
proc(pw:piecewise, S)
  local i;
begin
  if args(0)<>2 then
    error("Wrong number of arguments")
  end_if;
  _or((piecewise::condition(pw,i) and piecewise::expression(pw,i) in S )
      $i=1..extnops(pw))
end_proc:


piecewise::freeIndets:=
proc(pw)
  local i;
begin
  _union(freeIndets(piecewise::branch(pw, i), args(2..args(0))) $i=1..nops(pw)) minus {Otherwise}
end_proc:


// do not allow conversion to poly
piecewise::poly:= x -> FAIL:


piecewise::set2expr:=
proc(pw:piecewise, x)
  local i;
begin
  if args(0)<>2 then
    error("Wrong number of arguments")
  end_if;
  _or((piecewise::condition(pw,i) and x in piecewise::expression(pw,i))
      $i=1..extnops(pw))
end_proc:


// overload solvelib::getElement
// return FAIL if some conditions cannot be removed

piecewise::getElement:=
proc(pw:piecewise)
  local i, S;
begin
  S:= _intersect(piecewise::expression(pw, i) $i=1..extnops(pw));
  if type(S) = piecewise then
     FAIL
  else
     solvelib::getElement(S, args(2..args(0)))
  end_if;
end_proc:

piecewise::isFinite:=
proc(pw:piecewise):DOM_BOOL
  local i;
begin
  {solvelib::isFinite(piecewise::expression(pw, i)) $i=1..extnops(pw)};
  // is the result independent of the conditions ?
  if nops(%)=1 then
    op(%,1)
  else
    UNKNOWN
  end_if
end_proc:

piecewise::PWif := funcenv(hold(slot)(piecewise, "PWif"),
                           builtin(1097, output::Priority::Stmt,
                                   " if ", "piecewise::PWif")):
piecewise::PWotherwise := funcenv(hold(slot)(piecewise, "PWotherwise"),
                           builtin(1099, output::Priority::Stmt,
                                   " otherwise", "piecewise::PWotherwise")):
piecewise::print :=
proc(pw)
  local i, ind, nargs, pieces, other;
begin
  pieces := [extop(pw)];
  ind := contains(map(pieces, op, 1), Otherwise);
  if ind = 0 then
    other := null();
  else
    other := generate::sortSums(BRANCH::expression(pieces[ind]));
    delete pieces[ind];
    pw := new(dom, op(pieces));
  end_if;

  if PRETTYPRINT=FALSE then
    hold(piecewise)([generate::sortSums(piecewise::condition(pw,i)),
                     generate::sortSums(piecewise::expression(pw, i))]
                    $i=1..extnops(pw),
                    if other <> null() then [Otherwise, other]
                    else null() end)
  else
    nargs := extnops(pw);
    hold(piecewise)(subsop(piecewise::PWif
                           (hold(dummy),
                            generate::sortSums(piecewise::condition(pw, i))),
                           1 = generate::sortSums(piecewise::expression(pw, i)),
                           Unsimplified)
                    $i = 1..nargs,
                    if other <> null() then
                      piecewise::PWotherwise(other)
                    else null() end)
  end_if;
end_proc:


piecewise::TeX :=
proc(pw)
  local i;
begin
  "\\left\\{\n\\begin{array}{ccl}\n".
  _concat((generate::TeX(piecewise::expression(pw,i)).
          if piecewise::condition(pw, i) = Otherwise then
             "& \\text{\\rlap{otherwise}} &"
          else
             " & \\text{if} & ".
             generate::TeX(piecewise::condition(pw, i))
          end_if."\\\\\n") $i=1..extnops(pw)).
  "\\end{array}\n\\right."
end_proc:

// Typesetting output of piecewise
piecewise::Content :=
proc(Out, pw)
  local i, pieces, other;
begin
  pieces := [extop(pw)];
  i := contains(map(pieces, op, 1), Otherwise);
  if i = 0 then
    other := null()
  else
    other := Out::Cotherwise(Out(BRANCH::expression(pieces[i])));
    delete pieces[i];
    if nops(pieces) = 0 then
      // should not happen and would lead to a broken typeset output
      error("empty piecewise not expected");
    end;
  end_if;

  Out::Cpiecewise(Out::Cpiece(Out(BRANCH::expression(i)),
                              Out(BRANCH::condition(i)))
                  $ i in pieces,
                  other)
end_proc:

piecewise::maprec := pw -> piecewise::extmap(piecewise::mapConditions(pw,
    misc::maprec, args(2..args(0))),
  misc::maprec, args(2..args(0))):
