/* -----------------------------------------------------------
Calls:
   plotfunc2d(f1, f2, .. , <x = xmin .. xmax>, <a = amin .. amax>, 
              <Colors = [c1, c2, ..]>, <attributes>)

Parameters:
  f1, f2, ... : univariate functions: expressions in x and a
                or procedures accepting 1 parameter x or 2 paramters x, a
  x           : identifier or indexed identifier
  xmin, xmax  : real numerical values or expressions in the 
                animation parameter a.
                Default range is x = -5 .. 5
  a           : the animation parameter: indexed identifier
  amin, amax  : real numerical values
  c1, c2      : RGB or RGBa values. The colors are used cyclically
  attributes  : any plot attributes accepted by
                Canvas, Scene2d, CoordinateSystem2d or Function2d
          
Details: 
  *) plotfunc2d plots plot::Function2d(f.i), i = 1, 2, 
     in one graphical scene.   
  *) Some defaults of plot::Function2d are changed.
     plotfunc2d uses the defaults
           DiscontinuitySearch = TRUE
           AdaptiveMesh = 2
     The default Colors are 
           [Red,ForestGreen,Blue,Black,MuPADGold,Orange]
  *) Legends are set automatically
       
Examples:
>> plotfunc2d(x, x^2, sin(x), x=-3..3):
>> plotfunc2d(a*x, a*x^2, x = 0..1, a = 0..1)
>> plotfunc2d(x, x^2, x^3, Colors = [RGB::Red, RGB::Green, RGB::Blue])
>> plotfunc2d(x, x^2, x^3, Axes = Box, BackgroundColor = RGB::Black)
>> plotfunc2d(x, x^2, x^3, GridVisible = TRUE)
>> plotfunc2d(x, x^2, x^3, ViewingBoxYRange = -1 .. 1)
------------------------------------------------------------------------*/

plotfunc2d:= proc()
local object, other, attributes, ranges,
      adaptive, discont, xaxistitle, attrNames, legendvisible,
      functions, special_attributes, dummy, tmp, range,
      n, colors, eq, ncolors, f, i, Args, colorsdummy, outputArgs,
      Function2dAttributes, CanvasAndScene2dAttributes;
begin
  //---------------------------------------------------
  // set the defaults:
  //---------------------------------------------------
  adaptive:= AdaptiveMesh = 2;          //default for plotfunc2d
  discont:= DiscontinuitySearch = TRUE; //default for plotfunc2d
  legendvisible:= LegendVisible = TRUE; //default for plotfunc2d
/*
  colors:= [                            //default color list for plotfunc2d
            RGB::Red,      
            RGB::ForestGreen, 
            RGB::Blue, 
            RGB::Black, 
            RGB::MuPADGold, 
            RGB::Orange
           ]:
*/
  colors:= RGB::ColorList[1..10]:
  // xaxistitle is set further down below
  //---------------------------------------------------
  // split into standard graphical attributes versus
  // functions and special attributes (Colors)
  //---------------------------------------------------

  //---------------------------------------------------
  // kick out Colors for plot::Function2d::checkArgs
  //---------------------------------------------------
  [colorsdummy, Args, dummy] := split([args()],
                                      x -> type(x)="_equal" and
                                      op(x,1) = Colors):
  for eq in colorsdummy do
      if op(eq,1) = Colors and 
         domtype(op(eq, 2)) = DOM_LIST and
         map({op(op(eq, 2))}, domtype) = {DOM_LIST} then
         colors:= op(eq, 2):
         next;
      end_if;
      error("unexpected argument: ".expr2text(eq));
  end_for:

  //---------------------------------------------------
  // Allow YRange = ymin .. ymax as a short hand notation
  // for the clumsy ViewingBoxYRange in plotfunc2d:
  //---------------------------------------------------
  Args:= subs(Args, YRange = ViewingBoxYRange,
                    YMin = ViewingBoxYMin,
                    YMax = ViewingBoxYMax):

  //---------------------------------------------------
  // kick out Export options for plot::Function2d::checkArgs
  //---------------------------------------------------
  [outputArgs, Args, dummy] := split(Args,
                                      x -> type(x)="_equal" and
                                      (op(x,1) = OutputFile or
                                       op(x,1) = OutputOptions)):

  object:= plot::Function2d::checkArgs(["X"], op(Args)):
  other := object::other;
  //---------------------------------------------------
  // pick out the functions
  //---------------------------------------------------
  [functions, special_attributes, dummy]:= split(other, 
         x -> not (
                (type(x) = "_equal") or
                (domtype(x) = plot::StyleSheetEntry) or
                contains(plot::Function2d::allKnownAttributes, x)
               )); 
  //---------------------------------------------------
  // special_attributes are additional attributes for
  // plotfunc2d that plot::Function2d does not have.
  // Presently, there are none.
  //---------------------------------------------------
  if nops(special_attributes) <> 0 then
     error("unexpected argument: ".expr2text(special_attributes[1]));
  end_if:
  //---------------------------------------------------
  // Split the arguments into 'ranges' versus 'other attributes'.
  // The ordering of the range attributes is important
  // (x range versus animation range). Treat the ranges
  // separately before converting the lists to sets.
  //---------------------------------------------------
  attributes:= Args;
  ranges:= select(attributes, 
                  x -> _lazy_and(type(x) = "_equal",
                                 type(op(x, 2)) = "_range",
                                 not contains(plot::getOptionNames(), op(x, 1)
                                 )));
  attributes:= {op(attributes)} minus {op(ranges)};
  attributes:= attributes minus {op(functions)};
  attributes:= attributes minus {op(colorsdummy)};
  // attributes:= attributes minus {op(special_attributes)};
  //---------------------------------------------------
  // pick out some special attributes that shall have a
  // different default than plot::Function2d
  //---------------------------------------------------
  attrNames:= map(attributes, op, 1); 
  if contains(attrNames, AdaptiveMesh) then
       adaptive:= null();
  end_if;
  if contains(attrNames, DiscontinuitySearch) then
       discont:= null();
  end_if;
  if contains(attrNames, XAxisTitle) then
       xaxistitle:= null();
  elif nops(ranges) <> 0 then
       xaxistitle:= XAxisTitle = expr2text(op(ranges[1], 1));
  else xaxistitle:= null();
  end_if;
  if contains(attrNames, LegendVisible) then
       legendvisible:= null();
  end_if;
  //---------------------------------------------------
  // attributes = a sequence of standard graphical 
  // attributes including hints, not including 
  // functions, ranges, special_attributes
  //---------------------------------------------------
  attributes:= op(attributes);  
  //---------------------------------------------------
  // split 'attributes' in 'Function2dAttributes' versus
  // 'CanvasAndScene2dAttributes'
  //---------------------------------------------------
  [Function2dAttributes, CanvasAndScene2dAttributes, dummy]:= 
        split([attributes], x -> slot(plot::Function2d, expr2text(op(x, 1))) <> FAIL):

  n:= nops(functions);
  if n = 0 then
     // return an empty scene
    return(plot(op(outputArgs), plot::Scene2d(plot::CoordinateSystem2d(op(CanvasAndScene2dAttributes)))));
  end_if;
  if n = 1 then
       legendvisible:= null();
  end_if;

  ncolors:= nops(colors);
  //---------------------------------------------------
  // check consistency of parameter names
  //---------------------------------------------------
  tmp := numeric::indets(functions): 
  if nops(tmp) > 2 then
     error("too many symbolic parameters in the function(s). ".
           "Got: ".expr2text(numeric::indets(functions)));
  end_if;
  if nops(tmp) = 2 and 
     tmp minus {op(range, 1) $ range in ranges} <> {} then
     error("too many symbolic parameters in the function(s). ".
           "Got: ".expr2text(numeric::indets(functions)));
  end_if;

/*
  //---------------------------------------------------------
  // check if the user entered an animation range by mistake:
  //---------------------------------------------------------
  if nops(ranges) = 2 then
     a := op(ranges, [2, 1]):
     if (not has(functions, a)) and
        (not hastype(functions, DOM_PROC)) and
        (not has(op(ranges, 1), a)) then
        warning("you have entered an animation range ".expr2text(op(ranges, 2)).
                ", but the animation parameter ".expr2text(a). 
                " does neither turn up in the function(s) nor in the plot range.".
                " Did you wish to specify a vertical viewing range?".
                " Use YRange = ".expr2text(op(ranges, [2, 2])).
                " or ViewingBoxYRange = ".  expr2text(op(ranges, [2, 2]))
               );
      end_if;
  end_if;
*/
                
  //---------------------------------------------------
  // Create various plot::Function2d objects
  //---------------------------------------------------
  f:= [0 $ n]:
  for i from 1 to n do
      f[i]:= plot::Function2d(functions[i], 
                              op(ranges),
                              LineColor = colors[((i - 1) mod ncolors) + 1],
                              adaptive,
                              discont,
                              op(Function2dAttributes),
                              LegendText=expr2text(functions[i]),
                              LegendEntry=TRUE
                             );
  end_for:
  plot(op(outputArgs), op(f), xaxistitle, legendvisible, 
       op(CanvasAndScene2dAttributes));
end_proc:
