/* -----------------------------------------------------------
Calls:
   plotfunc3d(f1, f2, .. , <x = xmin .. xmax>, <y = ymin .. ymax>, 
              <a = amin .. amax>, <Colors = [c1, c2, ..]>, <attributes>)

Parameters:
  f1, f2, ... : bivariate functions: expressions in x, y and a
                or procedures accepting 2 parameters x, y or 
                3 paramters x, z, a
  x           : identifier or indexed identifier
  xmin, xmax  : real numerical values or expressions in the 
                animation parameter a.
                Default range is x = -5 .. 5
  y           : identifier or indexed identifier
  ymin, ymax  : real numerical values or expressions in the 
                animation parameter a.
                Default range is x = -5 .. 5
  a           : the animation parameter: indexed identifier
  amin, amax  : real numerical values
  c1, c2      : RGB or RGBa values. The colors are used cyclically
  attributes  : any plot attributes accepted by
                Canvas, Scene3d, CoordinateSystem3d or Function3d
          
Details: 
  *) plotfunc3d plots plot::Function3d(f.i), i = 1, 2, 
     in one graphical scene.   
  *) Some defaults of plot::Function3d are changed.
     The default Colors are 
           [Red,ForestGreen,Blue,Black,MuPADGold,Orange]
  *) Legends are set automatically
       
Examples:
>> plotfunc3d(x, y*x^2, sin(x - y), x=-3..3, y = 2..3):
>> plotfunc3d(a*x*y, a*x^2 + y, x = 0..1, y = 2..3, a = 0..1)
>> plotfunc3d(x*y, x^2*y^2, x^3 - y, Colors = [RGB::Red, RGB::Green, RGB::Blue])
>> plotfunc3d(y + x^2, y*x^3, Axes = Box, BackgroundColor = RGB::Black)
>> plotfunc3d(y + x^2, y*x^3, GridVisible = TRUE)
>> plotfunc3d(y + x^2, y*x^3, ViewingBoxZRange = -1 .. 1)
------------------------------------------------------------------------*/

plotfunc3d:= proc()
local object, other, attributes, ranges, attrNames,
      xaxistitle, yaxistitle, legendvisible,  //  adaptive, discont,
      functions, special_attributes, dummy, n, tmp, range,
      colors, eq, ncolors, f, i, Args, colorsdummy, outputArgs,
      Function3dAttributes, CanvasAndScene3dAttributes;
begin
  //---------------------------------------------------
  // set the defaults:
  //---------------------------------------------------
  // adaptive:= AdaptiveMesh = 2
  // discont:= DiscontinuitySearch = TRUE
  legendvisible:= LegendVisible = TRUE; //default for plotfunc3d
/*
  colors:= [                            //default color list for plotfunc3d
            RGB::Red, 
            RGB::ForestGreen, 
            RGB::Blue, 
            RGB::Black, 
            RGB::MuPADGold, 
            RGB::Orange
           ]:
*/

  // For consistency with plot::Function3d, we want
  // FillColor = RGB::Red for the first object.
  // Start with the red RGB::ColorList entry, then
  // proceed according to this list for further
  // objects.
  for i from 1 to nops(RGB::ColorList) do
      if RGB::ColorList[i] = RGB::Red then
         break;
      end_if:
  end_for:
  if i <= nops(RGB::ColorList) then
     colors := RGB::ColorList[i .. max(i + 9, nops(RGB::ColorList))]:
  else
     colors := RGB::ColorList[1 .. 10];
  end_if:

  // axes titles are set further down below
  //---------------------------------------------------
  // split into standard graphical attributes versus
  // functions and special attributes (Colors)
  //---------------------------------------------------

  //---------------------------------------------------
  // kick out Colors for plot::Function3d::checkArgs
  //---------------------------------------------------
  [colorsdummy, Args, dummy] := split([args()],
                                      x -> type(x)="_equal" and
                                      op(x,1) = Colors):
  for eq in colorsdummy do
      if op(eq,1) = Colors and 
         domtype(op(eq, 2)) = DOM_LIST and
         map({op(op(eq, 2))}, domtype) = {DOM_LIST} then
         colors:= op(eq, 2):
         next;
      end_if;
      error("unexpected argument: ".expr2text(eq));
  end_for:

  //---------------------------------------------------
  // Allow ZRange = zmin .. zmax as a short hand notation
  // for the clumsy ViewingBoxZRange in plotfunc3d:
  //---------------------------------------------------
  Args:= subs(Args, ZRange = ViewingBoxZRange,
                    ZMin = ViewingBoxZMin,
                    ZMax = ViewingBoxZMax):

  //---------------------------------------------------
   // kick out Export options for plot::Function3d::checkArgs
  //---------------------------------------------------
  [outputArgs, Args, dummy] := split(Args,
                                      x -> (type(x)="_equal" and
                                      (op(x,1) = OutputFile or
                                       op(x,1) = OutputOptions or
                                       op(x,1) = CameraDirection or
                                       op(x,1) = OrthogonalProjection))
                                       or x = OrthogonalProjection):
  // note: OrthogonalProjection = TRUE may also be just OrthogonalProjection
 
  object:= plot::Function3d::checkArgs(["X", "Y"], op(Args)):
  other := object::other;
  //---------------------------------------------------
  // pick out the functions
  //---------------------------------------------------
  [functions, special_attributes, dummy]:= split(other,
         x -> not (
                (type(x) = "_equal") or
                (domtype(x) = plot::StyleSheetEntry) or
                contains(plot::Function3d::allKnownAttributes, x)
               ));
  //---------------------------------------------------
  // special_attributes are additional attributes for
  // plotfunc3d that plot::Function3d does not have.
  // Presently, there are none.
  //---------------------------------------------------
  if nops(special_attributes) <> 0 then
     error("unexpected argument: ".expr2text(special_attributes[1]));
  end_if:
  //---------------------------------------------------
  // Split the arguments into 'ranges' versus 'other attributes'.
  // The ordering of the range attributes is important
  // (x range versus animation range). Treat the ranges
  // separately before converting the lists to sets.
  //---------------------------------------------------
  attributes:= Args;
  ranges:= select(attributes,
                  x -> _lazy_and(type(x) = "_equal",
                                 type(op(x, 2)) = "_range",
                                 not contains(plot::getOptionNames(), op(x, 1)
                                 )));
  attributes:= {op(attributes)} minus {op(ranges)};
  attributes:= attributes minus {op(functions)};
  attributes:= attributes minus {op(colorsdummy)};
  // attributes:= attributes minus {op(special_attributes)};
  //---------------------------------------------------
  // pick out some special attributes that shall have a
  // different default than plot::Function3d
  //---------------------------------------------------
  attrNames:= map(attributes, op, 1); 
/*
  if contains(attrNames, AdaptiveMesh) then
       adaptive:= null();
  end_if;
  if contains(attrNames, DiscontinuitySearch) then
       discont:= null();
  end_if;
*/
  if contains(attrNames, XAxisTitle) then
       xaxistitle:= null();
  elif nops(ranges) <> 0 then
       xaxistitle:= XAxisTitle = expr2text(op(ranges[1], 1));
  else xaxistitle:= null():
  end_if;
  if contains(attrNames, YAxisTitle) then
       yaxistitle:= null();
  elif nops(ranges) > 1 then
       yaxistitle:= YAxisTitle = expr2text(op(ranges[2], 1));
  else yaxistitle:= null():
  end_if;
  if contains(attrNames, LegendVisible) then
       legendvisible:= null();
  end_if;
  
  //---------------------------------------------------
  // attributes = a sequence of standard graphical 
  // attributes including hints, not including 
  // functions, ranges, special_attributes
  //---------------------------------------------------
  attributes:= op(attributes);  
  
  //---------------------------------------------------
  // split 'attributes' in 'Function3dAttributes' versus
  // 'CanvasAndScene3dAttributes'
  //---------------------------------------------------
  [Function3dAttributes, CanvasAndScene3dAttributes, dummy]:=
        split([attributes], x -> slot(plot::Function3d, expr2text(op(x, 1))) <> FAIL):

  n:= nops(functions);
  if n = 0 then
     // return an empty scene
     return(plot(op(outputArgs), plot::Scene3d(plot::CoordinateSystem3d(op(CanvasAndScene3dAttributes)))));
  end_if;
  if n = 1 then
       legendvisible:= null();
  end_if;

  ncolors:= nops(colors);
  //---------------------------------------------------
  // check consistency of parameter names
  //---------------------------------------------------
  tmp := numeric::indets(functions): 
  if nops(tmp) > 3 then
     error("too many symbolic parameters in the function(s). ".
           "Got: ".expr2text(numeric::indets(functions)));
  end_if;
  if nops(tmp) = 3 and 
     tmp minus {op(range, 1) $ range in ranges} <> {} then
     error("too many symbolic parameters in the function(s). ".
           "Got: ".expr2text(numeric::indets(functions)));
  end_if;

/*
  //---------------------------------------------------------
  // check if the user entered an animation range by mistake:
  //---------------------------------------------------------
  if nops(ranges) = 3 then
     a := op(ranges, [3, 1]):
     if (not has(functions, a)) and
        (not hastype(functions, DOM_PROC)) and
        (not has(op(ranges, 1), a)) and
        (not has(op(ranges, 2), a)) then
        warning("you have entered an animation range ".expr2text(op(ranges, 3)).
                ", but the animation parameter ".expr2text(a).
                " does neither turn up in the function(s) nor in the plot range.".
                " Did you wish to specify a viewing range in the z-direction?".
                " Use ZRange = ".expr2text(op(ranges, [3, 2])).
                " or ViewingBoxZRange = ".  expr2text(op(ranges, [3, 2]))
               );
      end_if;
  end_if;
*/

  //---------------------------------------------------
  // Create various plot::Function3d objects
  //---------------------------------------------------
  f:= [0 $ n]:
  for i from 1 to n do
      f[i]:= plot::Function3d(functions[i], 
                              op(ranges),
                              FillColor = colors[((i - 1) mod ncolors) + 1],
                          //  adaptive,
                          //  discont,
                              op(Function3dAttributes),
                              LegendText=expr2text(functions[i]),
                              LegendEntry=TRUE
                             );
  end_for:
  plot(op(outputArgs), op(f), xaxistitle, yaxistitle, legendvisible,
       op(CanvasAndScene3dAttributes));
end_proc:
