


/********************************************************************
rationalize(e, options)

e - any object
options - any of the following:
  FindRelations = types
  DescendInto = types
  StopOn = types
  Recursive  
  ShowDependencies
  Prefix = ..
  MinimalPolynomials
  StopOnConstants
  ReplaceTypes = types
  ReplaceHardToEval
  ApproximateFloats

Returns a sequence
  y, subst [, minpolys] 
where 
  - y is a rational function; or a set of list of such if e is a set or list 
  - subst is a (possibly empty) set of equations (substitutions)
  - minpolys is a set of expressions all of which represent zero (this only occurs if MinimalPolynomials was given)


For compatibility reasons, the old syntax is still accepted:
rationalize(e, l, s, _t) is the same as rationalize(e, DescendInto = l, StopOn = s, FindRelations = [op(_t)])

*********************************************************************/


rationalize:=
proc(e)
  local options, argv: DOM_LIST, opt, i, subst, dep, relations, replaced, repTypes, oldvars, minpolys, minpolysToFind;
begin
  if args(0) = 0 then
     error("rationalize called without arguments")
  end_if;

  argv:= [args()];
  
  // check whether the old or the new calling syntax is used
  if args(0) >= 2 and contains({DOM_NIL, DOM_FAIL, DOM_PROC, DOM_SET}, type(args(2))) then
    // old syntax
    // remove this code block when going from warning to error 
    warning("This calling syntax is obsolete. Instead of rationalize(x, t1, t2, types), please enter ".
    "rationalize(x, DescendInto = t1, StopOn = t2, FindRelations = [op(types)]). See the help page for details.");
    
    options:= rationalize::defaultOptions;
    argv:= [args(2..args(0))];
    for opt in {Recursive, ShowDependencies} do 
      if (i:= contains(argv, opt)) > 0 then
         options[opt] := TRUE;
         delete argv[i]
      end_if;  
    end_for;
    
    case nops(argv)
      of 3 do 
        if not testtype(argv[3], DOM_SET) then
          error("Illegal argument")
        end_if;
        options[FindRelations] := [op(argv[3])];
        // fall through
      of 2 do
        if argv[2] <> NIL and argv[2] <> FAIL then
          options[StopOn] := argv[2]
        end_if;
        // fall through
      of 1 do 
        if argv[1] <> NIL and argv[1] <> FAIL then 
          options[DescendInto] := argv[1]
        end_if;  
        // fall through
      of 0 do
        break
      otherwise
        error("Too many arguments")
    end_case;
    
  elif args(0) >=2 and type(args(2)) = DOM_TABLE then
    options:= args(2)
  else
    // new syntax
    options:= prog::getOptions(2, [args()], rationalize::defaultOptions,
                                 TRUE, rationalize::optionTypes)[1];
  end_if;
  
  if options[ReplaceHardToEval] then
    repTypes:= options[ReplaceTypes];
    if type(repTypes) = DOM_SET then
      options[ReplaceTypes]:= repTypes union rationalize::hardToEval
    else
      options[ReplaceTypes]:= X -> _lazy_or(repTypes(X), contains(rationalize::hardToEval, type(X)))
    end_if
  end_if;  
  
  if options[StopOnConstants] then
    if type(options[StopOn]) = DOM_SET then
      options[StopOn]:= options[StopOn] union {Type::Constant}
    else
      options[StopOn]:= options[StopOn] or (x -> testtype(x, Type::Constant))
    end_if
  end_if;
  
  
  if options[ReplaceTypes] <> {} then
    [e, replaced]:= rationalize::replaceTypes(e, options)
  else
    replaced:= {}
  end_if;  
  
  if options[ApproximateFloats] then
    e:= rationalize::approximateFloats(e)
  end_if;  
  
  // the main work
  [e, subst]:= rationalize::descend(e, options);
  
  if options[FindRelations] <> [] then
    relations:= rationalize::findRelations(subst, options);
    e:= subs(e, relations[1]); 
    oldvars:= map(relations[1], op, 1);
    // let only those substitutions survive where the left hand sides have not been replaced ...
    subst:= select(subst, eq -> not contains(oldvars, op(eq, 1)));
    
    // now add the newly found substitutions
    subst:= subst union relations[2];
    if contains(options[FindRelations], "_power") > 0 then 
    // if the existing vars in e can be expressed in terms of new vars, then do so 
      [e, subst]:= rationalize::useNewVars(e, subst, options)
    end_if;  
    
    // these are the elements where we will have to find the minimal polynomials
    minpolysToFind:= subst;
    minpolys:= relations[3];
    
  else
    minpolys:= {};
    minpolysToFind:= subst 
  end_if; 
  
  
  if options[MinimalPolynomials] then
    minpolys:= minpolys union map(minpolysToFind, rationalize::minimalPolynomial, options)
  end_if;
  
  if options[Recursive] then
    subst:= rationalize::recurse(subst, options);
    // remove duplicate substitutions
    [e, subst]:= rationalize::deduplicate(e, subst);
    // we do not compute minimal polynomials in the recursive step - they need not be updated
    // add replacements to substitutions
    subst:= [op(replaced)] . subs(subst, replaced)
  else
    subst:= replaced union subs(subst, replaced);
  end_if;  
  
  if options[ShowDependencies] then
    dep:= map(subst, rationalize::showDependencies);
    subst:= subs(subst, dep);
    e:= subs(e, dep);
    if options[MinimalPolynomials] then 
      minpolys:= subs(minpolys, dep)
    end_if   
  end_if;  
  
  if options[MinimalPolynomials] then
    // return value is a sequence of length 3
    e, subst, minpolys
  else
    e, subst 
  end_if
  
end_proc:

rationalize:= funcenv(rationalize):

rationalize::defaultDescend:=
proc(a)
  local t;
begin
  t:= type(a);
  _lazy_or(contains({"_mult", "_plus", DOM_SET, DOM_LIST}, t), t = "_power" and type(op(a, 2)) = DOM_INT)
end_proc:  


rationalize::defaultOptions:= 
table(
  FindRelations = [],
  DescendInto = rationalize::defaultDescend,
  StopOn = {DOM_INT, DOM_RAT, DOM_IDENT},
  Recursive  = FALSE,
  ShowDependencies = FALSE,
  Prefix = "X",
  MinimalPolynomials = FALSE,
  StopOnConstants = FALSE,
  ReplaceTypes = {},
  ReplaceHardToEval = FALSE,
  ApproximateFloats = FALSE
):

rationalize::optionTypes:= 
table(
  FindRelations = DOM_LIST,
  DescendInto = Type::AnyType,
  StopOn = Type::AnyType,
  Recursive = DOM_BOOL, 
  ShowDependencies = DOM_BOOL,
  Prefix = DOM_STRING,
  MinimalPolynomials = DOM_BOOL,
  StopOnConstants = DOM_BOOL,
  ReplaceTypes = Type::AnyType,
  ReplaceHardToEval = DOM_BOOL, 
  ApproximateFloats = DOM_BOOL
):



// types that are hard to evaluate and should be replaced
rationalize::hardToEval:= {"limit", "sum", "int"}:


/*
  rationalize::replaceTypes(e, options)
  
  e - arithmetical expression, or set or list of such
  options - table of options in the usual format

  returns a list [ee, replacements]
  
  where 
  - ee is an arithmetical expression, or a set or list of such
  - replacements is a set of equations
  such that
  -  no type specified by ReplaceTypes occurs in ee
  - subs(ee, replacements) gives e
  

  Example:
  T:= rationalize::defaultOptions;
  T[ReplaceTypes]:= {"sin"};
  rationalize::replaceTypes(x + sin(x) + sin(sin(x)), T)

  should give 

  [X22513 + X22514 + x, {X22513 = sin(sin(x)), X22514 = sin(x)}]

  where the newly generated identifiers may be different
  

*/
rationalize::replaceTypes:=
proc(e, options: DOM_TABLE)
  local types, replace: DOM_PROC, replacements, selector, ee;
begin
  // local method replace(u) 
  // adds an entry replacements[u]=X to replacements, and returns X
  // if such a substitution already exists in replacements, it returns that X
  // it is a pity that we cannot make this a global procedure that accepts a
  // pointer to replacements as an argument
  replace:=
  proc(u)
  begin
    if contains(replacements, u) then
      replacements[u]
    else
      // result of assignment used as a return value
      replacements[u]:= genident(options[Prefix])
    end_if
  end_proc;  
  
  // main program of replaceTypes
  types:= options[ReplaceTypes];
  replacements:= table();  
  // the following call descends into e and replaces each sub-tree of one of the given
  // types without descending into it (it is therefore important to use PostMap)
  selector:= types;

  ee:= misc::maprec(e, selector = replace, PostMap);
  [ee, rationalize::table2set(replacements)]
end_proc:

 // rationalize::table2set(T)
 // convert replacements from a table of entries u=X into a set of substitutions X=u
 
rationalize::table2set:=
proc(T)
  begin 
  map({op(T)}, X -> op(X, 2) = op(X, 1))
end_proc:  




/*
   rationalize::approximateFloats(e)

   returns an expression that contains no floats, and is a good rational approximation of e

*/
rationalize::approximateFloats:=
proc(e)
begin
  numeric::rationalize(e)
end_proc:  
  
/*
   rationalize::descend(e,  options)

   this is *the* main function of rationalize    

   e - arithmetical expression, or set or list of such
   options - table of options

   returns a list [ee, subst]
   such that
   - ee is an arithmetical expression and a rational function; or a set or list of such
   - subst is a set of equations
   - subs(ee, subst) = e
  


*/
rationalize::descend:=
proc(e, options: DOM_TABLE)
  local negateDescendInto, desc, i: DOM_INT, selector, stopOn, rueck, prefix;
begin
  
  // first, we have to construct the type of all objects that are
  // *not* selected by DescendInto
  
  desc:= options[DescendInto];
  if type(desc) = DOM_SET then
    negateDescendInto:= (x -> not _lazy_or(testtype(x, op(desc, i)) $i=1..nops(desc)))
  else
    negateDescendInto:= not desc
  end_if;
  
  
  stopOn:= options[StopOn];
  selector:= stopOn;

  if desc = rationalize::defaultOptions[DescendInto] then
    negateDescendInto:= "DefaultSelector";
  end_if:

  prefix:=options[Prefix];
  if prefix = rationalize::defaultOptions[Prefix] then
    while has(e, text2expr(prefix."_I")) do
      prefix:="X".prefix;
    end_while:
  end_if:

  rueck:=stdlib::rationalizeDescendHelper(e, prefix,
  selector          = id,
  negateDescendInto = "DefaultReplace");

  [rueck[1], rationalize::table2set(rueck[2])]
end_proc:


rationalize::isRational:=
proc(b, options)
begin
  bool(op(rationalize::descend(b, options), 1) = b)
end_proc:


/*
   rationalize::findRelations(subst, options)

   subst - set of substitutions {X1 = a1, X2 = a2, ...}

   returns 
   - a set of substitutions {Xj = bj} where all left hand sides occur as left hand sides in subst 
     (but not all left hand sides from subst need to occur). The bj may contain new variables Y1, Y2, ...
   - a set of substitutions {Y1 = c1, Y2 = c2, ..} where exactly the new variables Y1, Y2, ... occur on the left hand side.
     The right hand sides c1, c2, ... may only contain variables that also occur in one of a1, a2, ...
   - a set of minimal polynomials, involving the Yi and those of the Xj for which no substitution Xj = bj occurs in the first operand of the return value


*/
rationalize::findRelations:=
proc(subst: DOM_SET, options: DOM_TABLE)
  local types, newsubst: DOM_SET, Ysubst: DOM_SET, minpolys: DOM_SET, t, f, l;
begin
  types:= options[FindRelations];
  assert(domtype(types) = DOM_LIST);
  newsubst:= {};
  Ysubst:= {};
  minpolys:= {};
  
  for t in types do
    assert(type(t) = DOM_STRING);
    if type((f:= eval(text2expr(t)))) <> DOM_FUNC_ENV or 
      (f:= f::findRelations) = FAIL then
      warning("No slot for finding relations for ".t." exists")
    else
      l:= [f(subst /* union newsubst */, options)];
      // f has to return a (possibly empty) set of substitutions for replacing the left hand sides, such as 
      // {X1 = Y + X2 + Y^2 + 3*Y^4, ...}
      // as well as a set that indicates what the new variables represent, such as
      // {Y1 = sin(x) + 5, ...}
      // Here, the right hand sides may only involve variables occurring in e1, e2, ...
      // Finally, a third set consisting of minimal polynomials must be returned. 
      // These latter may involve the newly introduced variables Y1, Y2, .. and those of X1, X2, ... for which no substitution in terms of Y's is returned; 
      // they must be rational expressions according to the definition above and the options DescendInto and StopOn as provided by the caller.
       if map(l, type) <> [DOM_SET $3] then
         error("Wrong type of return value of findRelations - slot: must consist of three sets")
       end_if; 
       // do the elements of l[1] overwrite variables from newsubst? This we do not want
       l[1]:= select(l[1], equ -> not contains(map(newsubst, op, 1), op(equ, 1)));
       // apply the newly found substitutions to the existing right hand sides in newsubst
       newsubst:= subs(newsubst, l[1]);
       // and conversely!
       l[1]:= subs(l[1], newsubst);
       newsubst:= newsubst union l[1];
       Ysubst:= Ysubst union l[2];
       minpolys:= minpolys union l[3]
     end_if
  end_for;  
  
  newsubst, Ysubst, minpolys
end_proc:


/*
b: base
s: set of substitutions {X1 = b^e1, X2 = b^e2, ...} 
*/
rationalize::findPowerRelationsBaseFixed:=
proc(b, subst: DOM_SET, options: DOM_TABLE)
  local eqvclasses, i: DOM_INT, j: DOM_INT, ss, s, expo, found: DOM_BOOL, 
  q, qq, g, X: DOM_IDENT, Y: DOM_IDENT, newsubst, Ysubst, exponent, expos, X1, X2;
begin
  
    exponent:=
    proc(ss)
    begin
      if op(ss, 2) = b then
        1
      elif not iszero(b) and op(ss, 2) = 1/b then
        -1
      else 
        assert(type(op(ss, 2)) = "_power");
        op(ss, [2, 2])
      end_if;  
    end_proc;  
  
    // form equivalence classes w.r.t. the equivalence relation "has a rational quotient"
    // each equivalence class is stored as a list [exponent, set_of_substitutions, X belonging to that exponent]
    eqvclasses:= [];
    subst:= prog::sort([op(subst)], length);
    for i from 1 to nops(subst) do
      ss:= op(subst, i);
      assert(_lazy_or(type(op(ss,2)) = "_power", op(ss, 2) = b, b=0, op(ss, 2) = 1/b ));
      expo:= exponent(ss);
      found:= FALSE;
      for j from 1 to nops(eqvclasses) do
        if contains({DOM_RAT, DOM_INT}, type(expo/eqvclasses[j][1])) then
          found:= TRUE;
          eqvclasses[j][2]:= eqvclasses[j][2] union {ss};
          break  
        end_if;  
      end_for;   
      if not found then 
        // start a new equivalence class 
        eqvclasses:= append(eqvclasses, [expo, {ss}, op(ss, 1)])
      end_if     
    end_for;
   
    newsubst:= table();
    Ysubst:= table();
    // for each equivalence class [e, {X1 = b^(q1*e), X2 = b^(q2*e), ...] where the qi are rationals, determine the gcd of all qi's
    for i from 1 to nops(eqvclasses) do
      s:= eqvclasses[i];
      expo:= s[1];
      s:= s[2];  // set of substitutions
      if nops(s) = 1 then
        
        if stdlib::hasmsign(expo) /* testtype(expo, Type::Negative) = TRUE */ and not iszero(b) and not (type(b) = "_power" and stdlib::hasmsign(op(b, 2))) then 
           // we want to represent negative powers by 1/Y, where Y is a new variable
           Y:= genident(options[Prefix]); // new variable representing b^(-expo)
           Ysubst[Y]:= b^(-expo);
           newsubst[op(s, [1, 1])]:= 1/Y
        end_if;   
        
        next
      end_if;  
      // we now construct a table Xi = qi containing the cofactors
      q:= table((op(ss, 1) = exponent(ss)/expo) $ss in s);
      // list of equations Xi = qi
      qq:= [op(q)];  
      // list of exponents qi
      expos:= map([op(q)], op, 2);
      // gcd of exponents
      g:= gcd(op(expos));
      // we do not want to create polynomials of too high degree
      if max(map(expos, abs@_divide, g)) > 2^20 then
         next
      end_if;

       // prefer negative powers if the base is a negative power already
       // use the negative gcd if b is a negative powermod
      if stdlib::hasmsign(expo) and not iszero(b) or 
        type(b) = "_power" and stdlib::hasmsign(op(b, 2)) then
        g:= -g  
      end_if;  
      // write all exponents as multiples of g; that is, the corresponding Xi as integer powers of b^(g*e)
      // we have to do this only if g*e does not actually occur among the exponents
      if (j:= contains(map(qq, op, 2), g)) = 0 then
        if type(b^(g*expo)) = DOM_IDENT then
          Y:= b^(g*expo)
        else   
          Y:= genident(options[Prefix]);  // new variable representing b^(g*e)
          Ysubst[Y]:= b^(g*expo)
        end_if   
      else 
        Y:= op(qq[j], 1)
      end_if;  
        
      for j from 1 to nops(s) do
        X:= op(s, [j, 1]); // left hand side of j-th substitution
        // the j-th substitution is X = b^(q[X]*e); so X = Y^(q[X]/g)
        // we do not need to do that if X=Y
        if q[X] <> g then
           newsubst[X]:= Y^(q[X]/g)
        end_if;
      end_for; // j  
    end_for;  // i
    
    
    // second heuristic. If the base is a rational function, we may combine powers whose exponents differ by an integer, that is, given
    // X1 = b^a1, X2 = b^(a1 + g), we may write X2 = b^a1 * b^g = X1*b^g
    if not iszero(b) and rationalize::isRational(b, options) then 
        s:= subst;
        expo:= map(s, exponent);
        // mark all rational exponents as not interesting ( we have already worked on them)
        expo:= map(expo, u -> if type(u) = DOM_RAT or type(u) = DOM_INT then FAIL else u end_if);
        for i from 1 to nops(expo) do
        if expo[i] = FAIL then next end;
        X1:= op(s[i], 1);
        for j from i+1 to nops(expo) do
          if expo[j] <> FAIL and type((g:= expo[j] + expo[i])) = DOM_INT then 
            X2:= op(s[j], 1);
            if not contains(newsubst, X2) then 
              // add a new substitution for X2, and express in terms of X1
              X1:= subs(X1, op(newsubst));
              if not has(X1, X2) then
                newsubst:= subs(newsubst, X2 = b^g/X1);
                newsubst[X2]:= b^g/X1;
                expo[j]:= FAIL;
              end_if;
            elif not contains(newsubst, X1) then
              X2:= subs(X2, op(newsubst));
              if not has(X2, X1) then
                newsubst:= subs(newsubst, X1 = b^g/X2);
                newsubst[X1]:= b^g/X2;
                expo[i]:= FAIL;
                break // this i has been completed, do no more j's
              end_if;
            end_if;  
          end_if;
          if expo[j] <> FAIL and type((g:= expo[j] - expo[i])) = DOM_INT then 
            X2:= op(s[j], 1);
            if not contains(newsubst, X2) then 
              // add a new substitution for X2, and express in terms of X1
              X1:= subs(X1, op(newsubst));
              if not has(X1, X2) then
                newsubst:= subs(newsubst, X2 = X1 * b^g);
                newsubst[X2]:= X1 * b^g;
                expo[j]:= FAIL;
              end_if;
            elif not contains(newsubst, X1) then
              X2:= subs(X2, op(newsubst));
              if not has(X2, X1) then
                newsubst:= subs(newsubst, X1 = X2 * b^(-g));
                newsubst[X1]:= X2 * b^(-g);
                expo[i]:= FAIL;
                break // this i has been completed, do no more j's
              end_if;
            end_if;
          end_if;
         end_for;
       end_for;
    end_if;
    
    // tableToSet:
    newsubst:= {op(newsubst)};
    Ysubst:= {op(Ysubst)};
    newsubst, Ysubst
end_proc:

/*
    rationalize::findRelationsTrig(subst, options)

    the slot sin::findRelations and cos::findRelations

    recognizes algebraic dependencies between sin(x) and cos(x), but *not* yet between sin(x) and sin(x/2)

*/
rationalize::findRelationsTrig:=
proc(subst: DOM_SET, options: DOM_TABLE)
  local l, l1, xtab, x, g, T, newsubst, Ysubst, revsubst, rewriteTanHalf;
begin
  // local table
  rewriteTanHalf:= 
  table(
  "tan" = -(2*#T)/(#T^2 - 1),
  "cot" =  -(#T^2 - 1)/2/#T,
  "cos" = -(#T^2 - 1)/(#T^2 + 1),
  "sin" = 2*#T/(#T^2 + 1)
  ):
  
  subst:= select( subst, equ -> testtype(op(equ, 2), Type::Union("sin", "cos", "tan", "cot")));
  // table 
  revsubst:= table((op(x, 2) = op(x, 1)) $x in subst);
  
  // right hand sides: expressions of the form sin(..), cos(..), tan(..), cot(..)
  l:= map(subst, op, 2);
  // arguments 
  l1:= map(subst, op, [2, 1]);
  xtab:= table((x = (l intersect {sin(x),cos(x),tan(x),cot(x)})) $x in l1);
  // select those that appear in at least two functions 
  l1:= select(l1, x -> nops(xtab[x]) >= 2);
  
  newsubst:= {};
  Ysubst:= {};
  for x in l1 do 
    if xtab[x] intersect {sin(x), cos(x)} <> {} then
      // rewrite everything by tan(x/2)
      T:= genident(options[Prefix]);
      Ysubst:= Ysubst union {T = tan(x/2)};
      for g in xtab[x] do
        newsubst:= newsubst union {revsubst[g] = subs(rewriteTanHalf[type(g)], #T = T)}
      end_for;
    else 
      assert(xtab[x] = {tan(x), cot(x)});
      // rewrite cot by tan
      newsubst:= newsubst union {revsubst[cot(x)] = 1/revsubst[tan(x)]}
    end_if;
  end_for; // x in l1 
  // we do not detect minimal polynomials yet
  newsubst, Ysubst, {}
end_proc:

  
  
 




/*
   _power::findRelations - the slot handling rationalize(..., FindRelations = [..., "_power", ....])
*/
_power::findRelations:= 
proc(subst: DOM_SET, options: DOM_TABLE)
  local subst2, bases, bas, T: DOM_TABLE, equ: "_equal", s, ss, newsubst, b,
  Ysubst: DOM_SET, newsubst2, Ysubst2;
begin
  // extract powers
  subst2:= select(subst, equ -> type(op(equ, 2)) = "_power");
  // get bases
  bases:= map(subst2, op, [2, 1]);
  // get substitutions for each base
  T:= table((bas = select(subst2, equ -> (op(equ, [2, 1]) = bas))) $bas in bases);
  
  // possible to-do item: bases may be integer powers themselves
  // we do not recognize dependencies between x^2 and (x^4)^(1/2)
  // in general, we do not recognize dependencuies between powers having different bases
  
  newsubst:= {};
  Ysubst:= {};
  for equ in T do 
    s:= op(equ, 2);
    b:= op(s, [1, 2, 1]);
    assert(b = op(equ, 1));
    // we have extracted all substitutions Xi = b^ai; but we still need to look if there is some substitution Xi = b or Xi = 1/b
    // we must not do this for b=0 as 0 and 0^n are independent of each other 
    if not iszero(b) then
      ss:= select(subst, eq -> op(eq, 2) = b or op(eq, 2) = 1/b);
      if nops(ss) > 0 then
        assert(nops(ss) <= 2);
        s:= s union ss
      end_if;  
    end_if;
    [newsubst2, Ysubst2]:= [rationalize::findPowerRelationsBaseFixed(b, s, options)];  
    // avoid getting another substitution for an already substituted identifier
    // that is we neither want two alternative substitutions X=a1, X=a2 nor a substitution and its reciprocal, i.e. X=1/Y, Y=1/X
    newsubst2:= select(newsubst2, equ -> not has(newsubst, op(equ, 1))); 
    newsubst:= newsubst union newsubst2;
    Ysubst:= Ysubst union Ysubst2;
  end_for;  // equ in T
  // substitute recursively as right hand sides of substitutions may contain left hand sides of other substitutions
  newsubst:= map(newsubst, equ -> op(equ, 1) = subs(op(equ, 2), newsubst));
  newsubst, Ysubst, {}
end_proc:  

/*
rationalize::useNewVars(e, subst)

  returns an expression e2 with:
  e is equivalent to e2, but contains no variable Y for which 
  a substitution X = Y^(1/n) exists (but contains X^n instead)

*/
rationalize::useNewVars:=
proc(e, subst, options: DOM_TABLE)
  local i, equ, a, Y, XX, X, p, q;
begin
  subst:= prog::sort([op(subst)], length, Reverse);
  for i from 1 to nops(subst) do
    equ:= op(subst, i);
    X:= op(equ, 1);
    a:= op(equ, 2);
    if type(a) = "_power" and
      type(op(a, 2)) = DOM_RAT and 
      not contains({DOM_INT, DOM_RAT, DOM_COMPLEX}, type((Y:= op(a, 1)))) and 
      has(e, {Y, 1/Y}) then 
      [p, q]:= [op(op(a, 2))];
      // our substitution is of the form X = Y^(p/q)
      // create a new identifier XX representing Y^(1/q), and 
      // replace Y by XX^q and X by XX^p in the expression e
      // note that we do not need a new identifier if p=1
      if op(a, [2, 1]) = 1 then 
         e:= subs(e, Y = X^q, 1/Y = X^(-q))   
      else
        XX:= genident(options[Prefix]);
        e:= subs(e, X = XX^p, Y = XX^q, 1/Y = XX^(-q));
        subst:= subsop(subst, i = (XX = Y^(1/q)))
      end_if   
    end_if;  
  end_for;
  [e, {op(subst)}]
end_proc:
  


/*
   rationalize::minimalPolynomial(equ, options)

   - equ: equation of the form X = a
   - options: table of options

   returns a polynomial expression f(X) such that f(a) represents zero; 
   or the empty object if no such polynomial expression can be found

*/
rationalize::minimalPolynomial:=
proc(equ: "_equal", options)
  local X, a;
begin
  [X, a]:= [op(equ)];
  case type(a)
  of DOM_COMPLEX do 
    if type(op(a, 1)) = DOM_FLOAT then
      break
    end_if;  
    // the minimal polynomial of a complex number u + v*I equals
    // X^2 - 2*u*X + u^2 + v^2 
    // return(X^2 - 2*op(a, 1)*X + op(a, 1)^2 + op(a, 2)^2)
    // we are guaranteed that always I or -I is replaced by a new variable
    assert(a = I or a = -I);
    return(X^2 + 1)
  of "_power" do
    if type(op(a, 2)) = DOM_RAT and // contains({DOM_INT, DOM_RAT}, type(op(a, 1))) 
      rationalize::isRational(op(a, 1), options)
      then
      return(X^op(a, [2, 2]) - op(a, 1)^op(a, [2, 1]))
    else
      break
    end_if
  of "surd" do
    if type(op(a, 2)) = DOM_INT then
      return(X^op(a, 2) - op(a, 1))
    else
      break
    end_if
  of "exp" do 
    // detect roots of unity
    return(rationalize::minpolyExp(a, X))
  end_case;
  
  // default:
  null()
end_proc:

rationalize::minpolyExp:=
proc(a: "exp", X)
  local q;
begin
  // detect roots of unity
    if type((q:= op(a, 1)/(2*PI*I))) = DOM_RAT then
      return(expr(polylib::cyclotomic(op(q, 2), X)))
    else
      null()
    end_if  
end_proc:


/*
   rationalize::recurse(subst)

   subst - set of substitutions

   returns a list of substitutions such that 
   - the operands of every right hand side are rational functions
   - if the input contained n elements, then the output contains m>=n elements such that applying 
     the last m-n substitutions to the n first gives the original input

*/
rationalize::recurse:=
proc(subst: DOM_SET, options: DOM_TABLE): Type::ListOf("_equal")
  local i: DOM_INT, operands, options2: DOM_TABLE, subst2, newsubst: DOM_LIST, reinsert;
begin
  reinsert:= 
  proc(f, l)
  begin
    op(f, 1) = subsop(op(f, 2),  i = l[i] $i=1..nops(l), Unsimplified)
  end_proc;  
    
  if nops(subst) = 0 then
    // nothing to do
    return([])
  end_if;  
  
  // some functions (like radsimp) prefer to have the ouput sorted w.r.t. descending complexity
  subst:= prog::sort([op(subst)], x -> Simplify::defaultValuation(op(x, 2)), Reverse);
  // subst is  a list [X1 = a1, ..., Xn = an]
  // we have to rationalize the *operands* of a1, ..., an  
  // if some operand already occurs in a substitution, we want to use the same letter
  // atomic objects cannot be rationalized recursively even if the y are irrational (e.g., floats)
  // extract right hand sides
  operands:= map(subst, op, 2);
  // we extract the operands; atomic objects are treated as objects without operands
  operands:= map(operands, X -> if X = op(X, 1) or type(X) = DOM_COMPLEX then [] else [op(X)] end_if);
  // rationalize the operands
  // take care to descend into lists in any case!
  options2:= options;
  if type(options2[DescendInto]) = DOM_SET then
     options2[DescendInto] := options2[DescendInto] union {DOM_LIST}
  else
     options2[DescendInto] := X -> bool(options[DescendInto](X) or type(X) = DOM_LIST)
  end_if;
  [subst2, newsubst]:= rationalize::descend(operands, options2);
  // reinsert the rationalized operands into the substitutions they come from
  operands:= zip(subst, subst2, reinsert);
  newsubst:= rationalize::recurse(newsubst, options);
  operands.newsubst
end_proc:

/*
   rationalize::deduplicate(e, subst)

   given: an object e and a list subst of substitutions [X1=a1, ..., Xn=an]
   returns: e2, subst2 such that subs(e2, subst2) = subs(e, subst) if the substitutions are carried out from right to left
            subst 2 contains no two substitutions Xi=ai, Xj = aj with ai=aj
   
*/
rationalize::deduplicate:=
proc(e, subst: DOM_LIST): DOM_LIST
  local lhsides: DOM_LIST, rhsides: DOM_LIST, i: DOM_INT, j: DOM_INT;
begin
  rhsides:= map(subst, op, 2);
  lhsides:= map(subst, op, 1);
  for i from 1 to nops(subst) do
    // if the right hand side of subst[i] occurs again at a later position, use that substitution
    if (j:= contains(rhsides, rhsides[i], i+1)) > 0 then
      e:= subs(e, lhsides[i] = lhsides[j]);
      subst:= subs(subst, lhsides[i] = lhsides[j]);
      subst[i]:= FAIL 
    end_if;
  end_for;  
  subst:= select(subst, _unequal, FAIL);
  [e, subst]
end_proc:  
  


/*
  rationalize::showDependencies(subst)

  turns X = a into a substitution X = X(x1, x2,..) where x1, x2, ... are the free identifiers occurring in a
  this substitution must then be applied to both the expression and the substitutions

*/
rationalize::showDependencies:=
proc(subst: "_equal")
  local inds;
begin
  inds:= freeIndets(op(subst, 2));
  op(subst, 1) = op(subst, 1)(op(inds))
end_proc:
