/*
series -- series expansions

series(f,x)		f is an expression
series(f,x,n)		x is a variable
series(f,x=v,n)         n is an integer and v an expression independent of x
series(f,x=v,Right)
series(f,x=v,Left)
series(f,x=v,Real)
series(f,x=v,Undirected) (Undirected is default)
series(f,x=v,n,Right)
series(f,x=v,NoWarning) - supresses warning "Could not find ..."
series(f,x=v,n,NoWarning)
series(f,x=v,Real,NoWarning)
series(f,x=v,n,Real,NoWarning)

Note: option NoWarning is only relevant within the interface functions
series and Series::main; it is NOT passed to the series attributes.
Thus, if a series attribute itself calls series, then NoWarning has
no effect for this call.
*/

series:=
proc(f, x)
  local applyFloat, s, y, ord, dir, war, i,
        x0, t, a, b, a0, b0, i1, i2, i3, opt;
begin
  if args(0) = 0 then error("missing arguments") end_if;
  if f::dom::series <> FAIL then return(f::dom::series(args())) end_if;

  if testargs() then
    if domtype(f) <> DOM_POLY and domtype(f) <> RootOf then
      if testtype(f, Type::Arithmetical) <> TRUE then
        error("Illegal argument")
      end_if;
      if args(0) = 1 then
        error("missing series variable")
      end_if;
    end_if
  end_if;

  
  ord:= ORDER;
  dir:= Undirected;

  opt:= series::defaultOptions;
  

  

  for i from 3 to args(0) do
    case type(args(i))
      of DOM_INT do
      of stdlib::Infinity do
        ord:= args(i);
        break
      of DOM_IDENT do  
        if contains({Undirected, Real, Left, Right}, args(i)) then
          dir:= args(i)
        elif contains(series::optionnames, args(i)) then
          opt[args(i)] := TRUE
        else
          error("Unknown option")
        end_if;
        break
      of "_equal" do
        if not contains(series::optionnames, op(args(i), 1)) then
          error("Illegal option")
        end_if;
        if not testtype(op(args(i), 2), series::optiontypes[op(args(i), 1)])
          then
          error("Option has wrong type")
        end_if;
        opt[op(args(i), 1)] := op(args(i), 2);
        break
      of  DOM_TABLE do
        if map({op(args(i))}, op, 1) <> series::optionnames then
          error("Table of options has wrong format")
        else
          opt:= args(i)
        end_if;
        break
      otherwise
        error("Illegal option")
    end_case
  end_for;

  if opt[NoWarning] then
    war:= NoWarning
  else
    war:= null()
  end_if;

  case type(f)
  of "int" do
     if ((domtype(x) = DOM_IDENT and x = op(f, 2))) or
        ((type(x) = "_equal") and op(x, 1) = op(f, 2)) then
       // indefinite integral, and series variable is integration variable:
       // exchange int and series
       return(int(series(op(f, 1), x, ord, dir, opt), op(f, 2..nops(f))))
     end_if;
     // Juergen: otherwise: Series::unknown will handle this
     // Walter: yes, but the way Series::unknown works is:
     // differentiate (this includes evaluation), *then* subs(.. , x = x0);
     // This way, really complicated integrals will occur which takes
     // years until int gives up. Then x = x0 is substituted and int
     // tries again, ...
     // It is much faster to integrate the series term by term. 
     // Note that int handles series objects:
     // series( int(f(x, y), y), x = x0) 
     //   --->  int( series(f(x, y), x = x0), y)

     // Special case:
     // If the series variable is not in the boundaries of the integral:
     if not has([op(f, 2..nops(f))], op(x, 1)) then
       i1:= int(series(op(f, 1), x, ord, dir, opt), op(f, 2..nops(f)));
       if not has(i1, infinity) then
         return(i1)
       end_if;
     end_if;
     /* ------------
     // If the series variable is in the int boundaries, e.g.,
     //    series( int(f(x, y), y = a(x) .. b(x)), x = x0),
     // we better do sequential differentiations and substitutions
     // via use Series::unknown.
     // Unfortunately, this will compute only Taylor-Expansions:
     //  f:= int(exp(sqrt(x)), x=0..t):
     //  series(f, t = 1) will work, but
     //  series(f, t = 0) = t + 2/3*t^(3/2) + ...  won't ;-(
     break;
     ------------ */
     // Now the general case: Split the integral into 3 pieces.
     // The integrals i1, i3 can be handled via a bivariate Taylor
     // expansion (the integration interval has zero length),
     // the integral i2 does not have the series variable x in the 
     // boundaries.
     //   series(int(f(x, t), t = a(x ) .. b(x )), x = x0)
     // = series(int(f(x, t), t = a(x ) .. a(x0)), x = x0) // = series(i1(x),x=x0)
     // + series(int(f(x, t), t = a(x0) .. b(x0)), x = x0) // = series(i2(x),x=x0)
     // + series(int(f(x, t), t = b(x0) .. b(x )), x = x0) // = series(i3(x),x=x0)
     if type(x) = "_equal" then
        [x, x0]:= [op(x)];
     else
        x0:= 0;
     end_if:
     // split int(f(x, t), t = a .. b)
     [f, t, a, b]:= [ op(f, 1), op(f, [2, 1]), op(f, [2, 2, 1]), op(f, [2, 2, 2])];
     a0:= subs(a, x = x0, EvalChanges):
     b0:= subs(b, x = x0, EvalChanges):

     if a <> a0 then
          i1:= int(mtaylor(f, [x = x0, t = a0], ord, war), t = a .. a0);
          i1:= series(i1, x = x0, ord, dir, opt);
     else i1:= series(0, x = x0, ord, dir, opt):
     end_if;
     if a0 <> b0 then
       i2:= int(series(f, x = x0, ord, dir, opt), t = a0 .. b0);
       if has(i2, infinity) then
         return(procname(hold(int)(f, t=a..b), x=x0))
       end_if
     else
       i2:= series(0, x = x0, ord, dir, opt):
     end_if;
     if b <> b0 then
          i3:= int(mtaylor(f, [x = x0, t = b0], ord, war), t = b0 .. b);
          i3:= series(i3, x = x0, ord, dir, opt);
     else i3:= series(0, x = x0, ord, dir, opt):
     end_if;

     return(i1 + i2 + i3);

  of DOM_POLY do
    if args(0) = 1 then
       // expansion w.r.t. the main variable
       return(series(op(f, 1), op(f, [2, 1])))
    else
       return(series(op(f, 1), args(2..args(0))))
    end_if
  end_case;

 

  if args(0) = 1 then
    error("missing series variable")
  end_if;

  

  if ord < 0 then
    error("expecting nonnegative integer or infinity as order")
  end_if;
  

  applyFloat := id;
  // replace all floating point constants by rational approximations  
  s := misc::maprec(f, {DOM_FLOAT, DOM_COMPLEX} =
                        proc(f)
                          local r;
                        begin
                          r := numeric::rationalize(args(1), Minimize);
                          if r<>f then
                            applyFloat := float;
                          end;
                          r
                        end_proc);
  y := misc::maprec(x, {DOM_FLOAT, DOM_COMPLEX} =
                        proc(f)
                          local r;
                        begin
                          r := numeric::rationalize(args(1), Minimize);
                          if r<>f then
                            applyFloat := float;
                          end;
                          r
                        end_proc);

  s := Series::main(s, y, ord, dir, opt);

  if s = FAIL then procname(args()) else applyFloat(s) end_if
end_proc:

series := funcenv(series):
series::type := "series":
series::info := "series -- compute series expansions [try ?series for help]":
series::print := "series":

series::defaultOptions:=
table(
      NoWarning = FALSE,
      UseGseries = TRUE
      ):

series::optiontypes:=
table(
      NoWarning = DOM_BOOL,
      UseGseries = DOM_BOOL
      ):

series::optionnames:= {NoWarning, UseGseries}:

series::operandsToSimplify:= [1]:



//
// CHANGED (jngerhar) 24.02.01:
// - properties of the series variable are respected now
// - if a direction is given or the expansion is around +-infinity,
//   a corresponding property is attached to the series variable
// - uses Series::check_point
//
Series::main :=
proc(f, p, ord, direc, opt)
  local x, x0, x1, yofx, xofy, n, s, tmp, dir, s1, s2, ord2,
  orderTooSmall, ratio, fac;
begin

  assert(args(0) = 5);
  
   // check that p = x or p = (x - x0) specifices a valid series variable
   // and expansion point
   [x, x1] := Series::check_point(p);
  
   if type(x) = "_index" then
     s:= genident();
     tmp:= Series::main(f, s = x1, ord, direc, opt);
     return(new(tmp::dom, op(subs([extop(tmp)], s=x))))
   end_if;
                  
   if has(x1, infinity) and not contains({infinity, -infinity}, x1) then
      // rewrite series(f(x), x = I*infinity) as series(f(I*x), x = infinity)
      fac:= 1:
      if type(x1) = "_mult" or 
        (type(x1) = stdlib::Infinity and stdlib::Infinity::is_mult(x1)) then
         fac:= extop(x1);
         x1:= infinity:
         f:= subs(f, x = fac*x);
      end_if;
      if fac <> 1 then
         s:= Series::main(f, x = x1, ord, direc, opt);
         if s = FAIL then 
            return(FAIL)
         elif traperror((tmp:= s::dom::subs(s, x = x/fac))) = 0 then
            return(tmp);
         elif domtype(s) = Series::Puiseux then
            // If s is a Puiseux series with sin, cos etc,
            // subs(s, x = I*x) will not work and throw an
            // error. In such a case, convert to a gseries, 
            // first:
            s:= Series::gseries::convert(s):
            if s = FAIL then 
               return(FAIL)
            else
               return(s::dom::subs(s, x = x/fac));
            end_if;
         else
            return(FAIL);
         end_if;
      end_if;
   end_if;
   x0 := x1;

   // Prepare change of variable: x will be replaced by y(x) later,
   // such that after the change x is to tend to 0.
   // Moreover, modify properties of x accordingly, taking into account
   // the direction, if any

   dir := direc;

   // the following piece of code is identical to the corresponding
   // one in Series::Puiseux::new
   if domtype(x0) = stdlib::Infinity then
         if Series::sign(dir) = sign(x0) then
           // specified Left for -infinity or Right for infinity
           error("inconsistent direction")
         end_if;
         yofx := sign(x0)/x;
         xofy := yofx;
         tmp := getprop(yofx);
         dir := Right;
         assume(x > 0);
         if tmp <> C_ then
           assume(x, tmp, _and);
         end_if;
         x0 := complexInfinity;
   elif x0 = complexInfinity then
         yofx := 1/x;
         tmp := getprop(yofx);
         case dir
         of Left do // infinity
           dir := Right;
           assume(x > 0);
           x1 := infinity;
           break;
         of Right do // -infinity
           yofx := -1/x;
           tmp := getprop(yofx);
           dir := Right;
           assume(x > 0);
           x1 := -infinity;
           break;
         of Real do
           assume(x, Type::Real)
         end_case;
         if tmp <> C_ then
             assume(x, tmp, _and)
         end_if;
         xofy := yofx;
   else 
         yofx := x + x0;
         xofy := x - x0;
         tmp := getprop(x - x0);
         case dir
         of Right do
           assume(x > 0);
           break;
         of Left do
           assume(x < 0);
           break;
         of Real do
           assume(x, Type::Real);
           break;
         end_case;
         if tmp <> C_ then
           assume(x, tmp, _and)
         end_if;
   end_if;

   // perform change of variables
   if domtype(f) = RootOf then
     if ord = infinity then
       error("order = infinity not allowed for RootOf's")
     end_if;

     // perform change of variables only in the first operand
     s := RootOf(subs(op(f, 1), x = yofx), op(f,2));
     
     // compute the set of all series solutions around x = 0
     s1 := Series::algebraic(s, x, ord, dir, opt);

     // reverse change of variables and return the result
     // property of x is restored automatically
     if type(s1) = "_union" then
       return(map(s1, map, Series::Puiseux::set_var, x, x, x1));
     else
       return(map(s1, Series::Puiseux::set_var, x, x, x1));
     end_if;
   else
     s := FAIL;
     traperror((s := subs(f, x = yofx)));
   end_if;

   // try to simplify the input with the properties that were set
   // above.
   // commented out since it is much too slow (jngerhar) 24.02.01
   // s := simplify(s);

   // expansion with infinite precision, if possible
   if s <> FAIL and ord = infinity then

      if testtype(s, Type::PolyExpr(x)) then
        return(subs(expand(s), x = xofy, Unsimplified))
      end_if;
      n := solvelib::getIdent(Z_, indets(s) union {x});
      s1 := Series::expr2coeff(s,x,n);
      // should we do the following?
      // s1:= rewrite(s1, fact) assuming n in Z_ and n>=0;
      if s1 <> FAIL then
        if x0 = complexInfinity then
          s1 := freeze(sum)(s1*x^-n, n = 0..infinity)
        elif traperror(s1 | n=x0) <> 0 then
          // in some cases (e.g. s=exp(x)), we have to extract the first summand
          // TODO: can singularities occur in other summands, too ?
          s1:= (s | x=x0) +  freeze(sum)(s1*(x - x0)^n, n = 1..infinity) 
        else
          s1 := freeze(sum)(s1*(x - x0)^n, n = 0..infinity)
        end_if
      end_if;
      return(s1)
   end_if;


   s1 := FAIL;
   orderTooSmall := FALSE;
   if s <> FAIL and traperror((s1 := Series::series(s, x, ord, dir, opt))) <> 0 then
     if substring(getlasterror()[2], 1..22) = "Error: order too small" and ord > 0
        and opt[UseGseries] then
       orderTooSmall := TRUE;
       // try again with double precision
     else
       // property of x is restored automatically
       lasterror()
     end_if
   end_if;

   if s1 <> FAIL or orderTooSmall then

     // normalize s, stripping trailing zeroes
     if s1::dom::normal <> FAIL then
        s1 := s1::dom::normal(s1, 0); // quick normalization
     end_if;

     if domtype(s1) = Series::Puiseux then
       // try to lower the branch order and to convert to flag = 0 if possible
       ratio := extop(s1, 2); // old branching order
       s1 := Series::Puiseux::condense(s1);
       ratio := ratio / extop(s1, 2); // old branching order / new branching o.
     end_if;

     // if "limit" calls "series", it sets UseGseries = FALSE to indicate
     // that series should not use gseries. For efficiency reasons,
     // we don't do any precision management either in this case,
     // since limit is only interested in the leading term and
     // implements its own precision management anyway
     if opt[UseGseries] and
        (orderTooSmall or
         (domtype(s1) = Series::Puiseux and
          extop(s1, 4) - extop(s1, 3) < ord)) then
       // precision too small ==> try again with increased precision
       assert(ord > 0);
       if orderTooSmall then
         ord2 := 2*ord
       else
         ord2 := ord + (ord - (extop(s1, 4) - extop(s1, 3)))*ratio;
       end_if;
       s2 := FAIL;
       if traperror((s2 := Series::series(s, x, ord2, dir, opt))) <> 0
          and s1 = FAIL then
         // property of x is restored automatically
         lasterror()
       end_if;
          
       if s2 <> FAIL then
         s1 := s2
       end_if
     end_if;

     // reverse change of variables and return the result
     if s1 <> FAIL then
       // property of x is restored automatically
       return(s1::dom::set_var(s1, x, x, x1));
     end_if
   end_if;

   // Puiseux series computation failed

   if dir = Undirected then
      // undirected expansion requested
      if not opt[NoWarning] then
        warning("Could not find undirected series expansion; try option `Left', `Right', or `Real'");
      end_if;
      return(FAIL);
   elif dir = Real then
      // two-sided expansion requested
      if not opt[NoWarning] then
        warning("Could not find real series expansion; try option `Left' or `Right'");
      end_if;
      return(FAIL);
   end_if;

   // one-sided expansion requested ==> try gseries
   if opt[UseGseries] then
     if traperror
       (
        (s:=Series::gseries::new
         (f,p,ord,
          if direc = Left or direc = Right then
            direc
          else
            null()
          end_if)))=0 then
       return(s)
     end_if
   end_if;

   return(FAIL)
end_proc:

