/*
        solve  --  solve equations, inequalities, and systems

        solve(eq,x)			returns a set of solutions
        solve(eq,x=t)			returns only roots in domain t
                              (obsolete, use option Domain=...)                        
		
        solve(eq,S)        returns a list of assignments
                           [x1=a1,..,xn=an] if S is a set {x1,..,xn}
                                            or a list [x1,..,xn]
        solve(eq)          equivalent to solve(eq, S) where S= set
                           of indeterminates of eq                   

        Options:
        IgnoreProperties  do not care whether the solutions are compatible with
                          the assumptions on the variable to solve for
        Multiple		   return a multiset
        PrincipalValue	return only principal values
        VectorFormat = TRUE   return solutions as [x1, ..,xn] in n-dimensional set
        VectorFormat = FALSE  return solutions as set of lists of equations
        MaxDegree=n     solve by radicals up to degree n, use RootOf
                                              for higher degrees
        BackSubstitution=TRUE    enables back substitution
        BackSubstitution=FALSE   disables back substitution
        Domain=d        returns solutions in domain d
        DontRewriteBySystem  do not reduce solving an equation to solving
                             a system
        NoWarning = TRUE suppresses warning messages
        Real = TRUE  assumes that all subexpressions of any expression
                     must be real
        IgnoreAnalyticConstraints = TRUE  allow transformations in general
                        that are correct only under additional assumptions;
                        may give incorrect results
        DiscontCheck  (undocumented, internal option) In equations of the form f*g=0, return 
                      the zeroes of f and g, without checking whether they are discontinuities of the other factor

possible types of return values:

        RootOf   - polynomial equation could not be solved, or its
                   degree exceeds MaxDegree                             
        "solve"  - equation could not be solved
        DOM_SET  - finite number of solutions
                   or set of lists of equations (as a special
                   return type of the solver for systems)
        "_in" - var \in (some of the other sets mentioned)
        "_union"
        "_intersect"
        "_minus"
        "Union" i.e. solvelib::Union(...)                                        
        solvelib::BasicSet - one of Z, Q, R, C
        Dom::Interval
        Dom::ImageSet - something of the form { f(x); x \in S}
        piecewise  - the solution depends on case analysis                                         
        Dom::Multiset
                                                      
*/


alias(OPTIONS={Multiple, PrincipalValue, MaxDegree, BackSubstitution,
               VectorFormat, hold(DiscontCheck),
               Domain, IgnoreProperties, IgnoreSpecialCases,
                 DontRewriteBySystem, MaxRecLevel, NoWarning, Real,
                 IgnoreAnalyticConstraints}):


solve :=                   
proc(eq, var)
  name solve;
  local sol,options, eqs, vars, indep, dep, dummy, x,
  secondargisoption: DOM_BOOL,
  pw,
  propset, baseSet;
  save MAXEFFORT;
  
  
begin
  // m a i n   p r o g r a m
  
  if args(0)=0 then
    error("solve called without arguments")
  end_if;
  if args(0)>1 then
    eqs:=eq;
    vars:=var;  // to avoid warnings 
    userinfo(1, "solving ".expr2text(eqs)." for ".expr2text(vars));
  else
    eqs:=eq;
    userinfo(1, "Solving :".expr2text(eqs));
  end_if;


  if type(eqs) = "_equal" then
    if domtype(op(eqs, 1)) = DOM_LIST and
      domtype(op(eqs, 2)) = DOM_LIST then
      eqs:= zip(op(eqs, 1), op(eqs, 2), _equal, 0)
    else
      eqs:= op(eqs, 1) - op(eqs, 2);
      if eqs = undefined then
        return(procname(args()));
      end_if;
    end_if
  end_if;

// first try overloading
  if eqs::dom::solve<>FAIL then
    return(eqs::dom::solve(eqs, args(2..args(0))))
  end_if;


  // handle special cases of equations and illegal first arguments 

  case type(eqs)
    of DOM_ARRAY do
      return(solve([op(eqs)], args(2..args(0))))
    of DOM_TABLE do
      return(solve([map(op(eqs), op, 2)], args(2..args(0))))
    of DOM_NIL do
    of DOM_FAIL do
    of DOM_EXEC do
    of DOM_FUNC_ENV do
    of DOM_PROC do
    of DOM_PROC_ENV do
    of DOM_STRING do  
    of DOM_DOMAIN do
    of DOM_VAR do  
      error("Illegal first argument ".expr2text(eqs))
  end_case;

  if args(0)>=2 and indets(args(2)) intersect OPTIONS <> {} then
    options:=solvelib::getOptions(args(2..args(0)));
    secondargisoption:= TRUE
  else
    options:=solvelib::getOptions(args(3..args(0)));
    secondargisoption:= FALSE
  end_if;    


  
  
  if args(0)=1 or secondargisoption then
    if domtype(eqs)=DOM_POLY then
      if op(eqs,3)=Expr then
        vars:=op(eqs,2)
      elif has(op(eqs,3), IntMod) then
        return(map(solve(poly(eqs, Dom::IntegerMod(op(eqs,[3,1])))),
                   expr));
      else
        vars:=op(eqs,2);
        if options[Domain]=solvelib::defaultOptions[Domain]
          and options[Domain]<> op(eqs,3) then
          // solving over coefficient domain
          options[Domain]:=op(eqs,3);
          eqs:= expr(eqs)
        end_if
      end_if
    else
      vars:= solvelib::indets(eq);
      if nops(vars)=0 then
          error("Need variable to solve for")
      end_if
    end_if;
    if {op(vars)} intersect OPTIONS <> {} then
      warning("Name of option used as an identifier")
    end_if;
  end_if;


// maximal recursion depth reached ?
  // time over?
  if options[MaxRecLevel] <= 0 then
    return(procname(eqs, vars))
  end_if;




  // domain solver


  if options[Domain] <> solvelib::defaultOptions[Domain] then
    return(solvelib::domsolve(eqs, vars, options))
  end_if;

  
  // handle the option PrincipalValue separately
  
  if options[PrincipalValue]=TRUE then
    return(solvelib::principalValue(eqs, vars, options))
  end_if;
    
  
  if options[IgnoreSpecialCases] then
    pw:= solvelib::ignoreSpecialCases
  else
    pw:= piecewise::new
  end_if;


  if options[Real] then
    // attach the property of being real to all free parameters
    dep:= indets(eq) minus Type::ConstantIdents; // minus indets(vars);
    for x in dep do
      assumeAlso(x in R_)
    end_for;
    // re-evaluate eq
    if not has(eqs, hold(solve)) then
      eqs:= eval(eqs)
    end_if;
    baseSet:= R_
  else
    baseSet:= C_
  end_if;
  
  userinfo(10, "Solving for ".expr2text(vars));
  
  case domtype(vars)
    of DOM_EXPR do
      case type(vars)
        of "_equal" do
          // special case: equations x = a..b are handled like
          // x, with option Domain = Dom::Interval(a, b)
          if type(op(vars, 2)) <> "_range" then
            error("right hand side of second argument must be range");
          end_if;
          assume(op(vars, 1) in Dom::Interval([op(vars, [2, 1]),
                                             op(vars, [2, 2])]));
          sol:= solve(eqs, op(vars, 1), options);
          if type(sol) = "solve" then
            return(procname(args()))
          else
            return(sol)
          end_if
        of "_in" do
          assume(vars);
          sol:= solve(eqs, op(vars, 1), options);
          if type(sol) = "solve" then
            return(procname(args()))
          else
            return(sol)
          end_if
        of "function" do
          if type(op(vars, 0)) = "D" then
            // special case: solve for a derivative; rewrite eqs
            // to match vars
            eqs := rewrite(eqs, D);
          end_if;
          break;
        of "diff" do
          // see above
          eqs := rewrite(eqs, diff);
          break;
      end_case;
      userinfo(10, "Substituting solve variable by new ident");
      sol:= genident();
      dep:= subsex(eqs, vars = sol);
      //  if solvelib::indets(vars) intersect solvelib::indets(dep) <> {} then
      //    return(procname(args()))
      //  end_if;
      return(subs(solve(dep, sol, options), sol=vars, EvalChanges))
    of DOM_IDENT do
      // solve for one variable

      // special case: the variable has properties
      if property::hasprop(vars) and options[IgnoreProperties]=FALSE then
        // solve the equation, and output those solutions that
        // agree with the properties of the variable to solve for
        options[IgnoreProperties]:= TRUE;
        propset:= solve(_and(op(property::showprops(vars))), vars, options);
        sol:= solve(eqs, vars, options);
        if type(sol) =  "solve" then
          return(sol)
        else
          propset:= solvelib::avoidAliasProblem(propset, {vars});
          return(solvelib::solve_intersect(sol, propset))
        end_if
      end_if;
      
      case type(eqs)
        of "_in" do
          return(solvelib::solveIn(op(eqs, 1), vars, op(eqs, 2), options))
          // boolean cases
        of DOM_BOOL do
          if eqs=FALSE then
            return({})
          elif eqs=TRUE then
            return(C_)
          else
            error("Cannot solve boolean value UNKNOWN")
          end_if
        of "_or" do
          [dep, indep, dummy]:= split(eqs, has, vars);
          assert(dummy = FALSE);
          if type(dep) <> "_or" then
            dep:= [dep]
          else
            dep:= [op(dep)]
          end_if;
          return(pw([indep, C_],
                    [not indep,
                     solvelib::solve_union
                     (op(map(dep, solve, vars, options)))]
                    )
                 )
        of "_and" do
        of DOM_SET do
        of DOM_LIST do  
          [dep, indep, dummy]:= split([op(eqs)], has, vars);
          assert(dummy = []);
          indep:= map(indep,
                      proc(equ)
                      begin
                        if type(equ) = DOM_IDENT or
                          not testtype(equ, Type::Boolean) then
                          equ = 0
                        else
                          equ
                        end_if
                      end_proc
                      );
          indep:= _and(op(indep));
          MAXEFFORT:= MAXEFFORT/4;
          if options[IgnoreSpecialCases] then
            return(
                    pw
                   ([not indep, {}],
                    [indep,
                     piecewise::disregardPoints
                     (solvelib::solve_intersect
                      (op(map(dep, solve, vars, options))))
                     ])
                   )
          else  
            return(
                   pw
                   ([not indep, {}],
                    [indep,
                     solvelib::solve_intersect(op(map(dep, solve, vars, options)))
                     ])
                   )
          end_if
        of "_not" do
          return(C_ minus solve(op(eqs,1), vars, options ))
        of "_less" do
        of "_leequal" do
          sol:= solvelib::ineqs(eqs, op(vars), options);
          return(solvelib::avoidAliasProblem(sol, freeIndets(eq) union {vars}))
        of "_unequal" do
          MAXEFFORT:= MAXEFFORT/3;
          return(baseSet minus
                 solvelib::solve_eq(op(eqs,1)-op(eqs,2), vars, options)
                 minus discont(op(eqs, 1), vars, Undefined, options)
                 minus discont(op(eqs, 2), vars, Undefined, options)
                 )
        of "_equal" do
          // should not happen
          assert(FALSE);
        of "_subset" do 
          if op(eqs, 2) = C_ then
            return(C_)
          end_if;
          return(procname(args()))
        of "isEmpty" do
          return(procname(args()))
        of DOM_POLY do
          if op(eqs,3)=hold(Expr) then
            if contains(op(eqs, 2), vars) = 0 then
               eqs:= expr(eqs)
            end_if;
            break
          end_if;
          if op(op(eqs,3),0)= hold(IntMod) then
            options[Domain]:=Dom::IntegerMod(op(op(eqs,3)));
            return(map(solvelib::domsolve
                   (expr(eqs), vars, options),
                       expr))
          end_if;
          // coefficient ring is a domain 
          options[Domain]:= op(eqs,3);
          return(solvelib::domsolve(expr(eqs), vars, options))
        of Series::Puiseux do
          error("Cannot solve a series")
      end_case;
      sol:= solvelib::solve_eq(eqs, vars, options);
      return(solvelib::avoidAliasProblem(sol, freeIndets(eq) union {vars}))
    of DOM_SET do
      vars:= sort([op(vars)]);
      if options[VectorFormat] then
        return(matrix(vars) in solve(eqs, vars, options))
        // else fall through
      end_if;
    of DOM_LIST do
      if nops(vars)=0 then
        error("Need variable to solve for")
      else
        case type(eqs)
          of DOM_SET do
            eqs:= [op(eqs)];
            // fall through
          of DOM_LIST do
            // this handles also inequalities 
            sol:= solvelib::solve_sys(eqs, vars, options);
            break
          of DOM_BOOL do
            case eqs
              of FALSE do
                sol:= {};
                break
              of TRUE do
                sol:= solvelib::cartesianPower(C_, nops(vars));
                break
              of UNKNOWN do  
                error("Cannot handle boolean UNKNOWN")
            end_case;
            break
          otherwise
            // solve the original version of eqs, such that
            // the handling of equations is done there
            sol:= solvelib::solve_sys([eq], vars, options)
        end_case;
        sol:= solvelib::avoidAliasProblem(sol, freeIndets(eq) union {op(vars)});
        if options[VectorFormat] then
          return(sol)
        else
          return(solvelib::convertToAssignments(sol, vars))
        end_if
      end_if
    otherwise
      error("Illegal variable(s) to solve for: ".
            expr2text(vars))
  end_case;

  // should not be reached
  
  assert(FALSE);
  procname(args())
  
end_proc:


solve:= prog::remember(solve, property::depends, PreventRecursion,
                       ()-> hold(solve)(args())):


solve:= funcenv(solve):
solve:= slot(solve, "type", "solve"):
solve:= slot(solve, "print", "solve"):
solve:= slot(solve, "info",
    "solve -- solve equations and inequalities [try ?solve for options]"
):


solve::float:=
proc(F, x)
  local options, searchRange, getSearchRange: DOM_PROC,
        generalSearchRange, addRange: DOM_PROC, checkSolution: DOM_PROC, result, props, solveOptions, numericOptions;
begin


  /* getSearchRange: determines the search range for every variable 
     getSearchRange(x = a..b) returns x and writes a..b to the table of search ranges
     getSearchRange(x) returns x and tries to determine a search range forx from its properties
   */
  getSearchRange:=
  proc(X)
    local S;
  begin
    if type(X) = "_equal" then
       // the range given is used, not the properties of the variable
       searchRange[op(X, 1)]:= op(X, 2);
       return(op(X, 1))
    else    
       S:= getprop(X, "Targets" = {Dom::Interval});
       if S = R_ then
         searchRange[X] := -infinity..infinity
       elif type(S) = Dom::Interval then     
         searchRange[X] := _range(op(Dom::Interval::borders(S)))
       elif generalSearchRange <> FAIL then
         searchRange[X] := generalSearchRange
       end_if;
       return(X)
     end_if
  end_proc;

  // addRange: determines a suitable second argument for the call to numeric::solve
  addRange:=
  proc(X)
  begin
    if type(X) = DOM_LIST or type(X) = DOM_SET  then
       map(X, addRange)
    elif contains(searchRange, X) then
       X = searchRange[X]
    else
       X
    end_if
  end_proc;

  if args(0) = 0 then
    error("solve::float called without arguments")
  end_if;  
  
  if args(0) = 1 then
    x:= numeric::indets(F)
  end_if;  
  
  // split options into those of the symbolic and those of the numeric solver
  // use result as dummy
  numericOptions:= {args(3..args(0))} intersect 
  {RestrictedSearch, UnrestrictedSearch, Random, MultiSolutions, 
   FixedPrecision, SquareFree, Factor, AllRealRoots, Multiple,
   NoWarning};
  solveOptions:= {args(3..args(0))} minus numericOptions;
  
  options:= solvelib::getOptions(op(solveOptions));

  // determine the general domain to solve over (the same for all variables)
  if type(options[Domain]) = Dom::Interval then
     generalSearchRange:= _range(op(Dom::Interval::borders(options[Domain])))
  elif options[Real] or contains({R_, Dom::Real}, options[Domain]) then
     generalSearchRange:= -infinity..infinity
  else
     generalSearchRange:= FAIL
  end_if;

  // get search range for each variable
  searchRange:= table();
  if type(x) = DOM_LIST or type(x) = DOM_SET then
     x:= map(x, getSearchRange)
  else   
     x:= getSearchRange(x)
  end_if;

  // finally, apply the search ranges again to all variables
  

  // should we use further options of solve?
  // we might also want to use realroot
  result:= numeric::solve(F, addRange(x), op(numericOptions));
  

  checkSolution:=
  proc(a)
  begin
    if type(a) = DOM_LIST then
      bool(is(props | a) <> FALSE)
    else
      bool(is(props | x=a) <> FALSE)
    end_if
  end_proc;

  props:= _and(op(showprop(indets(F))));
  select(result, checkSolution)

end_proc:


// declare solve to be a set
solve::testtype:=
proc(x, T)
begin
  if T = Type::Set then 
     TRUE
  elif T = Type::Arithmetical then
     FALSE
  else
     FAIL
  end_if
end_proc:


// overload freeIndets
solve::freeIndets:=
proc(S: "solve")
begin
  freeIndets(op(S, 1), args(2..args(0))) 
  minus 
  (indets(op(S, 2)) union Type::ConstantIdents)
end_proc:

// overload evalAt
solve::evalAt:=
proc(f:"solve", subst: DOM_SET)
  local eq, vars, dontsubst, dummy;
begin
  // special cases
  eq:= op(f, 1);
  case type(eq)
    of rec do
      vars:= op(eq, 2);
      assert(type(vars) = "function");
      if contains(map(subst, op, 1), op(vars, 0)) then
        error("Cannot substitute dependent variable")
      end_if;
      [subst, dontsubst, dummy]:= split(subst, x -> op(x, 1) <> op(vars, 1));
      f:= subsop(f, 1= evalAt(op(f, 1), subst), Unsimplified);
      if dontsubst = {} then
        return(f)
      end_if;
      if ((eq:= rec::evaluateAt(eq, op(dontsubst, [1, 2])) )) <> FAIL then
        return({eq})
      else  
        return(hold(evalAt)(f, dontsubst))
      end_if
    of ode do
      vars:= op(eq, 2);
      if type(vars) <> DOM_SET then
        vars := {vars}
      end_if;
      vars:= vars union {op(eq, 3)}; // independent variable
      [subst, dontsubst, dummy]:= split(subst,
                                    equ -> not has(vars, op(equ, 1)));
      assert(dummy = {});
      // substitute parameters, but not the variables to solve for
      return(subsop(f, 1 = evalAt(op(f, 1), subst), Unsimplified))
  end_case;
  
  vars:= op(f, 2);
  case type(vars)
    of DOM_LIST do
      vars:= {op(vars)};
      break
    of DOM_SET do
      break
    otherwise
      vars:= {vars}
  end_case;
  [subst, dontsubst, dummy]:= split(subst,
                                    equ -> not has(vars, op(equ, 1)));
  assert(dummy = {});
  // substitute parameters, but not the variables to solve for
  eval(subsop(f, 1 = evalAt(op(f, 1), subst), Unsimplified))
end_proc:

// overload _plus
solve::_plus := stdlib::set_plus:

// overload _mult
solve::_mult := stdlib::set_mult:

// overload _power
solve::_power := stdlib::set_power:


// end of file 
