/*-

stringlib::formatf -- formats floating point numbers

Calling sequence:

stringlib::formatf(f, d)
stringlib::formatf(f, d, l)

Parameter:

f - a floating point number
d - an integer 
l - an non-zero positive integer

Summary:

A decimal fixed point representation of the floating point number 'f', 
rounded to the specified number 'd', is returned by stringlib::formatf.

If 'd' is a non-zero positive integer, a rounded fixed-point 
represenation with 'd' digits after the decimal point is returned.
If 'd' is zero, then a rounded fixed-point represenation with one
zero after the decimal point is returned.
If 'd' is negative, then 'f' is rounded to '-d' digits before the
decimal point and a fixed-point represenation with one zero after
the decimal point is returned.

If a third argument 'l' is specified, then the string returned has 'l'
characters. If the converted number 'f' requires less than 'l' digits,
then it is padded on the left with spaces.

Examples:

>> stringlib::formatf(52.356, 2);

                                  " 52.36"

>> stringlib::formatf(-52.356, 2);

                                  "-52.36"

>> stringlib::formatf(52.356, 0);

                                  " 52.0"

>> stringlib::formatf(52.356, -1);

                                  " 50.0"

>> stringlib::formatf(52.356, -2);

                                  " 100.0"

>> stringlib::formatf(52.5, 4);
 
                                 " 52.5000"

>> stringlib::formatf(52.356, 2, 10);

                               "     52.36"

>> stringlib::formatf(0.0123, 2);

				" 0.01"

>> stringlib::formatf(-1.123, 2);

				"-1.12"

>> stringlib::formatf(-0.123, 2);

				"-0.12"

-*/


stringlib::formatf := proc(x, d, l)
  local f, s, t;
begin

  case args(0)
    of 2 do
       if not testtype(x, DOM_FLOAT) then
          error("Floating point number as 1st argument expected")
       end_if;
       if not testtype(d, DOM_INT) then
          error("Integer as 2nd argument expected")
       end_if;
       l := 0;
       break;
    of 3 do
       if not testtype(l, DOM_INT) or l < 1 then
          error("Integer > 1 as 3rd argument expected")
       end_if;
       break;
    otherwise                                                
       error("Wrong number of arguments") 
  end_case;

  if x < 0 then
     s := "-"; x := abs(x)
  else
     s := " "
  end_if;

  if d > 0 then
     x := round(x*10^d)*10^(-d);
     t := trunc(x); f := frac(x);
     f := round(f * 10^d);
     if f = 0 then
        if t = 0 then
           s := " ".t.".".("0"$d)
        else
           s := s.t.".".("0"$d)
        end_if
     else
        s := s.t.".";
        while f<10^(d-1) do              
	   s:=s.0;                    
           d:=d-1;                   
	end_while;
        s:=s.f;
     end_if
  else 
     if d = 0 then
        t := round(x);
     else
        t := round(x * 10^d) * 10^(-d);
     end_if;
     if t = 0 then
        s := " 0.0"
     else
        s := s.t.".0"
     end_if
  end_if;

  if l > 0 then
     stringlib::format(s, l, Right)
  else
     s
  end_if;

end_proc:
