
/*

stringlib::match -- string matching 

Calling Sequence:

stringlib::match( s, t, n, m, k)

Parameter:

s - non-empty string ( s = s[a] s[a+1] ... s[b] , b = a+n-1)
t - non-empty string
n - positive integer (length of s, 1 <= n <= length(s))
m - positive integer (length of t, 1 <= m <= length(t))
k - integer (  a <= k <= b )

Summary:

stringlib::match returns an integer i , a <= i <= b, 
if the string s contains the string t and s[i..i+m-1] = t holds. 
Otherwise FAIL is returned. Note, that no check of the arguments is 
done.

The value a is determined by the integer variable stringlib::FirstIndex.
The default value of stringlib::FirstIndex is 0. All integers are valid
values of stringlib::FirstIndex.

The running time of stringlib::match( s, t, n, m, k) is O(m + n) (see [1]).
Note, that successive calls of stringlib::match( s, t, n, m, k) with
different values of k needs only O( n ), because the first step
of the algorithm is computed only once  (see stringlib::init_match, option 
remember). n = n - k + a

Examples:

>> s := "abcdabe": t := "ab": n := length(s): m := length(t):
>> i := stringlib::match(s, t, n, m, 1); substring(s, i, m);

           1 
		   
	   "ab"
>> i := stringlib::match(s, t, n, m, i+m); substring(s, i, m);

           5
		   
	   "ab"
		   
>> stringlib::match(s, t, n, m, i+m);

           FAIL
		   
See also:

strmatch, length, substring

References:

[1] D.E. Knuth, J.H. Morris, R.V. Pratt. Fast Pattern Matching in Strings, 
    SIAM Journal of Computing 6:2, 1977, p. 323-350

[2] R. Karp, M. Rabin. Efficient Randomized Pattern Matching Algorithms,
    IBM Journal res. Development, 1987
*/

stringlib::match :=
  proc(s, t, n = 0, m = 0, k = 1)
    local OFFSET, index, j;
  begin
    OFFSET := stringlib::FirstIndex-1;
    if n = 0 then n:= length(s) end_if; 
    if m = 0 then m:= length(t) end_if;
    index := stringlib::init_match(t, m, OFFSET);
    j := 1; 
    k := k-OFFSET;
    while j <= m and k <= n do
      while j > 0 and s[k+OFFSET] <> t[j+OFFSET] do
        j := index[j];
      end_while;
      k := k + 1; 
      j := j + 1;
    end_while;
    if j > m then k - m + OFFSET else FAIL end_if
  end_proc:

stringlib::init_match := proc( t, m, OFFSET)
   option remember;
   local  index, i, j;
begin
   index := array(1..m);
   i := 0; 
   j := 1; 
   index[1] := 0;
   while j < m do
         while i > 0 and t[j+OFFSET] <> t[i+OFFSET] do
               i := index[i];
         end_while;
         i := i + 1; 
         j := j + 1;
         if t[j+OFFSET] <> t[i+OFFSET] then
             index[j] := i
         else
             index[j] := index[i]
         end_if
   end_while;
   index
end_proc:

// End of algorithm D.E. Knuth, J.H. Morris, R.V. Pratt

