// 

/*- 

stringlib::subs -- substitute strings

Calling sequence:

stringlib::subs(s, t = r)
stringlib::subs(s, t = r, First)

Parameter:

s     - non-empty string
t     - non-empty string
r     - string
First - an optional name

Summary:

stringlib::subs(s, t = r, <t1 = r1, ...>) replaces all substrings t in the string s by
the string r (t1 by r1...). This new string is returned.
stringlib::subs(s, t = r, ..., First) replaces only the first occurrence of t in s by r
(t1 by r1...).

Examples:

>> stringlib::subs("abcdabe", "ab" = "X");

           "XcdXe"
		   
>> stringlib::subs("abcdabe", "ab" = "X", First);

           "Xcdabe"
		   
>> stringlib::subs("abcdabe", "ab" = "");

           "cde"
		   
>> stringlib::subs("abcdabe", "abd" = "X");

           "abcdabe"
		   
See also:

subs, stringlib::contains

-*/

stringlib::subs :=
proc(s = "")
begin
  if args(0) = 0 then
    error("wrong number of arguments");
  end_if;
  if domtype(s) <> DOM_STRING then
    error("illegal argument: '".expr2text(s)."'");
  end_if;
  stringlib::subs_regex(s,
                        op(map([args(2..args(0))],
                               proc(eq)
                                 // give local proc a name such that the error
                                 // shows the correct name
                                 name stringlib::subs;
                               begin
                                 if eq=hold(First) then eq;
                                 elif type(eq) <> "_equal" 
                                   then error("illegal argument: ".expr2text(eq)) 
                                 elif domtype(lhs(eq)) <> DOM_STRING or domtype(rhs(eq)) <> DOM_STRING 
                                   then error("illegal argument: ".expr2text(eq)) 
                                 else stringlib::maskMeta(lhs(eq))=rhs(eq) end_if
                               end_proc)));
end_proc:

stringlib::subs_regex :=
  proc(s = "")
    local i, l, r, t, SUBS, first, X;
  begin
    if s = "" then
      return(s)
    end_if;
    
    // examine arguments
    if args(args(0)) = hold(First) then
      if args(0) < 3 then
        return(s)
      end_if;
      first := TRUE;
      SUBS := [args(2..args(0) - 1)]
    else
      first := FALSE;
      SUBS := [args(2..args(0))]
    end_if;
    
    if testargs() then
      // check substitution equations
      X := select(SUBS, testtype, Type::Equation(DOM_STRING, DOM_STRING));
      if nops(X) <> nops(SUBS) then
        if not testtype(args(args(0)), Type::Equation(DOM_STRING, DOM_STRING)) and args(args(0)) <> hold(First) then
          error("illegal last argument")
        else
          error("wrong type of substitution argument")
        end_if
      end_if
    end_if;

    for X in SUBS do
      t := op(X, 1);
      r := op(X, 2);
      
      l := strmatch(s, t,
      if first then Index else Index, All end):
      if l = FALSE or l = {} then
        next
      end_if;
      if first then
        l := [l];
      else
        // sort index list in decreasing order
        l := prog::sort([op(l)], x->-op(x, 1));
      end_if;
      for i in l do
        s[i[1]..i[2]] := r;
      end_for;
    end_for;

    s
  end_proc:

