// Literature:
//   Concrete mathematics 2ed.; Graham, Knuth, Patashnik
//   Hypergeometric Summation; Wolfram Koepf          
//   JSC Vol. 16, 1993m pages 243-258, Improvement of the Degree Setting
//     in Gospher's Algorithm; Peter Lisonek, Peter Paule and Volker Strehl
//
// from GKP pages 224-226 
sum::gosper :=
proc(f,_n_) // returns g such that f = subs(g,n=n+1)-g 
  local rat,qn,rn,res;
begin
  userinfo(1,"enter Gosper");
  if not has(f,_n_) then
    f*_n_
  else
      /* write f(k+1)/f(k) = p(k+1)/p(k) * q(k)/r(k+1) (5.117)
        where roots of q and r do not differ by an integer */
    rat:=sum::ratio(f,_n_);
      // check for geometric summations 
    if not has(rat,_n_) then
      return(subs(f,_n_=0,EvalChanges)*
             piecewise([rat <> 1, rat^_n_/(rat-1)],
                       [rat = 1, _n_])
             )
    end_if;
    if has(rat,{binomial,fact}) then
      rat:=normal(expand(rat))
    end_if;
    userinfo(2,"f(".expr2text(_n_)." / f( ".expr2text(_n_-1)." = ".
             expr2text(rat));
    qn:=numer(rat);
    rn:=denom(rat);
    if not (testtype(qn,Type::PolyExpr(_n_,Type::AnyType))
            and testtype(rn,Type::PolyExpr(_n_,Type::AnyType))) then
      return(hold(sum)(f,_n_))
    end_if;
    userinfo(1,"input is hypergeometric");
    res:=sum::gosper2(1,qn,rn,_n_,{},f);
    if res=FAIL then
      userinfo(1,"Gosper's algorithm fails");
      hold(sum)(f,_n_)
    else
      normal(res*f)
    end_if
  end_if
end_proc:

/* returns R=g/f such that f = subs(g,n=n+1)-g for f(n)/f(n-1)=p(n)/p(n-1)*qn/rn
  with pn,qn,rn polynomials,
  inds is a set of auxiliary unknowns (used only in Zeilberger).
  As f[n] = g[n+1]-g[n]. we must have f[n]=R[n+1]*f[n+1]-R[n]*f[n],
  i.e. 1 = R[n]*p(n)/p(n-1)*qn/rn-R[n-1]
*/
sum::gosper2 :=
proc(pn,qn,rn,n,inds,f)
  local X,rnj,res,i,p_zeros,z,j,gn,gnj,dp,R,dR,Q,dQ,d,k0,fn,eq,F,ii,m,unk,ff,
        cond,tmp;
begin
  userinfo(3,"pn=",pn,"qn=",qn,"rn=",rn);
  X:= genident("X");
  rnj:= subs(rn/content(rn,[n]),n=n+X); // do not expand !
  if MAXEFFORT = RD_INF then
     res:= polylib::resultant(qn,rnj,n)
  elif traperror((res:=polylib::resultant(qn,rnj,n)), 
                 MaxSteps = max(3, ceil(MAXEFFORT/20000))) <> 0 then
     userinfo(1, "timeout in gosper2");
     return(FAIL)
  end_if;
  res:= factor(res);
  p_zeros:= [];
  for i from 1 to nops(res) div 2 do
    ff:= op(res,2*i);
    if degree(ff,X)=1 then
      z:= -coeff(ff,X,0)/coeff(ff,X,1);
      if testtype(z,Type::PosInt) then 
        p_zeros:=append(p_zeros,z);
      end_if
    end_if
  end_for;
  for j from nops(p_zeros) downto 1 do
    gn:= gcd(qn,subs(rn,n=n+p_zeros[j],EvalChanges));
    qn:= divide(qn,gn,Quo);
    gnj:= gn;
    for i from 1 to p_zeros[j] do
      pn:= expand(pn*gnj);
      gnj:= expand(subs(gn,n=n-i));
    end_for;
    if ((tmp:= divide(rn,gnj,Exact))) <> FAIL then 
      rn:= tmp;
    else  
      rn:= normal(rn/gnj); // divide(rn,gnj,Exact) ?
    end_if;  
  end_for;
      // make the polynoms monic (cf Prop. 3.2 of [LiPaSt93]) 
  qn:= qn/lcoeff(rn,[n]);
  rn:= rn/lcoeff(rn,[n]);
  qn:= expand(subs(qn,n=n+1));
  dp:= degree(pn,[n]);
  R:= expand(qn+rn); 
  dR:= degree(R,[n]);
  Q:= expand(qn-rn); 
  dQ:= degree(Q,[n]);
  if dR<=dQ then // case 1
    userinfo(2,"case 1 of Gosper's algorithm");
    if dQ=0 then // polynomial case
      d:= dp+1
    else
      d:= dp-dQ
    end_if
  else // case 2 : deg(qn)=deg(rn) and lcoeff(qn)=lcoeff(rn)
    m:= dR; // to have the same notation as [LiPaSt93]
    k0:= coeff(rn-qn,n,m-1); // (5.2) of [LiPaSt93]
    if type(k0)<>DOM_INT or m=1 // cf [LiPaSt93] page 255
       or testtype(f,Type::RatExpr(n,Type::AnyType)) then // case 2a
      userinfo(2,"case 2a of Gosper's algorithm");
      d:= dp-m+1
    else // case 2b
      userinfo(2,"case 2b of Gosper's algorithm");
      d:= max(k0,dp-m+1)
    end_if
  end_if;
  userinfo(2,"d=",d);
  if d>=0 then
    F:= genident("F");
    fn:= _plus(F[ii]*n^ii$ii=0..d);
    // eq. (GE) of [LiPaSt93] p. 248 
    // where qn is q(n+1) 
    // put minus sign before fn because of PR302-303-310 
    eq:= pn+qn*subs(-fn,n=n+1)+rn*fn;
    userinfo(2,"number of equations is",degree(eq,[n])+1);
    eq:= poly(eq, [n]);
    // necessary to avoid problems with e.g. eq=param^(-1)*n 
    eq:= {coeff(eq,All)};
    unk:= inds union {F[ii] $ ii=0..d};
    if nops(eq)<nops(unk) then
      eq:= eq union {F[0]=0}
    end_if;
    userinfo(3,"system to solve is",eq);
    eq:= linsolve(eq,unk,hold(ShowAssumptions));
    cond:= op(op(eq, 2)) and op(op(eq, 3));
    if hastype(cond, "_equal") then
      return(FAIL)
    end_if;
    eq:= op(eq, 1);
    userinfo(1,"solution is",eq);
    if eq = FAIL then
             // system has no solution, or solution is only valid if the
             // parameters satisfy certain constraints 
      return(FAIL)
    else
             /* do the substitution into pn too because it can
               contain some indeterminates C[l] in Zeilberger's alg. */
      fn:= sum::normal(subs(fn/pn,eq,EvalChanges)*rn);
      if has(fn,F) then
        return(FAIL)
      elif inds={} then
        return(piecewise([cond, fn]))   // for sum::gosper
      else
        return(piecewise([cond, fn]), eq)             // for sum::zeilberger
      end_if
    end_if
  else
    return( FAIL )
  end_if
end_proc:
