// A couple of routines computing laplace transforms.
// Originally written by Stephan H. Sept - Nov 2002

transform::laplace::compute_x_n_exp_c_over_x :=
proc(_a,_n,_c,_fu, var1, var2)
   // Computes laplace transform of _fu = (_a*var1)^(_n)*exp(_c/var1).
   // The transform exists:
   // - for all _n if Re(_c) < 0 or
   // - for _n with Re(_n) > -1 if only Re(_c) <= 0 and _c <> 0
   // Note that in Oberhettinger's table of laplace transforms
   // formula 5.29 holds only for Re(a) > 0 and
   // formula 5.34 holds also for Re(a) >= 0, a <> 0 and Re(nu) > 0
local bol1, bol2;
begin
    bol1:=FALSE;
    bol2:=FALSE;                   
    if testtype(_c, Type::Numeric) then
        if Re(_c) < 0 then 
                 bol1:=TRUE; 
                 bol2:=TRUE; 
        elif Re(_c) = 0 then    // _c = 0 should not happen
           if testtype(_n, Type::Numeric) then
              if Re(_n) > -1 then
                 bol1:=TRUE;
                 bol2:=TRUE;
              end_if;
           else // n is not numeric
                 bol1:=TRUE;
                 bol2:=is(Re(_n)>-1);
           end_if;
        end_if;
    else // _c is not numeric
      if testtype(_n, Type::Numeric) then
        if Re(_n) > -1 then
          bol1:=is(Re(_c)<=0);
          bol2:=TRUE;
        else // Re(_n) =< -1 
          bol1:=is(Re(_c)<0);
          bol2:=TRUE;
        end_if;
      else   //_n is not numeric
        if is(Re(_n) > -1) = TRUE then
          bol1:=is(Re(_c)<=0);
          bol2:=TRUE;
        else // _n is anything
          bol1:=is(Re(_c)<0);
          bol2:=TRUE;
        end_if;
      end_if;
    end_if;
    if bol1 = TRUE and bol2 = TRUE then
       return( _a^(_n)* 2*besselK(_n+1,2*sqrt(-_c*var2))*(-_c/var2)^(_n/2+1/2)  );
    elif bol1 = TRUE and bol2 = UNKNOWN then
       warning("cannot determine the sign of ".expr2text(1 + Re(_n)));
    elif bol2 = TRUE and bol1 = UNKNOWN then
       warning("cannot determine the sign of ".expr2text(Re(_c)));
    elif bol1 = UNKNOWN and bol2 = UNKNOWN then
       warning("cannot determine the signs of "
               .expr2text(1 + Re(_n))." and ".expr2text(Re(_c)));
    end_if;
    if is(_a >= 0) = TRUE or
       is(_n in Z_) = TRUE then
       return(_a^_n*hold(transform::laplace)(var1^(_n)*exp(_c/var1), var1, var2)); 
    else
       return(hold(transform::laplace)(_fu, var1, var2)); 
    end_if;
end_proc:

//==================================================
// Compute laplace transform of (_d*x)^(_k)*cos(_c*x)
// Walter, 5.12.08. Implemened in analogy to compute_x_n_sin_x 
transform::laplace::compute_x_n_cos_x := proc(_d,_k,_c,t,s)
begin

   if is(_k in Z_) = TRUE and is(_k <= 0) = TRUE then
     if is(_k = 0) = TRUE then
       return(s/(s^2 + _c^2));  
     elif is(_k = -1) = TRUE then
       // this is the expanded version of the general case
       return(hold(transform::laplace)(1/t, t, s)/_d - ln(1 - I*_c/s)/2/_d - ln(1 + I*_c/s)/2/_d);
//     return(hold(transform::laplace)(1/t, t, s)/_d - ln(1 + _c^2/s^2)/2/_d);
//     return(-EULER/_d - ln(s - I*_c)/2/_d - ln(s + I*_c)/2/_d);
     else
       return((-1)^(-_k-1)/2/(-_k-1)!*_d^_k*(s-I*_c)^(-_k-1)*(harmonic(-_k-1)-EULER-ln(s-I*_c))
             +(-1)^(-_k-1)/2/(-_k-1)!*_d^_k*(s+I*_c)^(-_k-1)*(harmonic(-_k-1)-EULER-ln(s+I*_c))):
     end_if;
   end_if;
   // This is the result for general _k apart from _k = -1, -2, -3, ....
   // The case _k = 0 has a nicer representation. It is treated above.
   // Note that the Laplace transform does not exist for Re(_k) <= -1
   // because the integrand exp(-s*x)*(_d*x)^(_k)*cos(_c*x) is singular 
   // at x=0. In analogy to ego's system, we return the following result,
   // anyway, since it is the analytic continuation w.r.t. to _k and
   // consistent with laplace(x*f(x),x,s)=-diff(laplace(f(x),x,s),s):
   return(_d^_k*gamma(1+_k)/2*((s+_c*I)^(-_k-1)+(s-_c*I)^(-_k-1)));
end_proc;
    
//==================================================
// Compute laplace transform of (_d*x)^(_k)*sin(_c*x)
// The transform exists for all _k with Re(_k) > -2 
// This version was written by Walter, 5.12.08.
transform::laplace::compute_x_n_sin_x := proc(_d,_k,_c,t,s)
begin
   if is(_k in Z_) = TRUE and is(_k <= 0) = TRUE then 
     if is(_k = 0) = TRUE then
       return(_c/(s^2 + _c^2));  // = 1/(2*I) * (1/(s-I*_c) - 1/(s+I*_c))
     elif is(_k = -1) = TRUE then
       return(arctan(_c/s)/_d);  // = I/2 * (ln(s - I*_c) - ln(s + I*_c))
     elif is(_k = -2) = TRUE then
       return(_c/_d^2*hold(transform::laplace)(1/t, t, s)
             +_c/_d^2 -_c/_d^2/2*ln(1 + _c^2/s^2) - s/_d^2*arctan(_c/s));
     else
       return(I*(-1)^(-_k-2)/2/(-_k-1)!*_d^_k*(s-_c*I)^(-_k-1)*(harmonic(-_k-1)-EULER-ln(s-I*_c))
             -I*(-1)^(-_k-2)/2/(-_k-1)!*_d^_k*(s+_c*I)^(-_k-1)*(harmonic(-_k-1)-EULER-ln(s+I*_c))):
//     return(( I)^(-_k-2)/2/(-_k-1)!*_d^_k*(_c+I*s)^(-_k-1)*(harmonic(-_k-1)-EULER-ln(s-I*_c))
//           +(-I)^(-_k-2)/2/(-_k-1)!*_d^_k*(_c-I*s)^(-_k-1)*(harmonic(-_k-1)-EULER-ln(s+I*_c))):
     end_if;
   end_if;
   // This is the result for general _k apart from _k = -1, -2, -3, ....
   // The case _k = 0 has a nicer representation. It is treated above.
   // Note that the Laplace transform does not exist for Re(_k) <= 2
   // because the integrand exp(-s*x)*(_d*x)^(_k)*sin(_c*x) is singular 
   // at x=0. In analogy to ego's system, we return the following result,
   // anyway, since it is the analytic continuation w.r.t. to _k and 
   // consistent with laplace(x*f(x),x,s)=-diff(laplace(f(x),x,s),s):
   return(I*_d^_k*gamma(1+_k)/2*((s+_c*I)^(-_k-1)-(s-_c*I)^(-_k-1)));
end_proc:

/* ============ The following is Stephan's original code:=================
transform::laplace::compute_x_n_sin_x :=
proc(_g,_d,_k,_c,_fu, var1, var2)
 // computes laplace transform of _fu = (_d*var1)^(_k)*sin(_c*var1)
 // The transform exists for all _k with Re(_k) > -2 
 // there are, however, roots involved leading to various
 // branches in the _c times _var2 plane
local branch, dummy, trafo;
begin
    case _k
      of  0 do
         return(_c/(var^2 + _c^2));  // = 1/(2*I) * (1/(s-I*c) - 1/(s+I*c))
      of -1 do
        return(arctan(_c/var2)/_d);  // = I/2 * (ln(s - I*c) - ln(s + I*c))
      of -1/2 do // pick the correct branch of the root
        branch := transform::laplace::pick_branch(_c, var1);
        // branch is only zero when evaluation of 
        // assumptions on symbolics might lead further
        // then do a simplify later
        if not iszero(branch) and testtype(branch, Type::Numeric) then
        return( branch * (sqrt(var2^2+_c^2)-var2)^(1/2)
          /sqrt(var2^2+_c^2)*_d^_k*sqrt(PI/2));
        end_if;
        break;
      of -3/2 do// pick the correct branch of the root
        branch := transform::laplace::pick_branch(_c, var1);
        // branch is only zero when evaluation of 
        // assumptions on symbolics might lead further
        // then do a simplify later
        if not iszero(branch) and testtype(branch, Type::Numeric) then
        return(branch* (sqrt(var2^2+_c^2)-var2)^(1/2)*_d^_k*sqrt(2*PI));
        end_if;
        break;
      otherwise
        if (testtype(_k, Type::Numeric) and Re(_k) > -2)
        // case _k = integer > 0 is already covered somewhere else
            or // symbolic and assumption
            (not testtype(_k, Type::Numeric) and (is(Re(_k) > -2) = TRUE)) then                                                                                                                                      
               return( gamma(_k+1)*(var2^2+_c^2)^(-(_k+1)/2)*
                       sin((_k+1)*arctan(_c/var2))*_d^_k)
        end_if;
      end_case;
      dummy := genident(); // if var2 is numeric 
      trafo:=(transform::laplace)(_g, var1, dummy); 
      userinfo(10, "sin-modulation");
      return( 1/(2*I)*(subsex(trafo, dummy=var2-I*_c)-
                       subsex(trafo, dummy=var2+I*_c)));
end_proc:

=====  the following utility was used by Stephan. It is obsolete now =======

transform::laplace::pick_branch :=
proc(_c, var2)
// in general the following holds with branch = + or - 1:
                        //
                        // sqrt(I*var2 +_c) - sqrt(I*var2 - _c)
                        //   = 2 * branch * sqrt(sqrt(var2^2 + _c^2) - var2)
                        //
                        // we compute branch for numerical _c in some cases we also
                        // need to know about the location of var2, we shall only do
                        // it for numericals then. 
                        // Otherwise (symbolic case + assumptions), 
                        // for reasons of speed
                        // simplify shall do it with a similar routine
                        // 

local branch, erg1, erg2;
begin
        branch := 0;
        if testtype(_c, Type::Numeric) then
        // only in the following cases we may pick a universal branch
                // -PI/4 < arg(_c) <= PI/4
                if  Re(_c) >= 0  and Im(_c) <= Re(_c) and -Im(_c) < Re(_c) then
                        branch :=1;
                elif        // 3 PI/4 < arg(_c) <= 5 PI/4
                    Re(_c) <= 0  and Im(_c) < -Re(_c) and -Im(_c) <= -Re(_c) then
                        branch :=-1;
                end_if;
        else // _c is symbolic, testing for assumptions on _c
        // again only in the following cases there is a univeral branch
                // -PI/4 < arg(_c) <= PI/4
                if  is(Re(_c) >= 0) = TRUE and
                        ( (is(Im(_c) >= 0) = TRUE and is(Im(_c) <= Re(_c)) = TRUE) or
                          (is(Im(_c) <= 0) = TRUE and is(-Im(_c) < Re(_c)) = TRUE) 
                        )    then
                        branch :=1;
                elif        // 3 PI/4 < arg(_c) <= 5 PI/4
                      is(Re(_c) <= 0) = TRUE and 
                        ( (is(Im(_c) >= 0) = TRUE and is(Im(_c) < -Re(_c)) = TRUE) or
                          (is(Im(_c) <= 0) = TRUE and is(-Im(_c) <= -Re(_c))= TRUE) 
                        ) then
                        branch :=-1;
                end_if;
        end_if;
        if iszero(branch) then
        // _c is numeric or symbolic and we don't know whether
        // it lies within the two above specified cones in the plane
        // just compute the branch:
                erg1 := sqrt(var2 + I*_c); // var2 must be restricted such that
                erg2 := sqrt(var2 - I*_c); // both ly in the right half-plane
                branch := sign(Im(erg1 - erg2)) - (1 - abs(sign(Im(erg1 - erg2))))
                                                        * sign(Re(erg1 - erg2));
                if branch <> 1 and branch <> -1 then
                        // it's an expression, we don't want to return that
                        branch := 0;        
                end_if;
        end_if;
        return(branch);

end_proc:
===================================================================== */

