
//-------------------------------------
transform::laplace:=
proc(_f, var1, var2)
   name  transform::laplace ;
   local back, dummy ;
   option hold;
begin
   if args(0) = 0 then error("no arguments given") end_if;

   // With option hold, we have to do our own flattening:
   if args(0) <> 3 then
      dummy:= context([args()]);
      if nops(dummy) = 3 then
        [_f, var1, var2]:= dummy;
      else
        error("expecting 3 arguments");
      end_if;
   end_if;

   // set assumptions on var1 and evaluate _f 
   // to simplify (using kernel properties).
   var1:= context(var1);
   assume(var1 >= 0):
   _f:= context(_f):
   var2:= context(var2);

   // allow overloading 
   if _f::dom::laplace<>FAIL then
     back := _f::dom::laplace(args());
     if back <> FAIL then
       return(back)
     else
       return(hold(transform::laplace)(args()))
     end_if
   end_if;
  
   // do the overloading for the funcenv transform::invlaplace because it
   // is no longer a domain
   if type(_f) = "transform::invlaplace" then
     back:= transform::invlaplace::laplace(args());
     if back <> FAIL then
       return(back)
     else
       return(hold(transform::laplace)(args()))
     end_if
   end_if:

   if testargs() then
    if type(_f) = "_equal" then
        if not testtype(op(_f, 1), Type::Arithmetical) or not testtype(op(_f, 2), Type::Arithmetical) then
            error("invalid 1st argument")
        end_if
    elif not testtype(_f,Type::Arithmetical) then
        error("1st argument must be of 'Type::Arithmetical'")
    end_if;

    if domtype(var1)<>DOM_IDENT then
      error("second argument must be of type DOM_IDENT");
    end_if;
   end_if;
 
   // --------------- S.H. August 2002 ------------------
   // var1 must not occur within var2, 
   if has(var2, var1) then
      error("third argument may not contain second argument");
   end_if;  

   // laplace::compute may use differentiation w.r.t. var2,
   // so make sure, that we can differentiate symbolically:
   if domtype(var2)<> DOM_IDENT 
     then dummy:= genident():
     else dummy:= var2;
   end_if;

   // -------------------------------------------------------
   // by Stephan Huckemann, May 2002: it might be wise to try
   // to compute explicitly before going for symbolic 
   // differentation:

   back:=transform::laplace::compute(_f, var1, var2);
   if back<> FAIL then
     return(back);
   end_if;
   // --------------------------------------------------------

   if dummy <> var2 then
     back:=transform::laplace::compute(_f, var1, dummy);
   end_if:
  
   if back=FAIL then
	// ----- S.H. Aug. 2002 -------------
	// if we messed up variables by internal
	// calls etc. untangle variables
	if has(var2, var1) then
		dummy := genident();
		_f := subs(_f, var1=dummy);
		var1 := dummy;
	end_if;
	// -----------------------------------
     return(hold(transform::laplace)(_f, var1, var2))
   else
     if dummy <> var2 then
        back:= subs(back, dummy = var2, EvalChanges);
     end_if:
     return(back)
   end_if;
end_proc:

transform::laplace:= funcenv(transform::laplace):
transform::laplace::type:= "transform::laplace":
transform::laplace::interface := {hold(addpattern)}:

transform::laplace::has:=proc() local T, v;
begin
  T:=args(1); v:=args(2);
  bool( has(extop(T,1), v) or
        has(extop(T,2), v) or
        has(extop(T,3), v) or
        v=hold(transform::laplace))
end_proc:

transform::laplace::numer:= id:
transform::laplace::denom:= 1:
transform::laplace::normal:=id:


// =========================================
//this part computes the laplacian transform
// =========================================
transform::laplace::compute:=
proc(_g, var1, var2)
  local _f, back, typ, i, tmp, C, a, b, c, hasDiff;
  save _X, _Y ;
begin 

  _f:=_g;

  if not has(_f, var1) then
    return(_f/var2);
  elif _f=var1 then
    return(1/var2^2);
  end_if;

  typ:=type(_f);

  if typ="_equal" then
    return(_equal(transform::laplace(op(_f,1), var1, var2),
            transform::laplace(op(_f,2), var1, var2)))
  end_if;

  // We first use the lookup mechanism
  C:=genident() ;
  delete _X, _Y;
  if var1 <>_X then _g:=subs(_g,[_X=C,var1=_X]) end_if;
  
  userinfo(2,"trying the lookup method for laplace");
  back:= transform::laplace::lookup(combine(_g), _X);
  if back<>FAIL then
    back:=subs(back, [_X=var1,_Y=var2,C=_X, 
      hold(transform::frozenLaplace) = hold(transform::laplace),
      hold(transform::frozenInvlaplace) = hold(transform::invlaplace)]) ;
    return(back);
  else
     _g:=subs(_g, [_X=var1,C=_X]) ;
  end_if;
                    
  if typ="_mult" or typ="_plus" then

    back:=transform::IntTrans::linear(hold(transform::laplace)(_f, var1, var2));

    // -------------------------------------------------
    // Walter, 4.5.2000
    // If no success, then try again with normalization 
    // inside IntTrans::linear. This is triggered by the
    // second (dummy) argument 0:
    if back = FAIL or hastype(back, "transform::laplace") then
       back:=transform::IntTrans::linear(hold(transform::laplace)(_f, var1, var2), 0);
    end_if:
    // -------------------------------------------------

    if not (back=FAIL) then
      return(back);
    end_if;
  end_if;

  if typ="function" and type(op(_f,0))="D" then
    _f:=transform::IntTrans::D2diff(_f);
  end_if;

  back:=transform::IntTrans::builtin(hold(transform::laplace)(_f, var1, var2));

  if not (back=FAIL) then
    return(back);
  end_if;

  // =====================================================================
  // Try laplace(x^n*f(x),x,y)  = (-1)^n*diff(laplace(f(x), x, y), y $ n)
  // =====================================================================
  if typ = "_mult" then
    for i from 1 to nops(_f) do
      _g:= op(_f, i);
      if testtype(_g, Type::PolyExpr(var1)) then 
         if domtype(var2) <> DOM_IDENT then
            // We need symbolic differentiation below.
            return(FAIL);
         end_if;
         tmp:= eval(subsop(_f, i = 1)):
         back:= transform::laplace(tmp, var1, var2);
         if not hastype(back, "transform::laplace") then
            assert(domtype(var2) = DOM_IDENT);
            _g:= poly(_g, [var1]):
            assert(has(_g, FAIL) = FALSE);
            _g:= poly2list(_g):
            return(normal(_plus(C[1]*(-1)^C[2]*diff(back, var2 $ C[2]) $ C in _g), Expand = FALSE));   
         elif hastype(_f, "diff") then
            // symbolic diffs can be removed from the laplace argument via:
            // laplace(x*f'(x), x, y) = y*laplace(x*f(x),x,y)-laplace(f(x),x,y)
            // Note that other parts of the laplace code apply the rule
            //    laplace(f'(x),x,y) = y*laplace(f(x),x,y) - f(0),
            // so we only need to implement
            // laplace(x*f'(x), x, y) = - diff(laplace(f'(x),x,y), y)
            //                        = y*laplace(x*f(x),x,y)-laplace(f(x),x,y)
            hasDiff:= FALSE;
            misc::maprec(back, {"transform::laplace"} = proc(x) begin 
                              if hastype(op(x, 1), "diff") then 
                                  hasDiff:= TRUE;
                                  misc::breakmap();
                               end_if; x end_proc);
            if not hasDiff then 
              assert(domtype(var2) = DOM_IDENT);
              _g:= poly(_g, [var1]):
              assert(has(_g, FAIL) = FALSE);
              _g:= poly2list(_g):
              return(normal(_plus(C[1]*(-1)^C[2]*diff(back, var2 $ C[2]) $ C in _g), Expand = FALSE));   
            end_if;
         end_if;
      end_if;
      if type(_g) = "_power" and
        op(_g, 1) = var1 and
        not has(op(_g, 2), var1) then 
        tmp:= eval(subsop(_f, i = 1)):
        if type(tmp) = "besselJ" then
          // =========================================================
          // laplace(x^(c)*besselJ(a, b*x), x, y) =
          //      b^a/2^a*gamma(2*a+n+1)/gamma(a+1)/y^(2*a+n+1)*
          //          hypergeom([a+(n+1)/2, a+(n+2)/2], [a+1], -b^2/y^2)
          // =========================================================
          [a, b, c]:= [op(tmp, 1), op(tmp, 2)/var1, op(_g, 2)];
          if not has([a, b, c], var1) then 
            return(b^a/2^a*gamma(a+c+1)/gamma(a+1)/var2^(a+c+1)*
                   hypergeom([a/2+c/2+1/2, a/2+c/2+1], [a+1], -b^2/var2^2)):
          end_if:
          // =========================================================
          // laplace(x^c*besselJ(a, b*sqrt(x)), x, y) =
          //      b^a/2^a*gamma(a/2+c+1)/gamma(a+1)/y^(a/2+c+1)*
          //          hypergeom([a/2+c+ 1], [a+1], -b^2/4/y)
          // =========================================================
          [a, b, c]:= [op(tmp, 1), op(tmp, 2)^2/var1, op(_g, 2)];
          if not has([a, b, c], var1) then 
             return(sqrt(b)^a/2^a*gamma(a/2+c+1)/gamma(a+1)/var2^(a/2+c+1)*
                        hypergeom([a/2+c+1], [a+1], -b/4/var2));
          end_if;
        end_if:
        if type(tmp) = "besselI" then
          // =========================================================
          // laplace(x^c)*besselJ(a, b*x), x, y) =
          //      b^a/2^a*gamma(a+c+1)/gamma(a+1)/y^(a+c+1)*
          //          hypergeom([a/2+c/2+1/2, a/2+c/2+1], [a+1], b^2/y^2)
          // =========================================================
          [a, b, c]:= [op(tmp, 1), op(tmp, 2)/var1, op(_g, 2)];
          if not has([a, b, c], var1) then 
            return(b^a/2^a*gamma(a+c+1)/gamma(a+1)/var2^(a+c+1)*
                   hypergeom([a/2+c/2+1/2, a/2+c/2+1], [a+1], b^2/var2^2)):
          end_if:
          // =========================================================
          // laplace(x^c*besselI(a, b*sqrt(x)), x, y) =
          //      b^a/2^a*gamma(a/2+c+1)/gamma(a+1)/y^(a/2+c+1)*
          //          hypergeom([a/2+c+ 1], [a+1], b^2/4/y)
          // =========================================================
          [a, b, c]:= [op(tmp, 1), op(tmp, 2)^2/var1, op(_g, 2)];
          if not has([a, b, c], var1) then 
             return(sqrt(b)^a/2^a*gamma(a/2+c+1)/gamma(a+1)/var2^(a/2+c+1)*
                        hypergeom([a/2+c+1], [a+1], b/4/var2));
          end_if;
        end_if:
      end_if;
    end_for;
  end_if:

  //---------------- making a product of sums a sum of products ----------------------
  // Stephan.H. June 2002: I don't understand why restriction to _power in first
  // and _plus in 2nd operand. So I changed the code slightly:

    if typ="_mult" and type(op(_f,2))="_plus" then
          back:=_plus(
              transform::laplace( op(op(_f,2), i)*op(_f, 1), var1, var2)
                 $ i=1..nops(op(_f,2)));
        if back<>FAIL then
          return(back)
        end_if;
    elif typ="_mult" and type(op(_f,1))="_plus" then
          back:=_plus(
             transform::laplace( op(op(_f,1), i)*op(_f, 2), var1, var2)
                 $ i=1..nops(op(_f,1)));
        if back<>FAIL then
          return(back)
        end_if;
    end_if; 

   //---------------------------------------------
   // Stephan, Jan. 2 .2002
   // the code above will produce a FAIL with
   // powers like f(x) = (x - 1)^2. 
   // expanding might help. Expanding should not be
   // done to early, though,  as e.g.
   // expand((sin 2t)^2) = 4 (cos t)^2 (sin t)^2

   if typ="_power" then
	tmp := op(_f,2);
	if testtype(tmp, DOM_INT) and tmp <> -1 then
      tmp := expand(_f);
      // only try with expanded argument, when expand
      // changed something.  Otherwise we get infinite
      // recursions
      if tmp <> _f then
        back := transform::laplace(tmp, var1, var2);
        return(back);
      end_if;
   end_if;
   end_if;
 
  // --------------------------------------------

 return(FAIL);
end_proc:


transform::laplace::diff:=proc() local f, e, x, S, vars, var, dummy;
begin
 
  // diff(anything) = anything
  if args(0) = 1 then return(args(1)) end_if;

  // analyze the arguments
  f:=args(1); // f = transform::laplace(e, x, S)
  e:= extop(f, 1); // this can be a multivariate expression 
  x:= extop(f, 2); // this is an identifier
  S:= extop(f, 3); // this can be some expression S(some variables)!
                 
  vars:= [args(2..args(0))]; // we are differentiating w.r.t. these vars 

  //----- S.H. Aug 2002: what if the variable we're integrating
  // with (and which should be gone once the integral is done)
  // carries by odd chance the same name as the variable
  // we are differentiating with. At this point we do not know
  // in advance the name of the integrating variable, so staying
  // on the safe side ....

  dummy:= x;
  for var in vars do
      if var = x then
         dummy := genident();
         e := subs(e, x = dummy);
      end_if;
      e:= -dummy*diff(S,var)*e + diff(e, var);
  end_for;

  return(transform::laplace(e, dummy, S)); 
end_proc:

// ==================================================================
// ------------------- S.H. Aug. 2002 ----------------------
transform::laplace::equal:=
proc(o1,o2)
  local f1, x1, t1, f2, x2, t2, dummy, l1, l2;
begin
  // undocumented, user is responsible for his errors
  // this will only work if
  // o1 = sum_i a_i*laplace(f_i, x_i, t)
  // o2 = sum_j b_i*laplace(g_j, y_j, s)
  // then we check
  // whether s = t and
  // sum_i a_i*f_i = sum_j b_j*g_j (making all x_i to x,
  //                                       all y_j to x)
  // we're trying to find laplacians in o1 and o2

  //create common variable for integrals
  l1:= o1;
  l2:= o2;        
  // each l1, l2 should be an unevaluated laplace object
  // l1 = laplace(f1, x1, t1)
  // l2 = laplace(f2, x2, t2)
  if testtype(l1, "transform::laplace") = FALSE or
     testtype(l2, "transform::laplace") = FALSE then 
     return(UNKNOWN);
  end_if;
  
  t1:= extop(l1, 3);
  t2:= extop(l2, 3);
  if t1 <> t2 then 
    return(FALSE); 
  else
    f1:= extop(l1, 1);
    x1:= extop(l1, 2);
    f2:= extop(l2, 1);
    x2:= extop(l2, 2);
    // make common range for the integral
    dummy := genident();
    f1:= subs(f1, x1 = dummy);
    f2:= subs(f2, x2 = dummy);
    if f1 <> f2 then
      return(FALSE);
    end_if;
  end_if;
  return(TRUE);
end_proc:

// ==================================================================
// Compute int(laplace(f(var1), var1, var2), t = a..b).
// complete overhaul by Walter, 20.4.2010
transform::laplace::int:=
proc()
  local F, f, a, b, t, var1, var2, AB, A, B, L;
begin
  // The input is: F = laplace(f(var1), var1, var2).
  // If var2 = t the result is:
  // = int(int(exp(-t*var1)*f(var1), var1 = 0..infinity), t= a..b)
  // = int(int(exp(-t*var1), t = a..b) * f(var1), var1 = 0..infinity)
  // = int((exp(-a*var1)-exp(-b*var1))*f(var1)/var1, var1 = 0..infinity)
  // laplace(f(var1)/var1, var1, a) - laplace(f(var1)/var1, var1, b) 
  F:= args(1);
  f:= extop(F,1);
  var1:= extop(F,2);
  var2:= extop(F,3);
  if has(f, var2) then
     return(hold(int)(F, args(2..args(0))));
  end_if;
  if type(args(2)) <> "_equal" then
     if not has(F, args(2)) then
       return(F*int(1, args(2..args(0))));
     else
       return(hold(int)(F, args(2..args(0))));
     end_if;
  end_if;
  t:= op(args(2),1);
  a:= op(op(args(2),2),1);
  b:= op(op(args(2),2),2);
  if (AB:= transform::TypeLinear(var2, [t])) <> FALSE then
    [A, B]:= AB:
    if not iszero(A) then 
      L:= eval(subsop(F, 1 = f/var1)); // = laplace(f(var1)/var1, var1, var2)
      return(subs(L, t = a, EvalChanges)/A 
            -subs(L, t = b, EvalChanges)/A);
    end_if;
  end_if;
  if not has(F, t) then
    return(F*int(1, args(2..args(0))));
  end_if;
  return(hold(int)(F, args(2..args(0))));
end_proc:

// ==================================================================
transform::laplace::invlaplace:=
proc(f: "transform::laplace", var1: DOM_IDENT, var2)
  local A, B, AB;
begin
  // f = transform::laplace(h(x), x, s)
  if has(extop(f, 1), var1) or
     has(extop(f, 2), var1) then
     return(hold(transform::invlaplace)(f, var1, var2));
  end_if:
  if extop(f, 3) = var1 then
     return(subs(extop(f,1), extop(f,2)=var2, EvalChanges))
  end_if;
  // laplace(f(x),x,A*s+B) = int(exp(-(A*s + B)*x)*f(x),x = 0..infinity)
  // = int(exp(-s*(A*x))*exp(-B*x)*f(x), x = 0..infinity)
  // = int(exp(-s*y)*exp(-B/A*y)*f(y/A)/A, y = 0..infinity) with y=A*x, A>0.
  // = laplace(exp(-B/A*y)*f(y/A), y, s).
  // ==> invlaplace(laplace(f(x),X,A*s+B),s,var2) = exp(-B/A*var2)*f(var2/A).
  if (AB:= transform::TypeLinear(extop(f, 3), [var1])) <> FALSE then
    [A, B]:= AB:
    if not iszero(A) and is(A > 0) = TRUE then 
      // laplace(f(x),x,A*s+B) = int(exp(-(A*s + B)*x)*f(x),x = 0..infinity)
      // = int(exp(-s*(A*x))*exp(-B*x)*f(x), x = 0..infinity)
      // = int(exp(-s*y)*exp(-B/A*y)*f(y/A)/A, y = 0..infinity) with y=A*x, A> 0.
      // = laplace(exp(-B/A*y)*f(y/A), y, A*s).
      // ==> invlaplace(laplace(f(x),X,A*s+B),s,var2) = exp(-B/A*var2)*f(var2/A).
      return(exp(-B/A*var2)/A*subs(extop(f,1), extop(f,2)=var2/A,EvalChanges));
    end_if;
  end_if:
  return(hold(transform::invlaplace)(f, var1, var2));
end_proc:

// ==================================================================
 // S.H.9/2002: computes laplace transform of _fu = (_a*var1)^(_n)*exp(_c/var1)
transform::laplace::compute_x_n_exp_c_over_x:=
   loadproc(transform::laplace::compute_x_n_exp_c_over_x,
            pathname("TRANS","LAPLACE"), "L_specials"):

 // S.H.10/2002: computes laplace transform of _fu = (_a*var1)^(_n)*sin(_c*var1)
transform::laplace::compute_x_n_sin_x:=
   loadproc(transform::laplace::compute_x_n_sin_x,
            pathname("TRANS","LAPLACE"), "L_specials"):

 // W.O.12/2008: computes laplace transform of _fu = (_a*var1)^(_n)*cos(_c*var1)
transform::laplace::compute_x_n_cos_x:=
   loadproc(transform::laplace::compute_x_n_cos_x,
            pathname("TRANS","LAPLACE"), "L_specials"):

// Necessary for the subs command after the lookup mechnism, because
// otherwise no further evaluation is done for substituted elements in a
// transform::laplace element
// laplace::subs:= e -> new(dom, subs(extop(e), args(2..args(0))));

//-------------------------------------
// utility for
// laplace(f(t)*delta(t - t0), t, s) = exp(-s*t0)*subs(f(t), t = t0)
//-------------------------------------
transform::laplace::evalAtPoint:= (f, t, t0) -> subs(f, t = t0, EvalChanges):

//------------------------------------------------
// overloading of evalAt is needed for testeq etc. 
// Ignore substitutions of the second operand!
//------------------------------------------------
transform::laplace::evalAt:= proc(f, subst)
local g, t, tt, s, ss;
begin
   assert(type(f) = "transform::laplace");
   [g, t, s]:= [extop(f)];
   // ignore substitutions of the bound variable t:
   subst:= select(subst, y -> (op(y, 1) <> t) );
   // when the bound variable x is in the rhs's of the
   // substitutions, change the name of the variable:
   if contains(freeIndets(map(subst, op, 2)), t) then
      tt:= genident(expr2text(t));
      f:= subs(f, t = tt);
      g:= subs(g, t = tt);
      t:= tt;
   end_if;
   g:= g | subst;
   ss:= s | subst;
   if extop(f, 1) <> g or s <> ss then
      return(transform::laplace(g, t, ss))
   else
      return(f)
   end_if;
end_proc:
//------------------------------------------------

transform::laplace::freeIndets:= proc(f)
begin
  assert(type(f) = "transform::laplace");
  freeIndets([extop(f, 1), extop(f, 3)], args(2..args(0))) minus {extop(f, 2)};
end_proc:

//------------------------------------------------
autoload(transform::laplace::lookup):
transform::laplace::patternFSA:=
loadproc(transform::laplace::patternFSA,
         pathname("TRANS","LAPLACE"), "load_patterns"):
autoload(transform::laplace::addpattern):

transform::laplace::userpatterns := []:
transform::laplace::userpatternsFSA := FAIL:

transform::frozenLaplace:= () -> procname(args()):

// end of file 
