/*++
Series - test for series expansions

Calls:

Type::Series(T)
Type::Series(T, x)
Type::Series(T, x = x0)
Type::Series(T, x,      dir)
Type::Series(T, x = x0, dir)

T:   one of Puiseux, Laurent, or Taylor
x:   identifier
x0:  arithmetical expression; if omitted, x0 = 0 is assumed
dir: one of Left, Right, Real, Undirected

See below for examples.
++*/

Type::Series :=
proc(T) option noDebug;
begin

    if args(0) < 1 or args(0) > 3 then
       error("wrong number of arguments")
    elif not contains({Puiseux, Laurent, Taylor}, T) then
       error("first argument 'Puiseux', 'Laurent', or 'Taylor' expected")
    end_if;

  new(Type, "Series",
      proc(T, x)
	name Type::Series;
	option noDebug;
      begin
	case T[1]
	  of Puiseux do Type::Puiseux(x, op(T, 2..nops(T))); break;
	  of Laurent do Type::Laurent(x, op(T, 2..nops(T))); break;
	  of Taylor  do Type::Taylor(x, op(T, 2..nops(T))); break;
	end_case
      end_proc,
      [args()], FALSE)
end_proc:

Type::Puiseux := proc(x, varpoint, dir) option noDebug; begin
  if domtype(x) <> Series::Puiseux // or variable occurs in the coefficients
     or extop(x, 1) <> 0 or has(extop(x, 5), Series::Puiseux::indet(x)) then
     FALSE
  elif args(0) = 1 then
     TRUE
  else // args(0) = 2 or 3
     // check for variable and expansion point
     bool(if type(varpoint) = "_equal" then
             if op(varpoint, 2) = infinity then
               extop(x, 6) = (op(varpoint, 1) = complexInfinity)
               and Series::min(extop(x, 7), Left) = Left
             elif op(varpoint, 2) = -infinity then
               extop(x, 6) = (op(varpoint, 1) = complexInfinity)
               and Series::min(extop(x, 7), Right) = Right
             else
               extop(x, 6) = varpoint
             end_if
          else
             extop(x, 6) = (varpoint = 0)
          end_if and
          if args(0) = 3 then
             // check whether the direction is compatible
             Series::min(extop(x, 7), dir) = dir
          else
             TRUE
          end_if)
  end_if
end_proc:

Type::Laurent := proc(x) option noDebug; begin
  if not Type::Puiseux(args()) then
     FALSE
  else
     // BranchOrder = 1 ?
     bool(extop(x, 2) = 1)
  end_if
end_proc:

Type::Taylor := proc(x) option noDebug; begin
  if not Type::Puiseux(args()) then
     FALSE
  else
     // BranchOrder = 1 and valuation >= 0 ?
     bool(extop(x, 2) = 1 and extop(x, 3) >= 0)
  end_if
end_proc:


/* Examples:
 
>> s:=series(sin(sqrt(x)),x);

                         3/2    5/2
                  1/2   x      x         7/2
                 x    - ---- + ---- + O(x   )
                          6     120

>> testtype(s, Type::Series(Puiseux));
 
                TRUE

>> testtype(s, Type::Series());
Error: wrong number of arguments [Type::Series]

>> testtype(s, Type::Series(Fourier));
Error: first argument 'Puiseux', 'Laurent', or 'Taylor' expect\
ed [Type::Series]

>> s:=series(sin(sqrt(-x)),x): testtype(s, Type::Series(Puiseux));
 
                FALSE

// since flag = 1
 
>> s:=series(ln(x)/(1 + x),x): testtype(s, Type::Series(Puiseux));
 
                FALSE

// since ln(x) occurs in the coefficients
 
>> s:=x+x^3*(-1/6)+x^5*1/120+O(x^6): testtype(s, Type::Series(Puiseux));
 
                FALSE

// since domtype(s) = DOM_EXPR

>> s:=series(exp(x), x = infinity): testtype(s, Type::Series(Puiseux));
 
                FALSE

// since domtype(s) = Series::gseries

>> s:=series(exp(x), x = complexInfinity, NoWarning):
   testtype(s, Type::Series(Puiseux));

                FALSE

// although type(s) = "series"

>> s:=series(sin(sqrt(x)),x):
   map([Puiseux, Laurent, Taylor], z -> testtype(s, Type::Series(z)));

                [TRUE, FALSE, FALSE]

>> s:=series(1/sin(x),x):
   map([Puiseux, Laurent, Taylor], z -> testtype(s, Type::Series(z)));

                [TRUE, TRUE, FALSE]

>> s:=series(cos(sqrt(x)), x):
   map([Puiseux, Laurent, Taylor], z -> testtype(s, Type::Series(z)));

                [TRUE, TRUE, TRUE]

>> s:=series(cos(sqrt(-x)), x):
   map([Puiseux, Laurent, Taylor], z -> testtype(s, Type::Series(z)));

                [TRUE, TRUE, TRUE]

>> s:=series(sin(x),x);

                          3     5
                         x     x       7
                     x - -- + --- + O(x )
                          6   120

>> map([Puiseux, Laurent, Taylor], z -> testtype(s, Type::Series(z)));
 
                [TRUE, TRUE, TRUE]
 
>> testtype(s, Type::Series(Taylor,y));

                FALSE

// wrong variable

>> testtype(s, Type::Series(Taylor,x));

                TRUE

>> testtype(s, Type::Series(Taylor,x = 0));

                TRUE

>> testtype(s, Type::Series(Taylor,x = 2));

                FALSE

// wrong expansion point

>> s:=series(sin(sqrt(x-2)),x=2);

                         3/2          5/2
            1/2   (x - 2)      (x - 2)               7/2
     (x - 2)    - ---------- + ---------- + O((x - 2)   )
                       6           120

>> testtype(s, Type::Series(Puiseux, y = 2));

                FALSE

// wrong variable

>> testtype(s, Type::Series(Puiseux, x));

                FALSE

// wrong expansion point

>> testtype(s, Type::Series(Puiseux, x = 3));

                FALSE

// wrong expansion point

>> testtype(s, Type::Series(Puiseux, x = 2));

                TRUE

>> map([Left,Right,Real,Undirected],
       d -> testtype(s, Type::Series(Puiseux, x = 2, d)));

                [TRUE, TRUE, TRUE, TRUE]

>> testtype(s, Type::Series(Puiseux, x = 2, Imaginary));

                FALSE

// invalid direction

>> s:=series(exp(1/z),z=infinity);

            1     1      1      1        1      /  1 \
        1 + - + ---- + ---- + ----- + ------ + O| -- |
            z      2      3       4        5    |  6 |
                2 z    6 z    24 z    120 z     \ z  /

>> testtype(s, Type::Series(Taylor,z));

		FALSE

// wrong expansion point

>> testtype(s,Type::Series(Taylor,z=infinity));

		TRUE

>> testtype(s,Type::Series(Taylor,z=-infinity));

		FALSE

// incompatible direction

>> testtype(s, Type::Series(Taylor,z=complexInfinity));

		TRUE

>> map([Left,Right,Real,Undirected],
       d -> testtype(s, Type::Series(Taylor,z=complexInfinity,d)));

                [TRUE, FALSE, FALSE, FALSE]

>> s:=series(exp(1/z),z=complexInfinity);

            1     1      1      1        1      /  1 \
        1 + - + ---- + ---- + ----- + ------ + O| -- |
            z      2      3       4        5    |  6 |
                2 z    6 z    24 z    120 z     \ z  /

>> testtype(s, Type::Series(Taylor,z));

		FALSE

// wrong expansion point

>> testtype(s,Type::Series(Taylor,z=infinity));

		TRUE

>> testtype(s,Type::Series(Taylor,z=-infinity));

		TRUE

>> testtype(s, Type::Series(Taylor,z=complexInfinity));

		TRUE

>> map([Left,Right,Real,Undirected],
       d -> testtype(s, Type::Series(Taylor,z=complexInfinity,d)));

                [TRUE, TRUE, TRUE, TRUE]

*/
